from . import field_parser

from ..base_bridges import Bridge

from ... import main_register
from ... import parser
from ... import parser_tools as parset
from ... import InvalidModuleImplementation
from ... import SampleBatchData

from ...parser_tools import ArgumentException
from ...environments import Environment
from ...misc import StreamContext, DomainPropertyParserWrapper

import abc
import os


class StopSamplingException(Exception):
    pass


class SamplerBridge(Bridge):
    """
    Superclass for all bridges to define access to sampling techniques for
    the sampler classes.
    """
    arguments = parset.ClassArguments('SamplerBridge', Bridge.arguments,
        ("tmp_dir", True, None, str,
         "Directory to store temporary data during the sampling"),
        ('fields', True, None, str,
         "Single or list of field names from the data to use (if loading only "
         "those fields are loaded, if also storing, only those fields are stored."
         "If the parameter is not defined, all fields are loaded (but their order"
         " will be unknown."),
        ('parse_kwargs', True, None, str,
         "Additional parameters for parsing the sampling fields. Map with field"
         "name as key and a list of maps as values. Parsing a field can consists "
         "of applying multiple parse functions after each other (e.g. first "
         "parsing a list, then each element to a state). The i-th map in the list"
         "belongs to the i-th parse function of the field to which the list "
         "belongs. The map contains keyword: value parameter pairs. It is not"
         " necessary to define for all fields additional keywords, neither is "
         "it necessary that if for a field additional keywords are defined that "
         "the list contains keywords for all parser used. If parsing a field "
         "consists of 5 consecutive parsers and the given list of kwargs has only"
         " 2 entries, then just the first 2 parsers get additional keywords."),
        ('unparse_kwargs', True, None, str,
         "Like 'parse_kwargs', but for the 'unparse' method which transforms a"
         " parsed fields data back into a string."),
        ('provide', True, True,
         parser.convert_bool,
         "Return the sampled data as object"),
        ("forget", True, 0.0, float,
         "Probability to 'forget' to return a sample entry from the data"),
        ('domain', True, None, str),
        ('domain_properties', True, None, main_register.get_register(DomainPropertyParserWrapper),
         "Wrapper for a DomainProperties object. Needed for some StateFormat conversion."),
        ('domain_properties_loader', True, None, None,
         "Cannot be passed by CMD. Provide a function which sets for every "
         "problem the domain properties. Gets as argument file_domain and "
         "file_problem. Shall return a domain properties object "
         "(or None if not available or you do not want it available)"),
        ("makedir", True, False, parser.convert_bool),
        ("environment", True, None, main_register.get_register(Environment)),
        order=["tmp_dir", "fields", "parse_kwargs", "unparse_kwargs",
               "provide", "forget", "domain", "domain_properties",
               "domain_properties_loader",
               "makedir", "environment", "id"])

    loaded_samples = property(lambda self: self._nb_loaded_samples)
    loaded_tasks = property(lambda self: self._nb_loaded_tasks)
    def __init__(self, tmp_dir=None, fields=None,
                 parse_kwargs=None, unparse_kwargs=None,
                 provide=True, forget=0.0,
                 domain=None, domain_properties=None,
                 domain_properties_loader=None, makedir=False,
                 environment=None, id=None):
        Bridge.__init__(self, id)
        self._tmp_dir = tmp_dir
        self._fields = [fields] if isinstance(fields, str) else fields
        self._parse_kwargs = parse_kwargs
        self._unparse_kwargs = unparse_kwargs
        self._provide = provide
        self._forget = forget
        self._domain = domain
        self._domain_properties = domain_properties
        self._domain_properties_loader = domain_properties_loader
        self._makedir = makedir
        self._environment = environment

        # Some Statistic variables
        self._nb_loaded_samples = 0
        self._nb_loaded_tasks = 0


    def _get_tmp_dir(self, dir, path_problem=None):
        dir = (dir if dir is not None else
               (self._tmp_dir if self._tmp_dir is not None else os.path.dirname(path_problem)))

        if not os.path.isdir(dir):
            if self._makedir:
                os.makedirs(dir)
            else:
                raise parset.ArgumentException("The required temporary directory"
                                               "is missing and permission "
                                               "for its creation was not "
                                               "granted: " + str(dir))
        return dir

    """
    def _get_path_sample(self, path_samples, path_problem=None):
        if path_samples is not None:
            return path_samples

        return (self._target_file if self._target_file is not None
                else ((os.path.splitext(path_problem)[0] + ".data")
                      if self._target_dir is None
                      else os.path.join(self._target_dir,
                                        os.path.splitext(
                                            os.path.basename(path_problem))[0]
                                        + ".data")))
    """
    def initialize(self):
        self._initialize()

    @abc.abstractmethod
    def _initialize(self):
        pass

    def get_default_container(self, data_container, path_problem="NA", **kwargs):
        nb_fields = None if self._fields is None else len(self._fields)
        return (data_container if data_container is not None else
                SampleBatchData(
                    nb_fields, None, self._fields, path_problem,
                    fields_from_first_entry=nb_fields is None, **kwargs))

    def sample(self, path_problem, path_dir_tmp=None,
               path_domain=None, data_container=None,
               skip_lines=None, return_counts=False):
        """
        Starts a sampling run.
        :param path_problem: Path to the original problem to sample from
        :param path_dir_tmp: Path to temporary storage directory (required if samples
                         shall be appended to sample_file). If neither given
                         here nor for the bridge construction, then the problem
                         directory will be used.
        :param path_domain: path to domain file. If not given automatically
                            looked for.
        :param append: Write sample_file via appending or overwritting
        :param data_container: Container with and add_entry(entry, type) method
                               in which the sampled entries shall be added.
        :return: A container containing the sampled entries. If data_container
                 was given, then data_container shall be returned.
        """
        first_round = True
        data = None
        for data in self.generator_sample(path_problem, path_dir_tmp,
                                          path_domain, data_container,
                                          skip_lines=skip_lines,
                                          return_counts=return_counts):
            assert first_round
            first_round = False
        if data is None:
            data = self.get_default_container(data_container, path_problem)

        return data

    def generator_sample(self, path_problem, path_dir_tmp=None,
                         path_domain=None, data_container=None,
                         batch_size=None, skip_lines=None, return_counts=False,
                         **kwargs):
        if self._domain_properties_loader is not None:
            self._domain_properties = self._domain_properties_loader(
                path_domain, path_problem)
        path_dir_tmp = self._get_tmp_dir(path_dir_tmp, path_problem)
        path_domain = self._domain if path_domain is None else path_domain

        for c_samples, c_problems, data in self._generator_sample(
                path_problem, path_dir_tmp,
                path_domain, data_container,
                skip_lines=skip_lines,
                count_lines=return_counts,
                batch_size=batch_size, **kwargs):

            self._nb_loaded_samples += c_samples
            self._nb_loaded_tasks += c_problems
            yield (c_samples, c_problems) if return_counts else data

        field_parser.clear_caches()

    @abc.abstractmethod
    def _generator_sample(self, path_problem, path_dir_tmp,
                path_domain, data_container, skip_lines, **kwargs):
        pass

    def finalize(self):
        self._finalize()

    @abc.abstractmethod
    def _finalize(self):
        pass

    @staticmethod
    def parse(tree, item_cache):
        obj = parser.try_lookup_obj(tree, item_cache, Bridge, None)
        if obj is not None:
            return obj
        else:
            raise ArgumentException("The definition of the base sampler bridge"
                                    " can only be used for look up of any "
                                    "previously defined condition via "
                                    "'SamplerBridge(id=ID)'")


main_register.append_register(SamplerBridge, "samplerbridge")


class DummySamplerBridge(SamplerBridge):
    arguments = parset.ClassArguments(
        'DummySamplerBridge', SamplerBridge.arguments,
        ('verbose', True, 0, int, "Verbosity level >= 0"),
        order=["verbose",
               "tmp_dir", "fields", "parse_kwargs",
               "unparse_kwargs",
               "provide", "forget", "domain", "domain_properties",
               "domain_properties_loader",
               "makedir", "environment", "id"])

    def __init__(self, verbose=0, fields=None,
                 parse_kwargs=None, unparse_kwargs=None,
                 tmp_dir=None, provide=True, forget=0.0,
                 domain=None, domain_properties=None,
                 domain_properties_loader=None, makedir=False,
                 environment=None, id=None):
        SamplerBridge.__init__(self, tmp_dir, fields, parse_kwargs,
                               unparse_kwargs, provide, forget,
                               domain, domain_properties,
                               domain_properties_loader, makedir, environment, id)
        self._verbose = verbose

    def _get_name_id(self):
        return "DummySamplerBridge(id=%s)" % self.id

    def _initialize(self):
        if self._verbose > 0:
            print("Initialize %s." % self._get_name_id())
        pass

    def _sample(self, path_problem, path_dir_tmp,
                path_domain, data_container):
        if self._verbose > 0:
            print("Sample %s." % self._get_name_id())

        if self._verbose >= 2:
            print("\tProblem: %s" % path_problem)
            print("\tDomain:  %s" % path_domain)

    def _finalize(self):
        if self._verbose > 0:
            print("Finalize %s." % self._get_name_id())

    @staticmethod
    def parse(tree, item_cache):
        return parser.try_whole_obj_parse_process(tree, item_cache,
                                                  DummySamplerBridge)


main_register.append_register(DummySamplerBridge, "dummy_sampler_bridge")
