#ifndef TASK_UTILS_SUCCESSOR_GENERATOR_FACTORY_H
#define TASK_UTILS_SUCCESSOR_GENERATOR_FACTORY_H

#include "../operator_id.h"
#include <memory>
#include <vector>

class TaskProxy;

class FactPair;

namespace successor_generator {
class GeneratorBase;

using GeneratorPtr = std::unique_ptr<GeneratorBase>;

struct OperatorRange {
    int begin;
    int end;

    OperatorRange(int begin, int end)
            : begin(begin), end(end) {
    }

    bool empty() const {
        return begin == end;
    }

    int span() const {
        return end - begin;
    }
};


class OperatorInfo {
    /*
      The attributes are not const because we must support
      assignment/swapping to sort vector<OperatorInfo>.
    */
    OperatorID op;
    std::vector<FactPair> precondition;
public:
    OperatorInfo(OperatorID op, std::vector<FactPair> precondition);

    bool operator<(const OperatorInfo &other) const;
    OperatorID get_op() const;
    // Returns -1 as a past-the-end sentinel.
    int get_var(int depth) const;
    int get_value(int depth) const;
};


enum class GroupOperatorsBy {
    VAR,
    VALUE
};


class OperatorGrouper {
    const std::vector<OperatorInfo> &operator_infos;
    const int depth;
    const GroupOperatorsBy group_by;
    OperatorRange range;

    const OperatorInfo &get_current_op_info() const;

    int get_current_group_key() const;

public:
    explicit OperatorGrouper(
            const std::vector<OperatorInfo> &operator_infos,
            int depth,
            GroupOperatorsBy group_by,
            OperatorRange range);

    bool done() const;

    std::pair<int, OperatorRange> next();
};



class SuccessorGeneratorFactory {
    using ValuesAndGenerators = std::vector<std::pair<int, GeneratorPtr>>;

    const TaskProxy &task_proxy;
    std::vector<OperatorInfo> operator_infos;

    GeneratorPtr construct_fork(std::vector<GeneratorPtr> nodes) const;
    GeneratorPtr construct_leaf(OperatorRange range) const;
    GeneratorPtr construct_switch(
        int switch_var_id, ValuesAndGenerators values_and_generators) const;
    GeneratorPtr construct_recursive(int depth, OperatorRange range) const;
public:
    explicit SuccessorGeneratorFactory(const TaskProxy &task_proxy);
    // Destructor cannot be implicit because OperatorInfo is forward-declared.
    ~SuccessorGeneratorFactory();
    GeneratorPtr create();
};
}

#endif
