#ifndef TASK_UTILS_ASSIGNMENT_COST_GENERATOR_FACTORY_H
#define TASK_UTILS_ASSIGNMENT_COST_GENERATOR_FACTORY_H

#include "successor_generator_factory.h"

#include "sampling.h"

#include <memory>
#include <vector>


class TaskProxy;
class PartialAssignment;

namespace assignment_cost_generator {
    using namespace successor_generator;
    using namespace sampling;


    class AssignmentInfo {
        /*
          The attributes are not const because we must support
          assignment/swapping to sort vector<OperatorInfo>.
        */
        size_t assignment_id;
        std::vector<FactPair> assignments;
        int cost;
    public:
        AssignmentInfo(
                size_t assignment_id,
                const PartialAssignment &assignment,
                int cost);

        bool operator<(const AssignmentInfo &other) const;
        size_t get_id() const;
        int get_cost() const;
        // Returns -1 as a past-the-end sentinel.
        int get_var(int depth) const;
        int get_value(int depth) const;
    };

    enum class GroupAssignmentsBy {
        VAR,
        VALUE
    };


    class AssignmentGrouper {
        const std::vector<AssignmentInfo> &assignment_infos;
        const int depth;
        const GroupAssignmentsBy group_by;
        OperatorRange range;

        const AssignmentInfo &get_current_assignment_info() const;

        int get_current_group_key() const;

    public:
        explicit AssignmentGrouper(
                const std::vector<AssignmentInfo> &assignment_infos,
                int depth,
                GroupAssignmentsBy group_by,
                OperatorRange range);

        bool done() const;

        std::pair<int, OperatorRange> next();
    };



    class BaseAssignmentCostGeneratorFactory {
    protected:
        using ValuesAndGenerators = std::vector<std::pair<int, GeneratorPtr>>;
        const TaskProxy &task_proxy;
        std::vector<AssignmentInfo> assignment_infos;

        GeneratorPtr construct_fork(std::vector<GeneratorPtr> nodes) const;
        GeneratorPtr construct_leaf(OperatorRange range) const;
        GeneratorPtr construct_switch(
                int switch_var_id, ValuesAndGenerators values_and_generators) const;
        GeneratorPtr construct_recursive(int depth, OperatorRange range) const;
    public:
        explicit BaseAssignmentCostGeneratorFactory(
                const TaskProxy &task_proxy);
        // Destructor cannot be implicit because OperatorInfo is forward-declared.
        ~BaseAssignmentCostGeneratorFactory();
        virtual GeneratorPtr create() = 0;
    };

    class AssignmentCostGeneratorFactory : public BaseAssignmentCostGeneratorFactory {
    protected:
        const PartialAssignmentRegistry &registry;
        const utils::HashMap<size_t, int> &id2costs;
    public:
        explicit AssignmentCostGeneratorFactory(
                const TaskProxy &task_proxy,
                const PartialAssignmentRegistry &registry,
                const utils::HashMap<size_t, int> &id2costs);
        // Destructor cannot be implicit because OperatorInfo is forward-declared.
        ~AssignmentCostGeneratorFactory();
        virtual GeneratorPtr create() override;
    };

    class AssignmentUnitCostGeneratorFactory : BaseAssignmentCostGeneratorFactory {
    protected:
        std::vector<PartialAssignment> assignments;
        const int cost;
    public:
        explicit AssignmentUnitCostGeneratorFactory(
                const TaskProxy &task_proxy,
                const std::vector<PartialAssignment> assignments,
                const int cost);
        // Destructor cannot be implicit because OperatorInfo is forward-declared.
        ~AssignmentUnitCostGeneratorFactory();
        virtual GeneratorPtr create() override;
    };

}

#endif
