#include "generator_search.h"

#include "../task_proxy.h"

#include "../algorithms/ordered_set.h"
#include "../task_utils/successor_generator.h"
#include "../task_utils/predecessor_generator.h"

#include <fstream>
#include <iostream>
#include <set>

#include <memory>
#include <numeric>
#include <string>

using namespace std;

namespace generator_search {
GeneratorSearch::GeneratorSearch(const Options &opts)
    : SearchEngine(opts),
      sampling_techniques(prepare_sampling_techniques(
                              opts.get_list<shared_ptr<sampling_technique::SamplingTechnique>>("techniques"))),
      current_technique(sampling_techniques.begin()),
      index_current_problem(0),
      count_total_problems(accumulate(
              sampling_techniques.begin(), sampling_techniques.end(), 0,
              [] (const int c,
                      const shared_ptr<sampling_technique::SamplingTechnique> t) {
                  return c + t->get_count();})),
      write_sas(opts.get<bool>("write_sas")),
      write_statistics(opts.get<bool>("write_statistics")),
      count_sample_frequency(opts.get<bool>("count_sample_frequency")),
      count_fact_frequencies(opts.get<int>("count_fact_frequencies")),
      cache_statistics(opts.get<int>("cache_statistics")),
      count_cache_statistics(0) {
    if (write_statistics && !(count_sample_frequency || count_fact_frequencies > 0)) {
        cerr << "No statistic to write defined!" << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }
    if (!write_statistics && (count_sample_frequency || count_fact_frequencies > 0)) {
        cerr << "I will not calculate some statistics, it you do not care for me to write them down." << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }

    if (count_fact_frequencies > 0) {
        for (int i = 1; i <= count_fact_frequencies; ++i) {
            fact_frequencies.push_back(utils::HashMap<vector<FactPair>, int>());
        }
    }
}


vector<shared_ptr<sampling_technique::SamplingTechnique>>
GeneratorSearch::prepare_sampling_techniques(
    vector<shared_ptr<sampling_technique::SamplingTechnique>> input) const {
    if (input.empty()) {
        input.push_back(make_shared<sampling_technique::TechniqueNull>());
    }
    return input;
}

void GeneratorSearch::initialize() {
    cout << "Initializing Generator Manager...";
    cout << "done." << endl;
}


void GeneratorSearch::write_sas_to_file() {
    string path_problem = plan_manager.get_plan_filename() +
                          to_string(index_current_problem);
    ofstream outfile(path_problem, ios::trunc);
    outfile << sampling_technique::modified_task->get_sas();
    outfile.close();
    cout << "Problem written to: " << path_problem << endl;
}

void GeneratorSearch::write_statistics_to_file() {
    ostringstream oss;
//    if (count_sample_frequency) {
//        oss << "# sample_frequencies: [[that a sample has been seen n times, has happend this often]]" << endl;
//    }
//    if (count_fact_frequencies){
//        oss << "# description of fact frequencies" << endl;
//    }


    oss << "{" << endl << "\"nb_samples\":" << (index_current_problem) << endl;
    if (count_sample_frequency) {
        oss << "," << endl;
        oss << "\"frequency_counts\": [" << endl;
        utils::HashMap<int, int> frequencies;
        for (auto iter = sample_frequency.begin(); iter != sample_frequency.end(); ++iter) {
            frequencies[iter->second]++;
        }
        for (auto iter = frequencies.begin(); iter != frequencies.end(); ++iter) {
            oss << "[" << iter->first << "," << iter->second << "]," << endl;
        }
        oss << "]";
    }

    if (count_fact_frequencies) {
        oss << "," << endl;
        oss << "\"fact_frequencies\" : [" << endl;

        for (size_t idx_fact_freq = 0; idx_fact_freq < fact_frequencies.size();
             idx_fact_freq++) {
            const auto &fact_freq = fact_frequencies[idx_fact_freq];
            oss << "{" << endl;
            for (auto iter = fact_freq.begin(); iter != fact_freq.end();) {
                oss << "\"";
                for (const FactPair & fp : iter->first) {
                    oss << task->get_fact_name(fp) << "@";
                }
                oss << "\":" << iter->second;
                iter++;
                oss << (((iter == fact_freq.end())) ? "": ",") << endl;
            }
            oss << "}" << ((idx_fact_freq < fact_frequencies.size() - 1) ? "," : "") << endl;
        }
        // [[facts, ...], count]
        oss << "]" << endl;
    }
    oss << "}";

    string path_problem = plan_manager.get_plan_filename() + "_statistics.json";
    ofstream outfile(
            path_problem,
            ios::trunc);
    outfile << oss.str();
    outfile.close();

    cout << "Statistics written to: " << path_problem << endl;
}

void GeneratorSearch::update_sample_frequencies(shared_ptr<AbstractTask> new_task) {
    vector<pair<int, int>> goal_facts;
    goal_facts.reserve(new_task->get_num_goals());
    for (size_t i = 0; i < (size_t)new_task->get_num_goals(); ++i) {
        FactPair fp = new_task->get_goal_fact(i);
        goal_facts.emplace_back(fp.var, fp.value);
    }
    pair<vector<int>, vector<pair<int, int>>> init_goal =
            {new_task->get_initial_state_values(), goal_facts};
    sample_frequency[init_goal]++;
}


void GeneratorSearch::update_fact_frequencies(shared_ptr<AbstractTask> new_task) {
    vector<int> values = new_task->get_initial_state_values();
    for (int tuple_size = 1; tuple_size <= count_fact_frequencies; tuple_size++){
        //iterate each x-tuple
        vector<int> selected(tuple_size);
        iota(selected.begin(), selected.end(), 0);

        while (true) {
            //add tuple count
            vector<FactPair> selected_facts;
            selected_facts.reserve(tuple_size);
            for (size_t idx_selected = 0; idx_selected < selected.size(); ++idx_selected) {
                selected_facts.emplace_back(selected[idx_selected], values[selected[idx_selected]]);
            }
            fact_frequencies[tuple_size - 1][selected_facts]++;

            //first index not to reset because of overflow
            int current_index = tuple_size - 1;
            while(current_index >= 0 &&
                    ((current_index == tuple_size - 1 &&
                        selected[current_index] >= (int) values.size() - 1) ||
                    (current_index < tuple_size - 1 &&
                        selected[current_index] + 1 >= selected[current_index + 1]))) {
                current_index--;
            }
            if (current_index < 0) {
                break;
            }
            //update selection
            selected[current_index]++;
            for (int higher_index = current_index + 1; higher_index < tuple_size; higher_index++){
                selected[higher_index] = selected[higher_index - 1] + 1;
            }
        }
    }
}
SearchStatus GeneratorSearch::step() {
    if ((*current_technique)->empty()) {
        current_technique++;
        if (current_technique == sampling_techniques.end()) {
            if (write_statistics) {
                write_statistics_to_file();
            }
            return SOLVED;
        }
    }

    shared_ptr<AbstractTask> new_task = (*current_technique)->next(task);

    if (write_sas) {
        write_sas_to_file();
    }

    if (count_sample_frequency) {
        update_sample_frequencies(new_task);
    }

    if (count_fact_frequencies > 0) {
        update_fact_frequencies(new_task);
    }
    index_current_problem++;
    count_cache_statistics++;
    if (cache_statistics > 0 && count_cache_statistics >= cache_statistics) {
        write_statistics_to_file();
        count_cache_statistics = 0;
    }
    return IN_PROGRESS;
}

void GeneratorSearch::add_generator_options(OptionParser &parser) {
    parser.add_list_option<shared_ptr < sampling_technique::SamplingTechnique >> (
        "techniques",
        "List of sampling technique definitions to use",
        "[]");
    parser.add_option<bool>(
            "write_sas",
            "write the generated task as sas files",
            "true");
    parser.add_option<bool>(
            "write_statistics",
            "writes down some statistics"
            "false"
            );
    parser.add_option<bool>(
            "count_sample_frequency",
            "counts how often every sample is seen.",
            "false"
            );
    parser.add_option<int>(
            "count_fact_frequencies",
            "counts frequencies of fact x-tuples (1 = just facts, 2 = pairs of facts, ...)",
            "0"
            );
    parser.add_option<int>(
            "cache_statistics",
            "every n tasks, the currently gathered statistics are written"
            "to disk. If this is not used, then the statistics are only written "
            "once at the end to disk (if the process stops early, they are "
            "not written to disk.",
            "-1"
            );
}
}
