#include "eager_search.h"

#include "search_common.h"

#include "../evaluation_context.h"
#include "../evaluator.h"
#include "../open_list_factory.h"
#include "../option_parser.h"
#include "../pruning_method.h"

#include "../algorithms/ordered_set.h"
#include "../heuristics/lm_cut_heuristic.h"
#include "../pruning/null_pruning_method.h"
#include "../tasks/modified_init_goals_task.h"
#include "../task_utils/successor_generator.h"
#include "../task_utils/task_properties.h"

#include <cassert>
#include <cstdlib>
#include <memory>
#include <set>
#include <sstream>
#include <iosfwd>

using namespace std;

namespace eager_search {
EagerSearch::EagerSearch(const Options &opts)
    : SearchEngine(opts),
      reopen_closed_nodes(opts.get<bool>("reopen_closed")),
      evaluators(opts.get_list<shared_ptr<Evaluator>>("evals", vector<shared_ptr<Evaluator>>())),
      open_list(opts.get<shared_ptr<OpenListFactory>>("open")->
                create_state_open_list()),
      f_evaluator(opts.get<shared_ptr<Evaluator>>("f_eval", nullptr)),
      preferred_operator_evaluators(opts.get_list<shared_ptr<Evaluator>>("preferred")),
      lazy_evaluator(opts.get<shared_ptr<Evaluator>>("lazy_evaluator", nullptr)),
      pruning_method(opts.get<shared_ptr<PruningMethod>>("pruning")),
      reference_evaluators(opts.get_list<shared_ptr<Evaluator>>("ref_evals", vector<shared_ptr<Evaluator>>())),
      reference_hstar(opts.get<bool>("ref_hstar", false)),
      choices_hstar(opts.get<bool>("choices_hstar", false)),
      batch_expansions(opts.get<int>("batch_expansions", 1)) {
    if (lazy_evaluator && !lazy_evaluator->does_cache_estimates()) {
        cerr << "lazy_evaluator must cache its estimates" << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }
    shared_ptr<Evaluator> eval = opts.get<shared_ptr<Evaluator>>("eval", nullptr);
    if (eval != nullptr) {
        evaluators.push_back(eval);
    }
    assert(!choices_hstar);
    assert(!reference_hstar);
//    assert(reference_evaluators.empty());
    if (batch_expansions <= 0) {
        cerr << "batch_expansions has to be at least 1" << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }
}

void EagerSearch::initialize() {
    cout << "Conducting best first search"
         << (reopen_closed_nodes ? " with" : " without")
         << " reopening closed nodes, (real) bound = " << bound
         << endl;
    assert(open_list);

    set<Evaluator *> evals;
    open_list->get_path_dependent_evaluators(evals);

    /*
      Collect path-dependent evaluators that are used for preferred operators
      (in case they are not also used in the open list).
    */
    for (const shared_ptr<Evaluator> &evaluator : preferred_operator_evaluators) {
        evaluator->get_path_dependent_evaluators(evals);
    }

    /*
      Collect path-dependent evaluators that are used in the f_evaluator.
      They are usually also used in the open list and will hence already be
      included, but we want to be sure.
    */
    if (f_evaluator) {
        f_evaluator->get_path_dependent_evaluators(evals);
    }

    /*
      Collect path-dependent evaluators that are used in the lazy_evaluator
      (in case they are not already included).
    */
    if (lazy_evaluator) {
        lazy_evaluator->get_path_dependent_evaluators(evals);
    }

    if (batch_expansions > 1 && evals.size() > 0) {
        cerr << "Cannot use batch expansions with path dependent evaluators"
             << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }
    path_dependent_evaluators.assign(evals.begin(), evals.end());

    const GlobalState &initial_state = state_registry.get_initial_state();
    for (Evaluator *evaluator : path_dependent_evaluators) {
        evaluator->notify_initial_state(initial_state);
    }

    /*
      Note: we consider the initial state as reached by a preferred
      operator.
    */
    EvaluationContext eval_context(initial_state, 0, true, &statistics);

    statistics.inc_evaluated_states();

    if (open_list->is_dead_end(eval_context)) {
        cout << "Initial state is a dead end." << endl;
    } else {
        if (search_progress.check_progress(eval_context)) {
            print_references(eval_context);
            print_checkpoint_line(0);
        }
        start_f_value_statistics(eval_context);
        SearchNode node = search_space.get_node(initial_state);
        node.open_initial();

        open_list->insert(eval_context, initial_state.get_id());
    }

    print_initial_evaluator_values(eval_context);

    pruning_method->initialize(task);
}

int EagerSearch::calculate_hstar(EvaluationContext &eval_context) {
    streambuf *old_cout = cout.rdbuf();
    stringstream dump;
    cout.rdbuf(dump.rdbuf());

    vector<int> new_init(eval_context.get_state().unpack().get_values());    //task->get_initial_state_values();
    vector<FactPair> new_goals;
    new_goals.reserve(task->get_num_goals());
    for (int i = 0; i < task->get_num_goals(); ++i) {
        new_goals.push_back(task->get_goal_fact(i));
    }
    shared_ptr<AbstractTask> new_task = make_shared<extra_tasks::ModifiedInitGoalsTask>(task, std::move(new_init), std::move(new_goals));

    Options new_eval_opts;
    new_eval_opts.set_unparsed_config("lmcut()");
    new_eval_opts.set<shared_ptr<AbstractTask>>("transform", new_task);
    new_eval_opts.set<string>("register", "None");
    new_eval_opts.set<bool>("cache_estimates", "true");
    shared_ptr<Evaluator> new_evaluator = make_shared<lm_cut_heuristic::LandmarkCutHeuristic>(new_eval_opts);

    Options opts;
    opts.set<shared_ptr<AbstractTask>>("transform", new_task);
    opts.set<int>("cost_type", cost_type);
    opts.set<double>("max_time", max_time);     //bad upper bound
    opts.set<int>("bound", std::numeric_limits<int>::max());
    opts.set<shared_ptr<PruningMethod>>("pruning", make_shared<null_pruning_method::NullPruningMethod>());
    opts.set<shared_ptr<Evaluator>>("eval", new_evaluator);

    auto temp = search_common::create_astar_open_list_factory_and_f_eval(opts);
    opts.set("open", temp.first);
    opts.set("f_eval", temp.second);
    opts.set("reopen_closed", true);
    vector<shared_ptr<Evaluator>> preferred_list;
    opts.set("preferred", preferred_list);

    shared_ptr<eager_search::EagerSearch> engine = make_shared<eager_search::EagerSearch>(opts);
    engine->search();

    cout.rdbuf(old_cout);

    if (engine->solution_found) {
        TaskProxy new_task_proxy(*new_task);
        return calculate_plan_cost(engine->get_plan(), new_task_proxy);
    } else {
        return -1;
    }
}

void EagerSearch::print_references(EvaluationContext &eval_context) {
    for (shared_ptr<Evaluator> &eval : reference_evaluators) {
        EvaluationResult eval_result = eval->compute_result(eval_context);
        cout << "New reference heuristic value for " << eval->get_description()
             << ": " << eval_result.get_evaluator_value() << endl;
    }
    if (reference_hstar) {
        int hstar = calculate_hstar(eval_context);
        cout << "New reference heuristic value for h*: ";
        if (hstar == -1) {
            cout << "Unsolvable";
        } else {
            cout << hstar;
        }
        cout << endl;
    }
}

void EagerSearch::print_plan_references() {

    const Plan &plan = get_plan();
    int plan_cost = calculate_plan_cost(plan, task_proxy);
    GlobalState current_state = state_registry.lookup_state(
        state_registry.get_initial_state().get_id());

    vector<shared_ptr<Evaluator>> all_evaluators;
    all_evaluators.reserve(evaluators.size() + reference_evaluators.size());
    all_evaluators.insert(all_evaluators.end(), evaluators.begin(), evaluators.end());
    all_evaluators.insert(all_evaluators.end(), reference_evaluators.begin(), reference_evaluators.end());

    cout << "Heuristic evolution during plan Heuristic(confidence): remaining plan cost";
    for (shared_ptr<Evaluator> eval : all_evaluators) {
        cout << "\t" << eval->get_description();
    }
    cout << endl;

    unsigned int idx = 0;
    while (true) {
        EvaluationContext eval_context(current_state, nullptr, false, true);
        cout << plan_cost;
        for (shared_ptr<Evaluator> &eval: all_evaluators) {
            cout << "\t";
            EvaluationResult eval_result = eval->compute_result(eval_context);
            cout << eval_result.get_evaluator_value()
            << "(" << eval_result.get_confidence() << ")";
        }
        cout << endl;
        if (idx >= plan.size()) {
            break;
        }
        current_state = state_registry.get_successor_state(
            current_state, task_proxy.get_operators()[plan[idx]]);
        plan_cost -= task_proxy.get_operators()[plan[idx]].get_cost();
        ++idx;
    }

    assert(plan_cost == 0);
    assert(task_properties::is_goal_state(task_proxy, current_state));
}

void EagerSearch::print_checkpoint_line(int g) const {
    cout << "[g=" << g << ", ";
    statistics.print_basic_statistics();
    cout << "]" << endl;
}

void EagerSearch::print_statistics() const {
    statistics.print_detailed_statistics();
    search_space.print_statistics();
    pruning_method->print_statistics();
}

SearchStatus EagerSearch::step1() {
    pair<SearchNode, bool> n = fetch_next_node();
    if (!n.second) {
        cout << "Completely explored state space -- no solution!" << endl;
        return FAILED;
    }
    SearchNode node = n.first;

    GlobalState s = node.get_state();
    if (check_goal_and_set_plan(s)) {
        print_plan_references();
        return SOLVED;
    }

    vector<OperatorID> applicable_ops;
    successor_generator.generate_applicable_ops(s, applicable_ops);

    /*
      TODO: When preferred operators are in use, a preferred operator will be
      considered by the preferred operator queues even when it is pruned.
    */
    pruning_method->prune_operators(s, applicable_ops);

    // This evaluates the expanded state (again) to get preferred ops
    EvaluationContext eval_context(s, node.get_g(), false, &statistics, true);
    ordered_set::OrderedSet<OperatorID> preferred_operators;
    for (const shared_ptr<Evaluator> &preferred_operator_evaluator : preferred_operator_evaluators) {
        collect_preferred_operators(eval_context,
                                    preferred_operator_evaluator.get(),
                                    preferred_operators);
    }

    for (OperatorID op_id : applicable_ops) {

        OperatorProxy op = task_proxy.get_operators()[op_id];
        if ((node.get_real_g() + op.get_cost()) >= bound)
            continue;

        GlobalState succ_state = state_registry.get_successor_state(s, op);
        statistics.inc_generated();
        bool is_preferred = preferred_operators.contains(op_id);

        SearchNode succ_node = search_space.get_node(succ_state);

        for (Evaluator *evaluator : path_dependent_evaluators) {
            evaluator->notify_state_transition(s, op_id, succ_state);
        }

        // Previously encountered dead end. Don't re-evaluate.
        if (succ_node.is_dead_end())
            continue;

        if (succ_node.is_new()) {
            // We have not seen this state before.
            // Evaluate and create a new node.

            // Careful: succ_node.get_g() is not available here yet,
            // hence the stupid computation of succ_g.
            // TODO: Make this less fragile.
            int succ_g = node.get_g() + get_adjusted_cost(op);

            EvaluationContext eval_context(
                succ_state, succ_g, is_preferred, &statistics);
            statistics.inc_evaluated_states();

            if (open_list->is_dead_end(eval_context)) {
                succ_node.mark_as_dead_end();
                statistics.inc_dead_ends();
                continue;
            }
            succ_node.open(node, op, get_adjusted_cost(op));

            open_list->insert(eval_context, succ_state.get_id());
            if (search_progress.check_progress(eval_context)) {
                print_references(eval_context);
                print_checkpoint_line(succ_node.get_g());
                reward_progress();
            }
        } else if (succ_node.get_g() > node.get_g() + get_adjusted_cost(op)) {
            // We found a new cheapest path to an open or closed state.
            if (reopen_closed_nodes) {
                if (succ_node.is_closed()) {
                    /*
                      TODO: It would be nice if we had a way to test
                      that reopening is expected behaviour, i.e., exit
                      with an error when this is something where
                      reopening should not occur (e.g. A* with a
                      consistent heuristic).
                    */
                    statistics.inc_reopened();
                }
                succ_node.reopen(node, op, get_adjusted_cost(op));

                EvaluationContext eval_context(
                    succ_state, succ_node.get_g(), is_preferred, &statistics);

                /*
                  Note: our old code used to retrieve the h value from
                  the search node here. Our new code recomputes it as
                  necessary, thus avoiding the incredible ugliness of
                  the old "set_evaluator_value" approach, which also
                  did not generalize properly to settings with more
                  than one evaluator.

                  Reopening should not happen all that frequently, so
                  the performance impact of this is hopefully not that
                  large. In the medium term, we want the evaluators to
                  remember evaluator values for states themselves if
                  desired by the user, so that such recomputations
                  will just involve a look-up by the Evaluator object
                  rather than a recomputation of the evaluator value
                  from scratch.
                */
                open_list->insert(eval_context, succ_state.get_id());

            } else {
                // If we do not reopen closed nodes, we just update the parent pointers.
                // Note that this could cause an incompatibility between
                // the g-value and the actual path that is traced back.
                succ_node.update_parent(node, op, get_adjusted_cost(op));
            }
            statistics.inc_evaluated_states();
        }
    }

    return IN_PROGRESS;
}

struct SuccessorData {
    const int use_case;
    const size_t parent_index;
    SearchNode succ_node;
    GlobalState succ_state;
    OperatorID op_id;

    SuccessorData(
            int use_case, size_t parentIndex,
            SearchNode &&succ_node,
            GlobalState &&succ_state,
            OperatorID op_id)
      : use_case(use_case),
        parent_index(parentIndex),
        succ_node(move(succ_node)),
        succ_state(move(succ_state)),
        op_id(op_id){}
};
SearchStatus EagerSearch::stepN() {
    assert(path_dependent_evaluators.empty());

    // Fetch nodes to expand
    vector<SearchNode> search_nodes;
    search_nodes.reserve(batch_expansions);
    for (int i = 0; i < batch_expansions; ++i) {
        pair<SearchNode, bool> n = fetch_next_node();
        if (!n.second) {
            break;
        }
        search_nodes.push_back(move(n.first));
    }
    if (search_nodes.empty()) {
        cout << "Completely explored state space -- no solution!" << endl;
        return FAILED;
    }

    // Check if those nodes corresponds to goal states
    vector<GlobalState> global_states;
    global_states.reserve(search_nodes.size());
    for (const auto &node : search_nodes) {
        global_states.push_back(node.get_state());

        if (check_goal_and_set_plan(global_states.back())) {
            print_plan_references();
            return SOLVED;
        }
    }

    // Calculate preferred operators
    vector<ordered_set::OrderedSet<OperatorID>> nodes_preferred_operators(global_states.size());
    if (!preferred_operator_evaluators.empty()) {
        assert(false); //Not tested"
        vector<EvaluationContext> pref_ops_evaluation_contexts;
        for (size_t i = 0; i < global_states.size(); ++i) {
            // This evaluates the expanded state (again) to get preferred ops
            pref_ops_evaluation_contexts.emplace_back(
                global_states[i], search_nodes[i].get_g(), false, &statistics, true);
        }

        for (const shared_ptr<Evaluator> &preferred_operator_evaluator : preferred_operator_evaluators) {

            vector<EvaluationResult> eval_results =
                    preferred_operator_evaluator->compute_results(pref_ops_evaluation_contexts);
            for (size_t i = 0; i < eval_results.size(); ++i) {
                if (!eval_results[i].is_infinite()) {
                    for (OperatorID op_id : eval_results[i].get_preferred_operators()) {
                        nodes_preferred_operators[i].insert(op_id);
                    }
                }
            }
        }
    }

    // All the temporary data to store because we want to evaluate in a batch...
    vector<GlobalState> succ_states;
    vector<SearchNode> succ_nodes;
    vector<OperatorProxy> succ_op;
    vector<EvaluationContext> succ_evaluation_contexts;
    vector<int> parent_nodes;
    vector<int> use_cases;

    // Process expansion till evaluation needed
    for (size_t i = 0; i < global_states.size(); ++i) {
        GlobalState &s = global_states[i];
        SearchNode &node = search_nodes[i];

        vector<OperatorID> applicable_ops;
        successor_generator.generate_applicable_ops(s, applicable_ops);

        /*
          TODO: When preferred operators are in use, a preferred operator will be
          considered by the preferred operator queues even when it is pruned.
        */
        pruning_method->prune_operators(s, applicable_ops);

        for (OperatorID op_id : applicable_ops) {
            succ_op.push_back(task_proxy.get_operators()[op_id]);
            OperatorProxy &op = succ_op.back();
            if ((node.get_real_g() + op.get_cost()) >= bound)
                continue;

            succ_states.push_back(state_registry.get_successor_state(s, op));
            GlobalState &succ_state = succ_states.back();
            statistics.inc_generated();
            bool is_preferred = nodes_preferred_operators[i].contains(op_id);

            succ_nodes.push_back(search_space.get_node(succ_states.back()));
            SearchNode &succ_node = succ_nodes.back();
            /* Path dependent evaluators are not supported
            for (Evaluator *evaluator : path_dependent_evaluators) {
                evaluator->notify_state_transition(s, op_id, succ_state);
            }*/

            // Previously encountered dead end. Don't re-evaluate.
            if (succ_node.is_dead_end()) {
                succ_states.pop_back();
                succ_nodes.pop_back();
                succ_op.pop_back();
                continue;
            }

            if (succ_node.is_new()) {
                succ_node.mark_as_dead_end();//hack to prevent duplicates in batch
                // We have not seen this state before.
                // Evaluate and create a new node.

                // Careful: succ_node.get_g() is not available here yet,
                // hence the stupid computation of succ_g.
                // TODO: Make this less fragile.
                int succ_g = node.get_g() + get_adjusted_cost(op);

                succ_evaluation_contexts.emplace_back(
                        succ_state, succ_g, is_preferred, &statistics);

                parent_nodes.push_back(i);
                use_cases.push_back(0);

                statistics.inc_evaluated_states();

            } else if (succ_node.get_g() > node.get_g() + get_adjusted_cost(op)) {
                // We found a new cheapest path to an open or closed state.
                if (reopen_closed_nodes) {
                    if (succ_node.is_closed()) {
                        /*
                          TODO: It would be nice if we had a way to test
                          that reopening is expected behaviour, i.e., exit
                          with an error when this is something where
                          reopening should not occur (e.g. A* with a
                          consistent heuristic).
                        */
                        statistics.inc_reopened();
                    }
                    succ_node.reopen(node, op, get_adjusted_cost(op));
                    succ_evaluation_contexts.emplace_back(
                            succ_state, succ_node.get_g(), is_preferred, &statistics);
                    parent_nodes.push_back(i);
                    use_cases.push_back(1);

                } else {
                    // If we do not reopen closed nodes, we just update the parent pointers.
                    // Note that this could cause an incompatibility between
                    // the g-value and the actual path that is traced back.
                    succ_node.update_parent(node, op, get_adjusted_cost(op));
                    succ_states.pop_back();
                    succ_nodes.pop_back();
                    succ_op.pop_back();
                }
                statistics.inc_evaluated_states();
            } else {
                succ_states.pop_back();
                succ_nodes.pop_back();
                succ_op.pop_back();
            }
        }
    }

    // Evaluate
    for (shared_ptr<Evaluator> evaluator : evaluators) {
        vector<EvaluationResult> succ_eval_results =
                evaluator->compute_results(succ_evaluation_contexts);
        for (size_t i = 0; i < succ_evaluation_contexts.size(); ++i) {
            succ_evaluation_contexts[i].insert_result(evaluator.get(), succ_eval_results[i]);
        }
    }

    assert(succ_states.size() == succ_nodes.size());
    assert(succ_states.size() == succ_op.size());
    assert(succ_states.size() == succ_evaluation_contexts.size());
    assert(succ_states.size() == parent_nodes.size());
    assert(succ_states.size() == use_cases.size());

    // insert into the open list as appropriated
    for (size_t i = 0; i < succ_states.size(); ++i) {
        int use_case = use_cases[i];
        EvaluationContext &eval_context = succ_evaluation_contexts[i];
        GlobalState &succ_state = succ_states[i];
        if (use_case == 0) {
            SearchNode &succ_node = succ_nodes[i];
            succ_node.mark_as_new();
            OperatorProxy &op = succ_op[i];
            if (open_list->is_dead_end(eval_context)) {
                succ_node.mark_as_dead_end();
                statistics.inc_dead_ends();
                continue;
            }
            succ_node.open(search_nodes[parent_nodes[i]],
                    op, get_adjusted_cost(op));

            open_list->insert(eval_context, succ_state.get_id());
            if (search_progress.check_progress(eval_context)) {
                print_references(eval_context);
                print_checkpoint_line(succ_node.get_g());
                reward_progress();
            }
        } else if (use_case == 1) {
            /*
              Note: our old code used to retrieve the h value from
              the search node here. Our new code recomputes it as
              necessary, thus avoiding the incredible ugliness of
              the old "set_evaluator_value" approach, which also
              did not generalize properly to settings with more
              than one evaluator.

              Reopening should not happen all that frequently, so
              the performance impact of this is hopefully not that
              large. In the medium term, we want the evaluators to
              remember evaluator values for states themselves if
              desired by the user, so that such recomputations
              will just involve a look-up by the Evaluator object
              rather than a recomputation of the evaluator value
              from scratch.
            */
            open_list->insert(eval_context, succ_state.get_id());
        } else {
            cerr << "Invalid case in batch eager greedy search" << endl;
            utils::exit_with(utils::ExitCode::SEARCH_CRITICAL_ERROR);
        }
    }
    return IN_PROGRESS;
}

SearchStatus EagerSearch::step() {
    if (batch_expansions == 1) {
        return step1();
    } else {
        return stepN();
    }
}


pair<SearchNode, bool> EagerSearch::fetch_next_node() {
    /* TODO: The bulk of this code deals with multi-path dependence,
       which is a bit unfortunate since that is a special case that
       makes the common case look more complicated than it would need
       to be. We could refactor this by implementing multi-path
       dependence as a separate search algorithm that wraps the "usual"
       search algorithm and adds the extra processing in the desired
       places. I think this would lead to much cleaner code. */

    while (true) {
        if (open_list->empty()) {
            // HACK! HACK! we do this because SearchNode has no default/copy constructor
            const GlobalState &initial_state = state_registry.get_initial_state();
            SearchNode dummy_node = search_space.get_node(initial_state);
            return make_pair(dummy_node, false);
        }
        StateID id = open_list->remove_min();
        // TODO is there a way we can avoid creating the state here and then
        //      recreate it outside of this function with node.get_state()?
        //      One way would be to store GlobalState objects inside SearchNodes
        //      instead of StateIDs
        GlobalState s = state_registry.lookup_state(id);
        SearchNode node = search_space.get_node(s);

        if (node.is_closed())
            continue;

        if (!lazy_evaluator)
            assert(!node.is_dead_end());

        if (lazy_evaluator) {
            /*
              With lazy evaluators (and only with these) we can have dead nodes
              in the open list.

              For example, consider a state s that is reached twice before it is expanded.
              The first time we insert it into the open list, we compute a finite
              heuristic value. The second time we insert it, the cached value is reused.

              During first expansion, the heuristic value is recomputed and might become
              infinite, for example because the reevaluation uses a stronger heuristic or
              because the heuristic is path-dependent and we have accumulated more
              information in the meantime. Then upon second expansion we have a dead-end
              node which we must ignore.
            */
            if (node.is_dead_end())
                continue;

            if (lazy_evaluator->is_estimate_cached(s)) {
                int old_h = lazy_evaluator->get_cached_estimate(s);
                /*
                  We can pass calculate_preferred=false here
                  since preferred operators are computed when the state is expanded.
                */
                EvaluationContext eval_context(s, node.get_g(), false, &statistics);
                int new_h = eval_context.get_evaluator_value_or_infinity(lazy_evaluator.get());
                if (open_list->is_dead_end(eval_context)) {
                    node.mark_as_dead_end();
                    statistics.inc_dead_ends();
                    continue;
                }
                if (new_h != old_h) {
                    open_list->insert(eval_context, id);
                    continue;
                }
            }
        }

        node.close();
        assert(!node.is_dead_end());
        update_f_value_statistics(node);
        statistics.inc_expanded();
        return make_pair(node, true);
    }
}

void EagerSearch::reward_progress() {
    // Boost the "preferred operator" open lists somewhat whenever
    // one of the heuristics finds a state with a new best h value.
    open_list->boost_preferred();
}

void EagerSearch::dump_search_space() const {
    search_space.dump(task_proxy);
}

void EagerSearch::start_f_value_statistics(EvaluationContext &eval_context) {
    if (f_evaluator) {
        int f_value = eval_context.get_evaluator_value(f_evaluator.get());
        statistics.report_f_value_progress(f_value);
    }
}

/* TODO: HACK! This is very inefficient for simply looking up an h value.
   Also, if h values are not saved it would recompute h for each and every state. */
void EagerSearch::update_f_value_statistics(const SearchNode &node) {
    if (f_evaluator) {
        /*
          TODO: This code doesn't fit the idea of supporting
          an arbitrary f evaluator.
        */
        EvaluationContext eval_context(node.get_state(), node.get_g(), false, &statistics);
        int f_value = eval_context.get_evaluator_value(f_evaluator.get());
        statistics.report_f_value_progress(f_value);
    }
}
}
