#!/usr/bin/env python3

from __future__ import print_function, division


from compare_parameter_changes import compare_parameters
import compare_h_evolutions
from compare_h_evolutions import compare_h_deviations
from compare_expansions import compare_expansions
from compare_base_statistics import compare_base_statistics

import argparse
import json
import numpy as np
import math
import matplotlib.pyplot as plt
import os
import re
import sys

NORMALIZE_COMPARISONS = True
PATTERN_PARAMETER_COMPARISON_CONFIGURATIONS = re.compile(
    r"(reg|cls)_(ns|full)_(bal|ubal)_(h5)_(sigmoid)_(init|inter|plan)_(gen_)?(opt_|sat_)?(drp0|drp25)(_rlt1e-2)?(_M?(Kall|K75|K50))?"
)
H_EVOLUTION_FILTERS = []
for b in ["bal", "ubal"]:
    for m in ["M", ""]:
        for k in ["Kall", "K75", "K50"]:
            H_EVOLUTION_FILTERS.append(
                ("init_inter_plan_%s_%s%s" % (b, m, k), re.compile(
                r"cls_ns_%s_h5_sigmoid_(init|inter|plan)_gen_opt_drp0_%s%s" % (b, m, k))))

for s in ["init","inter", "plan"]:
    H_EVOLUTION_FILTERS.append(
        ("%s_Kall_K75_K50" % s, re.compile(
        r"cls_ns_ubal_h5_sigmoid_%s_gen_opt_drp0_(Kall|K75|K50)" % s)))


H_EVOLUTION_FILTERS_PRUNE = re.compile(r"(cls_ns_ubal_h5_sigmoid_(inter|plan)_gen_opt_drp0_M?Kall(_pruneOff)?|cls_ns_bal_h5_sigmoid_(inter|plan)_gen_opt_drp0_Kall)")
H_EVOLUTION_COLUMNS_PRUNE = 2
H_EVOLUTION_COLUMN_PRUNE = lambda x: 2 if x.find("_inter_") > -1 else 1

H_EVOLUTION_FILTERS_DECREASE_SIZE = re.compile(r"cls_ns_ubal_h5_sigmoid_(inter|plan)_gen_(sat|opt)_drp0_K(all|\d+e\d+)$")
H_EVOLUTION_COLUMNS_DECREASE_SIZE = 4
H_EVOLUTION_COLUMN_DECREASE_SIZE = lambda x: (2 if x.find("_inter_") > -1 else 1) + 2 * (0 if x.find("_sat_") > -1 else 1)


BASELINE_ALGORITHMS_EXPANSIONS = ["EG-ALT-FF-RND", "EG-FF"]
FILTER_ALGORITHMS_EXPANSIONS = re.compile(r"(%s|DUAL_FF_(cls|reg)_ns_ubal_h5_sigmoid_init_gen_(opt|sat)|DUAL_FF_cls_ns_u?bal_h5_sigmoid_inter_gen_opt_drp0_MKall)" % "|".join(BASELINE_ALGORITHMS_EXPANSIONS))
FILTER_ALGORITHMS_EXPANSIONS_PRUNING = re.compile(
    r"(%s|cls_ns_ubal_h5_sigmoid_(inter|plan)_gen_(opt|sat)_drp0_M?Kall)" % "|".join(BASELINE_ALGORITHMS_EXPANSIONS))
COLUMN_ALGORITHMS_EXPANSIONS_PRUNING = lambda x: 1 + (0 if x.find("_inter_") > -1 else 1) + 2 * (1 if x.find("_opt_") > -1 else 0)
COLUMNS_ALGORITHMS_EXPANSIONS_PRUNING = 4


FILTER_ALGORITHMS_EXPANSIONS_DECREASING_SIZE = re.compile(
    r"(%s|cls_ns_ubal_h5_sigmoid_(inter|plan)_gen_(opt|sat)_drp0_K(all|\d+e\d+)$)" % "|".join(BASELINE_ALGORITHMS_EXPANSIONS))
COLUMN_ALGORITHMS_EXPANSIONS_DECREASING_SIZE = lambda x: 1 + (0 if x.find("_inter_") > -1 else 1) + 2 * (1 if x.find("_opt_") > -1 else 0)
COLUMNS_ALGORITHMS_EXPANSIONS_DECREASING_SIZE = 4

BASELINE_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff = ["EG-FF"]
FILTER_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff = re.compile(
    r"(%s|cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff(_cpu\d+)?$)" % "|".join(BASELINE_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff))
COLUMN_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff = lambda x: 1
COLUMNS_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff = 1




FILTERS_BASE_STATS = [
    ("init_inter_plan",
     re.compile(r"%s|(cls_ns_(bal|ubal)_h5_sigmoid_(init|inter|plan)_gen_opt_drp0_M?K(all|75|50))" % "|".join(BASELINE_ALGORITHMS_EXPANSIONS))),
]

ALGORITHM = "algorithm"
DOMAIN = "domain"
PROBLEM = "problem"
COVERAGE = "coverage"
TOTAL_TIME = "total_time"
EXPANSIONS = "expansions"
INITIAL_H_VALUE = "initial_h_value"

PATTERN_NMPUZZLE = re.compile(r"\dx\d")
PATTERN_NOMYSTERY = re.compile(r"f?l\d+_p\d+_c\d+\.\d+")
PATTERN_SOKOBAN = re.compile(r"n\d+_b\d+_w_?\d+")
PATTERN_DEPOT = re.compile(r"depot_p\d+")
PATTERN_GRID = re.compile(r"grid_prob\d+")
PATTERN_NPUZZLE = re.compile(r"npuzzle_n\d+")
PATTERN_BLOCKSWORLD = re.compile(r"probBLOCKS-\d+-\d+")
PATTERN_PIPESWORLD_NOTANKAGE  = re.compile(r"pipes_nt_p\d+")
PATTERN_PIPESWORLD_TANKAGE  = re.compile(r"pipes_t_p\d+")
PATTERN_TRANSPORT_FIX_ROADS = re.compile(r"fc\d+_t\d+_p\d+")
PATTERN_TRANSPORT_FIX_ROADS_FIX_GOALS = re.compile(r"c(\d+x)?\d+_t\d+_p\d+_gs\d+_rs\d+")
PATTERN_TRANSPORT_VAR_ROADS = re.compile(r"c\d+_t\d+_p\d+")
PATTERN_TRANSPORT_IPC = re.compile(r"transport_p\d+")
def get_domain_associated_fixed_universe(fixed_universe_name):
    if fixed_universe_name.startswith("blocks"):
        return ("blocksworld_fix_goals" if fixed_universe_name.find("_gs_") > -1
                else "blocksworld")

    elif fixed_universe_name[1:].startswith("_dhcs"):
        return "freecell"
    elif fixed_universe_name == "bak":
        return None
    else:
        for name, pattern in [
            ("tilepuzzle", PATTERN_NMPUZZLE),
            ("nomystery", PATTERN_NOMYSTERY),
            ("sokoban", PATTERN_SOKOBAN),
            ("transport_fix_roads", PATTERN_TRANSPORT_FIX_ROADS),
            ("transport_fix_roads_fix_goals", PATTERN_TRANSPORT_FIX_ROADS_FIX_GOALS),
            ("transport_var_roads", PATTERN_TRANSPORT_VAR_ROADS),
            ("transport_ipc", PATTERN_TRANSPORT_IPC),
            ("depot", PATTERN_DEPOT),
            ("grid", PATTERN_GRID),
            ("tilepuzzle", PATTERN_NPUZZLE),
            ("pipesworld-notankage", PATTERN_PIPESWORLD_NOTANKAGE),
            ("pipesworld-tankage", PATTERN_PIPESWORLD_TANKAGE),
            ("blocksworld", PATTERN_BLOCKSWORLD)
        ]:

            if pattern.match(fixed_universe_name):
                return name

    assert False, fixed_universe_name


def structure_properties(properties):
    """

    :param properties: {SomeKey: {Entry Properties}}
    :return: {Algorithm: {Domain: {Fixed Universe: {Problem: {Entry Properties}}}}}
    """
    def has_python_import_bug(entry):
        errors = entry.get("unexplained_errors")
        if errors is not None and len(errors) > 1:
            if errors[1].find("EOFError: EOF read where object expected") > -1:
                return True
        return False

    structured = {}
    python_import_bugs = []
    for entry in properties.values():
        if has_python_import_bug(entry):
            python_import_bugs.append(entry)
            continue
        algorithm = entry[ALGORITHM]
        fixed_universe = entry[DOMAIN]
        domain = get_domain_associated_fixed_universe(fixed_universe)
        if domain is None:
            continue
        problem = entry[PROBLEM]

        if algorithm not in structured:
            structured[algorithm] = {}
        if domain not in structured[algorithm]:
            structured[algorithm][domain] = {}
        if fixed_universe not in structured[algorithm][domain]:
            structured[algorithm][domain][fixed_universe] = {}
        structured[algorithm][domain][fixed_universe][problem] = entry
    print("Python import bugs:", len(python_import_bugs))
    for pib in python_import_bugs:
        print("\t", pib[ALGORITHM], pib[DOMAIN], pib[PROBLEM])
    return structured


def get_property_factory(key):
    def _get_property(entry):
        if entry is None:
            return None
        else:
            return entry[key] if key in entry else None
    return _get_property


def process_property_sum(values):
    return sum([0 if x is None else x for x in values])


def process_property_geometric_mean(values):
    values = [x for x in values if x is not None]
    if len(values) == 0:
        return None
    exp = 1.0 / len(values)
    prod = 1
    for value in [val ** exp for val in values]:
        prod *= value
    return prod


def compare_geometric_means_of_property(values1, values2):
    gm1 = process_property_geometric_mean(values1)
    gm2 = process_property_geometric_mean(values2)
    if gm1 is None or gm2 is None:
        return "NA"
    return gm1 - gm2


def compare_sums_of_properties(values1, values2):
    return (sum([x for x in values1 if x is not None]) -
            sum([x for x in values2 if x is not None]))


def compare_sums_of_properties_fractional(values1, values2):
    assert len(values1) == len(values2)
    return 0 if len(values1) == 0 else compare_sums_of_properties(values1, values2)/len(values1)


def make_property_overview(
        path_csv, structured,
        get_property, process_property, default_property_value=0):
    fixed_universes = set()
    for algo in structured:
        for domain in structured[algo]:
            fixed_universes.update(structured[algo][domain].keys())

    domains = {}
    for fixed_universe in fixed_universes:
        domain = get_domain_associated_fixed_universe(fixed_universe)
        if domain not in domains:
            domains[domain] = []
        domains[domain].append(fixed_universe)
    for domain in domains:
        domains[domain] = sorted(domains[domain])
    sorted_domains = sorted(domains.keys())
    sorted_algo = sorted(structured.keys())

    nb_rows = len(structured) + 2
    nb_cols = len(fixed_universes) + 1
    ary = np.ndarray(shape=(nb_rows, nb_cols), dtype=object)
    ary[:, :] = ""
    ary[1, 0] = "Algorithm"
    for no, algo in enumerate(sorted_algo):
        ary[no + 2, 0] = algo

    idx_col = 1
    for domain in sorted_domains:
        ary[0, idx_col] = domain
        for fixed_universe in domains[domain]:
            ary[1, idx_col] = fixed_universe

            for no, algo in enumerate(sorted_algo):
                value = default_property_value
                if domain in structured[algo] and fixed_universe in structured[algo][domain]:
                    problem_values = [
                        get_property(structured[algo][domain][fixed_universe][problem])
                        for problem in structured[algo][domain][fixed_universe]]
                    value = process_property(problem_values)

                ary[no + 2, idx_col] = value
            idx_col += 1

    np.savetxt(path_csv, ary, delimiter=";", fmt="%s")


def make_property_comparison(path_csv, structured, req_covered,
                             get_property, compare_properties):
    """

    :param path_csv:
    :param structured:
    :param req_covered: -2 = both have tried to solve the problem
                        -1 = at least one has tried to solve the problem
                         0 = use all entries,
                         1 = at least one algorithm has to cover the problem
                         2 = both algorithms have to cover the problem

    :return:
    """
    sorted_algo = sorted(structured.keys())
    nb_rows = len(sorted_algo) + 1
    nb_cols = len(sorted_algo) + 1
    ary = np.ndarray(shape=(nb_rows, nb_cols), dtype=object)
    ary[:, :] = ""
    for no, algo in enumerate(sorted_algo):
        ary[no + 1, 0] = algo
        ary[0, no + 1] = algo
        ary[no + 1, no + 1] = "-"

    def get_entry_keys(algo):
        entries = set()
        for domain in structured[algo]:
            for funiverse in structured[algo][domain]:
                for problem, entry in structured[algo][domain][funiverse].items():
                    if (req_covered <= 0 or
                            (req_covered > 0 and entry[COVERAGE] == 1)):
                        entries.add((domain, funiverse, problem))
        return entries

    def get_entry_values(algo, keys):
        values = []
        for domain, funiverse, problem in keys:
            entry = None
            if (domain in structured[algo] and
                    funiverse in structured[algo][domain] and
                    problem in structured[algo][domain][funiverse]):
                entry = structured[algo][domain][funiverse][problem]
            values.append(get_property(entry))
        return values


    for no_algo1, algo1 in enumerate(sorted_algo):
        entries_algo1 = get_entry_keys(algo1)
        for no_algo2 in range(no_algo1 + 1, len(sorted_algo)):
            algo2 = sorted_algo[no_algo2]
            entries_algo2 = get_entry_keys(algo2)

            if req_covered == 0 or req_covered == 1 or req_covered == -1:
                entry_keys = entries_algo1 | entries_algo2
            elif req_covered == 2 or req_covered == -2:
                entry_keys = entries_algo1 & entries_algo2
            else:
                assert False, "Argument req_covered: Unknown value %s" % str(req_covered)

            entry_keys = list(entry_keys)
            values_algo1 = get_entry_values(algo1, entry_keys)
            values_algo2 = get_entry_values(algo2, entry_keys)
            ary[no_algo1 + 1, no_algo2 + 1] = compare_properties(values_algo1,
                                                                 values_algo2)
            ary[no_algo2 + 1, no_algo1 + 1] = compare_properties(values_algo2,
                                                                 values_algo1)

    np.savetxt(path_csv, ary, delimiter=";", fmt="%s")


PATTERN_REMOVE_EXTEND = re.compile("(_EXTEND_[^-]+)")
def remove_EXTEND_(properties):
    for old_key in properties.keys():
        new_key = PATTERN_REMOVE_EXTEND.sub("", old_key)
        properties[new_key] = properties.pop(old_key)


def _get_optimal_plan_cost(structured):
    OPTIMAL_ALGORITHM = "A*-LMcut"
    assert OPTIMAL_ALGORITHM in structured
    opt_domains = structured["A*-LMcut"]
    optimal_costs = {}
    for domain, opt_universe in opt_domains.items():
        domain_costs = {}
        for universe, opt_problems in opt_universe.items():
            universe_costs = {}
            for problem, properties in opt_problems.items():
                #assert properties.get("cost") == properties.get("plan_length"), "%s/%s/%s" % (domain, universe, problem)
                # This is because I run it with uniform costs, but FastDownward
                # still shows the costs as if the action would have non uniform
                # cost
                if "plan_length" in properties:
                    universe_costs[problem] = properties.get("plan_length")
            if len(universe_costs) > 0:
                domain_costs[universe] = universe_costs
        if len(domain_costs) > 0:
            optimal_costs[domain] = domain_costs
    return optimal_costs


def _get_deviation_initial_h_value_hstar(structured, hstar):
    deviations = {}
    for algorithm, struct_domain in structured.items():
        d = [abs(properties[INITIAL_H_VALUE] - hstar[domain][universe][problem])
             for domain, struct_universes in struct_domain.items() if domain in hstar
             for universe, struct_problems in struct_universes.items() if universe in hstar[domain]
             for problem, properties in struct_problems.items() if (
                     problem in hstar[domain][universe] and INITIAL_H_VALUE in properties)]
        if len(d) > 0:
            deviations[algorithm] = d
    return deviations

def plot_deviation_initial_h_value(path, deviations):
    sorted_algorithms = sorted(deviations.keys(), key=lambda x: -np.median(deviations[x]))
    sorted_deviations = [deviations[key] for key in sorted_algorithms]
    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)
    ax.set_xlabel("Deviation")
    ax.set_title("Deviation of predicted initial h value to h* value")
    box = plt.boxplot(sorted_deviations, vert=0, labels=sorted_algorithms)

    for no, line in enumerate(box['medians']):
        # get position data for median line
        x, y = line.get_xydata()[1]  # top of median line
        # overlay median value
        ax.text(x, y, '%.1f' % np.median(sorted_deviations[no]),
                horizontalalignment='center', fontsize=5)

    fig.tight_layout()
    fig.savefig(path)
    plt.close(fig)

PATTERN_FOLD = re.compile(r"(.*/)?p(\d+)\.pddl")
def get_fold(problem):
    m = PATTERN_FOLD.match(problem)
    assert m is not None
    idx_problem = int(m.group(2))
    return math.floor((idx_problem - 1) / 20)


def get_initial_h_interval_per_fold(structured):
    intervals = {}  # {Algorithm: [Interval of Fold, ...]}
    for algorithm, struct_domain in structured.items():
        algorithm_intervals = []
        for domain, struct_universes in struct_domain.items():
            for universe, struct_problems in struct_universes.items():
                folds = {}
                for problem, properties in struct_problems.items():
                    if INITIAL_H_VALUE not in properties:
                        continue
                    fold = get_fold(problem)
                    if fold not in folds:
                        folds[fold] = []
                    folds[fold].append(properties[INITIAL_H_VALUE])
                algorithm_intervals.extend(v for v in folds.values())
        if len(algorithm_intervals) > 0:
            intervals[algorithm] = algorithm_intervals
    return intervals

def analyse_initial_h_value_intervals(path, structured):
    h_intervals = get_initial_h_interval_per_fold(structured)
    ary = np.ndarray(shape=(len(h_intervals) + 1, 6), dtype=object)
    ary[0, 0] = "Algorithm"
    ary[0, 1] = "min interval size"
    ary[0, 2] = "max interval size"
    ary[0, 3] = "25%-tile"
    ary[0, 4] = "50%-tile"
    ary[0, 5] = "75%-tile"

    for no, key in enumerate(sorted(h_intervals.keys())):
        data = [max(interval) - min(interval) for interval in h_intervals[key]]
        idx_row = no + 1
        ary[idx_row, 0] = key
        ary[idx_row, 1] = min(data)
        ary[idx_row, 2] = max(data)
        ary[idx_row, 3] = np.percentile(data, 25)
        ary[idx_row, 4] = np.percentile(data, 50)
        ary[idx_row, 5] = np.percentile(data, 75)

    np.savetxt(path, ary, fmt="%s", delimiter=";")

def run_evaluations_legacy(path_dir, structured):
    compare_expansions(
        os.path.join(path_dir,
                     "compare_expansions_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff.pdf"),
        structured,
        FILTER_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff,
        BASELINE_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff,
        COLUMNS_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff,
        COLUMN_ALGORITHMS_EXPANSIONS_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff,
        path_speed=os.path.join(path_dir,
                                "compare_expansions_speed_cls_ns_ubal_h5_sigmoid_inter_gen_sat_drp0_Kall_pruneOff.pdf"),
    )

    compare_expansions(
        os.path.join(path_dir, "compare_expansions_decreasing_size.pdf"),
        structured,
        FILTER_ALGORITHMS_EXPANSIONS_DECREASING_SIZE,
        BASELINE_ALGORITHMS_EXPANSIONS,
        COLUMNS_ALGORITHMS_EXPANSIONS_DECREASING_SIZE,
        COLUMN_ALGORITHMS_EXPANSIONS_DECREASING_SIZE
    )

    compare_expansions(
        os.path.join(path_dir, "compare_expansions_pruning.pdf"),
        structured,
        FILTER_ALGORITHMS_EXPANSIONS_PRUNING,
        BASELINE_ALGORITHMS_EXPANSIONS,
        COLUMNS_ALGORITHMS_EXPANSIONS_PRUNING,
        COLUMN_ALGORITHMS_EXPANSIONS_PRUNING
    )

    # Analyse h evolutions
    data_h_evolution = compare_h_evolutions.convert_to_relative_evolution(
        structured)

    path_h_evolution_prune = os.path.join(
        path_dir, "hstar_deviation_prune.pdf")
    compare_h_evolutions.plot_per_universe(
        path_h_evolution_prune, data_h_evolution, H_EVOLUTION_FILTERS_PRUNE,
        H_EVOLUTION_COLUMNS_PRUNE, H_EVOLUTION_COLUMN_PRUNE,
        skip_median_deviation=True
    )

    path_h_evolution_prune = os.path.join(
        path_dir, "hstar_deviation_decrease.pdf")
    compare_h_evolutions.plot_per_universe(
        path_h_evolution_prune, data_h_evolution,
        H_EVOLUTION_FILTERS_DECREASE_SIZE,
        H_EVOLUTION_COLUMNS_DECREASE_SIZE, H_EVOLUTION_COLUMN_DECREASE_SIZE,
        skip_median_deviation=True
    )

    for filter_name, filter_algorithms in H_EVOLUTION_FILTERS:
        path_h_evolution = os.path.join(
            path_dir, "hstar_deviations_%s.pdf" % filter_name)
        compare_h_evolutions.plot_per_universe(
            path_h_evolution, data_h_evolution, filter_algorithms)

    for filter_name, filter_algorithms in FILTERS_BASE_STATS:
        print(filter_name)
        compare_base_statistics(
            os.path.join(path_dir, "base_stats_%s.pdf" % filter_name),
            structured, filter_algorithms, BASELINE_ALGORITHMS_EXPANSIONS, None
        )

    compare_expansions(
        os.path.join(path_dir, "compare_expansions.pdf"),
        structured,
        FILTER_ALGORITHMS_EXPANSIONS,
        BASELINE_ALGORITHMS_EXPANSIONS
    )

    # Overview of coverages numbers per algorithm
    make_property_overview(
        os.path.join(path_dir, "coverages.csv", ), structured,
        get_property_factory(COVERAGE), process_property_sum
    )

    # Overview of expansions per algorithm
    make_property_overview(
        os.path.join(path_dir, "expansions.csv", ), structured,
        get_property_factory(EXPANSIONS), process_property_geometric_mean
    )

    # Overview of total time per algorithm
    make_property_overview(
        os.path.join(path_dir, "total_time.csv", ), structured,
        get_property_factory(TOTAL_TIME), process_property_geometric_mean
    )

    # Comparison of geometric means between two algorithms total times.
    # Comparison for algorithm A & B uses only problems, both have attempted
    # to solve.
    path_comp_total_time = os.path.join(path_dir, "comparison_total_time.csv")
    make_property_comparison(
        path_comp_total_time, structured, -2,
        get_property_factory(TOTAL_TIME), compare_geometric_means_of_property
    )
    compare_parameters([path_comp_total_time],
                       filters=[PATTERN_PARAMETER_COMPARISON_CONFIGURATIONS],
                       compare_0_1=False,
                       invert_comparison=True)
    compare_parameters([path_comp_total_time],
                       filters=[PATTERN_PARAMETER_COMPARISON_CONFIGURATIONS],
                       compare_0_1=True,
                       invert_comparison=True)

    # Comparison of two algorithms coverages.
    # Comparison for algorithm A & B uses only problems, both have attempted
    # to solve.
    path_comp_coverage = os.path.join(path_dir, "comparison_coverage.csv")
    make_property_comparison(
        path_comp_coverage, structured, -2,
        get_property_factory(COVERAGE),
        compare_sums_of_properties_fractional if NORMALIZE_COMPARISONS else compare_sums_of_properties
    )
    compare_parameters([path_comp_coverage],
                       filters=[PATTERN_PARAMETER_COMPARISON_CONFIGURATIONS],
                       compare_0_1=False,
                       invert_comparison=False)
    compare_parameters([path_comp_coverage],
                       filters=[PATTERN_PARAMETER_COMPARISON_CONFIGURATIONS],
                       compare_0_1=True,
                       invert_comparison=False)

    optimal_costs = _get_optimal_plan_cost(structured)
    deviations = _get_deviation_initial_h_value_hstar(structured, optimal_costs)

    plot_deviation_initial_h_value(
        os.path.join(path_dir, "initial_h_deviation.pdf"), deviations)

    analyse_initial_h_value_intervals(
        os.path.join(path_dir, "initial_h_intervals.csv"), structured)


def run_evaluations_h_evolution(path_dir, structured, base_name):
    base_name = base_name if base_name.find(".") > -1 else (base_name + ".pdf")
    skip_median_deviation = True
    data_h_evolution = compare_h_evolutions.convert_to_relative_evolution(
        structured)

    nb_columns = 1
    column_assigner = lambda x:1
    compare_h_evolutions.plot_per_universe(
        path=os.path.join(path_dir, base_name),
        data=data_h_evolution,
        nb_data_columns=nb_columns,
        data_column_assigner=column_assigner,
        skip_median_deviation=skip_median_deviation
    )




def run(path_dir, filter_eval=[], run_legacy=False, run_h_evolution=False):
    properties = {}  # {UniqueEntryKey: {Entry Properties like coverage}}
    for item in os.listdir(path_dir):
        if item.endswith("-eval") and all(regex.match(item) for regex in filter_eval):
            path_properties = os.path.join(path_dir, item, "properties")
            if not os.path.exists(path_properties):
                print("Warning> Expected to find a property file: %s" % path_properties)
                continue
            with open(path_properties, "r") as f:
                print("Load properties: %s" % path_properties)
                new_properties = json.load(f)
                if path_properties.find("_EXTEND_") > -1:
                    remove_EXTEND_(new_properties)
                properties.update(new_properties)

    # {Algorithm: {Domain: {Fixed Universe: {Problem: {Entry Properties}}}}}
    structured = structure_properties(properties)

    if run_legacy:
        run_evaluations_legacy(path_dir=path_dir, structured=structured)

    if run_h_evolution:
        run_evaluations_h_evolution(path_dir=path_dir, structured=structured,
                                    base_name=run_h_evolution)

    print("done")


def type_exists(arg):
    assert os.path.exists(arg), arg
    return arg

parser = argparse.ArgumentParser()
parser.add_argument(
    "directories", action="append", default=[], type=type_exists,
    help="Each given directory is an independent run")

parser.add_argument(
    "--filter-eval", action="append", default=[], type=re.compile,
    help="Only direct subdirectories which match all regexes are used.")
parser.add_argument(
    "--run-legacy", action="store_true",
    help="Run all the evaluations of the old script (new script runs only "
         "specified evaluations")
parser.add_argument(
    "--run-h-evolution", default=None,
    help="Run the h evolution evaluation. Provide name for the output file")

if __name__ == "__main__":
    options = parser.parse_args(sys.argv[1:])
    for path_dir in options.directories:
        run(path_dir,
            filter_eval=options.filter_eval,
            run_legacy=options.run_legacy,
            run_h_evolution=options.run_h_evolution)
