from collections import defaultdict
import re

COLOR = "color"
MARKER = "marker"
LINE_STYLE = "line_style"

GROUP_ALGORITHMS_BRACKETS = re.compile("^([^(]*)(?:\(([^)]*)\))?$")

COLORS_GROUPED = [
    ["black", "gray", "lightgray"],
    ["red", "firebrick", "lightcoral", "orangered"],
    ["yellow", "gold"],
    ["cocolate", "saddlebrown", "dark orange", "orange", "darkgoldenrod"],
    ["olivedrab", "lawngreen", "lightgreen", "forestgreen", "lime"],
    ["blue", "cyan", "dodgerblue", "teal", "lightblue"],
    ["slateblue", "darkviolet", "magenta", "deeppink"]
]
COLORS = []
for idx_color in range(max(len(x) for x in COLORS_GROUPED)):
    for group in COLORS_GROUPED:
        if len(group) > idx_color:
            COLORS.append(group[idx_color])


class StyleCache(object):
    DEFAULT_COLORS = COLORS
    DEFAULT_MARKERS = ["x", "+", "d", "s", "o", "v", "^"]
    DEFAULT_LINE_STYLES = ['-', "--", "-.", ":"]

    DEFAULT_MAIN_ALGORITHM_ATTRIBUTES = [COLOR, MARKER]
    DEFAULT_SUB_ALGORITHM_ATTRIBUTES = [LINE_STYLE]

    def __init__(self, colors=None, markers=None, line_styles=None,
                 algorithm_groups=None,
                 main_algorithm_attributes=None,
                 sub_algorithm_attributes=None):

        colors = StyleCache.DEFAULT_COLORS if colors is None else colors
        markers = StyleCache.DEFAULT_MARKERS if markers is None else markers
        line_styles = (StyleCache.DEFAULT_LINE_STYLES if line_styles is None
                       else line_styles)
        self._attributes = {
            COLOR: colors,
            MARKER: markers,
            LINE_STYLE: line_styles
        }
        self._algorithm_groups = algorithm_groups

        if main_algorithm_attributes is None:
            main_algorithm_attributes = \
                StyleCache.DEFAULT_MAIN_ALGORITHM_ATTRIBUTES
        self._main_algorithm_attributes = main_algorithm_attributes
        if sub_algorithm_attributes is None:
            sub_algorithm_attributes = \
                StyleCache.DEFAULT_SUB_ALGORITHM_ATTRIBUTES
        self._sub_algorithm_attributes = sub_algorithm_attributes
        assert len(self._main_algorithm_attributes) > 0, \
            self._main_algorithm_attributes
        assert (len(self._sub_algorithm_attributes) > 0 or
                self._algorithm_groups is None), self._sub_algorithm_attributes
        assert len(set(self._main_algorithm_attributes) &
                   set(self._sub_algorithm_attributes)) == 0, (
            self._main_algorithm_attributes, self._sub_algorithm_attributes)

        for aa in [self._main_algorithm_attributes,
                   self._sub_algorithm_attributes]:
            assert all(len(self._attributes[x]) == len(self._attributes[aa[0]])
                       for x in aa)

        self._next_algorithm = 0
        self._next_sub_algorithm = defaultdict(int)
        self._cache = {}

    def get_style(self, algorithm):
        if algorithm in self._cache:
            return self._cache[algorithm]

        if self._algorithm_groups is None:
            algorithm_main, algorithm_sub = algorithm, ""
        else:
            m = self._algorithm_groups.match(algorithm)
            assert m is not None
            algorithm_main, algorithm_sub = m.groups()

        if algorithm_sub == None:
            assert algorithm_main == algorithm
            style = {}
            for a in self._main_algorithm_attributes:
                style[a] = self._attributes[a][self._next_algorithm]
            for a in self._sub_algorithm_attributes:
                style[a] = self._attributes[a][
                    self._next_sub_algorithm[algorithm_main]]
            self._next_algorithm += 1
            self._next_sub_algorithm[algorithm_main] += 1
        else:
            assert algorithm_main in self._cache, (algorithm, algorithm_main, algorithm_sub)
            style = dict(self._cache[algorithm_main])
            for a in self._sub_algorithm_attributes:
                style[a] = self._attributes[a][
                    self._next_sub_algorithm[algorithm_main]]
            self._next_sub_algorithm[algorithm_main] += 1
        self._cache[algorithm] = style
        return style
