#! /usr/bin/env python
# -*- coding: utf-8 -*-
"""
./fast-sample.py PROBLEM -p -tmp tmp -f FD_A_01
./fast-sample.py ../benchmarks/gripper/prob01.pddl -p -tmp tmp -f FD_A_01
./fast-sample.py /home/ferber/repositories/DeePDown/data/FixedWorlds/opt/transport_var_roads/c10_t2_p2/p1.pddl -p -tmp tmp -f FD_A_01 --generator AUTO AUTO -o "gzip(file={FirstProblem:-5}.test.data.gz)"
"""
from __future__ import print_function
import datetime
import itertools
import os
import sys

DEBUG = False  # Set true for debug defaults


# Load dependency module w/o loading the whole package (otherwise,
# changing the dependencies will have no effect anymore
print("python ", sys.version_info)
path_dependency_script = os.path.join(os.path.dirname(__file__), "src/training/dependencies.py")
if sys.version_info >= (3, 5):
    import importlib.util
    spec = importlib.util.spec_from_file_location("src.training.dependencies", path_dependency_script)
    dependencies = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(dependencies)
    sys.modules["src.training.dependencies"] = dependencies
    dependencies.setup()
    dependencies.set_external(False, True)
elif sys.version_info < (3,):
    import imp
    dependencies = imp.load_source("src.training.dependencies", path_dependency_script)
    sys.modules["src.training.dependencies"] = dependencies
    dependencies.setup()
    dependencies.set_external(False, True)
else:
    print("Warning: Dependency preloading not supported by this python version."
          " All dependencies are require.")


from src.training import parser, parser_tools

from src.training.bridges import FastDownwardSamplerBridge
from src.training.samplers import IterableFileSampler
from src.training.samplers import DirectorySampler, GeneratorSampler
from src.training.bridges.sampling_bridges import StateFormat
from src.training.misc import DomainProperties, StreamContext, StreamDefinition

import argparse
import os
import re
import shlex
import subprocess
import sys
import logging
log = logging.getLogger()


# Which state to sample
SELECT_STATE_INIT = "INIT"
SELECT_STATE_INTER = "INTER"
SELECT_STATE_PLAN = "PLAN"
SELECT_STATES = [SELECT_STATE_INIT, SELECT_STATE_INTER, SELECT_STATE_PLAN]

# DEFAULT SEARCH CONFIGURATION
DEFAULT_DEAD_END_DETECTION = "[ff(transform=sampling_transform())]"
if not DEBUG:
    DEFAULT_SAMPLING_ITERATIONS = 20000
    DEFAULT_TRANSFORMATIONS = [
        "none_none(1,evals=%s)" % (DEFAULT_DEAD_END_DETECTION),
        #"uniform_none(%i,mutexes={mutexes},evals=%s)" % (DEFAULT_SAMPLING_ITERATIONS,
        #                                                  DEFAULT_DEAD_END_DETECTION),
        #"iforward_none(100,  mutexes={mutexes}, evals=%s, distribution=uniform_int_dist(5,15))" % (DEFAULT_DEAD_END_DETECTION),
        #"iforward_iforward(1,  mutexes={mutexes}, evals=%s, dist_init=uniform_int_dist(5,15),dist_goal=uniform_int_dist(1,5))" % (DEFAULT_DEAD_END_DETECTION),
    ]
    DEFAULT_TOTAL_TIME = "7200"
else:
    DEFAULT_SAMPLING_ITERATIONS = 5
    DEFAULT_TRANSFORMATIONS = [
        "none_none(1,evals=%s)" % (DEFAULT_DEAD_END_DETECTION),
        #"uniform_none(%i,mutexes={mutexes},evals=%s)" % (DEFAULT_SAMPLING_ITERATIONS,
        #                                                  DEFAULT_DEAD_END_DETECTION),
        #"iforward_none(100,  mutexes={mutexes}, evals=%s, distribution=uniform_int_dist(5,15))" % (DEFAULT_DEAD_END_DETECTION),
        #"iforward_iforward(1,  mutexes={mutexes}, evals=%s, dist_init=uniform_int_dist(5,15),dist_goal=uniform_int_dist(1,5))" % (DEFAULT_DEAD_END_DETECTION),
    ]
    DEFAULT_TOTAL_TIME = "60"

DEFAULT_STR_TRANSFORMATIONS_GENERATOR = "none_none(1,evals=%s)" % (DEFAULT_DEAD_END_DETECTION)
DEFAULT_STR_TRANSFORMATIONS = ",".join(DEFAULT_TRANSFORMATIONS)
DEFAULT_MAX_TIME_PER_SAMPLING = "1800"


# {Name: (str_predefinitions, str_search_arg)}
# Always hadd hff/hlmcut to search for additional statistics
DEFAULT_PLANNERS = {
    "gbfs-ff": (
        "--heuristic hff=ff(transform=sampling_transform()) ",
        "eager_greedy([hff],transform=sampling_transform(),max_time=%s)" %
        DEFAULT_MAX_TIME_PER_SAMPLING
    ),
    "lama-first": (
        ("--heuristic hff=ff(transform=sampling_transform()) " +
         "--heuristic hlmcut=lmcut(transform=sampling_transform()) " +
         "--evaluator hlm=lmcount(lm_factory=lm_rhw(reasonable_orders=true),"
         "transform=sampling_transform(),pref=false)"),
        ("lazy_greedy([hff,hlm],preferred=[hff,hlm],"
         "transform=sampling_transform(),cost_type=one,reopen_closed=false,"
         "max_time=%s)") % DEFAULT_MAX_TIME_PER_SAMPLING
    )
}
for _search_algorithm, _heuristic, _w in itertools.product(
        ["astar", "lazy_wastar"], ["lmcut", "ipdb"], [None, 2, 5]):
    if ((_search_algorithm == "astar" and _w is not None) or
            (_search_algorithm == "lazy_wastar" and _w is None)):
        continue
    _key = "%s%s-%s" % (_search_algorithm, "" if _w is None else _w, _heuristic)
    DEFAULT_PLANNERS[_key] = (
        ("--heuristic hff=ff(transform=sampling_transform()) " +
         "--heuristic hlmcut=lmcut(transform=sampling_transform())"),
        ("%s(%s%s(transform=sampling_transform())%s,"
         "transform=sampling_transform()%s,max_time=%s)") %
        (_search_algorithm, "[" if _search_algorithm == "lazy_wastar" else "",
         _heuristic, "]" if _search_algorithm == "lazy_wastar" else "",
         "" if _w is None else ",w=%s" % _w,
         DEFAULT_MAX_TIME_PER_SAMPLING)
    )

DEFAULT_PREDEFINITIONS = None
DEFAULT_SEARCH = None

def set_defaults(planner, optimal, satisficing, state_select=True,
                 total_time=DEFAULT_TOTAL_TIME, generator=None):
    global DEFAULT_PREDEFINITIONS, DEFAULT_SEARCH
    if planner is None and not optimal and not satisficing:
        optimal = True
    assert sum([planner is not None, optimal, satisficing]) == 1
    if optimal:
        predefinitions, search = DEFAULT_PLANNERS["astar-lmcut"]
    elif satisficing:
        predefinitions, search = DEFAULT_PLANNERS["lama-first"]
    else:
        assert planner in DEFAULT_PLANNERS, planner
        predefinitions, search = DEFAULT_PLANNERS[planner]


    if state_select == SELECT_STATE_INIT:
        store_initial = "true"
        store_intermediate = "false"
    elif state_select == SELECT_STATE_INTER:
        store_initial = "false"
        store_intermediate = "true"
    elif state_select == SELECT_STATE_PLAN:
        store_initial = "false"
        store_intermediate = "false"
    else:
        assert False

    transformations = (DEFAULT_STR_TRANSFORMATIONS
                       if generator is None else
                       DEFAULT_STR_TRANSFORMATIONS_GENERATOR)

    DEFAULT_PREDEFINITIONS = predefinitions
    DEFAULT_SEARCH = ("sampling(%s"
                  ",techniques=[%s]"
                  ",use_predefined_evaluators=[]"
                  ",transform=adapt_costs(ONE),max_time=%s,hash={hash}"
                  ",skip_second_state_field=true"
                  ",store_initial_state=%s"
                  ",store_intermediate_state=%s"
                  ",random_seed={rnd_integer1})"
                  % (search, transformations,
                     total_time,
                     store_initial, store_intermediate))

set_defaults("astar-lmcut", False, False, SELECT_STATE_INIT, DEFAULT_TOTAL_TIME)

# Argparser Configurations
DESCRIPTION = """Fast-Downward Data Sampling

Samples data using Fast-Downward. The script has the following two modes which
can be defined multiple times
\t--sample: Samples states for a given list of problems and options
\t--traverse: Traverses a directory and invokes sampling for the detected problems.

The script starts in the --sample mode and allows the user to define a 
sampling for a list of problems he/she manually provides. This can be repeated
multiple times.
The first --traverse block changes this behaviour. The --traverse block defines
how to traverse through a directory and detects problem files. ALL --sampling
blocks after a --traverse block, but before the next traverse block (if
existing) are called with the list of problems detected by the current traverse
block. A --traverse block without associated --sample block is skipped.

Use -h within a block to show the help menu for the block (this ends the processing
of the block and continues with the next block)

Example:
./SCRIPT PROBLEM1 PROBLEM 2 OPTIONS - sample from PROBLEM1 & PROBLEM2 with options
./SCRIPT --sample P1 P2 OPTIONS - sample from P1 & P2 with options
./SCRIPT P1 P2 OPTION1 --traverse OPTIONS2 --sample OPTION3 --sample OPTION4 
      - first samples from P1 & P2 with OPTION1, then traverse with OPTION2 and
        detects problem files p1, ..., pn. Afterwards, it calls sampling for
        p1, ..., pn with OPTION3 and finally with OPTION4. More --traverse and
        --sample blocks could follow follow"""

psample = argparse.ArgumentParser(description="Sample block arguments:")

psample.add_argument("problem", nargs="*", type=str, action="store",
                     help="Path to a problem to sample from. Multiple problems"
                         "can be given.")

psample.add_argument("-b", "--build", type=str,
                     action="store", default="release64dynamic",
                     help="Name of the build to use")
psample.add_argument("-d", "--domain",
                     type=str, action="store", default=None,
                     help="Path to the domain file used by all problems (if not"
                         "given, then the domain file is automatically searched"
                         "for every problem individually).")
psample.add_argument("-f", "--format", choices=StateFormat.get_format_names(),
                     action="store", default=StateFormat.FullTrue.name,
                     help="State format in the sampled file.")
psample.add_argument("-fd", "--fast-downward", type=str,
                     action="store", default="./fast-downward.py",
                     help="Path to the fast-downward script (Default"
                          "assumes in current directory fast-downward.py).")
psample.add_argument("-o", "--output", type=str,
                     action="append", default=None,
                     help="Define an output stream for the sampling"
                          "(use this option multiple times for multiple). "
                          "The following placeholder regex will be replaces:\n:"
                          "\t{FirstProblem(:(-)?(\d))?}: path of the first given "
                          "problem optionally till the -?(\d) character.\n"
                          "The "
                          " available streams can be checked in "
                          "training.misc.stream_contexts.py. If not given,"
                          " then 'gzip(suffix=.data.gz)' is used as default.")
psample.add_argument("-p", "--prune", action="store_true",
                     help="Prune duplicate entries")
psample.add_argument("-pre", "--predefine", action="store", type=str,
                     default=None,
                     help="Forward as predefinition to Fast Downward.")
psample.add_argument("-s", "--search", type=str,
                     action="store", default=None,
                     help="Search for sampling to start in Fast-Downward (the"
                         "given search has to perform the sampling, this script"
                         " is not wrapping your search in a sampling search).\n"
                          "If you want additionally the heuristic values of "
                          "other heuristics per entry, those heuristics have to "
                          "be defined via a predefinition, see --predefine.\n"
                          "To automatically add the problem hash, define in "
                          "the sampling search 'hash={hash}'.\n"
                          "To use the mutexes stored in 'mutexes.sas' in of the"
                          "same directory as the current problem, use 'mutexes="
                          "{mutexes}'. If the file is missing, the default "
                          "absent value is set.")
mxg_planner = psample.add_mutually_exclusive_group(required=False)
mxg_planner.add_argument("--satisficing", action="store_true",
                         help="Changes planner algorithm to lama-first")
mxg_planner.add_argument("--optimal", action="store_true",
                         help="Changes the defaults to  A* LMcut")
mxg_planner.add_argument("--planner", action="store", default=None,
                         choices=DEFAULT_PLANNERS.keys(),
                         help="Choose one of the following predefined planners:"
                              "%s" % ", ".join(sorted(DEFAULT_PLANNERS.keys())))


psample.add_argument("-tmp", "--temporary-folder", type=str,
                     action="store", default=None,
                     help="Folder to store temporary files. By default the same"
                         " directory is used where the used problem file is"
                         " stored.")
psample.add_argument("--memory-limit", type=str, action="store", default="3700",
                     help="Fast Downward overall memory limit")
psample.add_argument("--state-select", choices=SELECT_STATES, default=SELECT_STATE_INIT,
                     help="Defines which state(s) to sample from the solution. "
                          "{SELECT_STATE_INIT} samples the initial state of the "
                          "problem, {SELECT_STATE_INTER} samples a randomly "
                          "chosen intermediate state, {SELECT_STATE_PLAN} "
                          "samples all states in the solution.".format(**locals()))
psample.add_argument("--generator", nargs=2, type=str, action="store", default=None,
                     help="[GENERATOR SCRIPT|AUTO] [GENERATING ROUNDS|AUTO]."
                          "If set, then no existing problems are taken from the domain, "
                          "but this script is executed which solely outputs the "
                          "PDDL for the problem. Additionally, the default"
                          "sampling techniques are set to [none_none(1)], meaning "
                          "from every problem only the initial state generated "
                          "by the generator are sampled. The generator is"
                          "run [GENERATING ROUNDS] or  AUTO=" +
                          str(DEFAULT_SAMPLING_ITERATIONS) +
                          " times. If given the special "
                          "string \"AUTO\", then the generator script is looked "
                          "for in [DIRECTORY of 1. Problem]/[run_generator.sh] (It is "
                          "still suggested to provide the path to an example "
                          "problem, although that problem will not be used, "
                          "it can be used to interfere default values like domain"
                          "file path). The following special keys are defined and"
                          "can be used for the generator script:"
                          "{ProblemDir}: directory of the first problem."
                          "Remark, if using custom output streams,"
                          "those streams need slight modifications when used "
                          "with a generator.")
psample.add_argument("--ignore-generator-errors", action="store_true",
                     help="Ignore errors of the generator and still tries to "
                          "solve problems with errors (e.g. if the generator "
                          "writes stuff on the STDOUT)")
psample.add_argument("--total-time", type=int, action="store", default=DEFAULT_TOTAL_TIME,
                     help="Total time to do the sampling. Does not work with custom "
                          "'--search' argument, there define it yourself.")


PATTERN_STREAM_FIRST_PROBLEM = re.compile("{FirstProblem(:(-)?(\d+))?}")
PATTERN_STREAM_TEMPORARY_FOLDER = re.compile("{TMPDIR}")
def format_stream_definitions(definition, problems, temporary_folder):
    def modify_first_problem(iter):
        modification = problems[0]
        if iter.group(2) is not None:
            idx = int(iter.group(3))
            idx *= (1 if iter.group(2) is None else -1)
            modification = modification[:idx]
        return modification

    def modify_tmp_dir(_):
        return temporary_folder

    for (pattern, modifier) in [
            (PATTERN_STREAM_FIRST_PROBLEM, modify_first_problem),
            (PATTERN_STREAM_TEMPORARY_FOLDER, modify_tmp_dir)]:

        shift = 0
        for iter in pattern.finditer(definition):
            modification = modifier(iter)
            definition = (definition[:iter.start() + shift] +
                          modification +
                          definition[iter.end() + shift:])
            shift += len(modification) - (iter.end() - iter.start())
    return definition

def parse_sample_args(argv):
    global DEFAULT_SAMPLING_ITERATIONS
    options = psample.parse_args(argv)
    set_defaults(options.planner, options.optimal, options.satisficing,
                 options.state_select,
                 options.total_time,
                 options.generator)

    if options.generator is not None:
        if options.generator[1] == "AUTO":
            options.generator[1] = DEFAULT_SAMPLING_ITERATIONS
        else:
            options.generator[1] = int(options.generator[1])

    options.format = StateFormat.get(options.format)
    if options.predefine is None:
        options.predefine = DEFAULT_PREDEFINITIONS
    if options.search is None:
        options.search = DEFAULT_SEARCH

    if options.output is None or len(options.output) == 0:
        if options.generator is None:
            options.output = ["gzip(suffix=.data.gz)"]
        else:
            options.output = ["gzip(file={FirstProblem:-5}.data.gz)"]
        print("No output manually defined. Using now: %s" %
              ", ".join(options.output))
    assert options.domain is None or os.path.exists(options.domain)

    streams = []
    for definition in options.output:
        streams.append(parser.construct(
            parser_tools.ItemCache(),
            parser_tools.main_register.get_register(StreamDefinition),
            format_stream_definitions(
                definition, options.problem,
                "." if options.temporary_folder is None else
                options.temporary_folder)))

    streams = StreamContext(streams)

    domain_properties = None
    if (options.problem is not None and
        len(options.problem) > 0 and
        all(os.path.dirname(x) == os.path.dirname(options.problem[0])
            for x in options.problem)):

        path_domain_properties = os.path.join(
            os.path.dirname(options.problem[0]),
            "domain_properties.json")

        if os.path.exists(path_domain_properties):
            domain_properties = DomainProperties.sload(path_domain_properties)

    fdb = FastDownwardSamplerBridge(options.search, options.predefine,
                                    streams, None, None, None,
                                    format=options.format,
                                    build=options.build,
                                    memory_limit=options.memory_limit,
                                    tmp_dir=options.temporary_folder,
                                    provide=False, forget=0.0,
                                    domain=options.domain,
                                    domain_properties=domain_properties,
                                    makedir=True,
                                    fd_path=options.fast_downward,
                                    prune=options.prune,
                                    keep_comments=False)#options.generator is None)
    return fdb, options.problem, options

def get_generator_command(generator, problems):
    if generator[0].lower() != "auto":
        return [generator[0].format(ProblemDir=os.path.dirname(problems[0]))]
    else:
        return [os.path.join(os.path.dirname(problems[0]), "run_generator.sh")]


def find_generator_domain(options, problems):
    if options.domain is not None:
        return options.domain

    path_domain = os.path.join(os.path.dirname(problems[0]), "domain.pddl")
    if os.path.exists(path_domain):
        return path_domain

    raise ValueError("Unable to find domain for generator sampling run: %s" %
                     ",".join(problems))


def sample(argv):
    fdb, problems, options = parse_sample_args(argv)
    if len(problems) == 0 and options.generator is None:
        log.warning("No problems defined for sampling and no generator")
    sampler = None
    if options.generator is None:
        sampler = IterableFileSampler(fdb, problems, timeout=options.total_time)
    else:
        sampler = GeneratorSampler(
            fdb,
            get_generator_command(options.generator, problems),
            os.path.join("." if options.temporary_folder is None else options.temporary_folder,
                         "_automatically_generated_problem.pddl"),
            options.generator[1],
            path_domain=find_generator_domain(options, problems),
            ignore_errors=options.ignore_generator_errors,
            timeout=options.total_time)

    sampler.initialize()
    sampler.sample()
    sampler.finalize()


ptraverse = argparse.ArgumentParser(description="""Traverse block arguments:
If using --execute, the argument order for the called script is:
[--execute's value] [--array's value] [--args' value] [problems] [further call arguments]""")

ptraverse.add_argument("root", nargs="+", type=str, action="store",
                     help="Path to root directory of a traversal. Multiple"
                          "roots can be given, the traversals from each root"
                          "are independent.")
ptraverse.add_argument("-a", "--args", type=str,
                     action="store", default=None,
                     help="Single string describing a set of arguments to add"
                          "before the sampling call arguments (this is set even"
                          "before the list of problems and can be used for "
                          "passing arguments to an external script which "
                          "performs the sampling.")
ptraverse.add_argument("--array", action="store_true",
                       help="WORKS ONLY WITH --execute TOGETHER. If --batch is"
                            "set to a value greater 1 (also if not setting "
                            "--batch at all), then --array=0-X(X inclusive) is "
                            "added after --execute")
ptraverse.add_argument("--array-parallel", type=int, action="store", default=None,
                       help="If specified and using \"--array\", the maximum "
                            "number of jobs to run in parallel.")
ptraverse.add_argument("-b", "--batch", type=int,
                     action="store", default=None,
                     help="WORKS ONLY WITH --execute TOGETHER. Submit the"
                          "problems found during traversal in batches to the"
                          "script to execute.")
ptraverse.add_argument("-df", "--directory-filter", type=str,
                     action="append", default=[],
                     help="A subdirectory name has to match the regex otherwise"
                          "it is not traversed. By default no regex matches are"
                          "required. This argument can be given any number of"
                          "time to add additional filters (the directory name has"
                          "to match ALL regexes)")
ptraverse.add_argument("--slurm-dependency", action="store", default=None,
                       help="WORKS ONLY WITH --execute TOGETHER. Adds the given"
                            "ARGS as --dependency=ARGS, like --array.")
ptraverse.add_argument("-e", "--execute", type=str,
                     action="store", default=None,
                     help="Command or path to script to execute for the "
                          "sampling runs. If"
                          "none is given, then the scripts samples in its own"
                          "process, otherwise"
                          "it calls an external script in a subprocess.")
ptraverse.add_argument("-m", "--max-depth", type=int,
                     action="store", default=None,
                     help="Maximum depth from the root which is traversed ("
                          "default has no maximum, 0 means traversing no"
                          "subfolders, only the content of the root)")
ptraverse.add_argument("-p", "--problem-filter", type=str,
                     action="append", default=[],
                     help="A problem file name has to match the regex otherwise"
                          "it is not registered. By default no regex matches are"
                          "required. This argument can be given any number of"
                          "time to add additional filters (the file name has"
                          "to match ALL regexes)")
ptraverse.add_argument("-s", "--selection-depth", type=int,
                     action="store", default=None,
                     help="Minimum depth from the root which has to be traversed"
                          " before problem files are registered (default has "
                          "no minimum)")
ptraverse.add_argument("--skip-exists", type=str, action="store", default=None,
                       help="Skips sampling from a problem 'X.pddl' if already "
                            "a file XY exists with Y is the provided arguments.")
ptraverse.add_argument("--dry", action="store_true",
                     help="Show only the call arguments it should do, but does "
                          "not perform calls (and therefore, samplings)")


def traverse_directories(argv, sample_settings):
    options = ptraverse.parse_args(argv)

    options.args = [] if options.args is None else shlex.split(
        options.args.strip("\"").strip("'").replace(",", " "))
    for idx in range(len(sample_settings)):
        sample_settings[idx] = options.args + sample_settings[idx]
        if options.execute is None:
            sample_settings[idx] = parse_sample_args(sample_settings[idx])[0]
        else:
            sample_settings[idx].insert(0, options.execute)

    skip_exists_exec = None
    if options.skip_exists is not None:
        def skip_exists_exec(x):
            return not os.path.exists(x[:-len(".pddl")] + options.skip_exists)

    ds = DirectorySampler([] if options.execute is not None else sample_settings,
                          options.root,
                          options.directory_filter, options.problem_filter,
                          skip_exists_exec,
                          options.max_depth, options.selection_depth,
                          timeout=None)
    ds.initialize()



    if options.execute is None:
        if options.dry:
            print("Problems: ", ds._iterable)
            for setting in sample_settings:
                print("\tSetting: ", setting)
            return

        ds.sample()
        ds.finalize()
    else:
        for setting in sample_settings:
            local_settings = list(setting)
            start = 0
            step = len(ds._iterable) if options.batch is None else min(options.batch, len(ds._iterable))
            while start < len(ds._iterable):
                end = min(len(ds._iterable), start + step)
                nb_problems = (end-start)
                add_array = options.array and nb_problems > 1
                problem_idx = (
                        len(options.args) + 1 + (1 if add_array else 0) +
                        (2 if options.slurm_dependency is not None else 0))


                submit_setting = list(local_settings)
                if add_array:
                    param_array = "--array=0-%i" % (nb_problems - 1)
                    if options.array_parallel is not None:
                        param_array += "%%%i" % options.array_parallel

                    submit_setting[1:1] = [param_array]
                if options.slurm_dependency is not None:
                    submit_setting[1:1] = [
                        "--dependency=%s" % options.slurm_dependency,
                        "--kill-on-invalid-dep=yes"]
                submit_setting[problem_idx:problem_idx] = ds._iterable[start:end]
                start = end

                if options.dry:
                    print("Setting:\t", " ".join(submit_setting))
                else:
                    subprocess.call(submit_setting)


def nested_copy(obj):
    if isinstance(obj, dict):
        return {nested_copy(key): nested_copy(value) for key, value in obj.items()}
    if hasattr(obj, '__iter__'):
        return type(obj)(nested_copy(item) for item in obj)
    return obj


if __name__ == '__main__':
    if len(sys.argv) < 2 or sys.argv[1] in ["h", "help", "-h", "-help", "--help"]:
        print(DESCRIPTION, "\n\n")
        psample.print_help(sys.stdout)
        print("\n\n")
        ptraverse.print_help(sys.stdout)
    else:
        print("Startup time: %s" % str(datetime.datetime.now()))
        mode = 0 # 0=ind. samp, 1 = traverse block, 2=dep. sample block
        independent_sample_runs = []
        traversing = []
        last_idx = 1

        def process_block(idx):
            if last_idx == idx == 1:
                return
            block = sys.argv[last_idx:idx]

            if mode == 0:
                independent_sample_runs.append(block)
            elif mode == 1:
                traversing.append((block, []))
            elif mode == 2:
                traversing[-1][1].append(block)
            else:
                raise RuntimeError("Internal Error during separation of blocks")


        for idx in range(1, len(sys.argv)):
            next = sys.argv[idx]
            if next not in["--traverse", "--sample"]:
                continue

            process_block(idx)
            if next == "--traverse":
                mode = 1
            elif mode == 1:
                mode = 2

            last_idx = idx + 1
        process_block(len(sys.argv))

        for independent in independent_sample_runs:
            sample(independent)
        for (traverse_options, sample_settings) in traversing:
            for s in sample_settings:
                traverse_directories(traverse_options, sample_settings)
