####################################################
####################################################
# GLOBAL ASSESSMENT OF LIVESTOCK METHANE EMISSIONS #
####################################################
####################################################

####################################################
# Reference:
# Chang, J., Peng, S., Yin, Y., Ciais, P., Havlik, P., Herrero, M. (2021). 
# The key role of production efficiency changes in livestock methane emission mitigation. 
# AGU Advances, 2, e2021AV000391. DOI: https://doi. org/10.1029/2021AV000391
####################################################

##########################################
##########################################
# Part I Read and processing statistics from FAOSTAT and data for defining climate zones
##########################################
##########################################
#install.packages("rstudioapi")
library(rstudioapi)
# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)


##########################################
##########################################
# 1.1 Read and processing statistics from FAOSTAT 
##########################################
##########################################

##########################################
# 1.1.1 Read original file for livestock CH4 emissions, stock, and yield, slaughtered number, production, and GDP per capita
##########################################

# The files were directly downloaded from http://www.fao.org/faostat/en/#data
# Data include values for all countries/areas
# Note: All ' should be deleted/replaced
path_faodata <- paste0(path_data,"Source_FAODATA/")
csv.var <- c("EFCH4","MMCH4","Stock","Yield","Slaughtered","Production","GDP","Population")
for (mm in 1:8) { # variables
  if (mm == 1) {
    # 1.1 Enteric_Fermentation_CH4
    csv.name <- paste0(path_faodata,"Enteric_Fermentation_CH4_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 2) {
    # 1.2 Manure_Management_CH4
    csv.name <- paste0(path_faodata,"Manure_Management_CH4_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 3) {
    # 1.3 Live_Animals_Stocks
    csv.name <- paste0(path_faodata,"Live_Animals_Stocks_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 4) {
    # 1.4 Livestock_Primary_Yield
    csv.name <- paste0(path_faodata,"Livestock_Primary_Yield_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 5) {
    # 1.5 Livestock_Primary_Slaughtered
    csv.name <- paste0(path_faodata,"Livestock_Primary_Slaughtered_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 6) {
    # 1.6 Livestock_Primary_Production_Quantity
    csv.name <- paste0(path_faodata,"Livestock_Primary_Production_Quantity_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 7) {
    # 1.7 GDP
    csv.name <- paste0(path_faodata,"GDP_2010USD_2000-2018_FAOSTAT_data_9-22-2020.csv")
  } else if (mm == 8) {
    # 1.8 Population
    csv.name <- paste0(path_faodata,"Population_2000-2018_FAOSTAT_data_9-22-2020.csv")
  }
  # read csv file
  df.data <- read.table(csv.name,header=TRUE,sep=",",colClasses = "character")
  dim(df.data);lapply(df.data, class)
  # assign data.frame
  assign(paste0(csv.var[mm],".df.data"),df.data)
  # get levels of each column
  c.name <- colnames(df.data)
  for (k in 1:length(c.name)) {
    assign(paste0(csv.var[mm],".",c.name[k],".name"),levels(factor(df.data[,k])))
  }
} # variables

# GDP only has 2017 value
GDP.Year.name <- Population.Year.name

##########################################
# 1.1.2 Re-arrange data of livestock CH4 emissions, stock, and yield, slaughtered number, production, GDP, Popolation, and GDP per capita
##########################################

#### Read country/area name from world map
library(ncdf4)
path_map <- paste0(path_data,"Input_MAP/")
## get 243 country/area name and country id
ncfname <- paste(path_map,"regionname_world_country_0.5degree.nc",sep="")
ncin <- nc_open(ncfname)
regionid <- ncvar_get(ncin,"regionid")
idname <- ncvar_get(ncin,"idname")
nc_close(ncin)

# Read crosswalk table for mapping the countries from FAOSTAT to 243 countries/areas used in the map
# FAO country list is adapted from http://www.fao.org/countryprofiles/iso3list/en/
csv.name <- paste0(path_map,"map_regionid_FAO_List_of_countries_and_codes.csv")
df.crosswalk <- read.table(csv.name,header=TRUE,sep=",",colClasses = "character")
# convert from characters to numbers
df.crosswalk$GLEAM_Region <- as.numeric(df.crosswalk$GLEAM_Region)
df.crosswalk$Developed <- as.numeric(df.crosswalk$Developed)

# Re-arrange data from FAOSTAT to 243 countries/areas
for (mm in 1:8) {
  df.arranged <- eval(parse(text=paste0(csv.var[mm],".df.data")))
  df.arranged$regionid <- 9999
  Area.Code.name <- eval(parse(text=paste0(csv.var[mm],".Area.Code.name")))
  # check each country/area name fit
  for (i in 1:length(idname)) {
    #print(df.crosswalk[i,])
    # country.name
    for (j in 1:length(Area.Code.name)) {
      if (Area.Code.name[j] == df.crosswalk$FAOSTAT[i]) {
        index <- which(df.arranged$Area.Code == Area.Code.name[j])
        df.arranged$regionid[index] <- df.crosswalk$countryID[i]
      }
    }
  }
  
  # Countries/Areas that need combination
  if (csv.var[mm] != "Yield") {
    # 206 Sudan (former) (1961-2011), 277 South Sudan (2012-2017), 276 Sudan (2012-2017)
    index <- which(df.arranged$Area.Code == "206" | df.arranged$Area.Code == "277" | df.arranged$Area.Code == "276")
    df.arranged$regionid[index] <- df.crosswalk$countryID[127]
    # 272 Serbia, 186 Serbia and Montenegro, 273 Montenegro
    index <- which(df.arranged$Area.Code == "272" | df.arranged$Area.Code == "186" | df.arranged$Area.Code == "273")
    df.arranged$regionid[index] <- df.crosswalk$countryID[31]
  } else { # For Yield, we use that from one country
    # 206 Sudan (former) (1961-2011), 277 South Sudan (2012-2017), 276 Sudan (2012-2017)
    index <- which(df.arranged$Area.Code == "206" | df.arranged$Area.Code == "276")
    df.arranged$regionid[index] <- df.crosswalk$countryID[127]
    # 272 Serbia, 186 Serbia and Montenegro, 273 Montenegro
    index <- which(df.arranged$Area.Code == "186" | df.arranged$Area.Code == "272")
    df.arranged$regionid[index] <- df.crosswalk$countryID[31]
  }
  # detect the country/area that can not be allocated to the map of 243 countries/areas
  index <- which(df.arranged$regionid == 9999)
  df.left <- df.arranged[index,]
  FAO_rest_name <- levels(factor(df.left$Area))
  print(csv.var[mm]);print(FAO_rest_name)
  
  # allocate items to the matrix with 243 countries/areas
  # regionid,Item,Year
  mat.temp <- array(0.0,c(243,length(eval(parse(text=paste0(csv.var[mm],".Item.name")))),length(eval(parse(text=paste0(csv.var[mm],".Year.name"))))))
  # loop data points
  for (i in 1:length(df.arranged[,1])) {
    if (i%%1000 == 0) { print(paste(csv.var[mm],i)) }
    # loop country
    for (j_id in 1:243) {
      # if regionid 
      if (df.arranged$regionid[i] == j_id) {
        # loop Item
        for (j_Item in 1:length(eval(parse(text=paste0(csv.var[mm],".Item.name"))))) {
          if (df.arranged$Item[i] == eval(parse(text=paste0(csv.var[mm],".Item.name")))[j_Item]) {
            # loop Year
            for (j_Year in 1:length(eval(parse(text=paste0(csv.var[mm],".Year.name"))))) {
              if (df.arranged$Year[i] == eval(parse(text=paste0(csv.var[mm],".Year.name")))[j_Year]) {
                mat.temp[j_id,j_Item,j_Year] <- mat.temp[j_id,j_Item,j_Year]+as.numeric(df.arranged$Value[i])
              }
            }
          }
        }
      }
    }
  }
  
  assign(paste0(csv.var[mm],".mat"),mat.temp)
}

# calculate GDP per capita (million USD2010 / 1000 persons = 1000 USD2020 per capita)
GDP_per_capita.mat <- GDP.mat / Population.mat

# get year information 
year <- as.numeric(EFCH4.Year.name)

##########################################
# save data from FAOSTAT
##########################################
save.image(file=paste0(path_out,"FAOSTAT_Livestock_GDP.RData"))
# clear cache 
rm(list=ls())
gc()

##########################################
##########################################
# 1.2 Make maps of climate zone for calculating Manure Management CH4 emissions
##########################################
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

library(ncdf4)
path_map <- paste0(path_data,"Input_MAP/")
# Climate zone as defined in Chapter 3 of Volume 5 of 2019Refinement (also Annex 10A.2)
# it requires 
# 1) mean annual temperature
# 2) mean annual precipitation
# 3) elevation (from HWSD dataset)
# 4) PET (from Trabucco, Antonio; Zomer, Robert (2019): Global Aridity Index and Potential Evapotranspiration (ET0) Climate Database v2. figshare. Fileset. https://doi.org/10.6084/m9.figshare.7504448.v3)
# for the period of 1985-2015

##########################################
# 1.2.1 Read mean annual temperature and precipitation, mean elevation, and aridity index (P/PET)
##########################################

# 1.2.1.1 Read mean annual temperature and precipitation from CRU JRA v2.0 dataset
# 0.5 * 0.5 degree resolution
ncfname <- paste0(path_map,"MAT_MAP_CRUJRAv2.nc4")
ncin <- nc_open(ncfname)
MAT_1985_2015 <- ncvar_get(ncin,"MAT_1985_2015")
MAP_1985_2015 <- ncvar_get(ncin,"MAP_1985_2015")
MAP_1970_2000 <- ncvar_get(ncin,"MAP_1970_2000")
nc_close(ncin)
# downscale to 5 arc min resolution to match those of other dataset
MAT_8km <- array(NA,c(4320,2160))
MAP_8km <- array(NA,c(4320,2160))
for (mm in 1:720) {
  for (nn in 1:360) {
    MAT_8km[((mm-1)*6+1):(mm*6),((nn-1)*6+1):(nn*6)] <- MAT_1985_2015[mm,nn]
    MAP_8km[((mm-1)*6+1):(mm*6),((nn-1)*6+1):(nn*6)] <- MAP_1985_2015[mm,nn]
  }
}

# 1.2.1.2 Read mean elevation 8km resolution from HWSD database Terrain
# 5 arc min resolution
# Source: http://www.fao.org/soils-portal/soil-survey/soil-maps-and-databases/harmonized-world-soil-database-v12/en/
ascname <- paste0(path_map,"GloElev_5min.asc")
mat <- read.table(ascname,skip=6)
median_elevation <- t(mat)
median_elevation[median_elevation < 0] <- NA

# 1.2.1.3 Read Aridity Index 
# 5 arc min resolution
# aggregated from the Global Aridity Index (Global-Aridity_ET0) derived from 
# Trabucco, Antonio; Zomer, Robert (2019): Global Aridity Index and Potential Evapotranspiration (ET0) Climate Database v2. 
# figshare. Fileset. https://doi.org/10.6084/m9.figshare.7504448.v3; 
# https://cgiarcsi.community/2019/01/24/global-aridity-index-and-potential-evapotranspiration-climate-database-v2/"
ncfname <- paste0(path_map,"Global-Aridity_ET0_8km.nc4")
ncin <- nc_open(ncfname)
AI_data_8km <- ncvar_get(ncin,"AI_data_8km")
nc_close(ncin)
AI_data_8km <- AI_data_8km/1e4 # to get real value

##########################################
# 1.2.2 Create maps of climate zone following method of 2019Refinement
##########################################

# Here follows the order of Table 10.17 of 2019Refinement
name_climate_zone <- c(
  "Cool Temperate Moist", #Cool temperate moist: has > 0 oC mean annual temperature and a ratio of potential evapotranspiration to precipitation >1;   
  "Cool Temperate Dry", #Cool temperate dry: has > 0 oC mean annual temperature and a ratio of potential evapotranspiration to precipitation <1;   
  "Boreal Moist", #Boreal Moist: has < 0 oC mean annual temperature but some monthly temperatures > 10and a ratio of potential evapotranspiration >1;   
  "Boreal Dry", #Boreal dry: has < 0 oC mean annual temperature but some monthly temperatures > 10 and a ratio of potential evapotranspiration to precipitation <1;   
  "Warm Temperate Moist", #Warm temperate moist: has >10 oC mean annual temperature and a ratio of potential evapotranspiration to prescipitation > 1; 
  "Warm Temperate Dry", #Warm temperate dry: has >10 oC mean annual temperature and a ratio of potential evapotranspiration to precipitation < 1;   
  "Tropical Montane", #Tropical Montane: has >18oC mean annual temperature and at an elevation greater than 1000m;    
  "Tropical Wet", #Tropical Wet: has >18oC mean annual temperature and mean annual precipitation >2000mm;   
  "Tropical Moist", #Tropical Moist: has >18oC mean annual temperature and mean annual precipitation >1000mm;   
  "Tropical Dry") #Tropical Dry: has >18oC mean annual temperature and mean annual precipitation < 1000mm; 
#Polar moist: has < 0 oC mean annual temperature but all monthly temperatures < 10 and a ratio of potential evapotranspiration >1;
#Polar dry: has < 0 oC mean annual temperature but all monthly temperatures < 10 and a ratio of potential evapotranspiration to precipitation <1.

map_FAO_climate_zone <- array(NA,c(4320,2160))
#1 Cool temperate moist: has > 0 oC mean annual temperature and a ratio of potential evapotranspiration to precipitation >1;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE & 
                 MAT_8km > 0 & AI_data_8km > 0.65)
map_FAO_climate_zone[index] <- 1
#2 Cool temperate dry: has > 0 oC mean annual temperature and a ratio of potential evapotranspiration to precipitation <1;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE & 
                 MAT_8km > 0 & AI_data_8km <= 0.65)
map_FAO_climate_zone[index] <- 2
#3 Boreal Moist: has < 0 oC mean annual temperature but some monthly temperatures > 10and a ratio of potential evapotranspiration >1;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km <= 0 & AI_data_8km > 0.65)
map_FAO_climate_zone[index] <- 3
#4 Boreal dry: has < 0 oC mean annual temperature but some monthly temperatures > 10 and a ratio of potential evapotranspiration to precipitation <1;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km <= 0 & AI_data_8km <= 0.65)
map_FAO_climate_zone[index] <- 4
#5 Warm temperate moist: has >10 oC mean annual temperature and a ratio of potential evapotranspiration to prescipitation > 1;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km > 10 & AI_data_8km > 0.65)
map_FAO_climate_zone[index] <- 5
#6 Warm temperate dry: has >10 oC mean annual temperature and a ratio of potential evapotranspiration to precipitation < 1;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km > 10 & AI_data_8km <= 0.65)
map_FAO_climate_zone[index] <- 6
#7 Tropical Montane: has >18oC mean annual temperature and at an elevation greater than 1000m;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km > 18 & median_elevation > 1000)
map_FAO_climate_zone[index] <- 7
#8 Tropical Wet: has >18oC mean annual temperature and mean annual precipitation >2000mm;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km > 18 & MAP_8km > 2000 & median_elevation <= 1000)
map_FAO_climate_zone[index] <- 8
#9 Tropical Moist: has >18oC mean annual temperature and mean annual precipitation >1000mm;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km > 18 & MAP_8km > 1000 & MAP_8km <= 2000 & median_elevation <= 1000)
map_FAO_climate_zone[index] <- 9
#10 Tropical Dry: has >18oC mean annual temperature and mean annual precipitation < 1000mm;
index <- which(is.na(MAT_8km) == FALSE & is.na(MAP_8km) == FALSE & is.na(AI_data_8km) == FALSE & is.na(median_elevation) == FALSE &
                 MAT_8km > 18 & MAP_8km <= 1000 & median_elevation <= 1000)
map_FAO_climate_zone[index] <- 10

##########################################
# save data from maps of climate zone for calculating Manure Management CH4 emissions
##########################################
save.image(file=paste0(path_out,"map_FAO_climate_zone.RData"))
# clear cache 
rm(list=ls())
gc()


##########################################
##########################################
# Part II Calculate livestock CH4 emissions using the 2019MT method
##########################################
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

# Load FAOSTAT statistics on livestock
load(file=paste0(path_out,"FAOSTAT_Livestock_GDP.RData"))

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set parameter path
path_param_gleam <- paste0(path_data,"GLEAMv2_PARAM/")
path_param_dress <- paste0(path_data,"Opio2013_PARAM/")
path_param_2019refinement <- paste0(path_data,"2019Refinement_PARAM/")
# Load library
library(ncdf4)

##########################################
##########################################
# 2.1 Read livestock distribution maps for spatially explicit calculation
##########################################
##########################################

##########################################
# 2.1.1 read GLW3 livestock distribution at 5 arc min resolution
##########################################

path_param_GLW3 <- paste0(path_data,"GLW3/")
# livestock categories that GLW3 provide livestock distribution for the year 2010
live_cate <- c("cattle","sheep","goats","buffaloes",
               "pigs","chickens","ducks","horses")
live_abb <- c("Ct","Sh","Gt","Bf","Pg","Ch","Dk","Ho")
for (m in 1:length(live_cate)) {
  ncfname <- paste0(path_param_GLW3,"GLW3_livestock_fraction_",live_cate[m],".nc4")
  ncin <- nc_open(ncfname)
  tmp <- ncvar_get(ncin,"fraction")
  nc_close(ncin)
  assign(paste0("GLW3_fraction_",live_cate[m]),tmp)
}

##########################################
# 2.1.2 read country/area map at 5 arc min resolution
##########################################
path_map <- paste0(path_data,"Input_MAP/")
## get 243 country/area name and country id
ncfname <- paste(path_map,"regionname_world_country_8km.nc",sep="")
ncin <- nc_open(ncfname)
regionid_8km <- ncvar_get(ncin,"regionid")
area_8km <- ncvar_get(ncin,"area")
nc_close(ncin)

##########################################
##########################################
# 2.2 Calculate Metabolisable Energy Requirement by major ruminant livestock categories
##########################################
##########################################

# 1) Now we assume all milking animals are mature (at age at first calving) Table B2 of Opio et al., 2013
# 2) at any month, 1/12 of the annual slaughtered animal numbers was slaughtered
# 3) there is a replacement cow numbers calculated from replacement rate of adult females (i.e., at mean weight of calf weight + 1/2*(adult weight - calf weight) of milking animals)
# 4) for the rest of the animal (stock - milking - replacement), they produce meat and keep growing (i.e., at mean weight of calf weight + 1/2*(adult weight - calf weight) slaughtered animals)
# 5) in a more sophisticated way, assuming the stock is the same across the year and the milking animal keep milking for the whole year (or 300 day per year anyway), we can 
#i) have replacement female using replacement rate of each GLEAM region from Table B2 to B6 of Opio et al., 2013; 
#ii) replacement of female are evenly distributed from month/day 0 (at birth) to age at first calving (L)OR to the weight of slaughter (usually smaller than the adult); i.e., at any day of the year, 1/L of the replacement at age of 0, 1, ... L; L = AFC * (MFSkg - Ckg)/(FMkg - Ckg)
#iii) rest of animals are for meat production, and are also evenly distributed from day 0 (at birth) to age at slaughter L; L = AFC * (MMSkg - Ckg)/(AMkg - Ckg) 
# Here, since we only have a single slaughtered weight from FAOSTAT (MMSkg == mean slaughtered weight) and MFSkg == AFkg (adult milking weight)
# In this case, we actually neglect the different weight at slaughter for cow and beef cattle. For example, each year, there are a certain old cow were slaughtered (as replacement rate) at AFkg. But it should not cause much difference.
# Another issue is that we assume the situation changes each year (changes in slaughtered weight).

##########################################
# 2.2.1 Read Table 2.4 to 2.11 of GLEAMv2.0 for herd parameters
##########################################

# dairy cattle
csvname <- paste0(path_param_gleam,"GLEAMv2_Table2.4.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 2)
param_dairy <- mat[1:14,2:11]
# beef/other cattle
csvname <- paste0(path_param_gleam,"GLEAMv2_Table2.5.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 2)
param_beef <- mat[1:14,2:11]
# gapfill Russia
param_beef[,2] <- param_beef[,4]
# goats
csvname <- paste0(path_param_gleam,"GLEAMv2_Table2.11.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 2)
param_goats <- mat[1:13,2:11]
# sheep
csvname <- paste0(path_param_gleam,"GLEAMv2_Table2.10.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 2)
param_sheep <- mat[1:13,2:11]
# buffaloes
csvname <- paste0(path_param_gleam,"GLEAMv2_Table2.7.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 2)
param_bufalloes <- mat[1:14,2:11]
# gapfill Oceania < W Europe
param_bufalloes[,7] <- param_bufalloes[,3]
# gapfill SSA < NENA
param_bufalloes[,10] <- param_bufalloes[,5]

##########################################
# 2.2.2 Read 243 country information GLEAM region (FAO)
##########################################

region.name <- c("N. America","Russian Fed.","W. Europe","E. Europe","NENA","E & SE Asia","Oceania","South Asia","LAC","SSA")
for (i in 1:10) {
  assign(paste0("r",i), which(df.crosswalk$GLEAM_Region == i))
}
gleam.list <- list(r1,r2,r3,r4,r5,r6,r7,r8,r9,r10)

##########################################
# 2.2.3 Read new dressing percentage from Opio et al., 2013 Table B20 and B21
##########################################

csv.name <- paste0(path_param_dress,"TableB20.csv")
data.mat <- read.table(csv.name,header=TRUE,row.names=1,sep=",",colClasses = "character")
dress.cow <- as.vector(sapply(data.mat[,1],as.numeric))/100.
dress.beef <- as.vector(sapply(data.mat[,2],as.numeric))/100.
dress.dairybuffalo <- as.vector(sapply(data.mat[,3],as.numeric))/100.
dress.otherbuffalo <- as.vector(sapply(data.mat[,4],as.numeric))/100.
csv.name <- paste0(path_param_dress,"TableB21.csv")
data.mat <- read.table(csv.name,header=TRUE,row.names=1,sep=",",colClasses = "character")
dress.goats <- as.vector(sapply(data.mat[,1],as.numeric))/100.
dress.sheep <- as.vector(sapply(data.mat[,2],as.numeric))/100.

##########################################
# 2.2.4 Distribute regional parameters to country
##########################################

# AFkg Adult Female live weight 
# RRF Replacement female
# Ckg Calves at birth live weight
# AFC Age at first calving (years) mature
# AMkg Adult Male live weight
tmp <- array(NA,c(243))
param <- c(
  "AFkg_cow",  
  "RRF_cow",
  "Ckg_cow",
  "AFC_cow",
  "AMkg_beef",
  "AFkg_beef",
  "Ckg_beef",
  "AFC_beef",
  "AFkg_buffaloes",
  "RRF_buffaloes",
  "AMkg_buffaloes",
  "Ckg_buffaloes",
  "AFC_buffaloes",
  "AFkg_goats",
  "RRF_goats",
  "AMkg_goats",
  "Ckg_goats",
  "AFC_goats",
  "AFkg_sheep",
  "RRF_sheep",
  "AMkg_sheep",
  "Ckg_sheep",
  "AFC_sheep"
)
for (z in 1:length(param)) {
  assign(param[z],tmp)
}
for (i in 1:10) {
  index <- which(df.crosswalk$GLEAM_Region == i) 
  
  AFkg_cow[index] <- param_dairy[2,i]
  RRF_cow[index] <- param_dairy[8,i]
  Ckg_cow[index] <- param_dairy[4,i]
  AFC_cow[index] <- param_dairy[14,i]
  AFkg_beef[index] <- param_beef[2,i]
  AMkg_beef[index] <- param_beef[3,i]
  Ckg_beef[index] <- param_beef[4,i]
  AFC_beef[index] <- param_beef[14,i]
  
  AFkg_buffaloes[index] <- param_bufalloes[2,i]
  RRF_buffaloes[index] <- param_bufalloes[8,i]
  AMkg_buffaloes[index] <- param_bufalloes[3,i]
  Ckg_buffaloes[index] <- param_bufalloes[4,i]
  AFC_buffaloes[index] <- param_bufalloes[14,i]
  
  AFkg_goats[index] <-  param_goats[2,i]
  RRF_goats[index] <- param_goats[8,i]
  AMkg_goats[index] <- param_goats[3,i]
  Ckg_goats[index] <- param_goats[4,i]
  AFC_goats[index] <- param_goats[13,i]
  
  AFkg_sheep[index] <- param_sheep[2,i]
  RRF_sheep[index] <- param_sheep[8,i]
  AMkg_sheep[index] <- param_sheep[3,i]
  Ckg_sheep[index] <- param_sheep[4,i]
  AFC_sheep[index] <- param_sheep[13,i]
  
}

##########################################
# 2.2.5 Arrange livestock numbers and yield for cattle, goats, sheep and buffaloes
##########################################

# Cattle
j_Item <- which(Slaughtered.Item.name == "Milk, whole fresh cow")
Cattle_milk_head <- Slaughtered.mat[,j_Item,] # "Milk, whole fresh cow"
Cattle_replace_head <- Cattle_milk_head*rep(RRF_cow,length(year))/100 # row 1 1:243, row 2 1:243 ...
j_Item <- which(Stock.Item.name == "Cattle")
Cattle_residual_head <- Stock.mat[,j_Item,]-Cattle_milk_head-Cattle_replace_head # "Cattle"
j_Item <- which(Yield.Item.name == "Milk, whole fresh cow")
Cattle_milk_yield <- Yield.mat[,j_Item,] # "Milk, whole fresh cow"
j_Item <- which(Yield.Item.name == "Meat, cattle")
Cattle_meat_yield <- Yield.mat[,j_Item,] # "Meat, cattle"
# Goats
j_Item <- which(Slaughtered.Item.name == "Milk, whole fresh goat")
Goats_milk_head <- Slaughtered.mat[,j_Item,] # "Milk, whole fresh goat"
Goats_replace_head <- Goats_milk_head*rep(RRF_goats,length(year))/100
j_Item <- which(Stock.Item.name == "Goats")
Goats_residual_head <- Stock.mat[,j_Item,]-Goats_milk_head-Goats_replace_head # "Goats"
j_Item <- which(Yield.Item.name == "Milk, whole fresh goat" )
Goats_milk_yield <- Yield.mat[,j_Item,] # "Milk, whole fresh goat"
j_Item <- which(Yield.Item.name == "Meat, goat")
Goats_meat_yield <- Yield.mat[,j_Item,] # "Meat, goat"
# Sheep
j_Item <- which(Slaughtered.Item.name == "Milk, whole fresh sheep")
Sheep_milk_head <- Slaughtered.mat[,j_Item,] # "Milk, whole fresh sheep"
Sheep_replace_head <- Sheep_milk_head*rep(RRF_sheep,length(year))/100
j_Item <- which(Stock.Item.name == "Sheep")
Sheep_residual_head <- Stock.mat[,j_Item,]-Sheep_milk_head-Sheep_replace_head # "Sheep"
j_Item <- which(Yield.Item.name == "Milk, whole fresh sheep")
Sheep_milk_yield <- Yield.mat[,j_Item,] # "Milk, whole fresh sheep" 
j_Item <- which(Yield.Item.name == "Meat, sheep")
Sheep_meat_yield <- Yield.mat[,j_Item,] # "Meat, sheep"
# Buffaloes
j_Item <- which(Slaughtered.Item.name == "Milk, whole fresh buffalo")
Buffalo_milk_head <- Slaughtered.mat[,j_Item,] # "Milk, whole fresh buffalo"
Buffalo_replace_head <- Buffalo_milk_head*rep(RRF_buffaloes,length(year))/100
j_Item <- which(Stock.Item.name == "Buffaloes")
Buffalo_residual_head <- Stock.mat[,j_Item,]-Buffalo_milk_head-Buffalo_replace_head # "Buffaloes"
j_Item <- which(Yield.Item.name == "Milk, whole fresh buffalo")
Buffalo_milk_yield <- Yield.mat[,j_Item,] # "Milk, whole fresh buffalo"
j_Item <- which(Yield.Item.name == "Meat, buffalo")
Buffalo_meat_yield <- Yield.mat[,j_Item,] # "Meat, buffalo"

# Will be used for India Herd structure only
j_Item <- which(Slaughtered.Item.name == "Meat, cattle")
Cattle_slaughter_head <- Slaughtered.mat[,j_Item,] # "Meat, cattle"
j_Item <- which(Slaughtered.Item.name == "Meat, buffalo")
Buffalo_slaughter_head <- Slaughtered.mat[,j_Item,] # "Meat, buffalo"
# Wool
j_Item <- which(Production.Item.name == "Wool, greasy")
Sheep_wool_production <- Production.mat[,j_Item,] # "Wool, greasy"

##########################################
# 2.2.6 Gather the statistics for further processing
##########################################

animal.country <- array(NA,c(20,243,length(year)))
animal.country[1,,] <- Cattle_milk_head
animal.country[2,,] <- Cattle_replace_head
animal.country[3,,] <- Cattle_residual_head
animal.country[4,,] <- Cattle_milk_yield
animal.country[5,,] <- Cattle_meat_yield
animal.country[6,,] <- Goats_milk_head
animal.country[7,,] <- Goats_replace_head
animal.country[8,,] <- Goats_residual_head
animal.country[9,,] <- Goats_milk_yield
animal.country[10,,] <- Goats_meat_yield
animal.country[11,,] <- Sheep_milk_head
animal.country[12,,] <- Sheep_replace_head
animal.country[13,,] <- Sheep_residual_head
animal.country[14,,] <- Sheep_milk_yield
animal.country[15,,] <- Sheep_meat_yield
animal.country[16,,] <- Buffalo_milk_head
animal.country[17,,] <- Buffalo_replace_head
animal.country[18,,] <- Buffalo_residual_head
animal.country[19,,] <- Buffalo_milk_yield
animal.country[20,,] <- Buffalo_meat_yield
# reset to 0 for residual
animal.country[which(animal.country <= 0.0)] <- 0.0
animal.country[which(is.na(animal.country) | is.infinite(animal.country))] <- 0.0

##########################################
# 2.2.7 Calculate regional mean milk and meat yield for gapfilling
##########################################

# cattle,goats,sheep,buffalo
meanmilk <- array(NA,c(4,10,length(year)))
meanmeat <- array(NA,c(4,10,length(year)))
for (i in 1:4) {
  for (j in 1:length(year)) {
    tmp.nb.milk <- animal.country[i*5-4,,j]
    # use resdual livestock number for meat
    tmp.nb.meat <- animal.country[i*5-2,,j]
    tmp.milk <- animal.country[i*5-1,,j]
    tmp.meat <- animal.country[i*5,,j]
    for (k in 1:10) {
      aa <- sum(tmp.nb.milk[tmp.nb.milk > 0 & tmp.milk > 0 & df.crosswalk$GLEAM_Region == k])
      if (aa > 0 & is.na(aa) == FALSE) {
        meanmilk[i,k,j] <- sum(tmp.milk[tmp.nb.milk > 0 & tmp.milk > 0 & df.crosswalk$GLEAM_Region == k]*
                                 tmp.nb.milk[tmp.nb.milk > 0 & tmp.milk > 0 & df.crosswalk$GLEAM_Region == k])/
          sum(tmp.nb.milk[tmp.nb.milk > 0 & tmp.milk > 0 & df.crosswalk$GLEAM_Region == k])
      }
      bb <- sum(tmp.nb.meat[tmp.nb.meat > 0 & tmp.meat > 0 & df.crosswalk$GLEAM_Region == k])
      if (bb > 0 & is.na(bb) == FALSE) { 
        meanmeat[i,k,j] <- sum(tmp.meat[tmp.nb.meat > 0 & tmp.meat > 0 & df.crosswalk$GLEAM_Region == k]*
                                 tmp.nb.meat[tmp.nb.meat > 0 & tmp.meat > 0 & df.crosswalk$GLEAM_Region == k])/
          sum(tmp.nb.meat[tmp.nb.meat > 0 & tmp.meat > 0 & df.crosswalk$GLEAM_Region == k])
      }
    }
  }
}
# Gapfilling with regional productivities
# Milk
# region 1 N. America goats and sheep milk empty using region 3 W. Europe
# region 7 Oceania sheep milk empty using region 4 E. Europe
# North America Sheep empty using region 3 W. Europe
meanmilk[3,1,] <- meanmilk[3,3,]
# Russia Sheep sometimes empty using region 4 E. Europe
meanmilk[3,2,] <- meanmilk[3,4,]
# Oceania sheep milk empty using region 4 E. Europe
meanmilk[3,7,] <- meanmilk[3,4,]
# region 1 N. America milk using region 3 W. Europe
# region 2 Russian Fed. milk using region 4 E. Europe
# region 7 Oceania milk using region 3 W. Europe
# region 9,10 milk using region 6 E & SE ASia
meanmilk[4,1,] <- meanmilk[4,3,]
meanmilk[4,2,] <- meanmilk[4,4,]
meanmilk[4,7,] <- meanmilk[4,3,]
meanmilk[4,9,] <- meanmilk[4,6,]
meanmilk[4,10,] <- meanmilk[4,6,]
# Meat
# region 1 N. America meat using region 3 W. Europe
# region 2 Russian Fed. meat using region 4 E. Europe
# region 9,10 meat using region 6 E & SE ASia
meanmeat[4,1,] <- meanmeat[4,3,]
meanmeat[4,2,] <- meanmeat[4,4,]
meanmeat[4,9,] <- meanmeat[4,6,]
meanmeat[4,10,] <- meanmeat[4,6,]

# Gapfilling where with animal but not yield using regional mean value
# cattle,goats,sheep,buffalo
for (i in 1:4) {
  for (j in 1:length(year)) {
    tmp.nb.milk <- animal.country[i*5-4,,j]
    tmp.nb.meat <- animal.country[i*5-3,,j]
    tmp.nb.residual <- animal.country[i*5-2,,j]
    tmp.milk <- animal.country[i*5-1,,j]
    tmp.meat <- animal.country[i*5,,j]
    for (k in 1:10) {
      if (any(tmp.nb.milk > 0 & tmp.milk <= 0 & df.crosswalk$GLEAM_Region == k)) {
        print("milk error");print(i);print(j);print(k)
        tmp.milk[tmp.nb.milk > 0 & tmp.milk <= 0 & df.crosswalk$GLEAM_Region == k] <- meanmilk[i,k,j]
        animal.country[i*5-1,,j] <- tmp.milk
      }  
      # it is possible that buffaloes only have stock but no production
      # need to use regional mean yield
      if (any((tmp.nb.meat > 0 | tmp.nb.residual > 0) & tmp.meat <= 0 & df.crosswalk$GLEAM_Region == k)) {
        print("meat error");print(i);print(j);print(k)
        tmp.meat[(tmp.nb.meat > 0 | tmp.nb.residual > 0) & tmp.meat <= 0 & df.crosswalk$GLEAM_Region == k] <- meanmeat[i,k,j]
        animal.country[i*5,,j] <- tmp.meat
      }
    }
  }
}

##########################################
# 2.2.8 Calculate annual total country/area scale Metabolisable Energy (ME) for cattle, goats, sheep, and buffaloes
##########################################

# 2.2.8.1 Set ME variables
ME.cattle <- array(NA,c(243,length(year)))
ME.goats <- array(NA,c(243,length(year)))
ME.sheep <- array(NA,c(243,length(year)))
ME.buffalo <- array(NA,c(243,length(year)))
nb.cattle <- array(NA,c(243,length(year)))
nb.goats <- array(NA,c(243,length(year)))
nb.sheep <- array(NA,c(243,length(year)))
nb.buffalo <- array(NA,c(243,length(year)))
ME.dairycattle <- array(NA,c(243,length(year)))
nb.dairycattle <- array(NA,c(243,length(year)))
ME.othercattle <- array(NA,c(243,length(year)))
nb.othercattle <- array(NA,c(243,length(year)))
ME.buffalo_nonprod <- array(NA,c(243,length(year)))
nb.buffalo_nonprod <- array(NA,c(243,length(year)))
ME.othercattle_nonprod <- array(NA,c(243,length(year)))
nb.othercattle_nonprod <- array(NA,c(243,length(year)))
# for calculating emissions from livestock producing meat and/or milk
cate <- c("cattle","goats","sheep","buffalo","dairycattle","othercattle")
tmp <- array(NA,c(243,length(year)))
for (rr in 1:length(cate)) {
  assign(paste0("ME.maint.",cate[rr]),tmp)
  assign(paste0("ME.grow.",cate[rr]),tmp)
}
# for calculating emissions from non-producing livestock (draft animals)
nonprod_cate <- c("buffalo_nonprod","othercattle_nonprod")
for (rr in 1:length(nonprod_cate)) {
  assign(paste0("ME.maint.",nonprod_cate[rr]),tmp)
  assign(paste0("ME.grow.",nonprod_cate[rr]),tmp)
}

# 2.2.8.2 ME calculation
# for cattle it is dairycattle + other cattle
for (m in 2:6) { # animal type cate
  # 2.2.8.2.1 Asia specific weight of cattle and buffaloes that do not produce meat or milk
  # Typical cattle and Buffaloes weight from 2019Refinement Table 10A.5
  weight.Asia <- array(0.0,c(243))
  if (m == 4) { # Buffaloes
    index <- which(df.crosswalk$GLEAM_Region == 5) # NENA
    weight.Asia[index] <- 381 
    index <- which(df.crosswalk$GLEAM_Region == 6) # E & SE Asia
    weight.Asia[index] <- 336 
    index <- which(df.crosswalk$GLEAM_Region == 8) # South Asia
    weight.Asia[index] <- 321
    index <- which(df.crosswalk$GLEAM_Region == 9) # LAC
    weight.Asia[index] <- 315
    index <- which(df.crosswalk$GLEAM_Region == 10) # SSA
    weight.Asia[index] <- 339
  } else if (m == 6) { # Other Cattle
    index <- which(df.crosswalk$GLEAM_Region == 5) # NENA
    weight.Asia[index] <- 275 
    index <- which(df.crosswalk$GLEAM_Region == 6) # E & SE Asia
    weight.Asia[index] <- 299 
    index <- which(df.crosswalk$GLEAM_Region == 8) # South Asia
    weight.Asia[index] <- 226
    index <- which(df.crosswalk$GLEAM_Region == 9) # LAC
    weight.Asia[index] <- 303
    index <- which(df.crosswalk$GLEAM_Region == 10) # SSA
    weight.Asia[index] <- 236
  }
  
  # 2.2.8.2.2 Start ME calculation
  for (j in 1:length(year)) { # year
    # 2.2.8.2.2.1 Gathering inputs (livestock numbers, yield, parameters etc.)
    # i is used for allocating livestock numbers and yield from animal.country
    if (m <= 4) { # goats, sheep and buffalo
      i <- m 
    } else { # cattle
      i <- 1
    }
    # milking animal
    tmp.nb.milk <- animal.country[i*5-4,,j]
    # replacement of female milking animal
    tmp.nb.replace <- animal.country[i*5-3,,j]
    # other animal (assumed for meat)
    tmp.nb.res <- animal.country[i*5-2,,j]
    # India specific number
    if (m == 4) {
      tmp.nb.slaughtered <- Buffalo_slaughter_head[,j]
    } else if (m == 6) {
      tmp.nb.slaughtered <- Cattle_slaughter_head[,j]
    } else {
      tmp.nb.slaughtered <- array(0.0,c(243))
    }
    if (m == 5) { # dairycattle 
      tmp.nb.replace[1:243] <- 0.0
      tmp.nb.res[1:243] <- 0.0
    } else if (m == 6) { # othercattle
      tmp.nb.milk[1:243] <- 0.0
    }
    tmp.milk <- animal.country[i*5-1,,j]
    tmp.meat <- animal.country[i*5,,j]
    # Convert Yield unit from Hg to kg
    tmp.milk <- tmp.milk/10.
    tmp.meat <- tmp.meat/10.
    # get country/area level dressing percentage
    dress.milk <- array(NA,c(243))
    dress.meat <- array(NA,c(243))
    weight.milk <- array(NA,c(243))
    weight.meat <-array(NA,c(243))
    for (k in 1:10) {
      if (i == 1) {
        dress.milk[df.crosswalk$GLEAM_Region == k] <- dress.cow[k]
        dress.meat[df.crosswalk$GLEAM_Region == k] <- dress.beef[k]
      } else if (i == 2) {
        dress.milk[df.crosswalk$GLEAM_Region == k] <- dress.goats[k]
        dress.meat[df.crosswalk$GLEAM_Region == k] <- dress.goats[k]
      } else if (i == 3) {
        dress.milk[df.crosswalk$GLEAM_Region == k] <- dress.sheep[k]
        dress.meat[df.crosswalk$GLEAM_Region == k] <- dress.sheep[k]
      } else { # buffalo
        dress.milk[df.crosswalk$GLEAM_Region == k] <- dress.dairybuffalo[k]
        dress.meat[df.crosswalk$GLEAM_Region == k] <- dress.otherbuffalo[k]
      }
    }
    # weight of milking animal (using regional weight of adult female animal)
    if (m == 1) {
      weight.milk <- AFkg_cow
    } else if (m == 2) {
      weight.milk <- AFkg_goats
    } else if (m == 3) {
      weight.milk <- AFkg_sheep
    } else if (m == 4) {
      weight.milk <- AFkg_buffaloes
    } else if (m == 5) {
      weight.milk <- AFkg_cow
    } else if (m == 6) {
      weight.milk <- AFkg_cow
    }
    # weight of slaughtered animal (for meat)
    weight.meat <- tmp.meat/dress.meat
    # standard liveweight just for Livestock Unit (LU) calculation
    # Here, we assumed one LU is 600 kg Cow producing 3000 kg milk per year
    w.cattle <- 600 # standart cattle
    w.goats <- 30 # standart small ruminant
    w.sheep <- 30 # standart small ruminant
    w.buffalo <- 600 # standart cattle
    
    # 2.2.8.2.2.2 Start calculating metabolisable energy demand
    # We calculate total energy for all animals, for each country/area
    for (k in 1:243) {
      # print(k)
      # pre-calculation of the common herd parameters to be used
      if (m == 2) {
        # days from birth to mature milking animal always > 1 year
        AFC_day <- ceiling(AFC_goats[k]*365)
        # daily growth rate
        WG.resF <- (AFkg_goats[k]-Ckg_goats[k])/ceiling(AFC_goats[k]*365)
        AFC_day2 <- ceiling((weight.meat[k]-Ckg_goats[k])/WG.resF)
        WG.resM <- (AMkg_goats[k]-Ckg_goats[k])/ceiling(AFC_goats[k]*365)
        AFC_day3 <- ceiling((weight.meat[k]-Ckg_goats[k])/WG.resM)
      } else if (m == 3) {
        # days from birth to mature milking animal always > 1 year
        AFC_day <- ceiling(AFC_sheep[k]*365)
        # daily growth rate
        WG.resF <- (AFkg_sheep[k]-Ckg_sheep[k])/ceiling(AFC_sheep[k]*365)
        AFC_day2 <- ceiling((weight.meat[k]-Ckg_sheep[k])/WG.resF)
        WG.resM <- (AMkg_sheep[k]-Ckg_sheep[k])/ceiling(AFC_sheep[k]*365)
        AFC_day3 <- ceiling((weight.meat[k]-Ckg_sheep[k])/WG.resM)
      } else if (m == 4) {
        # days from birth to mature milking animal always > 1 year
        AFC_day <- ceiling(AFC_buffaloes[k]*365)
        # daily growth rate
        BW.resF <- Ckg_buffaloes[k]+(AFkg_buffaloes[k]-Ckg_buffaloes[k])*0.5
        WG.resF <- (AFkg_buffaloes[k]-Ckg_buffaloes[k])/ceiling(AFC_buffaloes[k]*365)
        BW.resM <- Ckg_buffaloes[k]+(AMkg_buffaloes[k]-Ckg_buffaloes[k])*0.5
        WG.resM <- (AMkg_buffaloes[k]-Ckg_buffaloes[k])/ceiling(AFC_buffaloes[k]*365)
        AFC_day2 <- ceiling((weight.meat[k]-Ckg_buffaloes[k])/WG.resF)
        AFC_day3 <- ceiling((weight.meat[k]-Ckg_buffaloes[k])/WG.resM)
      } else if (m == 5 | m == 6) {
        # days from birth to mature milking animal always > 1 year
        AFC_day <- ceiling(AFC_cow[k]*365)
        # daily growth rate
        BW.resF <- Ckg_beef[k]+(AFkg_beef[k]-Ckg_beef[k])*0.5
        WG.resF <- (AFkg_beef[k]-Ckg_beef[k])/ceiling(AFC_beef[k]*365)
        BW.resM <- Ckg_beef[k]+(AMkg_beef[k]-Ckg_beef[k])*0.5
        WG.resM <- (AMkg_beef[k]-Ckg_beef[k])/ceiling(AFC_beef[k]*365)
        AFC_day2 <- ceiling((weight.meat[k]-Ckg_beef[k])/WG.resF)
        AFC_day3 <- ceiling((weight.meat[k]-Ckg_beef[k])/WG.resM)
      }
      
      # in case that reaching weight.meat need less than 1 year
      if (AFC_day2 >= 365) {
        DayF <- 365
      } else {
        DayF <- AFC_day2
      }
      if (AFC_day3 >= 365) {
        DayM <- 365
      } else {
        DayM <- AFC_day3
      }
      
      # check if there are draft animal/working livestock (for Asian developing countries only)
      tmp.nb.slF <- tmp.nb.slaughtered[k]/2/365*AFC_day2
      tmp.nb.slM <- tmp.nb.slaughtered[k]/2/365*AFC_day3  
      # real res which are draft animal; if it is positive, we can apply the new strategy to treat it as typical livestock that do not growth
      real.tmp.nb.res <- tmp.nb.res[k]-tmp.nb.slF-tmp.nb.slM
      real.tmp.nb.res[real.tmp.nb.res < 0 | is.na(real.tmp.nb.res) == TRUE] <- 0
      
      # 1) maintainace
      if (m == 5 | m == 6) { # Table 10.4 (Updated) Cow
        # lacting cow Cfi 0.386 (10 months) normal cattle Cfi 0.322 (2 months)
        # assume all milking cow has the same milking weight
        NEm.milk <- (0.386*(weight.milk[k]^0.75)*305+0.322*(weight.milk[k]^0.75)*60)*tmp.nb.milk[k]
        # replace Cfi 0.322 non lacting cows
        # assuming even weight distribution between calves and milking cow
        NEm.replace <- 0.0
        for (ff in 1:AFC_day) {
          NEm.replace <- NEm.replace+0.322*((Ckg_cow[k]+(weight.milk[k]-Ckg_cow[k])*ff/AFC_day)^0.75)*365*(tmp.nb.replace[k]/AFC_day)
        }
        # res assumed to be half female Cfi 0.322 half male Cfi 0.370
        # they were both slaughtered at weight of weigt.meat 
        NEm.res <- 0.0
        # there could be possible that reaching weight.meat need less than 1 year 
        # So if reaching slaughter weight in less than 1 year using actual growing days
        # female
        # even age distribution from 0 to AFC_day2, where if AFC_day2 < 365, using DayF for actural emission days
        for (ff in 1:AFC_day2) {
          NEm.res <- NEm.res+0.322*((Ckg_beef[k]+(weight.meat[k]-Ckg_beef[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.res[k]/AFC_day2)/2
        }
        # male
        for (ff in 1:AFC_day3) {
          NEm.res <- NEm.res+0.370*((Ckg_beef[k]+(weight.meat[k]-Ckg_beef[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.res[k]/AFC_day3)/2
        }
        # LU cow 
        NEm.standard <- (0.386*(w.cattle^0.75)*305+0.322*(w.cattle^0.75)*60)
        #######################
        # Asia specific number
        # Only for SAS EAS developing countries and real res number > 0
        if ((df.crosswalk$GLEAM_Region[k] == 8 | df.crosswalk$GLEAM_Region[k] == 6) & df.crosswalk$Developed[k] == 0 & real.tmp.nb.res > 0)  {
          # Considering a majority of the cattle did not produce any products, we should not evenly distribute tmp.nb.res 
          # but only for followers of slaughtered numbers i.e., tmp.nb.slaughtered/365*AFC_day2 or AFC_day3
          # res assumed to be half female Cfi 0.322 half male Cfi 0.370
          # they were both slaughtered at weight of weigt.meat 
          NEm.res <- 0.0
          # there could be possible that reaching weight.meat need less than 1 year 
          # So if reaching slaughter weight in less than 1 year using actual growing days
          # female
          # even age distribution from 0 to AFC_day2, where if AFC_day2 < 365, using DayF for actural emission days
          for (ff in 1:AFC_day2) {
            NEm.res <- NEm.res+0.322*((Ckg_beef[k]+(weight.meat[k]-Ckg_beef[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.slF/AFC_day2)
          }
          # male
          for (ff in 1:AFC_day3) {
            NEm.res <- NEm.res+0.370*((Ckg_beef[k]+(weight.meat[k]-Ckg_beef[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.slM/AFC_day3)
          }
          # for the rest of the cattle, assume they are mature in typical weight of cattle in India Wolf et al., 2017 281 kg
          # number = res - female meat - male meat
          NEm.res <- NEm.res+(0.322+0.370)/2*weight.Asia[k]^0.75*365*real.tmp.nb.res
        }
        #######################
        # mature cattle that can be draft animal
        NEm.real.mature <- (0.322+0.370)/2*weight.Asia[k]^0.75*365*real.tmp.nb.res
      } else if (m == 2) { # Table 10.4 (Updated) Goats
        NEm.milk <- 0.315*(weight.milk[k]^0.75)*365*tmp.nb.milk[k]
        NEm.replace <- 0.0
        for (ff in 1:AFC_day) {
          NEm.replace <- NEm.replace+0.315*((Ckg_goats[k]+(weight.milk[k]-Ckg_goats[k])*ff/AFC_day)^0.75)*365*(tmp.nb.replace[k]/AFC_day)
        }
        NEm.res <- 0.0
        # there could be possible that reaching weight.meat need less than 1 year
        # female
        for (ff in 1:AFC_day2) {
          NEm.res <- NEm.res+0.315*((Ckg_goats[k]+(weight.meat[k]-Ckg_goats[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.res[k]/AFC_day2)/2
        }
        # male
        for (ff in 1:AFC_day3) {
          NEm.res <- NEm.res+0.315*((Ckg_goats[k]+(weight.meat[k]-Ckg_goats[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.res[k]/AFC_day3)/2
        }
        NEm.standard <- 0.315*(w.goats^0.75)*365
      } else if (m == 3) { # Table 10.4 (Updated) Sheep
        NEm.milk <- 0.236*(weight.milk[k]^0.75)*365*tmp.nb.milk[k]
        NEm.replace <- 0.0
        AFC_day <- ceiling(AFC_sheep[k]*365)
        for (ff in 1:AFC_day) {
          if (ff <= 365) { # lamb to 1 year Cfi 0.236
            NEm.replace <- NEm.replace+0.236*((Ckg_sheep[k]+(weight.milk[k]-Ckg_sheep[k])*ff/AFC_day)^0.75)*365*(tmp.nb.replace[k]/AFC_day)
          } else { # sheep older than 1 year Cfi 0.217
            NEm.replace <- NEm.replace+0.217*((Ckg_sheep[k]+(weight.milk[k]-Ckg_sheep[k])*ff/AFC_day)^0.75)*365*(tmp.nb.replace[k]/AFC_day)
          }
        }
        NEm.res <- 0.0
        # there could be possible that reaching weight.meat need less than 1 year
        # female
        for (ff in 1:AFC_day2) {
          if (ff <= 365) { # lamb to 1 year Cfi 0.236
            NEm.res <- NEm.res+0.236*((Ckg_sheep[k]+(weight.meat[k]-Ckg_sheep[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.res[k]/AFC_day2)/2
          } else { # sheep older than 1 year Cfi 0.217
            NEm.res <- NEm.res+0.217*((Ckg_sheep[k]+(weight.meat[k]-Ckg_sheep[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.res[k]/AFC_day2)/2
          }
        }
        # male
        for (ff in 1:AFC_day3) {
          if (ff <= 365) { # lamb to 1 year Cfi 0.236
            NEm.res <- NEm.res+0.236*((Ckg_sheep[k]+(weight.meat[k]-Ckg_sheep[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.res[k]/AFC_day3)/2
          } else { # sheep older than 1 year Cfi 0.217
            NEm.res <- NEm.res+0.217*((Ckg_sheep[k]+(weight.meat[k]-Ckg_sheep[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.res[k]/AFC_day3)/2
          }
        }
        NEm.standard <- 0.236*(w.sheep^0.75)*365
      } else if (m == 4) { # Table 10.4 (Updated) Buffaloes
        NEm.milk <- (0.386*(weight.milk[k]^0.75)*305+0.322*(weight.milk[k]^0.75)*60)*tmp.nb.milk[k]
        NEm.replace <- 0.0
        for (ff in 1:AFC_day) {
          NEm.replace <- NEm.replace+0.322*((Ckg_buffaloes[k]+(weight.milk[k]-Ckg_buffaloes[k])*ff/AFC_day)^0.75)*365*(tmp.nb.replace[k]/AFC_day)
        }
        NEm.res <- 0.0
        # there could be possible that reaching weight.meat need less than 1 year
        # So if reaching slaughter weight in less than 1 year using actual growing days
        # female
        for (ff in 1:AFC_day2) {
          NEm.res <- NEm.res+0.322*((Ckg_buffaloes[k]+(weight.meat[k]-Ckg_buffaloes[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.res[k]/AFC_day2)/2
        }
        # male
        for (ff in 1:AFC_day3) {
          NEm.res <- NEm.res+0.370*((Ckg_buffaloes[k]+(weight.meat[k]-Ckg_buffaloes[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.res[k]/AFC_day3)/2
        }
        NEm.standard <- 0.370*(w.buffalo^0.75)*365
        #######################
        # Only for SAS EAS developing countries and real res number > 0
        if ((df.crosswalk$GLEAM_Region[k] == 8 | df.crosswalk$GLEAM_Region[k] == 6) & df.crosswalk$Developed[k] == 0 & real.tmp.nb.res > 0)  {
          # Considering a majority of the cattle did not produce any products, we should not evenly distribute tmp.nb.res 
          # but only for followers of slaughtered numbers i.e., tmp.nb.slaughtered/365*AFC_day2 or AFC_day3
          # res assumed to be half female Cfi 0.322 half male Cfi 0.370
          # they were both slaughtered at weight of weigt.meat 
          NEm.res <- 0.0
          # there could be possible that reaching weight.meat need less than 1 year 
          # So if reaching slaughter weight in less than 1 year using actual growing days
          # female
          # even age distribution from 0 to AFC_day2, where if AFC_day2 < 365, using DayF for actural emission days
          for (ff in 1:AFC_day2) {
            NEm.res <- NEm.res+0.322*((Ckg_buffaloes[k]+(weight.meat[k]-Ckg_buffaloes[k])*ff/AFC_day2)^0.75)*DayF*(tmp.nb.slF/AFC_day2)
          }
          # male
          for (ff in 1:AFC_day3) {
            NEm.res <- NEm.res+0.370*((Ckg_buffaloes[k]+(weight.meat[k]-Ckg_buffaloes[k])*ff/AFC_day3)^0.75)*DayM*(tmp.nb.slM/AFC_day3)
          }
          # for the rest of the cattle, assume they are mature in typical weight of cattle in India Wolf et al., 2017 281 kg
          # number = res - female meat - male meat
          NEm.res <- NEm.res+(0.322+0.370)/2*weight.Asia[k]^0.75*365*real.tmp.nb.res
        }
        #######################
        # mature cattle that can be draft animal
        NEm.real.mature <- (0.322+0.370)/2*weight.Asia[k]^0.75*365*real.tmp.nb.res
      }
      
      # 2) activity
      if (m == 4 | m == 5 | m == 6) { # Table 10.5 (Updated) Cattle and Buffaloes
        # cattle NEa=Ca*NEm using pasture value Ca 0.17
        # Based on Table 10A.1 of 2019Refinement
        # Only Oceania, Latin America and South Asia are dominated Pasture/Range feed dairy cow
        if ((df.crosswalk$GLEAM_Region[k] == 7 | df.crosswalk$GLEAM_Region[k] == 8 | df.crosswalk$GLEAM_Region[k] == 9)) {
          NEa.milk <- 0.17*NEm.milk
        } else {
          NEa.milk <- 0.0
        }
        # while for replacement and other cattle, they are all dominated Pasture/Range feed
        NEa.replace <- 0.17*NEm.replace
        NEa.res <- 0.17*NEm.res
        NEa.standard <- 0.17*NEm.standard
        #######################
        # India specific number
        # all animals have activity, no changes here
        #######################
        # mature cattle that can be draft animal
        NEa.real.mature <- 0.17*NEm.real.mature
        #######################
      } else if (m == 2) { # Table 10.5 (Updated) # Goats
        # goats NEa=Ca*(weight) using lowland Ca 0.019
        NEa.milk <- 0.019*weight.milk[k]*365*tmp.nb.milk[k]
        NEa.replace <- 0.0
        for (ff in 1:AFC_day) {      
          NEa.replace <- NEa.replace+0.0107*(Ckg_goats[k]+(weight.milk[k]-Ckg_goats[k])*ff/AFC_day)*365*(tmp.nb.replace[k]/AFC_day)
        }
        NEa.res <- 0.0
        # female
        for (ff in 1:AFC_day2) {
          NEa.res <- NEa.res+0.019*(Ckg_goats[k]+(weight.meat[k]-Ckg_goats[k])*ff/AFC_day2)*DayF*(tmp.nb.res[k]/AFC_day2)/2
        }
        # male
        for (ff in 1:AFC_day3) {
          NEa.res <- NEa.res+0.019*(Ckg_goats[k]+(weight.meat[k]-Ckg_goats[k])*ff/AFC_day3)*DayM*(tmp.nb.res[k]/AFC_day3)/2
        }
        NEa.standard <- 0.019*w.goats*365
      } else if (m == 3) { # Table 10.5 (Updated) # Sheep
        # sheep NEa=Ca*(weight) using grazing flat pasture Ca 0.0107
        NEa.milk <- 0.0107*weight.milk[k]*365*tmp.nb.milk[k]
        NEa.replace <- 0.0
        for (ff in 1:AFC_day) {
          NEa.replace <- NEa.replace+0.0107*(Ckg_sheep[k]+(weight.milk[k]-Ckg_sheep[k])*ff/AFC_day)*365*(tmp.nb.replace[k]/AFC_day)
        }
        NEa.res <- 0.0
        # female
        for (ff in 1:AFC_day2) {
          NEa.res <- NEa.res+0.019*(Ckg_sheep[k]+(weight.meat[k]-Ckg_sheep[k])*ff/AFC_day2)*DayF*(tmp.nb.res[k]/AFC_day2)/2
        }
        # male
        for (ff in 1:AFC_day3) {
          NEa.res <- NEa.res+0.019*(Ckg_sheep[k]+(weight.meat[k]-Ckg_sheep[k])*ff/AFC_day3)*DayM*(tmp.nb.res[k]/AFC_day3)/2
        }
        NEa.standard <- 0.0107*w.sheep*365
      }
      # 3) growth
      if (m == 5 | m == 6) { # Equation 10.6 replacement and beef cattle 
        # assuming even distribution, mean weight = half between calves and milking weight 
        BW.replace <- Ckg_cow[k]+(weight.milk[k]-Ckg_cow[k])*0.5
        WG.replace <- (AFkg_cow[k]-Ckg_cow[k])/ceiling(AFC_cow[k]*365)
        NEg.replace <- 22.02*(BW.replace/(0.8*AFkg_cow[k]))^0.75*(WG.replace)^1.097*365*tmp.nb.replace[k]
        # assuming even distribution of age and half male C=1.2 half female C=0.8
        NEg.res <- 22.02*(BW.resF/(0.8*AFkg_beef[k]))^0.75*(WG.resF)^1.097*DayF*tmp.nb.res[k]/2+
          22.02*(BW.resM/(1.2*AMkg_beef[k]))^0.75*(WG.resM)^1.097*DayM*tmp.nb.res[k]/2
        #######################
        # India specific number
        # Only for SAS EAS developing countries and real res number > 0
        if ((df.crosswalk$GLEAM_Region[k] == 8 | df.crosswalk$GLEAM_Region[k] == 6) & df.crosswalk$Developed[k] == 0 & real.tmp.nb.res > 0)  {
          NEg.res <- 0.0
          NEg.res <- 22.02*(BW.resF/(0.8*AFkg_beef[k]))^0.75*(WG.resF)^1.097*DayF*tmp.nb.slF+
            22.02*(BW.resM/(1.2*AMkg_beef[k]))^0.75*(WG.resM)^1.097*DayM*tmp.nb.slM
        }
        #######################
      } else if (m == 2) { # Equation 10.7 Goats Table 10.6 (Updated)
        NEg.replace <- (weight.milk[k]-Ckg_goats[k])*(5.0+0.5*0.33*(Ckg_goats[k]+weight.meat[k]))/365*365*tmp.nb.replace[k]
        NEg.res <- (weight.meat[k]-Ckg_goats[k])*(5.0+0.5*0.33*(Ckg_goats[k]+weight.meat[k]))/365*DayF*tmp.nb.res[k]/2+
          (weight.meat[k]-Ckg_goats[k])*(5.0+0.5*0.33*(Ckg_goats[k]+weight.meat[k]))/365*DayM*tmp.nb.res[k]/2
      } else if (m == 3) { # Equation 10.7 Sheep Table 10.6 (Updated)
        # replacement female sheep
        NEg.replace <- (weight.milk[k]-Ckg_sheep[k])*(2.1+0.5*0.45*(Ckg_sheep[k]+weight.meat[k]))/365*365*tmp.nb.replace[k]
        NEg.res <- (weight.meat[k]-Ckg_sheep[k])*(2.1+0.5*0.45*(Ckg_sheep[k]+weight.meat[k]))/365*DayF*tmp.nb.res[k]/2 +
          (weight.meat[k]-Ckg_sheep[k])*(2.5+0.5*0.35*(Ckg_sheep[k]+weight.meat[k]))/365*DayM*tmp.nb.res[k]/2
      } else if (m == 4) { # Equation 10.6 Buffaloes
        # assuming even distribution, mean weight = half between calves and milking weight
        BW.replace <- Ckg_buffaloes[k]+(weight.milk[k]-Ckg_buffaloes[k])*0.5
        WG.replace <- (AFkg_buffaloes[k]-Ckg_buffaloes[k])/ceiling(AFC_buffaloes[k]*365)
        NEg.replace <- 22.02*(BW.replace/(0.8*AFkg_buffaloes[k]))^0.75*(WG.replace)^1.097*365*tmp.nb.replace[k]
        NEg.res <- 22.02*(BW.resF/(0.8*AFkg_buffaloes[k]))^0.75*(WG.resF)^1.097*DayF*tmp.nb.res[k]/2+
          22.02*(BW.resM/(1.2*AMkg_buffaloes[k]))^0.75*(WG.resM)^1.097*DayM*tmp.nb.res[k]/2
        #######################
        # India specific number
        # Only for SAS EAS developing countries and real res number > 0
        if ((df.crosswalk$GLEAM_Region[k] == 8 | df.crosswalk$GLEAM_Region[k] == 6) & df.crosswalk$Developed[k] == 0 & real.tmp.nb.res > 0)  {
          NEg.res <- 0.0
          NEg.res <- 22.02*(BW.resF/(0.8*AFkg_beef[k]))^0.75*(WG.resF)^1.097*DayF*tmp.nb.slF+
            22.02*(BW.resM/(1.2*AMkg_beef[k]))^0.75*(WG.resM)^1.097*DayM*tmp.nb.slM
        }
        #######################
      }
      # 4) lactation
      if (m == 4 | m == 5 | m == 6) { # Equation 10.8 (Updated) Cattle and Buffaloes
        # cattle NEl=milk*(1.47+0.4*Fat) Fat 3.5 for UK 3-4 in FAO
        NEl.milk <- tmp.milk[k]*(1.47+0.4*3.5)*tmp.nb.milk[k]
        NEl.standard <- 3000.*(1.47+0.4*3.5) 
      } else if (m == 2) { # Equation 10.9 (Updated) Goats
        # goats NEl=milk*EVmilk EVmilk 3.0 MJ/kg
        NEl.milk <- tmp.milk[k]*3.0*tmp.nb.milk[k]
        NEl.standard <- 300.*3.0
      } else if (m == 3) { # Equation 10.9 (Updated) Sheep
        # sheep NEl=milk*EVmilk EVmilk 4.6 MJ/kg
        NEl.milk <- tmp.milk[k]*4.6*tmp.nb.milk[k]
        NEl.standard <- 300.*4.6
      }
      # 5) work
      NEw.res <- 0.0
      if (m == 6) { # Equation 10.11 Beef cattle
        # mature res cattle only in developing countries NEwork=0.1*NEm*hours of work
        # we use NEm.real.mature rather than all res NEm
        if (df.crosswalk$Developed[k] == 0) {
          NEw.res <- 0.1*NEm.real.mature*1.0
        }
      } else if (m == 4) { # buffalo  # Table 10.11
        # For developing countries, a typical draft animal are assumed to work 40 days per year (Congress, 1991), which equates to 2.6 hours per day of work annually. 
        # SI of Dangal et al., 2017 GCB
        # 40 days * 10 hours per day / 365 days = 1.09589 per day
        # we use NEm.real.mature rather than all res NEm
        if (df.crosswalk$Developed[k] == 0) {
          NEw.res <- 0.1*NEm.real.mature*1.1
        }
      }
      #######################
      # mature cattle that can be draft animal
      NEw.real.mature <- NEw.res
      # 6) Wool # Equation 10.12 (Updated)
      NEwool.res <- 0.0
      if (m == 3) {
        NEwool.res <- 24 * Sheep_wool_production[k,j]*1000./365*365 # tonnes to kg yr-1/365 * MJ day-1 * 365 = MJ year-1
      }
      # 7) pregnancy
      if (m == 4 | m == 5 | m == 6) { # Table 10.7 (Updated)
        # cattle NEp=Cpregnancy*NEm C=0.1 only for milking cow
        NEp.milk <- 0.1 * NEm.milk
        NEp.standart <- 0.1 * NEm.standard
      } else if (m == 2 | m == 3) { # Table 10.7 (Updated)
        # goats/sheep NEp=Cpregnancy*NEm C=0.077 single birth 0.126 duble birth
        NEp.milk <- 0.077 * NEm.milk
        NEp.standart <- 0.077 * NEm.standard
      }
      # 8) Total ME
      ME.t <-  NEm.milk+NEm.replace+NEm.res+ # Maintenance
        NEa.milk+NEa.replace+NEa.res+ # Activity
        NEg.replace+NEg.res+ # growth
        NEl.milk+ # lactation
        NEw.res+ # work
        NEwool.res+ # wool
        NEp.milk # preganancy
      ME.maint.t <- NEm.milk+NEm.replace+NEm.res+ # Maintenance
        NEa.milk+NEa.replace+NEa.res+ # Activity
        NEl.milk+ # lactation
        NEw.res+ # work
        NEp.milk # preganancy
      ME.grow.t <- NEg.replace+NEg.res+ # growth
        NEwool.res # wool
      nb.t <- tmp.nb.milk[k]+tmp.nb.replace[k]+tmp.nb.res[k]
      
      # for non-producing draft animal    
      if (m == 4 | m == 6) {
        ME.t_nonprod <- NEm.real.mature+NEa.real.mature+NEw.real.mature
        ME.maint.t_nonprod <- NEm.real.mature+NEa.real.mature+NEw.real.mature
        ME.grow.t_nonprod <- 0.0
      }
      # for standard animal (LU)
      if (m == 5 | m == 6) {
        ME.LU.standard <- NEm.standard+NEa.standard+NEl.standard+NEp.standart 
        ME.LU.standard.nomilk <- NEm.standard+NEa.standard+NEp.standart
      } else if (m == 2) {
        ME.goats.standard <- NEm.standard+NEa.standard+NEl.standard+NEp.standart
        ME.goats.standard.nomilk <- NEm.standard+NEa.standard+NEp.standart
      } else if (m == 3) {
        ME.sheep.standard <- NEm.standard+NEa.standard+NEl.standard+NEp.standart
        ME.sheep.standard.nomilk <- NEm.standard+NEa.standard+NEp.standart
      }
      # 9) assign total ME for each category in a country/area
      if (m == 2) {
        ME.goats[k,j] <- ME.t
        ME.maint.goats[k,j] <- ME.maint.t
        ME.grow.goats[k,j] <- ME.grow.t
        nb.goats[k,j] <- nb.t
      } else if (m == 3) {
        ME.sheep[k,j] <- ME.t
        ME.maint.sheep[k,j] <- ME.maint.t
        ME.grow.sheep[k,j] <- ME.grow.t
        nb.sheep[k,j] <- nb.t 
      } else if (m == 4) {
        # Note: ME.XXX_noprod is for draft animal, while ME.XXX includes ME.XXX_noprod
        ME.buffalo[k,j] <- ME.t 
        ME.maint.buffalo[k,j] <- ME.maint.t
        ME.grow.buffalo[k,j] <- ME.grow.t
        nb.buffalo[k,j] <- nb.t 
        ME.buffalo_nonprod[k,j] <- ME.t_nonprod 
        ME.maint.buffalo_nonprod[k,j] <- ME.maint.t_nonprod
        ME.grow.buffalo_nonprod[k,j] <- ME.grow.t_nonprod
        nb.buffalo_nonprod[k,j] <- real.tmp.nb.res
      } else if (m == 5) {
        ME.dairycattle[k,j] <- ME.t 
        ME.maint.dairycattle[k,j] <- ME.maint.t
        ME.grow.dairycattle[k,j] <- ME.grow.t
        nb.dairycattle[k,j] <- nb.t 
      } else if (m == 6) {
        # Note: ME.XXX_noprod is for draft animal, while ME.XXX includes ME.XXX_noprod
        ME.othercattle[k,j] <- ME.t 
        ME.maint.othercattle[k,j] <- ME.maint.t
        ME.grow.othercattle[k,j] <- ME.grow.t
        nb.othercattle[k,j] <- nb.t
        ME.othercattle_nonprod[k,j] <- ME.t_nonprod 
        ME.maint.othercattle_nonprod[k,j] <- ME.maint.t_nonprod
        ME.grow.othercattle_nonprod[k,j] <- ME.grow.t_nonprod
        nb.othercattle_nonprod[k,j] <- real.tmp.nb.res
      }
    } # k country/area
  } # j year
} # m animal type cate

# 2.2.8.3 Aggregate cattle ME
ME.cattle <- ME.dairycattle+ME.othercattle
ME.maint.cattle <- ME.maint.dairycattle+ME.maint.othercattle
ME.grow.cattle <- ME.grow.dairycattle+ME.grow.othercattle
nb.cattle <- nb.dairycattle+nb.othercattle

# 2.2.8.4 Avoid NA
cate <- c("cattle","goats","sheep","buffalo","dairycattle","othercattle","buffalo_nonprod","othercattle_nonprod")
for (rr in 1:length(cate)) {
  tmp <- eval(parse(text=paste0("ME.",cate[rr])))
  tmp[is.na(tmp) == TRUE | is.finite(tmp) == FALSE] <- 0.0
  assign(paste0("ME.",cate[rr]),tmp)
  tmp <- eval(parse(text=paste0("ME.maint.",cate[rr])))
  tmp[is.na(tmp) == TRUE | is.finite(tmp) == FALSE] <- 0.0
  assign(paste0("ME.maint.",cate[rr]),tmp)
  tmp <- eval(parse(text=paste0("ME.grow.",cate[rr])))
  tmp[is.na(tmp) == TRUE | is.finite(tmp) == FALSE] <- 0.0
  assign(paste0("ME.grow.",cate[rr]),tmp)
  tmp <- eval(parse(text=paste0("nb.",cate[rr])))
  tmp[is.na(tmp) == TRUE | is.finite(tmp) == FALSE] <- 0.0
  assign(paste0("nb.",cate[rr]),tmp)
}

# 2.2.8.5 Calculate regional ME
tmp <- array(NA,c(11,length(year)))
for (rr in 1:length(cate)) {
  assign(paste0("regionalME.",cate[rr]),tmp)
  assign(paste0("regionalME.maint.",cate[rr]),tmp)
  assign(paste0("regionalME.grow.",cate[rr]),tmp)
  assign(paste0("regionalnb.",cate[rr]),tmp)
}
var <- c("ME.","ME.maint.","ME.grow.","nb.")
for (vv in 1:length(var)) {
  for (rr in 1:length(cate)) {
    regionaltmp <- array(NA,c(11,length(year)))
    tmp <- eval(parse(text=paste0(var[vv],cate[rr])))
    for (i in 1:10) {
      index <- which(df.crosswalk$GLEAM_Region == i)
      if (i == 2) {
        regionaltmp[i,] <- tmp[i,]
      } else {
        regionaltmp[i,] <- apply(tmp[index,],c(2),sum,na.rm=TRUE)
      }
    }
    i <- 11
    regionaltmp[i,] <- apply(tmp[,],c(2),sum,na.rm=TRUE)
    assign(paste0("regional",var[vv],cate[rr]),regionaltmp)
  }
}

# 2.2.8.6 Calculate country/area and regional number of Livestock Unit (LU)
LU.cattle <- ME.cattle/ME.LU.standard
LU.goats <- ME.goats/ME.LU.standard
LU.sheep <- ME.sheep/ME.LU.standard
LU.buffalo <- ME.buffalo/ME.LU.standard
LU.dairycattle <- ME.dairycattle/ME.LU.standard
LU.othercattle <- ME.othercattle/ME.LU.standard

##########################################
##########################################
# 2.3 Calculate Enteric Fermentation CH4 emissions from all livestock categories
##########################################
##########################################

##########################################
# 2.3.1 Set functions and parameters (DE and Ym) for major ruminant
##########################################

# 2.3.1.1 Functions on energy calculation
# REM represents the ratio of net energy available in diet for maintenance to digestible energy consumed
REM <- function(DE) {
  REM <- 1.123- (4.092 * 1e-3 * (DE)) + (1.126*1e-5*(DE)^2) - 25.4/(DE)
  return(REM)
}
# REG represents the ratio of net energy available for growth in a diet to digestible energy consumed
REG <- function(DE) {
  REG <- 1.164- (5.16 * 1e-3 * (DE)) + (1.308*1e-5*(DE)^2) - 37.4/(DE)
  return(REG)
}
# 2.3.1.2 Parameters
# Number of Monte Carlo ensembles for DE and Ym
MC.DE <- 1e4 
# assuming a energy content of 18.45MJ/kg DM feedstuff
DGE <- 18.45 # energy content of 18.45MJ/kg DM
# Regional DE from Table B13 of Opio et al., 2013
DE_regional_dairycattle <- c(71.8, 72.6, 77.0, 73.5, 56.1, 59.0, 72.9, 52.6, 62.2, 57.3)
DE_regional_othercattle <- c(68.3, 73.8, 76.1, 73.8, 57.7, 57.4, 72.9, 50.7, 62.7, 57.2)
DE_regional_small <-       c(66.2, 65.4, 69.7, 67.4, 55.5, 56.3, 69.8, 54.1, 58.9, 55.5)
DE_regional_buffalo <-     c(73.4, 70.7, 75.5, 72.8, 52.0, 56.0, 75.5, 52.1, 60.5, 52.1)

# Ym depends on DE as in Table 4.12 of GLEAM v2.0 Documentation
Ym_dairycattle <- 9.75 - 0.05 * DE_regional_dairycattle
Ym_othercattle <- 9.75 - 0.05 * DE_regional_othercattle
Ym_small <- 9.75 - 0.05 * DE_regional_small
Ym_buffalo <- 9.75 - 0.05 * DE_regional_buffalo
# Uncertainty for Ym for cattle is about 20% (SD)
dist_general <- rnorm(MC.DE,mean=1,sd=0.2)
dist_general[dist_general < 0.0] <- 0.0
# for sheep SD = 0.9/6.7 (*100%)
dist_sheep <- dist_general
# for goats SD = 1.0/5.5 (*100%)
dist_goats <- dist_general
for (i in 1:length(dist_general)) {
  # expand distance 
  tmp_dis <- dist_general[i]-1
  new_dis_sheep <- tmp_dis/0.2*0.9/6.7
  dist_sheep[i] <- new_dis_sheep+1
  new_dis_goats <- tmp_dis/0.2*1.0/5.5
  dist_goats[i] <- new_dis_goats+1
}
# Country/Area specific parameters
DE_animal <- array(NA,c(8,243,length(year)))
Ym_animal <- array(NA,c(MC.DE,8,243,length(year)))
for (m in 1:8) { # livestock category
  tmp_DE <- array(NA,c(243,length(year)))
  tmp_Ym <- array(NA,c(MC.DE,243,length(year)))
  for (j in 1:length(year)) {
    # assign DE
    tmp <- array(NA,c(243))
    for (i in 1:10) { # region
      index <- which(df.crosswalk$GLEAM_Region == i)
      if (m == 2) {
        tmp[index] <- DE_regional_small[i] 
      } else if (m == 3) {
        tmp[index] <- DE_regional_small[i]
      } else if (m == 4) {
        tmp[index] <- DE_regional_buffalo[i]
      } else if (m == 5) {
        tmp[index] <- DE_regional_dairycattle[i]
      } else if (m == 6) {
        tmp[index] <- DE_regional_othercattle[i]
      } else if (m == 7) { # buffalo_nonprod
        tmp[index] <- DE_regional_buffalo[i]
      } else if (m == 8) { # othercattle_nonprod
        tmp[index] <- DE_regional_othercattle[i]
      } else if (m == 1) {
        tmp[index] <- DE_regional_othercattle[i]
      }
    } # i region
    tmp_DE[,j] <- tmp
    # assign Ym
    tt_Ym <- array(NA,c(MC.DE,243))
    if (m == 2) { # goats
      # use regional Ym calculated from DE
      tt <- array(NA,c(243))
      for (i in 1:10) {
        index <- which(df.crosswalk$GLEAM_Region == i)
        tt[index] <- Ym_small[i]
      }
      for (i in 1:243) {
        tt_Ym[,i] <- tt[i]*dist_goats
      }
    } else if (m == 3) { # sheep
      # use regional Ym calculated from DE
      tt <- array(NA,c(243))
      for (i in 1:10) {
        index <- which(df.crosswalk$GLEAM_Region == i)
        tt[index] <- Ym_small[i]
      }
      for (i in 1:243) {
        tt_Ym[,i] <- tt[i]*dist_sheep
      }
    } else if (m == 6 | m == 8) { # othercattle
      tt <- array(NA,c(243))
      for (i in 1:10) {
        index <- which(df.crosswalk$GLEAM_Region == i)
        tt[index] <- Ym_othercattle[i]        
      }
      for (i in 1:243) {
        tt_Ym[,i] <- tt[i]*dist_general
      }
    } else if (m == 4 | m == 7) { # buffaloes
      tt <- array(NA,c(243))
      for (i in 1:10) {
        index <- which(df.crosswalk$GLEAM_Region == i)
        tt[index] <- Ym_buffalo[i]        
      }
      for (i in 1:243) {
        tt_Ym[,i] <- tt[i]*dist_general
      }  
    } else if (m == 5) { # dairycattle
      tt <- array(NA,c(243))
      for (i in 1:10) {
        index <- which(df.crosswalk$GLEAM_Region == i)
        tt[index] <- Ym_dairycattle[i]
      }
      for (i in 1:243) {
        tt_Ym[,i] <- tt[i]*dist_general
      }
    }
    tmp_Ym[,,j] <- tt_Ym
  } # j year
  DE_animal[m,,] <- tmp_DE
  Ym_animal[,m,,] <- tmp_Ym
} # m livestock category

##########################################
# 2.3.2 Calculate Gross Energy, Dry Matter Intake and Enteric Fermentation CH4 emissions for major ruminant livestock
##########################################

# 2.3.2.1 Calculate country/area values for each livestock category
MC_GE_used <- array(NA,c(MC.DE,6,243,length(year)))
MC_DM_all_used <- array(NA,c(MC.DE,6,243,length(year)))
MC_FCH4_used <- array(NA,c(MC.DE,6,243,length(year)))
for (m in 2:6) {
  ME_maint <- eval(parse(text=paste0("ME.maint.",cate[m])))
  ME_grow <- eval(parse(text=paste0("ME.grow.",cate[m])))
  for (z in 1:MC.DE) { # feed DE and Ym
    #print(paste(m,z))
    for (j in 1:length(year)) {
      for (i in 1:243) {
        DE_tmp <- DE_animal[m,i,j]
        Ym_tmp <- Ym_animal[z,m,i,j]
        REM_tmp <- REM(DE_tmp)
        REG_tmp <- REG(DE_tmp)
        GE_tmp <- (ME_maint[i,j]/REM_tmp+ME_grow[i,j]/REG_tmp)/(DE_tmp/100)
        MC_GE_used[z,m,i,j] <- GE_tmp # MJ
        MC_DM_all_used[z,m,i,j] <- GE_tmp/DGE # kg 
        MC_FCH4_used[z,m,i,j] <- GE_tmp*(Ym_tmp/100)/55.65 # kg CH4
      }
    }
  } 
}
# Aggregated to cattle
MC_GE_used[,1,,] <- MC_GE_used[,5,,]+MC_GE_used[,6,,]
MC_DM_all_used[,1,,] <- MC_DM_all_used[,5,,]+MC_DM_all_used[,6,,]
MC_FCH4_used[,1,,] <- MC_FCH4_used[,5,,]+MC_FCH4_used[,6,,]

# 2.3.2.2 Calculate country/area values for draft animals
MC_nonprod_GE_used <- array(NA,c(MC.DE,2,243,length(year)))
MC_nonprod_DM_all_used <- array(NA,c(MC.DE,2,243,length(year)))
MC_nonprod_FCH4_used <- array(NA,c(MC.DE,2,243,length(year)))
for (m in 7:8) {
  ME_maint <- eval(parse(text=paste0("ME.maint.",cate[m])))
  ME_grow <- eval(parse(text=paste0("ME.grow.",cate[m])))
  for (z in 1:MC.DE) { # feed DE and Ym
    #print(paste(m,z))
    for (j in 1:length(year)) {
      for (i in 1:243) {
        DE_tmp <- DE_animal[m,i,j]
        Ym_tmp <- Ym_animal[z,m,i,j]
        REM_tmp <- REM(DE_tmp)
        REG_tmp <- REG(DE_tmp)
        GE_tmp <- (ME_maint[i,j]/REM_tmp+ME_grow[i,j]/REG_tmp)/(DE_tmp/100)
        MC_nonprod_GE_used[z,m-6,i,j] <- GE_tmp # MJ
        MC_nonprod_DM_all_used[z,m-6,i,j] <- GE_tmp/DGE # kg 
        MC_nonprod_FCH4_used[z,m-6,i,j] <- GE_tmp*(Ym_tmp/100)/55.65 # kg CH4
      }
    }
  } 
}

# 2.3.2.3 calculate global and regional total FCH4
MC_global_FCH4_used <- apply(MC_FCH4_used[,1:6,1:243,],c(1,2,4),sum,na.rm=TRUE)/1e9 # convert from kg CH4 to Tg CH4
MC_regional_FCH4_used <- array(NA,c(MC.DE,6,10,length(year)))
for (k in 1:10) { # region
  if (k == 2) { # Russian Fed.
    MC_regional_FCH4_used[,1:6,k,] <- MC_FCH4_used[,1:6,gleam.list[[k]],]/1e9 # convert from kg CH4 to Tg CH4
  } else { # not Russian Fed.
    MC_regional_FCH4_used[,1:6,k,] <- apply(MC_FCH4_used[,1:6,gleam.list[[k]],],c(1,2,4),sum,na.rm=TRUE)/1e9 # convert from kg CH4 to Tg CH4
  } # not Russian Fed.
} # k region

##########################################
# 2.3.3 Calculate Enteric Fermentation CH4 emissions for other livestock
##########################################

# For all livestock has enteric fermentation emissions (except pigs), we use constant emission factor
# Because they are not major protein food sources, thus slaughtered weight change probabaly did not reflect the changes in mean liveweight
# Livestock with enteric fermentation emissions
# Asses 10 kg CH4 head-1 yr-1
# Camels 46 kg CH4 head-1 yr-1
# Horses 18 kg CH4 head-1 yr-1
# Mules 10 kg CH4 head-1 yr-1
# Swine, breeding 1 kg CH4 head-1 yr-1 for some regions
# Swine, market 1 kg CH4 head-1 yr-1
# Llamas 8

# 2.3.3.1 For Pigs, we use slaughtered weight change to extrapolate emission changes
# Given the fact that enteric fermentation emission factor depends on total DMI or GE
# and further depends on Metabolisable energy (assuming no changes in DE)
# The latter is dorminated comprised of maintenence emissions, which is correlated to liveweight^0.75
# Enteric fermentation from pigs
j_Item <- which(Yield.Item.name == "Meat, pig")
pig_yield <- Yield.mat[,j_Item,] # "Meat, pig"
pig_liveweight <- array(NA,c(243,length(year)))
# dressing percentage from GLEAMv2.0 Table 9.2
# 65% for backyard system, 75% for intermediate and industrial system
dress_pig <- array(0.65,c(243))
index <- which(df.crosswalk$Developed == 1)
dress_pig[index] <- 0.75
for (j in 1:length(year)) {
  tmp <- pig_yield[,j]*0.1/dress_pig # Hg to kg
  pig_liveweight[,j] <- tmp
}
j_Item <- which(Stock.Item.name == "Pigs")
pig_stock <- Stock.mat[,j_Item,] # "Pigs"
# set default low productivity system mean liveweight from Table 10.10 (Updated) 2019 Refinement
pig_liveweight[is.na(pig_liveweight) == TRUE] <- 52

EFCH4_pig <- array(NA,c(243,length(year)))
for (j in 1:length(year)) {
  for (i in 1:243) {
    if (df.crosswalk$Developed == 1) {
      EFCH4_pig[i,j] <- 1.5*((pig_liveweight[i,j]/2)^0.75)/(72^0.75)*pig_stock[i,j]/1e6 # kg to Gg
    } else {
      EFCH4_pig[i,j] <- 1.0*((pig_liveweight[i,j]/2)^0.75)/(52^0.75)*pig_stock[i,j]/1e6 # kg to Gg
    }    
  }
}

# 2.3.3.2 other livestock horses, camels, mules, asses, and llamas, rescale emission factor with regional liveweight from Table 10A.5 of 2019 Refinement
EFCH4_horses <- array(NA,c(243,length(year)))
EFCH4_camels <- array(NA,c(243,length(year)))
EFCH4_mules <- array(NA,c(243,length(year)))
EFCH4_asses <- array(NA,c(243,length(year)))
EFCH4_llamas <- array(NA,c(243,length(year)))
j_Item <- which(Stock.Item.name == "Horses")
stock_horses <- Stock.mat[,j_Item,] # "Horses"
j_Item <- which(Stock.Item.name == "Camels")
stock_camels <- Stock.mat[,j_Item,] # "Camels"
j_Item <- which(Stock.Item.name == "Mules")
stock_mules <- Stock.mat[,j_Item,] # "Mules" 
j_Item <- which(Stock.Item.name == "Asses")
stock_asses <- Stock.mat[,j_Item,] # "Asses"
j_Item <- which(Stock.Item.name == "Camelids, other")
stock_llamas <- Stock.mat[,j_Item,] # "Camelids, other"
for (j in 1:length(year)) {
  for (i in 1:243) {
    if (df.crosswalk$GLEAM_Region[i] <= 4 | df.crosswalk$GLEAM_Region[i] == 7) {  
      EFCH4_horses[i,j] <- 18*(377/550)^0.75*stock_horses[i,j]/1e6 # kg to Gg
    } else {
      EFCH4_horses[i,j] <- 18*(238/550)^0.75*stock_horses[i,j]/1e6 # kg to Gg
    } 
    EFCH4_camels[i,j] <- 46*(217/570)^0.75*stock_camels[i,j]/1e6 # kg to Gg
    EFCH4_mules[i,j] <- 10*(130/245)^0.75*stock_mules[i,j]/1e6 # kg to Gg
    EFCH4_asses[i,j] <- 10*(130/245)^0.75*stock_asses[i,j]/1e6 # kg to Gg
    EFCH4_llamas[i,j] <- 8*(65/65)^0.75*stock_llamas[i,j]/1e6 # kg to Gg
  }
}

##########################################
# 2.3.4 Enteric Fermentation CH4 emissions from major ruminant livestock
##########################################

mean_EFCH4_ruminant <- apply(MC_FCH4_used[,,,],c(2,3,4),mean,na.rm=TRUE)/1e6 # kg to Gg

##########################################
# 2.3.5 gethering all Enteric Fermentation CH4 emissions from livestock
##########################################

country_EFCH4_2019MT <- array(NA,c(243,16,length(year)))
country_EFCH4_2019MT[,1,] <- EFCH4_asses # "Asses"
country_EFCH4_2019MT[,2,] <- mean_EFCH4_ruminant[4,,] # "Buffaloes"
country_EFCH4_2019MT[,3,] <- EFCH4_horses # "Camels"
country_EFCH4_2019MT[,4,] <- mean_EFCH4_ruminant[5,,] # "Cattle, dairy" 
country_EFCH4_2019MT[,5,] <- mean_EFCH4_ruminant[6,,] # "Cattle, non-dairy"
country_EFCH4_2019MT[,6,] <- 0.0 # "Chickens, broilers"
country_EFCH4_2019MT[,7,] <- 0.0 # "Chickens, layers"
country_EFCH4_2019MT[,8,] <- 0.0 # "Ducks"
country_EFCH4_2019MT[,9,] <- mean_EFCH4_ruminant[2,,] # "Goats" 
country_EFCH4_2019MT[,10,] <- EFCH4_horses # "Horses"
country_EFCH4_2019MT[,11,] <- EFCH4_llamas # "Llamas"
country_EFCH4_2019MT[,12,] <- EFCH4_mules # "Mules"
country_EFCH4_2019MT[,13,] <- mean_EFCH4_ruminant[3,,] # "Sheep"
country_EFCH4_2019MT[,14,] <- 0.0 # "Swine, breeding"
country_EFCH4_2019MT[,15,] <- EFCH4_pig # "Swine, market"
country_EFCH4_2019MT[,16,] <- 0.0 # "Turkeys" 

# check regional value
#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 
spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")
regional_EFCH4_2019MT <- array(NA,c(11,16,length(year)))
for (i in 1:16) {
  for (k in 1:10) {
    if (k == 2) {
      regional_EFCH4_2019MT[k,,] <- country_EFCH4_2019MT[gleam.list[[k]],,]
    } else {
      regional_EFCH4_2019MT[k,,] <- apply(country_EFCH4_2019MT[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
    }
  }
  k <- 11
  regional_EFCH4_2019MT[k,,] <- apply(country_EFCH4_2019MT[,,],c(2,3),sum,na.rm=TRUE)
}

##########################################
# save output of Enteric Fermentation CH4 emissions using the 2019 MT method
##########################################
save.image(file=paste0(path_out,"EFCH4_2019MT.RData"))

##########################################
# 2.3.6 calculate spatially explicit Enteric Fermentation CH4 emissions from all livestock categories
##########################################

spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")

for (m in c(1:16)) {
  # spatial pattern of gridded livestock fraction within each country/area
  print(spatial_cate_name[m])
  if (m %in% c(4,5)) { # "CattleDairy","CattleNonDairy"
    frac_live <- GLW3_fraction_cattle
  } else if (m %in% c(13)) { # "Sheep" 
    frac_live <- GLW3_fraction_sheep
  } else if (m %in% c(9)) { # "Goats"
    frac_live <- GLW3_fraction_goats
  } else if (m %in% c(2)) { # Buffaloes"
    frac_live <- GLW3_fraction_buffaloes
  } else if (m %in% c(14,15)) { # "SwineBreeding","SwineMarket"
    frac_live <- GLW3_fraction_pigs
  } else if (m %in% c(6,7,16)) { # "ChickensBroilers","ChickensLayers","Turkeys"
    frac_live <- GLW3_fraction_chickens
  } else if (m %in% c(8)) { # "Ducks"
    frac_live <- GLW3_fraction_ducks
  } else if (m %in% c(10)) { # "Horses"
    frac_live <- GLW3_fraction_horses
  } else if (m %in% c(1,3,11,12)) {
    frac_live <- GLW3_fraction_cattle
  }
  # calculate gridded EFCH4_livestock
  country_EFCH4_livestock <- country_EFCH4_2019MT[,m,]
  EFCH4_livestock <- array(NA,c(4320,2160,length(year)))
  for (j in 1:length(year)) {
    tmp_EFCH4_livestock <- array(NA,c(4320,2160))
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        tmp_EFCH4_livestock[index] <- country_EFCH4_livestock[i,j]*1e6*frac_live[index]/1e6 # Gg to kg
      }
    }
    EFCH4_livestock[,,j] <- tmp_EFCH4_livestock
  }
  # save spatial pattern to netcdf file
  fillvalue <- -1e20; threshold <- 1e8
  # define dimensions
  lon <- seq(-180,180,length=4320)
  lat <- seq(90,-90,length=2160)
  londim <- ncdim_def("lon","degrees_east",as.double(lon), longname="Longitude")
  latdim <- ncdim_def("lat","degrees_north",as.double(lat), longname="Latitude")
  tyears <- seq.int(0,(length(year)-1),by=1)
  tunits <- c("year")
  timedim <- ncdim_def("time",tunits,as.double(tyears), unlim=TRUE,
                       create_dimvar=TRUE, calendar="noleap", longname="Time axis")
  # define variables
  dlname <- c("EFCH4_livestock")
  var1.def <-  ncvar_def(dlname[1],"kg CH4 yr-1",list(londim,latdim,timedim),
                         fillvalue,dlname[1],prec="single",compression=5)
  ncfnameout <- paste0(path_out,"EFCH4_livestock_2019MT_spatial_",spatial_cate_name[m],".nc4")
  ncout <- nc_create(ncfnameout,list(var1.def),force_v4=TRUE)
  ncvar_put(ncout,var1.def,EFCH4_livestock)
  ncatt_put(ncout,"lon","axis","X") #,verbose=FALSE) #,definemode=FALSE)
  ncatt_put(ncout,"lat","axis","Y")
  for (k in 1:length(dlname)) {
    ncatt_put(ncout,dlname[k],"missing_value",fillvalue,prec="float")
  }
  ncatt_put(ncout,0,"contact","changjf@zju.edu.cn")
  ncatt_put(ncout,0,"description","Enteric Fermentation CH4 emissions from livestock using 2019MT method")
  ncatt_put(ncout,0,"source","livestock numbers from FAOSTAT, emission factors from 2019 Refinement of IPCC 2006 Guidelines (Combination of Tier 1 and Tier 2), spatial pattern of livestock from GLW3")
  nc_close(ncout)
  # clean temporary variables
  rm(frac_live,EFCH4_livestock,tmp_EFCH4_livestock)
}  

##########################################
##########################################
# 2.4 Calculate Manure Management CH4 emissions from all livestock categories
##########################################
##########################################

## Tier 2 methodology from 2019 Refinement of IPCC Guidelines were used here.
## CH4 emissions from manure management were calculated using Equation 10.23 of 2019Refinement
## EF = (VS * 365) * (B0 * 0.67 * sum(MCF/100*AWMS))
## Parameter used are 
## 1) VS: Volatile solid excreted
## 2) B0: maximum methane producing capacity for manure produced by livestock
## 3) MCF: methane conversion factors
## 4) AWMS: Animal waste management system characteristics

##########################################
# 2.4.1 Climate zone as defined in Chapter 3 of Volume 5 of 2019Refinement (also Annex 10A.2)
##########################################

# See Create_map_FAO_climate_zone.R
# it requires
# 1) mean annual temperature
# 2) mean annual precipitation
# 3) elevation (from HWSD dataset)
# 4) AI / PET (from Trabucco, Antonio; Zomer, Robert (2019): Global Aridity Index and Potential Evapotranspiration (ET0) Climate Database v2. figshare. Fileset. https://doi.org/10.6084/m9.figshare.7504448.v3)
# load climate zone
load(file=paste0(path_out,"map_FAO_climate_zone.RData"))

##########################################
# 2.4.2 Calculate VS
##########################################

# 2.4.2.1 For Dairy cattle, other cattle, goats, sheep, and buffaloes 
# VS was calculated from GE and DE (Equation 10.24 of 2019Refinement), which were also used for calculating EFCH4
# VS = (GE * (1-DE/100) + (UE*GE))*((1-ASH)/18.45)
mean_GE_used <- apply(MC_GE_used,c(2,3,4),mean,na.rm=TRUE)
VS_ruminant <- array(NA,c(6,243,length(year)))
VS_ruminant <- (mean_GE_used * (1 - DE_animal[1:6,,]/100) + (0.04*mean_GE_used)) * ((1 - 0.06)/18.45)

# 2.4.2.2 other livestock
# 2.4.2.2.1 read VSrate
csvname <- paste0(path_param_2019refinement,"Table10.13A.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
regional_VSrate <- sapply(mat[2:11,2:18],as.numeric)
#name_VSrate <- c(Dairy cattle4    Other cattle4    Buffalo4    Swine3    Finishing    Breeding    Poultry3    Hens ±1 yr    Pullets    Broilers    Turkeys8    Ducks8    Sheep3    Goats5    Horses8    Mules/ Asses8    Camels8)

# 2.4.2.2.2 For pigs (Swine) 
# We use Tier 1 method to calculate VS following Equation 10.22A VS = (VSrate * TAM/1000)*365
# VSrate comes from Table 10.13A of 2019Refinement
# VS = VSrate * TAM/1000 * 365
j_Item <- which(Yield.Item.name == "Meat, pig")
pig_yield <- Yield.mat[,j_Item,] # "Meat, pig"
pig_liveweight <- array(NA,c(243,length(year)))
# dressing percentage from GLEAMv2.0 Table 9.2
# 65% for backyard system, 75% for intermediate and industrial system
dress_pig <- array(0.65,c(243))
index <- which(df.crosswalk$Developed == 1)
dress_pig[index] <- 0.75
for (j in 1:length(year)) {
  tmp <- pig_yield[,j]*0.1/dress_pig # Hg to kg
  pig_liveweight[,j] <- tmp
}
j_Item <- which(Stock.Item.name == "Pigs")
pig_stock <- Stock.mat[,j_Item,] # "Pigs"
# set default low productivity system mean liveweight from Table 10.10 (Updated) 2019 Refinement
pig_liveweight[is.na(pig_liveweight) == TRUE] <- 52
TAM_pig <- pig_liveweight/2 # mean weight is half of slaughtered weight
regional_VSrate_pig <- regional_VSrate[,4]
regional_VSrate_pig[2] <- regional_VSrate_pig[4] # Russia
VS_pig <- array(NA,c(243,length(year)))
for (i in 1:243) {
  for (k in 1:10) {
    if (df.crosswalk$GLEAM_Region[i] == k) {
      for (j in 1:length(year)) {
        VS_pig[i,j] <- regional_VSrate_pig[k]*TAM_pig[i,j]/1000*365*pig_stock[i,j] # kg VS yr-1 in a country
      }
    }
  }
}

# 2.4.2.2.3 For poultry, we also use Tier 1 method to calculate VS as done for pigs
# But need to use slaughtered number and do not use 365 days
# default weight of turkey and duck from Table 10A.5 of 2019Refinement
# regional adult females at the start/end of laying period Table 2.20 of GLEAMv2.0
layer_wstart <- c(1.3,1.3,1.6,1.5,1.32,1.5,1.5,1.3,1.4,1.5)
layer_wend <- c(1.5,2.0,1.9,1.9,1.9,1.91,2,1.6,1.6,1.9)
layer_wmean <- (layer_wstart+layer_wend)/2
# eggs per cycle and mean egg weight Table 2.20 of GLEAMv2.0
eggspercycle <- c(279,320,305,297,299,285,300,292,312,271)
meaneggweight <- c(54,57,57,57,49,53,47,52,51,45)

# get yield
j_Item <- which(Yield.Item.name == "Meat, chicken")
broiler_yield <- Yield.mat[,j_Item,] # "Meat, chicken" 0.1g/An
j_Item <- which(Yield.Item.name == "Eggs, hen, in shell")
layer_yield <- Yield.mat[,j_Item,] # "Eggs, hen, in shell" 0.1g/An
j_Item <- which(Yield.Item.name == "Meat, turkey")
turkey_yield <- Yield.mat[,j_Item,] # "Meat, turkey" 0.1g/An
j_Item <- which(Yield.Item.name == "Meat, duck")
duck_yield <- Yield.mat[,j_Item,] # "Meat, duck" 0.1g/An
# calculate mean liveweight
broiler_liveweight <- array(NA,c(243,length(year)))
layer_liveweight <- array(NA,c(243,length(year)))
turkey_liveweight <- array(NA,c(243,length(year)))
duck_liveweight <- array(NA,c(243,length(year)))
# dressing percentage from GLEAMv2.0 Table 9.2
# 55% for layers
dress_broiler <- c(68,70,67,70,67,63,66,68,88,65)/100
for (i in 1:243) {
  for (k in 1:10) {
    if (df.crosswalk$GLEAM_Region[i] == k) {
      for (j in 1:length(year)) {
        broiler_liveweight[i,j] <- broiler_yield[i,j]/dress_broiler[k]/1e4 # 0.1g -> kg 
        layer_liveweight[i,j] <- layer_wmean[k] # original kg
        turkey_liveweight[i,j] <- turkey_yield[i,j]/dress_broiler[k]/1e4 # 0.1g -> kg
        duck_liveweight[i,j] <- duck_yield[i,j]/dress_broiler[k]/1e4 # 0.1g -> kg
      }
    }
  }
}
# Gapfilling
broiler_liveweight[is.na(broiler_liveweight) == TRUE] <- 1.0
layer_liveweight[is.na(layer_liveweight) == TRUE] <- 1.4
turkey_liveweight[is.na(turkey_liveweight) == TRUE] <- 6.8 # Table 10A.5
duck_liveweight[is.na(duck_liveweight) == TRUE] <- 2.7 # Table 10A.5
# calculate TAM
TAM_broiler <- broiler_liveweight/2 # mean weight is half of slaughtered weight
TAM_layer <- layer_liveweight # mean weight is the mean layer weight
TAM_turkey <- turkey_liveweight/2
TAM_duck <- duck_liveweight/2
# calculate VSrate
regional_VSrate_broiler <- regional_VSrate[,10]
regional_VSrate_broiler[2] <- regional_VSrate_broiler[4]
regional_VSrate_layer <- regional_VSrate[,8]
regional_VSrate_layer[2] <- regional_VSrate_layer[4]
regional_VSrate_turkey <- regional_VSrate[,11]
regional_VSrate_turkey[2] <- regional_VSrate_turkey[4]
regional_VSrate_duck <- regional_VSrate[,12]
regional_VSrate_duck[2] <- regional_VSrate_duck[4]
# get stock of broiler, layers, turkey and duck
j_Item <- which(Slaughtered.Item.name == "Meat, chicken")
broiler_slaughtered <- Slaughtered.mat[,j_Item,]*1000 # original 1000 head
j_Item <- which(Slaughtered.Item.name == "Eggs, hen, in shell")
layer_producing <- Slaughtered.mat[,j_Item,]*1000
j_Item <- which(Stock.Item.name == "Chickens")
chicken_stock <- Stock.mat[,j_Item,]*1000
layer_stock <- array(NA,c(243,length(year))) 
for (i in 1:243) {
  for (k in 1:10) {
    if (df.crosswalk$GLEAM_Region[i] == k) {
      for (j in 1:length(year)) {
        # yield / meaneggweight = eggs per layer (actual) 
        # the full cycle year eggspercycle (per year)
        # thus stock = eggs per layer/eggspercycle (unit: yr-1) 
        layer_stock[i,j] <- layer_producing[i,j] * (layer_yield[i,j]/1e4/(meaneggweight[k]/1e3)/eggspercycle[k])
      }
    }
  }
}
layer_stock[is.na(layer_stock) == TRUE] <- 0.0
layer_stock[layer_stock > chicken_stock & is.na(chicken_stock) == FALSE] <- chicken_stock[layer_stock > chicken_stock & is.na(chicken_stock) == FALSE]
broiler_stock <- chicken_stock-layer_stock
broiler_stock[broiler_stock < 0.0] <- 0.0
j_Item <- which(Stock.Item.name == "Turkeys")
turkey_stock <- Stock.mat[,j_Item,]*1000
j_Item <- which(Stock.Item.name == "Ducks")
duck_stock <- Stock.mat[,j_Item,]*1000
# calculate VS
VS_broiler <- array(NA,c(243,length(year)))
VS_layer <- array(NA,c(243,length(year)))
VS_turkey <- array(NA,c(243,length(year)))
VS_duck <- array(NA,c(243,length(year)))
for (i in 1:243) {
  for (k in 1:10) {
    if (df.crosswalk$GLEAM_Region[i] == k) {
      for (j in 1:length(year)) {
        VS_broiler[i,j] <- regional_VSrate_broiler[k]*TAM_broiler[i,j]/1000*365*broiler_stock[i,j] # kg VS yr-1 in a country
        VS_layer[i,j] <- regional_VSrate_layer[k]*TAM_layer[i,j]/1000*365*layer_stock[i,j] # kg VS yr-1 in a country
        VS_turkey[i,j] <- regional_VSrate_turkey[k]*TAM_turkey[i,j]/1000*365*turkey_stock[i,j] # kg VS yr-1 in a country
        VS_duck[i,j] <- regional_VSrate_duck[k]*TAM_duck[i,j]/1000*365*duck_stock[i,j] # kg VS yr-1 in a country
      }
    }
  }
}

# 2.4.2.2.4 For other livestock we use typical liveweight from Table 10A.5 of 2019Refinement
VS_asses <- array(NA,c(243,length(year)))
VS_camels <- array(NA,c(243,length(year)))
VS_horses <- array(NA,c(243,length(year)))
VS_mules <- array(NA,c(243,length(year)))
VS_llamas <- array(NA,c(243,length(year)))
TAM_asses <- rep(130,10)
TAM_camels <- rep(217,10)
TAM_horses <- rep(238,10)
TAM_horses[c(1,2,3,4,7)] <- 377
TAM_mules <- rep(130,10)
TAM_llamas <- rep(65,10) # Table 10.10 of 2019Refinement
regional_VSrate_asses <- regional_VSrate[,16]
regional_VSrate_camels <- regional_VSrate[,17]
regional_VSrate_horses <- regional_VSrate[,15]
regional_VSrate_mules <- regional_VSrate[,16]
regional_VSrate_llamas <- regional_VSrate[,16]

j_Item <- which(Stock.Item.name == "Asses")
asses_stock <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Camels")
camels_stock <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Horses")
horses_stock <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Mules")
mules_stock <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Camelids, other")
llamas_stock <- Stock.mat[,j_Item,] # Camelids, other
for (i in 1:243) {
  for (k in 1:10) {
    if (df.crosswalk$GLEAM_Region[i] == k) {
      for (j in 1:length(year)) {
        VS_asses[i,j] <- regional_VSrate_asses[k]*TAM_asses[k]/1000*365*asses_stock[i,j]
        VS_camels[i,j] <- regional_VSrate_camels[k]*TAM_camels[k]/1000*365*camels_stock[i,j]
        VS_horses[i,j] <- regional_VSrate_horses[k]*TAM_horses[k]/1000*365*horses_stock[i,j]
        VS_mules[i,j] <- regional_VSrate_mules[k]*TAM_mules[k]/1000*365*mules_stock[i,j]
        VS_llamas[i,j] <- regional_VSrate_llamas[k]*TAM_llamas[k]/1000*365*llamas_stock[i,j]
      }
    }
  }
}

VS_animal <- array(NA,c(16,243,length(year)))
VS_animal[1,,] <- VS_asses # "Asses"
VS_animal[2,,] <- VS_ruminant[4,,] # "Buffaloes"
VS_animal[3,,] <- VS_camels # "Camels"
VS_animal[4,,] <- VS_ruminant[5,,] # "Cattle, dairy" 
VS_animal[5,,] <- VS_ruminant[6,,] # "Cattle, non-dairy"
VS_animal[6,,] <- VS_broiler # "Chickens, broilers"
VS_animal[7,,] <- VS_layer # "Chickens, layers"
VS_animal[8,,] <- VS_duck # "Ducks"
VS_animal[9,,] <- VS_ruminant[2,,] # "Goats" 
VS_animal[10,,] <- VS_horses # "Horses"
VS_animal[11,,] <- VS_llamas # "Llamas"
VS_animal[12,,] <- VS_mules # "Mules"
VS_animal[13,,] <- VS_ruminant[3,,] # "Sheep"
VS_animal[14,,] <- 0.0 # "Swine, breeding"
VS_animal[15,,] <- VS_pig # "Swine, market"
VS_animal[16,,] <- VS_turkey # "Turkeys" 

##########################################
# 2.4.3 calculate Manure Management CH4 emissions from all livestock categories
##########################################

# 2.4.3.1 share of High Productivity (HP) in total production system
csvname <- paste0(path_param_2019refinement,"Table10A.5.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
mean_weight <- sapply(mat[2:11,3:22],as.numeric)
HP_weight <- sapply(mat[12:21,3:22],as.numeric)
LP_weight <- sapply(mat[22:31,3:22],as.numeric)
# share of HP and mean can be derived from mean, HP and mean weight
# x * HP + (1-x)*mean = mean
# x = (mean - mean)/(HP-mean)
share_HP <- (mean_weight-LP_weight)/(HP_weight-LP_weight)
share_HP[c(1,2,3,4,7)] <- 1.0

# name_weight <- c(
# Dairy cattle2 Other cattle2   Buffalo2        Swine3  Finishing
# Breeding      Poultry3        Hens >/= 1 yr   Pullets Broilers
# Turkeys4 Ducks4       Sheep3  Goats5  Horses4
# Mules and asses4 Camels4      Ostrich5        Deer5   Reindeer5
###
regional_mean_TAM_asses <- mean_weight[,16]
regional_mean_TAM_buffaloes <- mean_weight[,3]
regional_mean_TAM_camels <- mean_weight[,17]
regional_mean_TAM_cattledairy <- mean_weight[,1]
regional_mean_TAM_cattleother <- mean_weight[,2]
regional_mean_TAM_broiler <- mean_weight[,10]
regional_mean_TAM_layer <- mean_weight[,8]
regional_mean_TAM_duck <- mean_weight[,12]
regional_mean_TAM_goats <- mean_weight[,14]
regional_mean_TAM_horses <- mean_weight[,15]
regional_mean_TAM_llamas <- mean_weight[,16]
regional_mean_TAM_mules <- mean_weight[,16]
regional_mean_TAM_sheep <- mean_weight[,13]
regional_mean_TAM_pig <- mean_weight[,4]
regional_mean_TAM_turkey <- mean_weight[,11]

regional_HP_TAM_asses <- HP_weight[,16]
regional_HP_TAM_buffaloes <- HP_weight[,3]
regional_HP_TAM_camels <- HP_weight[,17]
regional_HP_TAM_cattledairy <- HP_weight[,1]
regional_HP_TAM_cattleother <- HP_weight[,2]
regional_HP_TAM_broiler <- HP_weight[,10]
regional_HP_TAM_layer <- HP_weight[,8]
regional_HP_TAM_duck <- HP_weight[,12]
regional_HP_TAM_goats <- HP_weight[,14]
regional_HP_TAM_horses <- HP_weight[,15]
regional_HP_TAM_llamas <- HP_weight[,16]
regional_HP_TAM_mules <- HP_weight[,16]
regional_HP_TAM_sheep <- HP_weight[,13]
regional_HP_TAM_pig <- HP_weight[,4]
regional_HP_TAM_turkey <- HP_weight[,11]

regional_LP_TAM_asses <- LP_weight[,16]
regional_LP_TAM_buffaloes <- LP_weight[,3]
regional_LP_TAM_camels <- LP_weight[,17]
regional_LP_TAM_cattledairy <- LP_weight[,1]
regional_LP_TAM_cattleother <- LP_weight[,2]
regional_LP_TAM_broiler <- LP_weight[,10]
regional_LP_TAM_layer <- LP_weight[,8]
regional_LP_TAM_duck <- LP_weight[,12]
regional_LP_TAM_goats <- LP_weight[,14]
regional_LP_TAM_horses <- LP_weight[,15]
regional_LP_TAM_mules <- LP_weight[,16]
regional_LP_TAM_llamas <- LP_weight[,16]
regional_LP_TAM_sheep <- LP_weight[,13]
regional_LP_TAM_pig <- LP_weight[,4]
regional_LP_TAM_turkey <- LP_weight[,11]


country_mean_TAM <- array(NA,c(16,243))
country_HP_TAM <- array(NA,c(16,243))
country_LP_TAM <- array(NA,c(16,243))
for (i in 1:243) {
  # region number
  rr <- df.crosswalk$GLEAM_Region[i]
  
  country_mean_TAM[1,i] <- regional_mean_TAM_asses[rr]
  country_mean_TAM[2,i] <- regional_mean_TAM_buffaloes[rr]
  country_mean_TAM[3,i] <- regional_mean_TAM_camels[rr]
  country_mean_TAM[4,i] <- regional_mean_TAM_cattledairy[rr]
  country_mean_TAM[5,i] <- regional_mean_TAM_cattleother[rr]
  country_mean_TAM[6,i] <- regional_mean_TAM_broiler[rr]
  country_mean_TAM[7,i] <- regional_mean_TAM_layer[rr]
  country_mean_TAM[8,i] <- regional_mean_TAM_duck[rr]
  country_mean_TAM[9,i] <- regional_mean_TAM_goats[rr]
  country_mean_TAM[10,i] <- regional_mean_TAM_horses[rr]
  country_mean_TAM[11,i] <- regional_mean_TAM_llamas[rr]
  country_mean_TAM[12,i] <- regional_mean_TAM_mules[rr]
  country_mean_TAM[13,i] <- regional_mean_TAM_sheep[rr]
  country_mean_TAM[14,i] <- regional_mean_TAM_pig[rr]
  country_mean_TAM[15,i] <- regional_mean_TAM_pig[rr]
  country_mean_TAM[16,i] <- regional_mean_TAM_turkey[rr]
  
  country_HP_TAM[1,i] <- regional_HP_TAM_asses[rr]
  country_HP_TAM[2,i] <- regional_HP_TAM_buffaloes[rr]
  country_HP_TAM[3,i] <- regional_HP_TAM_camels[rr]
  country_HP_TAM[4,i] <- regional_HP_TAM_cattledairy[rr]
  country_HP_TAM[5,i] <- regional_HP_TAM_cattleother[rr]
  country_HP_TAM[6,i] <- regional_HP_TAM_broiler[rr]
  country_HP_TAM[7,i] <- regional_HP_TAM_layer[rr]
  country_HP_TAM[8,i] <- regional_HP_TAM_duck[rr]
  country_HP_TAM[9,i] <- regional_HP_TAM_goats[rr]
  country_HP_TAM[10,i] <- regional_HP_TAM_horses[rr]
  country_HP_TAM[11,i] <- regional_HP_TAM_llamas[rr]
  country_HP_TAM[12,i] <- regional_HP_TAM_mules[rr]
  country_HP_TAM[13,i] <- regional_HP_TAM_sheep[rr]
  country_HP_TAM[14,i] <- regional_HP_TAM_pig[rr]
  country_HP_TAM[15,i] <- regional_HP_TAM_pig[rr]
  country_HP_TAM[16,i] <- regional_HP_TAM_turkey[rr]
  
  country_LP_TAM[1,i] <- regional_LP_TAM_asses[rr]
  country_LP_TAM[2,i] <- regional_LP_TAM_buffaloes[rr]
  country_LP_TAM[3,i] <- regional_LP_TAM_camels[rr]
  country_LP_TAM[4,i] <- regional_LP_TAM_cattledairy[rr]
  country_LP_TAM[5,i] <- regional_LP_TAM_cattleother[rr]
  country_LP_TAM[6,i] <- regional_LP_TAM_broiler[rr]
  country_LP_TAM[7,i] <- regional_LP_TAM_layer[rr]
  country_LP_TAM[8,i] <- regional_LP_TAM_duck[rr]
  country_LP_TAM[9,i] <- regional_LP_TAM_goats[rr]
  country_LP_TAM[10,i] <- regional_LP_TAM_horses[rr]
  country_LP_TAM[11,i] <- regional_LP_TAM_llamas[rr]
  country_LP_TAM[12,i] <- regional_LP_TAM_mules[rr]
  country_LP_TAM[13,i] <- regional_LP_TAM_sheep[rr]
  country_LP_TAM[14,i] <- regional_LP_TAM_pig[rr]
  country_LP_TAM[15,i] <- regional_LP_TAM_pig[rr]
  country_LP_TAM[16,i] <- regional_LP_TAM_turkey[rr]
}

country_share_HP <- array(NA,c(16,243))
country_share_HP <- (country_mean_TAM-country_LP_TAM)/(country_HP_TAM-country_LP_TAM)
# if a country/area has the same weight for HP and LP, we first set the share to NA
country_share_HP[which(country_HP_TAM == country_LP_TAM)] <- NA
for (m in 1:16) {
  for (i in 1:243) {
    if (is.na(country_share_HP[m,i]) == TRUE & df.crosswalk$Developed[i] == 1) {
      country_share_HP[m,i] <- 1.0
    } else if (is.na(country_share_HP[m,i]) == TRUE & df.crosswalk$Developed[i] == 0) {
      country_share_HP[m,i] <- 0.0
    }
  }
}
# ducks share the same as broiler
country_share_HP[8,] <- country_share_HP[6,]
# Turkeys all as HP

# 2.4.3.2 load parameters 
# B0 from Table 10.16 of 2019Refinement for each region, each livestock category
csvname <- paste0(path_param_2019refinement,"Table10.16.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
regional_B0 <- sapply(mat[2:11,2:13],as.numeric)
# region (1:10), livestock category (1:12)
#Dairy cattle    Non dairy cattle    Buffalo    Swine    Chicken-Layer    Chicken-Broilers    Sheep    Goats    Horses    Mules/ Asses    Camels    All Animals PRP

# MCF from Table 10.17 of 2019Refinement for each manure management system in each climate zone
csvname <- paste0(path_param_2019refinement,"Table10.17.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
regional_MCF <- sapply(mat[2:29,3:12],as.numeric)
# manure management systems, climate zone (1:10)

#Cool Temperate Moist    Cool Temperate Dry    Boreal Moist    Boreal Dry    Warm Temperate Moist    Warm Temperate Dry    Tropical Montane    Tropical Wet    Tropical Moist    Tropical Dry

# AWMS from Table 10A.6 to 10A.9 of 2019Refinement for each livestock category
csvname <- paste0(path_param_2019refinement,"Table10A.6.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_dairycattle <- sapply(mat[2:11,3:11],as.numeric)
AWMS_dairycattle[2,] <- AWMS_dairycattle[4,]
AWMS_othercattle <- sapply(mat[12:21,3:11],as.numeric)
AWMS_othercattle[2,] <- AWMS_othercattle[4,]
AWMS_dairybuffalo <- sapply(mat[22:31,3:11],as.numeric)
AWMS_dairybuffalo[2,] <- AWMS_dairybuffalo[4,]
AWMS_dairybuffalo[7,] <- AWMS_dairybuffalo[1,]
AWMS_dairybuffalo[10,] <- AWMS_dairybuffalo[5,]
AWMS_otherbuffalo <- sapply(mat[32:41,3:11],as.numeric)
AWMS_otherbuffalo[1,] <- AWMS_dairybuffalo[1,]
AWMS_otherbuffalo[3,] <- AWMS_dairybuffalo[3,]
AWMS_otherbuffalo[7,] <- AWMS_dairybuffalo[7,]
AWMS_otherbuffalo[10,] <- AWMS_otherbuffalo[5,]
#Lagoon    "Liquid/Slurry"    "Solid storage"    Drylot    "Pasture/Range/ Paddock"    "Daily spread"    Digester    "Burned for fuel"    Other

csvname <- paste0(path_param_2019refinement,"Table10A.7.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_growingswine_High <- sapply(mat[2:11,3:11],as.numeric)
AWMS_growingswine_Low <- sapply(mat[12:21,3:11],as.numeric)
AWMS_breedingswine_High <- sapply(mat[22:31,3:11],as.numeric)
AWMS_breedingswine_Low <- sapply(mat[32:41,3:11],as.numeric)
#Lagoon    Liquid/Slurry    Solid storage    Drylot    "Pit <1"    Pit > 1    Daily spread    Digester    Pasture
AWMS_growingswine_Low[c(1,2,3,4,7),] <- AWMS_growingswine_High[c(1,2,3,4,7),]
AWMS_breedingswine_Low[c(1,2,3,4,7),] <- AWMS_breedingswine_High[c(1,2,3,4,7),]

csvname <- paste0(path_param_2019refinement,"Table10A.8.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_othersheep <- sapply(mat[2:11,3:11],as.numeric)
AWMS_othersheep[2,] <- AWMS_othersheep[4,]
AWMS_dairysheep <- sapply(mat[12:21,3:11],as.numeric)
AWMS_dairysheep[1,] <- AWMS_dairysheep[3,]
AWMS_dairysheep[2,] <- AWMS_dairysheep[4,]
AWMS_dairysheep[7,] <- AWMS_dairysheep[3,]
AWMS_goats <- sapply(mat[22:31,3:11],as.numeric)
#Lagoon    Liquid/Slurry    Solid storage    Drylot    Pasture/Range/ Paddock    Daily spread    Digester    Burned for fuel    Other

csvname <- paste0(path_param_2019refinement,"Table10A.9.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_layer <- sapply(mat[2:11,3:11],as.numeric)
AWMS_broiler_High <- sapply(mat[12:21,3:11],as.numeric)
AWMS_broiler_Low <- sapply(mat[22:31,3:11],as.numeric)
#Lagoon    "Liquid/Slurry"    Solid storage    Drylot    "Pasture/Range/Paddock"    Pit >1 month    Daily spread    Digester    Other (Poultry manure with litter)

# 2.4.3.3 calculate spatially explicit Manure Management CH4 emissions from all livestock categories
# EF = (VS * 365) * (B0 * 0.67 * sum(MCF/100*AWMS))
## Parameter used are
## 1) VS: Volatile solid excreted
## 2) B0: maximum methane producing capacity for manure produced by livestock (B0 is region and livestock specific)
## 3) MCF: methane conversion factors (manure management system and climate zone specific)
## 4) AWMS: Animal waste management system characteristics (region and manure management system specific)
#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 
spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")

country_MMCH4_2019MT <- array(NA,c(243,16,length(year)))

for (m in c(1:16)) {
  # spatial pattern of gridded livestock fraction within each country/area
  print(spatial_cate_name[m])
  if (m %in% c(4,5)) { # "CattleDairy","CattleNonDairy"
    frac_live <- GLW3_fraction_cattle
  } else if (m %in% c(13)) { # "Sheep" 
    frac_live <- GLW3_fraction_sheep
  } else if (m %in% c(9)) { # "Goats"
    frac_live <- GLW3_fraction_goats
  } else if (m %in% c(2)) { # Buffaloes"
    frac_live <- GLW3_fraction_buffaloes
  } else if (m %in% c(14,15)) { # "SwineBreeding","SwineMarket"
    frac_live <- GLW3_fraction_pigs
  } else if (m %in% c(6,7,16)) { # "ChickensBroilers","ChickensLayers","Turkeys"
    frac_live <- GLW3_fraction_chickens
  } else if (m %in% c(8)) { # "Ducks"
    frac_live <- GLW3_fraction_ducks
  } else if (m %in% c(10)) { # "Horses"
    frac_live <- GLW3_fraction_horses
  } else if (m %in% c(1,3,11,12)) {
    frac_live <- GLW3_fraction_cattle
  }
  
  # spread AWMS to grid level
  # lon,lat,livestock category, AWMS
  AWMS_grid_HP <- array(NA,c(4320,2160,9))
  AWMS_grid_LP <- array(NA,c(4320,2160,9))
  AWMS_regional_tmp <- array(NA,c(10,9))
  AWMS_regional_tmp_low <- array(NA,c(10,9))
  if (m %in% c(4)) { # "CattleDairy"
    AWMS_regional_tmp <- AWMS_dairycattle 
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(5)) { # "CattleNonDairy"
    AWMS_regional_tmp <- AWMS_othercattle
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(13)) { # "Sheep"
    AWMS_regional_tmp <- AWMS_othersheep
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(9)) { # "Goats"
    AWMS_regional_tmp <- AWMS_goats
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(2)) { # "Buffaloes"
    AWMS_regional_tmp <- AWMS_otherbuffalo
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(14)) { # "SwineBreeding"
    AWMS_regional_tmp <- AWMS_breedingswine_High
    AWMS_regional_tmp_low <- AWMS_breedingswine_Low
  } else if (m %in% c(15)) { # "SwineMarket"
    AWMS_regional_tmp <- AWMS_growingswine_High
    AWMS_regional_tmp_low <- AWMS_growingswine_Low
  } else if (m %in% c(6,8,16)) { # "ChickensBroilers","Ducks","Turkeys"
    AWMS_regional_tmp <- AWMS_broiler_High
    AWMS_regional_tmp_low <- AWMS_broiler_Low
  } else if (m %in% c(7)) { # "ChickensLayers"
    AWMS_regional_tmp <- AWMS_layer 
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(1,3,10,11,12)) { #"Asses","Camels","Horses","Llamas","Mules"
    # footnote of Table 10A.9 of 2019 Refinement
    AWMS_regional_tmp <- AWMS_goats
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  }  
  # spread, be careful of the high low
  for (w in 1:9) {
    print(paste("AWMS",w))
    tmp_map_AWMS_HP <- array(NA,c(4320,2160))
    tmp_map_AWMS_LP <- array(NA,c(4320,2160))
    
    for (i in 1:243) {
      for (k in 1:10) {
        if (df.crosswalk$GLEAM_Region[i] == k) {
          index <- which(regionid_8km == i)
          if (length(index) >= 1) {
            tmp_map_AWMS_HP[index] <- AWMS_regional_tmp[k,w]
            tmp_map_AWMS_LP[index] <- AWMS_regional_tmp_low[k,w]
          }
        }
      }
    }
    AWMS_grid_HP[,,w] <- tmp_map_AWMS_HP
    AWMS_grid_LP[,,w] <- tmp_map_AWMS_LP
  }
  # clean temporary variables
  rm(tmp_map_AWMS_HP,tmp_map_AWMS_LP)
  
  # spread MCF to grid level
  # lon,lat,livestock category, AWMS
  MCF_grid <- array(NA,c(4320,2160,9))
  # See the footnote of Table 10.17 for the default value that should be used
  # Lagoon: footnote 7, 12 months assumed (No. 1)
  # Liquid/Slurry: footnote 1, 6 months (No. 5)
  # Solid storage: (No. 9)
  # Drylot: (No. 13)
  # Pasture/Range/Paddock: (No. 19)
  # Daily spread: (No. 14)
  # Digester: (No. 26) Considering they are mainly for Swine manure and for low-income countries, we consider it to be High leakage, low quality technology, high quanlity gastight storage technology, which gives a value of 9.59% median value for different digesters
  # Burned for fuel: (No. 22)
  # Other: assume 0.0? as Aerobic treatment? assume as Pasture/Range/Paddock
  # Other (Poultry manure with litter): (No. 20)
  
  order_regional_MCF <- array(NA,c(9,10)) # 9 AWMS, 10 climate zone NOTE: not region
  if (m %in% c(2,4,5)) { # "Buffaloes","CattleDairy","CattleNonDairy"
    #Lagoon    "Liquid/Slurry"    "Solid storage"    Drylot    "Pasture/Range/ Paddock"    "Daily spread"    Digester    "Burned for fuel"    Other
    order_regional_MCF[1:9,1:10] <- regional_MCF[c(1,5,9,13,19,14,26,22,19),1:10]
  } else if (m %in% c(14,15)) { # "SwineBreeding","SwineMarket"
    #Lagoon    Liquid/Slurry    Solid storage    Drylot    "Pit <1"    Pit > 1    Daily spread    Digester    Pasture
    order_regional_MCF[1:9,1:10] <- regional_MCF[c(1,5,9,13,2,5,14,26,19),1:10]
  } else if (m %in% c(6,7,8,16)) { # "ChickensBroilers","ChickensLayers","Ducks","Turkeys" 
    #Lagoon    "Liquid/Slurry"    Solid storage    Drylot    "Pasture/Range/Paddock"    Pit >1 month    Daily spread    Digester    Other (Poultry manure with litter)
    order_regional_MCF[1:9,1:10] <- regional_MCF[c(1,5,9,13,19,5,14,26,20),1:10]
  } else if (m %in% c(1,3,9,10,11,12,13)) { # "Asses","Camels","Goats","Horses","Mules","Sheep","Llamas"
    #Lagoon    Liquid/Slurry    Solid storage    Drylot    Pasture/Range/ Paddock    Daily spread    Digester    Burned for fuel    Other
    order_regional_MCF[1:9,1:10] <- regional_MCF[c(1,5,9,13,19,14,26,22,19),1:10]
  }
  
  for (z in 1:9) {
    print(paste("MCF",z))
    tmp_MCF_grid <- array(NA,c(4320,2160))
    for (qq in 1:10) { # FAO climate zone
      index <- which(map_FAO_climate_zone == qq)
      if (length(index) >= 1) {  
        tmp_MCF_grid[index] <- order_regional_MCF[z,qq]
      }
    }
    MCF_grid[,,z] <- tmp_MCF_grid
  }
  
  # spread B0 to grid level
  B0_grid <- array(NA,c(4320,2160))
  if (m %in% c(4)) { # Dairy cattle
    order_regional_B0 <- regional_B0[1:10,1]
  } else if (m %in% c(5)) { # Non dairy cattle
    order_regional_B0 <- regional_B0[1:10,2] 
  } else if (m %in% c(2)) { # Buffalo
    order_regional_B0 <- regional_B0[1:10,3]
  } else if (m %in% c(14,15)) { # Swine
    order_regional_B0 <- regional_B0[1:10,4]
  } else if (m %in% c(7)) { # Chicken-Layer
    order_regional_B0 <- regional_B0[1:10,5]
  } else if (m %in% c(6,8,16)) { # Chicken-Broilers
    order_regional_B0 <- regional_B0[1:10,6]
  } else if (m %in% c(13)) { # Sheep
    order_regional_B0 <- regional_B0[1:10,7]
  } else if (m %in% c(9)) { # Goats
    order_regional_B0 <- regional_B0[1:10,8]
  } else if (m %in% c(10)) { # Horses
    order_regional_B0 <- regional_B0[1:10,9]
  } else if (m %in% c(1,12)) { # Mules/ Asses
    order_regional_B0 <- regional_B0[1:10,10]
  } else if (m %in% c(3)) { # Camels
    order_regional_B0 <- regional_B0[1:10,11]
  } else if (m %in% c(11)) { # All Animals PRP
    order_regional_B0 <- regional_B0[1:10,12]
  }
  for (i in 1:243) {
    for (k in 1:10) {
      if (df.crosswalk$GLEAM_Region[i] == k) {
        index <- which(regionid_8km == i)
        if (length(index) >= 1) {
          B0_grid[index] <- order_regional_B0[k]
        }
      }
    }
  }
  # set empty value to zero
  AWMS_grid_HP[is.na(AWMS_grid_HP) == TRUE | is.finite(AWMS_grid_HP) == FALSE | AWMS_grid_HP < 0.0] <- 0.0
  AWMS_grid_LP[is.na(AWMS_grid_LP) == TRUE | is.finite(AWMS_grid_LP) == FALSE | AWMS_grid_LP < 0.0] <- 0.0
  MCF_grid[is.na(MCF_grid) == TRUE | is.finite(MCF_grid) == FALSE | MCF_grid < 0.0] <- 0.0
  B0_grid[is.na(B0_grid) == TRUE | is.finite(B0_grid) == FALSE | B0_grid < 0.0] <- 0.0
  # spread VS to grid level
  # lon,lat,livestock category,year
  VS_grid <- array(NA,c(4320,2160,length(year)))
  for (j in 1:length(year)) {
    print(paste("VS",j))
    VS_tmp <- array(NA,c(4320,2160))
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        VS_tmp[index] <- VS_animal[m,i,j]*frac_live[index]/1e6 # kg # frac_XXX needs to be /1e6 for calculation
      }
    }
    VS_grid[,,j] <- VS_tmp
  }
  # calculate EF
  # EF = (VS * 365) * (B0 * 0.67 * sum(MCF/100*AWMS))
  MMCH4_livestock <- array(NA,c(4320,2160,length(year))) 
  # footnote 2 of the Table 10.17 Pasture Range and Paddock MCFs must always be used in conjunction with a B0 value of 0.19 m3 CH4 kg-1 of VS excreted to maintain consistency with the data in the in updated version of Cai et al. (2017) database (see Annex 10B.6)
  for (j in 1:length(year)) {
    print(paste("VS",j))
    # spread VS to grid level
    # lon,lat,livestock category,year
    VS_grid_HP <- array(NA,c(4320,2160))
    VS_grid_LP <- array(NA,c(4320,2160))
    # Country level VS
    #VS_animal_HP_LP <- array(NA,c(2,16,243,length(year)))
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        VS_grid_HP[index] <- VS_animal[m,i,j]*country_share_HP[m,i]*frac_live[index]/1e6 # kg # frac_XXX needs to be /1e6 for calculation
        VS_grid_LP[index] <- VS_animal[m,i,j]*(1-country_share_HP[m,i])*frac_live[index]/1e6 # kg # frac_XXX needs to be /1e6 for calculation
      }
    }
    VS_grid_HP[is.na(VS_grid_HP) == TRUE | is.finite(VS_grid_HP) == FALSE | VS_grid_HP < 0.0] <- 0.0
    VS_grid_LP[is.na(VS_grid_LP) == TRUE | is.finite(VS_grid_LP) == FALSE | VS_grid_LP < 0.0] <- 0.0
    
    ######
    # calculate final emissions
    print(paste("MMCH4",j))
    tmp_MMCH4_livestock <- array(0.0,c(4320,2160))
    for (z in 1:9) {
      tmp_B0_grid <- B0_grid
      # Force B0 for Pasture Range and Paddock
      if (z == 5) {
        tmp_B0_grid <- array(0.19,c(4320,2160))
      } 
      tmp_MCF_grid <- MCF_grid[,,z]
      tmp_MMCH4_livestock <- tmp_MMCH4_livestock + VS_grid_HP * tmp_B0_grid * 0.67 * tmp_MCF_grid/100 * AWMS_grid_HP[,,z]
      tmp_MMCH4_livestock <- tmp_MMCH4_livestock + VS_grid_LP * tmp_B0_grid * 0.67 * tmp_MCF_grid/100 * AWMS_grid_LP[,,z]
      
    }
    MMCH4_livestock[,,j] <- tmp_MMCH4_livestock 
  }
  # clean temporary variables
  rm(frac_live,AWMS_grid_HP,AWMS_grid_LP,tmp_MCF_grid,MCF_grid,B0_grid,VS_grid,VS_grid_HP,VS_grid_LP,tmp_MMCH4_livestock)
  
  for (j in 1:length(year)) {
    tmp <- MMCH4_livestock[,,j]
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        country_MMCH4_2019MT[i,m,j] <- sum(tmp[index],na.rm=TRUE)/1e6 # kg to Gg
      }
    }
  }
  
  # save spatial pattern to netcdf file
  fillvalue <- -1e20; threshold <- 1e8
  # define dimensions
  lon <- seq(-180,180,length=4320)
  lat <- seq(90,-90,length=2160)
  londim <- ncdim_def("lon","degrees_east",as.double(lon), longname="Longitude")
  latdim <- ncdim_def("lat","degrees_north",as.double(lat), longname="Latitude")
  tyears <- seq.int(0,(length(year)-1),by=1)
  tunits <- c("year")
  timedim <- ncdim_def("time",tunits,as.double(tyears), unlim=TRUE,
                       create_dimvar=TRUE, calendar="noleap", longname="Time axis")
  # define variables
  dlname <- c("MMCH4_livestock")
  var1.def <-  ncvar_def(dlname[1],"kg CH4 yr-1",list(londim,latdim,timedim),
                         fillvalue,dlname[1],prec="single",compression=5)
  ncfnameout <- paste0(path_out,"MMCH4_livestock_2019MT_spatial_",spatial_cate_name[m],".nc4")
  ncout <- nc_create(ncfnameout,list(var1.def),force_v4=TRUE)
  ncvar_put(ncout,var1.def,MMCH4_livestock)
  ncatt_put(ncout,"lon","axis","X") #,verbose=FALSE) #,definemode=FALSE)
  ncatt_put(ncout,"lat","axis","Y")
  for (k in 1:length(dlname)) {
    ncatt_put(ncout,dlname[k],"missing_value",fillvalue,prec="float")
  }
  ncatt_put(ncout,0,"contact","changjf@zju.edu.cn")
  ncatt_put(ncout,0,"description","Manure Management CH4 emissions from livestock using 2019MT method")
  ncatt_put(ncout,0,"source","livestock numbers from FAOSTAT, emission factors from 2019 Refinement of IPCC 2006 Guidelines (Combination of Tier 1 and Tier 2), spatial pattern of livestock from GLW3")
  nc_close(ncout)
  # clean temporary variables
  rm(MMCH4_livestock)
}

spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")
regional_MMCH4_2019MT <- array(NA,c(11,16,length(year)))
for (i in 1:16) {
  for (k in 1:10) {
    if (k == 2) {
      regional_MMCH4_2019MT[k,,] <- country_MMCH4_2019MT[gleam.list[[k]],,]
    } else {
      regional_MMCH4_2019MT[k,,] <- apply(country_MMCH4_2019MT[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
    }
  }
  k <- 11
  regional_MMCH4_2019MT[k,,] <- apply(country_MMCH4_2019MT[,,],c(2,3),sum,na.rm=TRUE)
}

##########################################
# save output of Manure Management CH4 emissions using the 2019 MT method
##########################################
save.image(file=paste0(path_out,"MMCH4_2019MT.RData"))
# clear cache 
rm(list=ls())
gc()

##########################################
##########################################
# Part III Calculate livestock CH4 emissions using the 2019T1 and T1a method
##########################################
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

# Load FAOSTAT statistics on livestock
load(file=paste0(path_out,"FAOSTAT_Livestock_GDP.RData"))

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set parameter path
path_param_gleam <- paste0(path_data,"GLEAMv2_PARAM/")
path_param_dress <- paste0(path_data,"Opio2013_PARAM/")
path_param_2019refinement <- paste0(path_data,"2019Refinement_PARAM/")
path_param_GLW3 <- paste0(path_data,"GLW3/")
# Load library
library(ncdf4)

##########################################
##########################################
# 3.1 Read livestock distribution maps for spatially explicit calculation
##########################################
##########################################

##########################################
# 3.1.1 read GLW3 livestock distribution at 5 arc min resolution
##########################################

# livestock categories that GLW3 provide livestock distribution for the year 2010
live_cate <- c("cattle","sheep","goats","buffaloes",
               "pigs","chickens","ducks","horses")
live_abb <- c("Ct","Sh","Gt","Bf","Pg","Ch","Dk","Ho")
for (m in 1:length(live_cate)) {
  ncfname <- paste0(path_param_GLW3,"GLW3_livestock_fraction_",live_cate[m],".nc4")
  ncin <- nc_open(ncfname)
  tmp <- ncvar_get(ncin,"fraction")
  nc_close(ncin)
  assign(paste0("GLW3_fraction_",live_cate[m]),tmp)
}

##########################################
# 3.1.2 read country/area map at 5 arc min resolution
##########################################
path_map <- paste0(path_data,"Input_MAP/")
## get 243 country/area name and country id
ncfname <- paste(path_map,"regionname_world_country_8km.nc",sep="")
ncin <- nc_open(ncfname)
regionid_8km <- ncvar_get(ncin,"regionid")
area_8km <- ncvar_get(ncin,"area")
nc_close(ncin)

##########################################
# 3.1.3 Read 243 country information GLEAM region (FAO)
##########################################

region.name <- c("N. America","Russian Fed.","W. Europe","E. Europe","NENA","E & SE Asia","Oceania","South Asia","LAC","SSA")
for (i in 1:10) {
  assign(paste0("r",i), which(df.crosswalk$GLEAM_Region == i))
}
gleam.list <- list(r1,r2,r3,r4,r5,r6,r7,r8,r9,r10)

##########################################
##########################################
# 3.2 Read Stocks from FAOSTAT
##########################################
##########################################

#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 

# Read livestock numbers
livenumber_cate <- array(NA,c(243,16,length(year)))
j_Item <- which(Stock.Item.name == "Asses")
livenumber_cate[,1,] <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Buffaloes")
livenumber_cate[,2,] <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Camels")
livenumber_cate[,3,] <- Stock.mat[,j_Item,]
j_Item <- which(Slaughtered.Item.name == "Milk, whole fresh cow")
livenumber_cate[,4,] <- Slaughtered.mat[,j_Item,]
# Other cattle = Cattle - milking cows
tmp_cow <- Slaughtered.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Cattle")
tmp_cattle <- Stock.mat[,j_Item,]
livenumber_cate[,5,] <- tmp_cattle-tmp_cow
j_Item <- which(Stock.Item.name == "Chickens")
livenumber_cate[,6,] <- Stock.mat[,j_Item,]*1000
j_Item <- which(Stock.Item.name == "Chickens")
livenumber_cate[,7,] <- 0.0
j_Item <- which(Stock.Item.name == "Ducks")
livenumber_cate[,8,] <- Stock.mat[,j_Item,]*1000
j_Item <- which(Stock.Item.name == "Goats")
livenumber_cate[,9,] <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Horses")
livenumber_cate[,10,] <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Camelids, other")
livenumber_cate[,11,] <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Mules")
livenumber_cate[,12,] <- Stock.mat[,j_Item,]
j_Item <- which(Stock.Item.name == "Sheep")
livenumber_cate[,13,] <- Stock.mat[,j_Item,]
# Breeding swine is assumed to be 10% of total pigs
j_Item <- which(Stock.Item.name == "Pigs")
livenumber_cate[,14,] <- Stock.mat[,j_Item,]*0.1
j_Item <- which(Stock.Item.name == "Pigs")
livenumber_cate[,15,] <- Stock.mat[,j_Item,]*0.9
j_Item <- which(Stock.Item.name == "Turkeys")
livenumber_cate[,16,] <- Stock.mat[,j_Item,]*1000

##########################################
##########################################
# 3.3 Calculate Enteric Fermentation CH4 emissions from all livestock categories
##########################################
##########################################

##########################################
# 3.3.1 share of High Productivity (HP) in total production system
##########################################
csvname <- paste0(path_param_2019refinement,"Table10A.5.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
mean_weight <- sapply(mat[2:11,3:22],as.numeric)
HP_weight <- sapply(mat[12:21,3:22],as.numeric)
LP_weight <- sapply(mat[22:31,3:22],as.numeric)
# share of HP and mean can be derived from mean, HP and mean weight
# x * HP + (1-x)*mean = mean
# x = (mean - mean)/(HP-mean)
share_HP <- (mean_weight-LP_weight)/(HP_weight-LP_weight)
share_HP[c(1,2,3,4,7)] <- 1.0

# name_weight <- c(
# Dairy cattle2 Other cattle2   Buffalo2        Swine3  Finishing
# Breeding      Poultry3        Hens >/= 1 yr   Pullets Broilers
# Turkeys4 Ducks4       Sheep3  Goats5  Horses4
# Mules and asses4 Camels4      Ostrich5        Deer5   Reindeer5
regional_mean_TAM_asses <- mean_weight[,16]
regional_mean_TAM_buffaloes <- mean_weight[,3]
regional_mean_TAM_camels <- mean_weight[,17]
regional_mean_TAM_cattledairy <- mean_weight[,1]
regional_mean_TAM_cattleother <- mean_weight[,2]
regional_mean_TAM_broiler <- mean_weight[,10]
regional_mean_TAM_layer <- mean_weight[,8]
regional_mean_TAM_duck <- mean_weight[,12]
regional_mean_TAM_goats <- mean_weight[,14]
regional_mean_TAM_horses <- mean_weight[,15]
regional_mean_TAM_llamas <- mean_weight[,16]
regional_mean_TAM_mules <- mean_weight[,16]
regional_mean_TAM_sheep <- mean_weight[,13]
regional_mean_TAM_pig <- mean_weight[,4]
regional_mean_TAM_turkey <- mean_weight[,11]

regional_HP_TAM_asses <- HP_weight[,16]
regional_HP_TAM_buffaloes <- HP_weight[,3]
regional_HP_TAM_camels <- HP_weight[,17]
regional_HP_TAM_cattledairy <- HP_weight[,1]
regional_HP_TAM_cattleother <- HP_weight[,2]
regional_HP_TAM_broiler <- HP_weight[,10]
regional_HP_TAM_layer <- HP_weight[,8]
regional_HP_TAM_duck <- HP_weight[,12]
regional_HP_TAM_goats <- HP_weight[,14]
regional_HP_TAM_horses <- HP_weight[,15]
regional_HP_TAM_llamas <- HP_weight[,16]
regional_HP_TAM_mules <- HP_weight[,16]
regional_HP_TAM_sheep <- HP_weight[,13]
regional_HP_TAM_pig <- HP_weight[,4]
regional_HP_TAM_turkey <- HP_weight[,11]

regional_LP_TAM_asses <- LP_weight[,16]
regional_LP_TAM_buffaloes <- LP_weight[,3]
regional_LP_TAM_camels <- LP_weight[,17]
regional_LP_TAM_cattledairy <- LP_weight[,1]
regional_LP_TAM_cattleother <- LP_weight[,2]
regional_LP_TAM_broiler <- LP_weight[,10]
regional_LP_TAM_layer <- LP_weight[,8]
regional_LP_TAM_duck <- LP_weight[,12]
regional_LP_TAM_goats <- LP_weight[,14]
regional_LP_TAM_horses <- LP_weight[,15]
regional_LP_TAM_mules <- LP_weight[,16]
regional_LP_TAM_llamas <- LP_weight[,16]
regional_LP_TAM_sheep <- LP_weight[,13]
regional_LP_TAM_pig <- LP_weight[,4]
regional_LP_TAM_turkey <- LP_weight[,11]


country_mean_TAM <- array(NA,c(16,243))
country_HP_TAM <- array(NA,c(16,243))
country_LP_TAM <- array(NA,c(16,243))
for (i in 1:243) {
  # region number
  rr <- df.crosswalk$GLEAM_Region[i]
  
  country_mean_TAM[1,i] <- regional_mean_TAM_asses[rr]
  country_mean_TAM[2,i] <- regional_mean_TAM_buffaloes[rr]
  country_mean_TAM[3,i] <- regional_mean_TAM_camels[rr]
  country_mean_TAM[4,i] <- regional_mean_TAM_cattledairy[rr]
  country_mean_TAM[5,i] <- regional_mean_TAM_cattleother[rr]
  country_mean_TAM[6,i] <- regional_mean_TAM_broiler[rr]
  country_mean_TAM[7,i] <- regional_mean_TAM_layer[rr]
  country_mean_TAM[8,i] <- regional_mean_TAM_duck[rr]
  country_mean_TAM[9,i] <- regional_mean_TAM_goats[rr]
  country_mean_TAM[10,i] <- regional_mean_TAM_horses[rr]
  country_mean_TAM[11,i] <- regional_mean_TAM_llamas[rr]
  country_mean_TAM[12,i] <- regional_mean_TAM_mules[rr]
  country_mean_TAM[13,i] <- regional_mean_TAM_sheep[rr]
  country_mean_TAM[14,i] <- regional_mean_TAM_pig[rr]
  country_mean_TAM[15,i] <- regional_mean_TAM_pig[rr]
  country_mean_TAM[16,i] <- regional_mean_TAM_turkey[rr]
  
  country_HP_TAM[1,i] <- regional_HP_TAM_asses[rr]
  country_HP_TAM[2,i] <- regional_HP_TAM_buffaloes[rr]
  country_HP_TAM[3,i] <- regional_HP_TAM_camels[rr]
  country_HP_TAM[4,i] <- regional_HP_TAM_cattledairy[rr]
  country_HP_TAM[5,i] <- regional_HP_TAM_cattleother[rr]
  country_HP_TAM[6,i] <- regional_HP_TAM_broiler[rr]
  country_HP_TAM[7,i] <- regional_HP_TAM_layer[rr]
  country_HP_TAM[8,i] <- regional_HP_TAM_duck[rr]
  country_HP_TAM[9,i] <- regional_HP_TAM_goats[rr]
  country_HP_TAM[10,i] <- regional_HP_TAM_horses[rr]
  country_HP_TAM[11,i] <- regional_HP_TAM_llamas[rr]
  country_HP_TAM[12,i] <- regional_HP_TAM_mules[rr]
  country_HP_TAM[13,i] <- regional_HP_TAM_sheep[rr]
  country_HP_TAM[14,i] <- regional_HP_TAM_pig[rr]
  country_HP_TAM[15,i] <- regional_HP_TAM_pig[rr]
  country_HP_TAM[16,i] <- regional_HP_TAM_turkey[rr]
  
  country_LP_TAM[1,i] <- regional_LP_TAM_asses[rr]
  country_LP_TAM[2,i] <- regional_LP_TAM_buffaloes[rr]
  country_LP_TAM[3,i] <- regional_LP_TAM_camels[rr]
  country_LP_TAM[4,i] <- regional_LP_TAM_cattledairy[rr]
  country_LP_TAM[5,i] <- regional_LP_TAM_cattleother[rr]
  country_LP_TAM[6,i] <- regional_LP_TAM_broiler[rr]
  country_LP_TAM[7,i] <- regional_LP_TAM_layer[rr]
  country_LP_TAM[8,i] <- regional_LP_TAM_duck[rr]
  country_LP_TAM[9,i] <- regional_LP_TAM_goats[rr]
  country_LP_TAM[10,i] <- regional_LP_TAM_horses[rr]
  country_LP_TAM[11,i] <- regional_LP_TAM_llamas[rr]
  country_LP_TAM[12,i] <- regional_LP_TAM_mules[rr]
  country_LP_TAM[13,i] <- regional_LP_TAM_sheep[rr]
  country_LP_TAM[14,i] <- regional_LP_TAM_pig[rr]
  country_LP_TAM[15,i] <- regional_LP_TAM_pig[rr]
  country_LP_TAM[16,i] <- regional_LP_TAM_turkey[rr]
}

country_share_HP <- array(NA,c(16,243))
country_share_HP <- (country_mean_TAM-country_LP_TAM)/(country_HP_TAM-country_LP_TAM)
# if a country/area has the same weight for HP and LP,
# we should use mean enteric emission factors
country_share_HP[which(country_HP_TAM == country_LP_TAM)] <- NA
# ducks share the same as broiler
country_share_HP[8,] <- country_share_HP[6,]
# Turkeys all as HP

##########################################
# 3.3.2 read emission factors
##########################################

csv.name <- paste0(path_param_2019refinement,"IPCC_Tier1_Tier1a_Enteric_Fermentation_CH4_Emission_Factors.csv")
data.mat <- read.table(csv.name,header=TRUE,sep=",",colClasses = "character")
# kg CH4 / head / year
EF2006T1 <- sapply(data.mat[2:11,2:17],as.numeric)
EF2019T1 <- sapply(data.mat[13:22,2:17],as.numeric)
EF2019T1a_LP <- sapply(data.mat[24:33,2:17],as.numeric)
EF2019T1a_HP <- sapply(data.mat[35:44,2:17],as.numeric)

##########################################
# 3.3.3 calculate Enteric fermentation emissions using 2019T1 and T1a methods
##########################################

country_EFCH4_2019T1 <- array(NA,c(243,16,length(year)))
country_EFCH4_2019T1a <- array(NA,c(243,16,length(year)))
for (i in 1:243) {
  rr <- df.crosswalk$GLEAM_Region[i]
  for (j in 1:length(year)) {
    for (m in 1:16) {
      # Note: the dimension of emission and livestock numbers are different
      country_EFCH4_2019T1[i,m,j] <- livenumber_cate[i,m,j]*EF2019T1[rr,m]/1e6
      if (is.na(country_share_HP[m,i] == TRUE)) {
        # For countries/areas with the same weight for HP and LP (country_share_HP == NA as assumed above)
        # mean emission factors should be used, because it means no HP and LP is differentiated for this livestock category and country/area
        country_EFCH4_2019T1a[i,m,j] <- livenumber_cate[i,m,j]*EF2019T1[rr,m]/1e6
      } else {
        # For countries/areas with HP and LP differentiated for this livestock category and country/area (country_share_HP != NA)
        country_EFCH4_2019T1a[i,m,j] <- livenumber_cate[i,m,j]*country_share_HP[m,i]*EF2019T1a_HP[rr,m]/1e6 + livenumber_cate[i,m,j]*(1-country_share_HP[m,i])*EF2019T1a_LP[rr,m]/1e6
      }
    }
  }
}

# check regional value
#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 
spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")
regional_EFCH4_2019T1 <- array(NA,c(11,16,length(year)))
regional_EFCH4_2019T1a <- array(NA,c(11,16,length(year)))
for (i in 1:16) {
  for (k in 1:10) {
    if (k == 2) {
      regional_EFCH4_2019T1[k,,] <- country_EFCH4_2019T1[gleam.list[[k]],,]
      regional_EFCH4_2019T1a[k,,] <- country_EFCH4_2019T1a[gleam.list[[k]],,]
    } else {
      regional_EFCH4_2019T1[k,,] <- apply(country_EFCH4_2019T1[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
      regional_EFCH4_2019T1a[k,,] <- apply(country_EFCH4_2019T1a[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
    }
  }
  k <- 11
  regional_EFCH4_2019T1[k,,] <- apply(country_EFCH4_2019T1[,,],c(2,3),sum,na.rm=TRUE)
  regional_EFCH4_2019T1a[k,,] <- apply(country_EFCH4_2019T1a[,,],c(2,3),sum,na.rm=TRUE)
}

##########################################
# save output of Enteric Fermentation CH4 emissions using the 2019 T1 method
##########################################
save.image(file=paste0(path_out,"EFCH4_2019T1.RData"))


##########################################
# 3.3.4 calculate spatially explicit Enteric Fermentation CH4 emissions from all livestock categories
##########################################

#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 
spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")

for (m in c(1:16)) {
  # spatial pattern of gridded livestock fraction within each country/area
  print(spatial_cate_name[m])
  if (m %in% c(4,5)) { # "CattleDairy","CattleNonDairy"
    frac_live <- GLW3_fraction_cattle
  } else if (m %in% c(13)) { # "Sheep" 
    frac_live <- GLW3_fraction_sheep
  } else if (m %in% c(9)) { # "Goats"
    frac_live <- GLW3_fraction_goats
  } else if (m %in% c(2)) { # Buffaloes"
    frac_live <- GLW3_fraction_buffaloes
  } else if (m %in% c(14,15)) { # "SwineBreeding","SwineMarket"
    frac_live <- GLW3_fraction_pigs
  } else if (m %in% c(6,7,16)) { # "ChickensBroilers","ChickensLayers","Turkeys"
    frac_live <- GLW3_fraction_chickens
  } else if (m %in% c(8)) { # "Ducks"
    frac_live <- GLW3_fraction_ducks
  } else if (m %in% c(10)) { # "Horses"
    frac_live <- GLW3_fraction_horses
  } else if (m %in% c(1,3,11,12)) {
    frac_live <- GLW3_fraction_cattle
  }
  # 2. calculate gridded EFCH4_livestock
  country_EFCH4_livestock <- country_EFCH4_2019T1[,m,]
  EFCH4_livestock <- array(NA,c(4320,2160,length(year)))
  for (j in 1:length(year)) {
    tmp_EFCH4_livestock <- array(NA,c(4320,2160))
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        tmp_EFCH4_livestock[index] <- country_EFCH4_livestock[i,j]*1e6*frac_live[index]/1e6 # Gg to kg
      }
    }
    EFCH4_livestock[,,j] <- tmp_EFCH4_livestock
  }
  # save spatial pattern to netcdf file
  fillvalue <- -1e20; threshold <- 1e8
  # define dimensions
  lon <- seq(-180,180,length=4320)
  lat <- seq(90,-90,length=2160)
  londim <- ncdim_def("lon","degrees_east",as.double(lon), longname="Longitude")
  latdim <- ncdim_def("lat","degrees_north",as.double(lat), longname="Latitude")
  tyears <- seq.int(0,(length(year)-1),by=1)
  tunits <- c("year")
  timedim <- ncdim_def("time",tunits,as.double(tyears), unlim=TRUE,
                       create_dimvar=TRUE, calendar="noleap", longname="Time axis")
  # define variables
  dlname <- c("EFCH4_livestock")
  var1.def <-  ncvar_def(dlname[1],"kg CH4 yr-1",list(londim,latdim,timedim),
                         fillvalue,dlname[1],prec="single",compression=5)
  ncfnameout <- paste0(path_out,"EFCH4_livestock_2019T1_spatial_",spatial_cate_name[m],".nc4")
  ncout <- nc_create(ncfnameout,list(var1.def),force_v4=TRUE)
  ncvar_put(ncout,var1.def,EFCH4_livestock)
  ncatt_put(ncout,"lon","axis","X") #,verbose=FALSE) #,definemode=FALSE)
  ncatt_put(ncout,"lat","axis","Y")
  for (k in 1:length(dlname)) {
    ncatt_put(ncout,dlname[k],"missing_value",fillvalue,prec="float")
  }
  ncatt_put(ncout,0,"contact","changjf@zju.edu.cn")
  ncatt_put(ncout,0,"description","Enteric Fermentation CH4 emissions from livestock using 2019T1 method")
  ncatt_put(ncout,0,"source","livestock numbers from FAOSTAT, emission factors from 2019 Refinement of IPCC 2006 Guidelines (Tier 1), spatial pattern of livestock from GLW3")
  nc_close(ncout)
  # clean temporary variables
  rm(frac_live,EFCH4_livestock,tmp_EFCH4_livestock)
}  


##########################################
##########################################
# 3.4 Calculate Manure Management CH4 emissions by major ruminant livestock categories
##########################################
##########################################

##########################################
# 3.4.1 Climate zone as defined in Chapter 3 of Volume 5 of 2019Refinement (also Annex 10A.2)
##########################################

# See Create_map_FAO_climate_zone.R
# it requires
# 1) mean annual temperature
# 2) mean annual precipitation
# 3) elevation (from HWSD dataset)
# 4) AI / PET (from Trabucco, Antonio; Zomer, Robert (2019): Global Aridity Index and Potential Evapotranspiration (ET0) Climate Database v2. figshare. Fileset. https://doi.org/10.6084/m9.figshare.7504448.v3)
# load climate zone
load(file=paste0(path_out,"map_FAO_climate_zone.RData"))

##########################################
# 3.4.2 read parameters 
##########################################

# 3.4.2.1 read VSrate
csvname <- paste0(path_param_2019refinement,"Table10.13A.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
regional_mean_VSrate <- sapply(mat[2:11,2:18],as.numeric)
regional_LP_VSrate <- sapply(mat[13:22,2:18],as.numeric)
regional_HP_VSrate <- sapply(mat[24:33,2:18],as.numeric)
#name_VSrate <- c(Dairy cattle4    Other cattle4    Buffalo4    Swine3    Finishing    Breeding    Poultry3    Hens ±1 yr    Pullets    Broilers    Turkeys8    Ducks8    Sheep3    Goats5    Horses8    Mules/Asses8    Camels8)

# assign regional and country/area VSrate for mean, LP and HP
for (k in 1:3) {
  if (k == 1) {
    regional_VSrate <- regional_mean_VSrate
  } else if (k == 2) {
    regional_VSrate <- regional_LP_VSrate
  } else if (k == 3) {
    regional_VSrate <- regional_HP_VSrate
  }
  regional_VSrate_asses <- regional_VSrate[,16]
  regional_VSrate_buffaloes <- regional_VSrate[,3]
  regional_VSrate_buffaloes[1] <- regional_VSrate_buffaloes[3]
  regional_VSrate_buffaloes[2] <- regional_VSrate_buffaloes[4]
  regional_VSrate_buffaloes[7] <- regional_VSrate_buffaloes[3]
  regional_VSrate_camels <- regional_VSrate[,17]
  regional_VSrate_cattledairy <- regional_VSrate[,1]
  regional_VSrate_cattledairy[2] <- regional_VSrate_cattledairy[4]
  regional_VSrate_cattleother <- regional_VSrate[,2]
  regional_VSrate_cattleother[2] <- regional_VSrate_cattleother[4]
  regional_VSrate_broiler <- regional_VSrate[,10]
  regional_VSrate_broiler[2] <- regional_VSrate_broiler[4]
  regional_VSrate_layer <- regional_VSrate[,8]
  regional_VSrate_layer[2] <- regional_VSrate_layer[4]  
  regional_VSrate_duck <- regional_VSrate[,12]
  regional_VSrate_duck[2] <- regional_VSrate_duck[4]
  regional_VSrate_goats <- regional_VSrate[,14]
  regional_VSrate_horses <- regional_VSrate[,15]
  regional_VSrate_llamas <- regional_VSrate[,16]
  regional_VSrate_mules <- regional_VSrate[,16]
  regional_VSrate_sheep <- regional_VSrate[,13]
  # same for market and breeding
  regional_VSrate_pig <- regional_VSrate[,4]
  regional_VSrate_pig[2] <- regional_VSrate_pig[4] # Russia
  regional_VSrate_turkey <- regional_VSrate[,11]
  regional_VSrate_turkey[2] <- regional_VSrate_turkey[4]
  
  country_VSrate <- array(NA,c(16,243))
  for (i in 1:243) {
    # region number 
    rr <- df.crosswalk$GLEAM_Region[i]
    country_VSrate[1,i] <- regional_VSrate_asses[rr]
    country_VSrate[2,i] <- regional_VSrate_buffaloes[rr]
    country_VSrate[3,i] <- regional_VSrate_camels[rr]
    country_VSrate[4,i] <- regional_VSrate_cattledairy[rr]  
    country_VSrate[5,i] <- regional_VSrate_cattleother[rr]
    country_VSrate[6,i] <- regional_VSrate_broiler[rr]
    country_VSrate[7,i] <- regional_VSrate_layer[rr]
    country_VSrate[8,i] <- regional_VSrate_duck[rr]
    country_VSrate[9,i] <- regional_VSrate_goats[rr]
    country_VSrate[10,i] <- regional_VSrate_horses[rr]
    country_VSrate[11,i] <- regional_VSrate_llamas[rr]
    country_VSrate[12,i] <- regional_VSrate_mules[rr]
    country_VSrate[13,i] <- regional_VSrate_sheep[rr]
    country_VSrate[14,i] <- regional_VSrate_pig[rr]
    country_VSrate[15,i] <- regional_VSrate_pig[rr]
    country_VSrate[16,i] <- regional_VSrate_turkey[rr]
  }
  
  if (k == 1) {
    country_mean_VSrate <- country_VSrate
  } else if (k == 2) {
    country_LP_VSrate <- country_VSrate
  } else if (k == 3) {
    country_HP_VSrate <- country_VSrate
  }
}


# 3.4.2.2 share of High Productivity (HP) in total production system
csvname <- paste0(path_param_2019refinement,"Table10A.5.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
mean_weight <- sapply(mat[2:11,3:22],as.numeric)
HP_weight <- sapply(mat[12:21,3:22],as.numeric)
LP_weight <- sapply(mat[22:31,3:22],as.numeric)
# share of HP and mean can be derived from mean, HP and mean weight
# x * HP + (1-x)*mean = mean
# x = (mean - mean)/(HP-mean)
share_HP <- (mean_weight-LP_weight)/(HP_weight-LP_weight)
share_HP[c(1,2,3,4,7)] <- 1.0

# name_weight <- c(
# Dairy cattle2 Other cattle2   Buffalo2        Swine3  Finishing
# Breeding      Poultry3        Hens >/= 1 yr   Pullets Broilers
# Turkeys4 Ducks4       Sheep3  Goats5  Horses4
# Mules and asses4 Camels4      Ostrich5        Deer5   Reindeer5
regional_mean_TAM_asses <- mean_weight[,16]
regional_mean_TAM_buffaloes <- mean_weight[,3]
regional_mean_TAM_camels <- mean_weight[,17]
regional_mean_TAM_cattledairy <- mean_weight[,1]
regional_mean_TAM_cattleother <- mean_weight[,2]
regional_mean_TAM_broiler <- mean_weight[,10]
regional_mean_TAM_layer <- mean_weight[,8]
regional_mean_TAM_duck <- mean_weight[,12]
regional_mean_TAM_goats <- mean_weight[,14]
regional_mean_TAM_horses <- mean_weight[,15]
regional_mean_TAM_llamas <- mean_weight[,16]
regional_mean_TAM_mules <- mean_weight[,16]
regional_mean_TAM_sheep <- mean_weight[,13]
regional_mean_TAM_pig <- mean_weight[,4]
regional_mean_TAM_turkey <- mean_weight[,11]

regional_HP_TAM_asses <- HP_weight[,16]
regional_HP_TAM_buffaloes <- HP_weight[,3]
regional_HP_TAM_camels <- HP_weight[,17]
regional_HP_TAM_cattledairy <- HP_weight[,1]
regional_HP_TAM_cattleother <- HP_weight[,2]
regional_HP_TAM_broiler <- HP_weight[,10]
regional_HP_TAM_layer <- HP_weight[,8]
regional_HP_TAM_duck <- HP_weight[,12]
regional_HP_TAM_goats <- HP_weight[,14]
regional_HP_TAM_horses <- HP_weight[,15]
regional_HP_TAM_llamas <- HP_weight[,16]
regional_HP_TAM_mules <- HP_weight[,16]
regional_HP_TAM_sheep <- HP_weight[,13]
regional_HP_TAM_pig <- HP_weight[,4]
regional_HP_TAM_turkey <- HP_weight[,11]

regional_LP_TAM_asses <- LP_weight[,16]
regional_LP_TAM_buffaloes <- LP_weight[,3]
regional_LP_TAM_camels <- LP_weight[,17]
regional_LP_TAM_cattledairy <- LP_weight[,1]
regional_LP_TAM_cattleother <- LP_weight[,2]
regional_LP_TAM_broiler <- LP_weight[,10]
regional_LP_TAM_layer <- LP_weight[,8]
regional_LP_TAM_duck <- LP_weight[,12]
regional_LP_TAM_goats <- LP_weight[,14]
regional_LP_TAM_horses <- LP_weight[,15]
regional_LP_TAM_mules <- LP_weight[,16]
regional_LP_TAM_llamas <- LP_weight[,16]
regional_LP_TAM_sheep <- LP_weight[,13]
regional_LP_TAM_pig <- LP_weight[,4]
regional_LP_TAM_turkey <- LP_weight[,11]


country_mean_TAM <- array(NA,c(16,243))
country_HP_TAM <- array(NA,c(16,243))
country_LP_TAM <- array(NA,c(16,243))
for (i in 1:243) {
  # region number
  rr <- df.crosswalk$GLEAM_Region[i]
  
  country_mean_TAM[1,i] <- regional_mean_TAM_asses[rr]
  country_mean_TAM[2,i] <- regional_mean_TAM_buffaloes[rr]
  country_mean_TAM[3,i] <- regional_mean_TAM_camels[rr]
  country_mean_TAM[4,i] <- regional_mean_TAM_cattledairy[rr]
  country_mean_TAM[5,i] <- regional_mean_TAM_cattleother[rr]
  country_mean_TAM[6,i] <- regional_mean_TAM_broiler[rr]
  country_mean_TAM[7,i] <- regional_mean_TAM_layer[rr]
  country_mean_TAM[8,i] <- regional_mean_TAM_duck[rr]
  country_mean_TAM[9,i] <- regional_mean_TAM_goats[rr]
  country_mean_TAM[10,i] <- regional_mean_TAM_horses[rr]
  country_mean_TAM[11,i] <- regional_mean_TAM_llamas[rr]
  country_mean_TAM[12,i] <- regional_mean_TAM_mules[rr]
  country_mean_TAM[13,i] <- regional_mean_TAM_sheep[rr]
  country_mean_TAM[14,i] <- regional_mean_TAM_pig[rr]
  country_mean_TAM[15,i] <- regional_mean_TAM_pig[rr]
  country_mean_TAM[16,i] <- regional_mean_TAM_turkey[rr]
  
  country_HP_TAM[1,i] <- regional_HP_TAM_asses[rr]
  country_HP_TAM[2,i] <- regional_HP_TAM_buffaloes[rr]
  country_HP_TAM[3,i] <- regional_HP_TAM_camels[rr]
  country_HP_TAM[4,i] <- regional_HP_TAM_cattledairy[rr]
  country_HP_TAM[5,i] <- regional_HP_TAM_cattleother[rr]
  country_HP_TAM[6,i] <- regional_HP_TAM_broiler[rr]
  country_HP_TAM[7,i] <- regional_HP_TAM_layer[rr]
  country_HP_TAM[8,i] <- regional_HP_TAM_duck[rr]
  country_HP_TAM[9,i] <- regional_HP_TAM_goats[rr]
  country_HP_TAM[10,i] <- regional_HP_TAM_horses[rr]
  country_HP_TAM[11,i] <- regional_HP_TAM_llamas[rr]
  country_HP_TAM[12,i] <- regional_HP_TAM_mules[rr]
  country_HP_TAM[13,i] <- regional_HP_TAM_sheep[rr]
  country_HP_TAM[14,i] <- regional_HP_TAM_pig[rr]
  country_HP_TAM[15,i] <- regional_HP_TAM_pig[rr]
  country_HP_TAM[16,i] <- regional_HP_TAM_turkey[rr]
  
  country_LP_TAM[1,i] <- regional_LP_TAM_asses[rr]
  country_LP_TAM[2,i] <- regional_LP_TAM_buffaloes[rr]
  country_LP_TAM[3,i] <- regional_LP_TAM_camels[rr]
  country_LP_TAM[4,i] <- regional_LP_TAM_cattledairy[rr]
  country_LP_TAM[5,i] <- regional_LP_TAM_cattleother[rr]
  country_LP_TAM[6,i] <- regional_LP_TAM_broiler[rr]
  country_LP_TAM[7,i] <- regional_LP_TAM_layer[rr]
  country_LP_TAM[8,i] <- regional_LP_TAM_duck[rr]
  country_LP_TAM[9,i] <- regional_LP_TAM_goats[rr]
  country_LP_TAM[10,i] <- regional_LP_TAM_horses[rr]
  country_LP_TAM[11,i] <- regional_LP_TAM_llamas[rr]
  country_LP_TAM[12,i] <- regional_LP_TAM_mules[rr]
  country_LP_TAM[13,i] <- regional_LP_TAM_sheep[rr]
  country_LP_TAM[14,i] <- regional_LP_TAM_pig[rr]
  country_LP_TAM[15,i] <- regional_LP_TAM_pig[rr]
  country_LP_TAM[16,i] <- regional_LP_TAM_turkey[rr]
}

country_share_HP <- array(NA,c(16,243))
country_share_HP <- (country_mean_TAM-country_LP_TAM)/(country_HP_TAM-country_LP_TAM)
# if a country/area has the same weight for HP and LP, we first set the share to NA
country_share_HP[which(country_HP_TAM == country_LP_TAM)] <- NA
# ducks share the same as broiler
country_share_HP[12,] <- country_share_HP[6,]
# Turkeys all as HP

# 3.4.2.3 calculate VS for HP and LP respectively 
# since the emission factors given by Table 10.14 depending on production system
VS_animal_HP_LP <- array(NA,c(2,243,16,length(year)))
#livenumber_cate <- array(NA,c(243,16,length(year)))
#country_mean_VSrate <- array(NA,c(16,243))
#country_HP_VSrate <- array(NA,c(16,243))
#country_LP_VSrate <- array(NA,c(16,243))
#country_mean_TAM <- array(NA,c(16,243))
#country_HP_TAM <- array(NA,c(16,243))
#country_LP_TAM <- array(NA,c(16,243))
#country_share_HP <- array(NA,c(16,243))
for (i in 1:243) {
  for (j in 1:length(year)) {
    for (m in 1:16) {
      ## VS(T,P) = VSrate(T,P) * TAM(T,P) /1000 *365 
      ## Head * % * kg VS per 1000 kg animal mass per day * (TAMkg / 1000) * 365 days = kg VS per country per year per livestock category
      if (is.na(country_share_HP[m,i]) == TRUE & df.crosswalk$Developed[i] == 1) {
        # For developed countries/areas with the same weight for HP and LP (country_share_HP == NA as assumed above)
        # mean VSrate and TAM should be used and assume all HP, because it means no HP and LP is differentiated for this livestock category and country/area
        VS_animal_HP_LP[1,i,m,j] <- livenumber_cate[i,m,j]*1.0 * (country_mean_VSrate[m,i]*country_mean_TAM[m,i]/1000*365)
        VS_animal_HP_LP[2,i,m,j] <- 0.0
      } else if (is.na(country_share_HP[m,i]) == TRUE & df.crosswalk$Developed[i] == 0) {
        # For developing countries/areas with the same weight for HP and LP (country_share_HP == NA as assumed above)
        # mean VSrate and TAM should be used and assume all LP, because it means no HP and LP is differentiated for this livestock category and country/area
        VS_animal_HP_LP[1,i,m,j] <- 0.0
        VS_animal_HP_LP[2,i,m,j] <- livenumber_cate[i,m,j]*1.0 * (country_mean_VSrate[m,i]*country_mean_TAM[m,i]/1000*365)
      } else {
        # For countries/areas with HP and LP differentiated for this livestock category and country/area (country_share_HP != 1)
        VS_animal_HP_LP[1,i,m,j] <- livenumber_cate[i,m,j]*country_share_HP[m,i] * (country_HP_VSrate[m,i]*country_HP_TAM[m,i]/1000*365)
        VS_animal_HP_LP[2,i,m,j] <- livenumber_cate[i,m,j]*(1-country_share_HP[m,i]) * (country_LP_VSrate[m,i]*country_LP_TAM[m,i]/1000*365)
      }
    }
  }
}

# 3.4.2.4 read other parameters
# Emission factor from Table 10.14 of 2019Refinement for each region, each livestock category
csvname <- paste0(path_param_2019refinement,"Table10.14.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
regional_EFMM <- sapply(mat[2:86,4:13],as.numeric)
# manure management systems, climate zone (1:10)

#Cool Temperate Moist    Cool Temperate Dry    Boreal Moist    Boreal Dry    Warm Temperate Moist    Warm Temperate Dry    Tropical Montane    Tropical Wet    Tropical Moist    Tropical Dry
# to keep the dimension as MCF (9*10) 9 management system * 10 climate zone ranspose it
# Unit : g CH4 kg VS-1
EFMM_dairycattle_HP <- sapply(mat[2:10,4:13],as.numeric)
EFMM_dairycattle_LP <- sapply(mat[11:19,4:13],as.numeric)
EFMM_othercattle_HP <- sapply(mat[20:28,4:13],as.numeric)
EFMM_othercattle_LP <- sapply(mat[29:37,4:13],as.numeric)
EFMM_pig_HP <- sapply(mat[38:46,4:13],as.numeric)
EFMM_pig_LP <- sapply(mat[47:55,4:13],as.numeric)
EFMM_poultry_HP <- sapply(mat[56:64,4:13],as.numeric)
##
# for Poultry Other manure management system means "Poultry manure with and without litter22"
# as in Table 10.17 with MCF = 1.5%
# Also in the footnote 22 Judgement of 2006 IPCC Expert Group. MCFs are similar to solid storage or to dry lot but with generally constant warm temperatures.
# SO we chose drylot temperate climate zone value as other of poultry 3.9
EFMM_poultry_HP[9,] <- 3.9
EFMM_poultry_LP <- array(0.0,c(9,10))
tmp <- sapply(mat[65,4:13],as.numeric)
for (k in 1:9) {
  EFMM_poultry_LP[k,] <- tmp
}
EFMM_sheep_HP <- array(0.0,c(9,10))
EFMM_sheep_LP <- array(0.0,c(9,10))
EFMM_goats_HP <- array(0.0,c(9,10))
EFMM_goats_LP <- array(0.0,c(9,10))
EFMM_camels_HP <- array(0.0,c(9,10))
EFMM_camels_LP <- array(0.0,c(9,10))
EFMM_horses_HP <- array(0.0,c(9,10))
EFMM_horses_LP <- array(0.0,c(9,10))
EFMM_asses_HP <- array(0.0,c(9,10))
EFMM_asses_LP <- array(0.0,c(9,10))
# Solid storage and Drylot
EFMM_sheep_HP[3:4,] <- sapply(mat[66:67,4:13],as.numeric)
EFMM_sheep_LP[3:4,] <- sapply(mat[68:69,4:13],as.numeric)
EFMM_goats_HP[3:4,] <- sapply(mat[70:71,4:13],as.numeric)
EFMM_goats_LP[3:4,] <- sapply(mat[72:73,4:13],as.numeric)
EFMM_camels_HP[3:4,] <- sapply(mat[74:75,4:13],as.numeric)
EFMM_camels_LP[3:4,] <- sapply(mat[76:77,4:13],as.numeric)
EFMM_horses_HP[3:4,] <- sapply(mat[78:79,4:13],as.numeric)
EFMM_horses_LP[3:4,] <- sapply(mat[80:81,4:13],as.numeric)
EFMM_asses_HP[3:4,] <- sapply(mat[82:83,4:13],as.numeric)
EFMM_asses_LP[3:4,] <- sapply(mat[84:85,4:13],as.numeric)
# Pasture/Range/ Paddock all 0.6
EFMM_sheep_HP[5,] <- 0.6
EFMM_sheep_LP[5,] <- 0.6 
EFMM_goats_HP[5,] <- 0.6 
EFMM_goats_LP[5,] <- 0.6 
EFMM_camels_HP[5,] <- 0.6
EFMM_camels_LP[5,] <- 0.6
EFMM_horses_HP[5,] <- 0.6
EFMM_horses_LP[5,] <- 0.6
EFMM_asses_HP[5,] <- 0.6 
EFMM_asses_LP[5,] <- 0.6 

# AWMS from Table 10A.6 to 10A.9 of 2019Refinement for each livestock category
csvname <- paste0(path_param_2019refinement,"Table10A.6.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_dairycattle <- sapply(mat[2:11,3:11],as.numeric)
AWMS_dairycattle[2,] <- AWMS_dairycattle[4,]
AWMS_othercattle <- sapply(mat[12:21,3:11],as.numeric)
AWMS_othercattle[2,] <- AWMS_othercattle[4,]
AWMS_dairybuffalo <- sapply(mat[22:31,3:11],as.numeric)
AWMS_dairybuffalo[2,] <- AWMS_dairybuffalo[4,]
AWMS_dairybuffalo[7,] <- AWMS_dairybuffalo[1,]
AWMS_dairybuffalo[10,] <- AWMS_dairybuffalo[5,]
AWMS_otherbuffalo <- sapply(mat[32:41,3:11],as.numeric)
AWMS_otherbuffalo[1,] <- AWMS_dairybuffalo[1,]
AWMS_otherbuffalo[3,] <- AWMS_dairybuffalo[3,]
AWMS_otherbuffalo[7,] <- AWMS_dairybuffalo[7,]
AWMS_otherbuffalo[10,] <- AWMS_otherbuffalo[5,]
#Lagoon    "Liquid/Slurry"    "Solid storage"    Drylot    "Pasture/Range/ Paddock"    "Daily spread"    Digester    "Burned for fuel"    Other

csvname <- paste0(path_param_2019refinement,"Table10A.7.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_growingswine_High <- sapply(mat[2:11,3:11],as.numeric)
AWMS_growingswine_Low <- sapply(mat[12:21,3:11],as.numeric)
AWMS_breedingswine_High <- sapply(mat[22:31,3:11],as.numeric)
AWMS_breedingswine_Low <- sapply(mat[32:41,3:11],as.numeric)

# NOTE AWMS of pigs are very different from other livestock
# the EFMM in Table 10.14 has been modified to match the AWMS here
#Lagoon    Liquid/Slurry    Solid storage    Drylot    "Pit <1"    Pit > 1    Daily spread    Digester    Pasture
AWMS_growingswine_Low[c(1,2,3,4,7),] <- AWMS_growingswine_High[c(1,2,3,4,7),]
AWMS_breedingswine_Low[c(1,2,3,4,7),] <- AWMS_breedingswine_High[c(1,2,3,4,7),]

csvname <- paste0(path_param_2019refinement,"Table10A.8.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_othersheep <- sapply(mat[2:11,3:11],as.numeric)
AWMS_othersheep[2,] <- AWMS_othersheep[4,]
AWMS_dairysheep <- sapply(mat[12:21,3:11],as.numeric)
AWMS_dairysheep[1,] <- AWMS_dairysheep[3,]
AWMS_dairysheep[2,] <- AWMS_dairysheep[4,]
AWMS_dairysheep[7,] <- AWMS_dairysheep[3,]
AWMS_goats <- sapply(mat[22:31,3:11],as.numeric)
#Lagoon    Liquid/Slurry    Solid storage    Drylot    Pasture/Range/ Paddock    Daily spread    Digester    Burned for fuel    Other

csvname <- paste0(path_param_2019refinement,"Table10A.9.csv")
mat <- read.table(csvname,header = FALSE, sep = ";",skip = 1,colClasses = "character")
AWMS_layer <- sapply(mat[2:11,3:11],as.numeric)
AWMS_broiler_High <- sapply(mat[12:21,3:11],as.numeric)
AWMS_broiler_Low <- sapply(mat[22:31,3:11],as.numeric)
#Lagoon    "Liquid/Slurry"    Solid storage    Drylot    "Pasture/Range/Paddock"    Pit >1 month    Daily spread    Digester    Other (Poultry manure with litter)

##########################################
# 3.4.3 calculate spatially explicit Manure Management CH4 emissions from all livestock categories
##########################################

# Emission = EFMM * VS
## Parameter used are
## 1) EFMM: emission factor from Table 10.14
## 2) VS: Volatile solid excreted calculated from TAM from Table 10A.5 and VSrate from Table 10.13a
## 4) AWMS: Animal waste management system characteristics (region and manure management system specific)
#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 
spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")

country_MMCH4_2019T1 <- array(NA,c(243,16,length(year)))

for (m in 1:16) {
  # spatial pattern of gridded livestock fraction within each country/area
  print(spatial_cate_name[m])
  if (m %in% c(4,5)) { # "CattleDairy","CattleNonDairy"
    frac_live <- GLW3_fraction_cattle
  } else if (m %in% c(13)) { # "Sheep" 
    frac_live <- GLW3_fraction_sheep
  } else if (m %in% c(9)) { # "Goats"
    frac_live <- GLW3_fraction_goats
  } else if (m %in% c(2)) { # Buffaloes"
    frac_live <- GLW3_fraction_buffaloes
  } else if (m %in% c(14,15)) { # "SwineBreeding","SwineMarket"
    frac_live <- GLW3_fraction_pigs
  } else if (m %in% c(6,7,16)) { # "ChickensBroilers","ChickensLayers","Turkeys"
    frac_live <- GLW3_fraction_chickens
  } else if (m %in% c(8)) { # "Ducks"
    frac_live <- GLW3_fraction_ducks
  } else if (m %in% c(10)) { # "Horses"
    frac_live <- GLW3_fraction_horses
  } else if (m %in% c(1,3,11,12)) {
    frac_live <- GLW3_fraction_cattle
  }
  
  # spread AWMS to grid level
  # lon,lat,livestock category, AWMS
  AWMS_grid_HP <- array(NA,c(4320,2160,9))
  AWMS_grid_LP <- array(NA,c(4320,2160,9))
  AWMS_regional_tmp <- array(NA,c(10,9))
  AWMS_regional_tmp_low <- array(NA,c(10,9))
  if (m %in% c(4)) { # "CattleDairy"
    AWMS_regional_tmp <- AWMS_dairycattle 
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(5)) { # "CattleNonDairy"
    AWMS_regional_tmp <- AWMS_othercattle
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(13)) { # "Sheep"
    AWMS_regional_tmp <- AWMS_othersheep
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(9)) { # "Goats"
    AWMS_regional_tmp <- AWMS_goats
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(2)) { # "Buffaloes"
    AWMS_regional_tmp <- AWMS_otherbuffalo
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(14)) { # "SwineBreeding"
    AWMS_regional_tmp <- AWMS_breedingswine_High
    AWMS_regional_tmp_low <- AWMS_breedingswine_Low
  } else if (m %in% c(15)) { # "SwineMarket"
    AWMS_regional_tmp <- AWMS_growingswine_High
    AWMS_regional_tmp_low <- AWMS_growingswine_Low
  } else if (m %in% c(6,8,16)) { # "ChickensBroilers","Ducks","Turkeys"
    AWMS_regional_tmp <- AWMS_broiler_High
    AWMS_regional_tmp_low <- AWMS_broiler_Low
  } else if (m %in% c(7)) { # "ChickensLayers"
    AWMS_regional_tmp <- AWMS_layer 
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  } else if (m %in% c(1,3,10,11,12)) { #"Asses","Camels","Horses","Llamas","Mules"
    # footnote of Table 10A.9 of 2019 Refinement
    AWMS_regional_tmp <- AWMS_goats
    AWMS_regional_tmp_low <- AWMS_regional_tmp
  }  
  # spread, be careful of the high low
  for (w in 1:9) {
    print(paste("AWMS",w))
    tmp_map_AWMS_HP <- array(NA,c(4320,2160))
    tmp_map_AWMS_LP <- array(NA,c(4320,2160))
    
    for (i in 1:243) {
      for (k in 1:10) {
        if (df.crosswalk$GLEAM_Region[i] == k) {
          index <- which(regionid_8km == i)
          if (length(index) >= 1) {
            tmp_map_AWMS_HP[index] <- AWMS_regional_tmp[k,w]
            tmp_map_AWMS_LP[index] <- AWMS_regional_tmp_low[k,w]
          }
        }
      }
    }
    AWMS_grid_HP[,,w] <- tmp_map_AWMS_HP
    AWMS_grid_LP[,,w] <- tmp_map_AWMS_LP
  }
  # clean temporary variables
  rm(tmp_map_AWMS_HP,tmp_map_AWMS_LP)
  
  # spread EFMM to grid level
  # lon,lat,livestock category, AWMS
  EFMM_grid_HP <- array(NA,c(4320,2160,9))
  EFMM_grid_LP <- array(NA,c(4320,2160,9))
  # See the footnote of Table 10.17 for the default value that should be used
  # Lagoon: footnote 7, 12 months assumed (No. 1)
  # Liquid/Slurry: footnote 1, 6 months (No. 5)
  # Solid storage: (No. 9)
  # Drylot: (No. 13)
  # Pasture/Range/Paddock: (No. 19)
  # Daily spread: (No. 14)
  # Digester: (No. 26) Considering they are mainly for Swine manure and for low-income countries, we consider it to be High leakage, low quality technology, high quanlity gastight storage technology, which gives a value of 9.59% median value for different digesters
  # Burned for fuel: (No. 22)
  # Other: assume 0.0? as Aerobic treatment? assume as Pasture/Range/Paddock
  # Other (Poultry manure with litter): (No. 20)
  
  order_EFMM_HP <- array(NA,c(9,10)) # 9 AWMS, 10 climate zone NOTE: not region
  order_EFMM_LP <- array(NA,c(9,10)) # 9 AWMS, 10 climate zone NOTE: not region
  #> MMCH4.Item.name
  #[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
  #[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
  #[15] "Swine, market"      "Turkeys" 
  spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                         "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                         "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                         "Turkeys")
  if (m %in% c(1,12)) { # "Asses" "Mules" 
    order_EFMM_HP[1:9,1:10] <- EFMM_asses_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_asses_LP[1:9,1:10]
  } else if (m %in% c(3)) { # "Camels"
    order_EFMM_HP[1:9,1:10] <- EFMM_camels_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_camels_LP[1:9,1:10]
  } else if (m %in% c(4)) { # "Cattle, dairy"
    order_EFMM_HP[1:9,1:10] <- EFMM_dairycattle_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_dairycattle_LP[1:9,1:10]
  } else if (m %in% c(5)) { # "Cattle, non-dairy"
    order_EFMM_HP[1:9,1:10] <- EFMM_othercattle_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_othercattle_LP[1:9,1:10]
  } else if (m %in% c(6,7,8,16)) { # "Chickens, broilers" "Chickens, layers" "Ducks" "Turkeys"
    order_EFMM_HP[1:9,1:10] <- EFMM_poultry_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_poultry_LP[1:9,1:10]
  } else if (m %in% c(9,11)) { # "Goats" "Llamas"
    order_EFMM_HP[1:9,1:10] <- EFMM_goats_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_goats_LP[1:9,1:10]
  } else if (m %in% c(10)) { # "Horses"
    order_EFMM_HP[1:9,1:10] <- EFMM_horses_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_horses_LP[1:9,1:10]
  } else if (m %in% c(13)) { # "Sheep"
    order_EFMM_HP[1:9,1:10] <- EFMM_sheep_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_sheep_LP[1:9,1:10]
  } else if (m %in% c(2)) { # "Buffaloes"
    order_EFMM_HP[1:9,1:10] <- EFMM_othercattle_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_othercattle_LP[1:9,1:10]
  } else if (m %in% c(14,15)) { # "Swine, breeding"    "Swine, market
    order_EFMM_HP[1:9,1:10] <- EFMM_pig_HP[1:9,1:10]
    order_EFMM_LP[1:9,1:10] <- EFMM_pig_LP[1:9,1:10]
  } 
  
  for (z in 1:9) {
    print(paste("EFMM",z))
    tmp_EFMM_HP <- array(NA,c(4320,2160))
    tmp_EFMM_LP <- array(NA,c(4320,2160))
    
    for (qq in 1:10) { # FAO climate zone
      index <- which(map_FAO_climate_zone == qq)
      if (length(index) >= 1) {  
        tmp_EFMM_HP[index] <- order_EFMM_HP[z,qq]
        tmp_EFMM_LP[index] <- order_EFMM_LP[z,qq]
      }
    }
    EFMM_grid_HP[,,z] <- tmp_EFMM_HP
    EFMM_grid_LP[,,z] <- tmp_EFMM_LP
  }
  
  # lon,lat, AWMS
  #AWMS_grid_HP <- array(NA,c(4320,2160,9))
  #AWMS_grid_LP <- array(NA,c(4320,2160,9))
  #EFMM_grid_HP <- array(NA,c(4320,2160,9))
  #EFMM_grid_LP <- array(NA,c(4320,2160,9))
  
  AWMS_grid_HP[is.na(AWMS_grid_HP) == TRUE | is.finite(AWMS_grid_HP) == FALSE | AWMS_grid_HP < 0.0] <- 0.0
  AWMS_grid_LP[is.na(AWMS_grid_LP) == TRUE | is.finite(AWMS_grid_LP) == FALSE | AWMS_grid_LP < 0.0] <- 0.0
  EFMM_grid_HP[is.na(EFMM_grid_HP) == TRUE | is.finite(EFMM_grid_HP) == FALSE | EFMM_grid_HP < 0.0] <- 0.0
  EFMM_grid_LP[is.na(EFMM_grid_LP) == TRUE | is.finite(EFMM_grid_LP) == FALSE | EFMM_grid_LP < 0.0] <- 0.0
  
  MMCH4_livestock <- array(NA,c(4320,2160,length(year))) 
  for (j in 1:length(year)) {
    print(paste("VS",j))
    # spread VS to grid level
    # lon,lat,livestock category,year
    VS_grid_HP <- array(NA,c(4320,2160))
    VS_grid_LP <- array(NA,c(4320,2160))
    # Country level VS
    #VS_animal_HP_LP <- array(NA,c(2,16,243,57))
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        VS_grid_HP[index] <- VS_animal_HP_LP[1,i,m,j]*frac_live[index]/1e6 # kg # frac_XXX needs to be /1e6 for calculation
        VS_grid_LP[index] <- VS_animal_HP_LP[2,i,m,j]*frac_live[index]/1e6 # kg # frac_XXX needs to be /1e6 for calculation
      }
    }
    VS_grid_HP[is.na(VS_grid_HP) == TRUE | is.finite(VS_grid_HP) == FALSE | VS_grid_HP < 0.0] <- 0.0
    VS_grid_LP[is.na(VS_grid_LP) == TRUE | is.finite(VS_grid_LP) == FALSE | VS_grid_LP < 0.0] <- 0.0
    
    ######
    # calculate final emissions
    # Emission = EFMM * VS for each system
    print(paste("MMCH4",j))
    tmp_MMCH4_livestock <- array(0.0,c(4320,2160))
    for (z in 1:9) {
      # kg VS * % * gCH4 /kg VS /1e3 = kg CH4
      tmp_MMCH4_livestock <- tmp_MMCH4_livestock + VS_grid_HP * (AWMS_grid_HP[,,z]/100) * EFMM_grid_HP[,,z]/1e3
      tmp_MMCH4_livestock <- tmp_MMCH4_livestock + VS_grid_LP * (AWMS_grid_LP[,,z]/100) * EFMM_grid_LP[,,z]/1e3
    }
    MMCH4_livestock[,,j] <- tmp_MMCH4_livestock 
  }
  # clean temporary variables
  rm(frac_live,VS_grid_HP,VS_grid_LP,AWMS_grid_HP,AWMS_grid_LP,EFMM_grid_HP,EFMM_grid_LP,tmp_MMCH4_livestock)
  
  for (j in 1:length(year)) {
    tmp <- MMCH4_livestock[,,j]
    for (i in 1:243) {
      index <- which(regionid_8km == i)
      if (length(index) >= 1) {
        country_MMCH4_2019T1[i,m,j] <- sum(tmp[index],na.rm=TRUE)/1e6 # kg to Gg
      }
    }
  }
  
  # save spatial pattern to netcdf file
  fillvalue <- -1e20; threshold <- 1e8
  # define dimensions
  lon <- seq(-180,180,length=4320)
  lat <- seq(90,-90,length=2160)
  londim <- ncdim_def("lon","degrees_east",as.double(lon), longname="Longitude")
  latdim <- ncdim_def("lat","degrees_north",as.double(lat), longname="Latitude")
  tyears <- seq.int(0,(length(year)-1),by=1)
  tunits <- c("year")
  timedim <- ncdim_def("time",tunits,as.double(tyears), unlim=TRUE,
                       create_dimvar=TRUE, calendar="noleap", longname="Time axis")
  # define variables
  dlname <- c("MMCH4_livestock")
  var1.def <-  ncvar_def(dlname[1],"kg CH4 yr-1",list(londim,latdim,timedim),
                         fillvalue,dlname[1],prec="single",compression=5)
  ncfnameout <- paste0(path_out,"MMCH4_livestock_2019T1_spatial_",spatial_cate_name[m],".nc4")
  ncout <- nc_create(ncfnameout,list(var1.def),force_v4=TRUE)
  ncvar_put(ncout,var1.def,MMCH4_livestock)
  ncatt_put(ncout,"lon","axis","X") #,verbose=FALSE) #,definemode=FALSE)
  ncatt_put(ncout,"lat","axis","Y")
  for (k in 1:length(dlname)) {
    ncatt_put(ncout,dlname[k],"missing_value",fillvalue,prec="float")
  }
  ncatt_put(ncout,0,"contact","changjf@zju.edu.cn")
  ncatt_put(ncout,0,"description","Manure Management CH4 emissions from livestock using 2019T1 method")
  ncatt_put(ncout,0,"source","livestock numbers from FAOSTAT, emission factors from 2019 Refinement of IPCC 2006 Guidelines (Tier 1), spatial pattern of livestock from GLW3")
  nc_close(ncout)
  # clean temporary variables
  rm(MMCH4_livestock)
}

spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")
regional_MMCH4_2019T1 <- array(NA,c(11,16,length(year)))
for (i in 1:16) {
  for (k in 1:10) {
    if (k == 2) {
      regional_MMCH4_2019T1[k,,] <- country_MMCH4_2019T1[gleam.list[[k]],,]
    } else {
      regional_MMCH4_2019T1[k,,] <- apply(country_MMCH4_2019T1[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
    }
  }
  k <- 11
  regional_MMCH4_2019T1[k,,] <- apply(country_MMCH4_2019T1[,,],c(2,3),sum,na.rm=TRUE)
}

##########################################
# save output of Manure Management CH4 emissions using the 2019 T1 method
##########################################
save.image(file=paste0(path_out,"MMCH4_2019T1.RData"))
# clear cache 
rm(list=ls())
gc()


##########################################
##########################################
# Part IV Calculate livestock CH4 emission intensity per protein produced using the emissions from FAOSTAT (2006T1), 2019T1 and 2019MT method
##########################################
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

##########################################
##########################################
# 4.1 Read livestock CH4 emissions from the 2019MT and 2019T1 methods
##########################################
##########################################

##########################################
# 4.1.1 Load emissions from 2019MT method, including emissions from draft animals
##########################################

#load(file=paste0(path_out,"EFCH4_2019MT.RData"))
load(file=paste0(path_out,"MMCH4_2019MT.RData"))
#country_EFCH4_2019MT <- array(NA,c(243,16,length(year)))
#country_MMCH4_2019MT <- array(NA,c(243,16,length(year)))

# draft buffaloes and other cattle
#MC_nonprod_FCH4_used <- array(NA,c(MC.DE,2,243,length(year)))
mean_EFCH4_draft <- apply(MC_nonprod_FCH4_used[,,,],c(2,3,4),mean,na.rm=TRUE)/1e6 # kg to Gg
country_EFCH4_2019MT_draft <- array(NA,c(243,16,length(year)))
# Buffaloes
country_EFCH4_2019MT_draft[,2,] <- mean_EFCH4_draft[1,,]
# Other cattle
country_EFCH4_2019MT_draft[,5,] <- mean_EFCH4_draft[2,,]
# Here, we assume Manure Management emissions from draft animals comprise the same share as for Enteric Fermentation emissions
country_MMCH4_2019MT_draft <- country_EFCH4_2019MT_draft/country_EFCH4_2019MT*country_MMCH4_2019MT
country_MMCH4_2019MT_draft[is.infinite(country_MMCH4_2019MT_draft) | is.na(country_MMCH4_2019MT_draft) | country_MMCH4_2019MT_draft < 0.0] <- 0.0

##########################################
# 4.1.2 Load emissions from 2019T1 method
##########################################

#load(file=paste0(path_out,"EFCH4_2019T1.RData"))
load(file=paste0(path_out,"MMCH4_2019T1.RData"))
#country_EFCH4_2019T1 <- array(NA,c(243,16,length(year)))
#country_MMCH4_2019T1 <- array(NA,c(243,16,length(year)))

##########################################
# 4.1.3 Only save variables that will be used
##########################################
var.name.all <- ls()
var.save <- c("livenumber_cate","country_EFCH4_2019MT","country_MMCH4_2019MT","country_EFCH4_2019MT_draft","country_MMCH4_2019MT_draft","country_EFCH4_2019T1","country_EFCH4_2019T1a","country_MMCH4_2019T1")
for (kk in 1:length(var.save)) {
  var.name.all <- var.name.all[var.name.all != var.save[kk]]
}
rm(list=var.name.all)

##########################################
# 4.1.3 Re-initiate paths
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

# Load FAOSTAT statistics on livestock
load(file=paste0(path_out,"FAOSTAT_Livestock_GDP.RData"))
# convert from characters to numbers
df.crosswalk$GLEAM_Region <- as.numeric(df.crosswalk$GLEAM_Region)
df.crosswalk$Developed <- as.numeric(df.crosswalk$Developed)
region.name <- c("N. America","Russian Fed.","W. Europe","E. Europe","NENA","E & SE Asia","Oceania","South Asia","LAC","SSA")
for (i in 1:10) {
  assign(paste0("r",i), which(df.crosswalk$GLEAM_Region == i))
}
gleam.list <- list(r1,r2,r3,r4,r5,r6,r7,r8,r9,r10)

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set parameter path
path_param_gleam <- paste0(path_data,"GLEAMv2_PARAM/")
path_param_dress <- paste0(path_data,"Opio2013_PARAM/")
path_param_2019refinement <- paste0(path_data,"2019Refinement_PARAM/")
path_param_GLW3 <- paste0(path_data,"GLW3/")
# Load library
library(ncdf4)

##########################################
##########################################
# 4.2 Read livestock CH4 emissions from FAOSTAT
##########################################
##########################################

##########################################
# 4.2.1 get Enteric fermentation Emissions
##########################################

country_EFCH4_FAOSTAT <- array(NA,c(243,16,length(year)))
#> EFCH4.Item.name
#[1] "Asses"             "Buffaloes"         "Camels"            "Cattle, dairy"     "Cattle, non-dairy"
#[6] "Goats"             "Horses"            "Llamas"            "Mules"             "Sheep"            
#[11] "Swine, breeding"   "Swine, market" 
country_EFCH4_FAOSTAT[,c(1:5),] <- EFCH4.mat[,c(1:5),]
country_EFCH4_FAOSTAT[,c(9,10,11,12,13,14,15),] <- EFCH4.mat[,c(6,7,8,9,10,11,12),]

##########################################
# 4.2.2 get Manure Management emissions
##########################################

country_MMCH4_FAOSTAT <- array(NA,c(243,16,length(year)))
#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys" 
country_MMCH4_FAOSTAT <- MMCH4.mat

# check global values
apply(country_EFCH4_FAOSTAT,c(3),sum,na.rm=TRUE)
apply(country_MMCH4_FAOSTAT,c(3),sum,na.rm=TRUE)
apply(country_EFCH4_2019T1,c(3),sum,na.rm=TRUE)
apply(country_MMCH4_2019T1,c(3),sum,na.rm=TRUE)
apply(country_EFCH4_2019MT,c(3),sum,na.rm=TRUE)
apply(country_MMCH4_2019MT,c(3),sum,na.rm=TRUE)
apply(country_EFCH4_2019MT_draft,c(3),sum,na.rm=TRUE)

##########################################
##########################################
# 4.3 calculate regional livestock CH4 emissions
##########################################
##########################################

##########################################
# 4.3.1 set all NA values to 0 for further calculations
##########################################

var.reset <- c("country_EFCH4_FAOSTAT","country_MMCH4_FAOSTAT","country_EFCH4_2019T1","country_MMCH4_2019T1","country_EFCH4_2019MT","country_MMCH4_2019MT","country_EFCH4_2019MT_draft","country_MMCH4_2019MT_draft")
for (kk in 1:length(var.reset)) {
  tmp <- eval(parse(text=var.reset[kk]))
  tmp[is.infinite(tmp) | is.na(tmp) | tmp < 0.0] <- 0.0
  assign(var.reset[kk],tmp)
}

##########################################
# 4.3.2 gather livestock CH4 emissions from Enteric Fermentation and Manure Management
##########################################

country_CH4_FAOSTAT <- array(NA,c(3,243,16,length(year)))
country_CH4_FAOSTAT[1,,,] <- country_EFCH4_FAOSTAT
country_CH4_FAOSTAT[2,,,] <- country_MMCH4_FAOSTAT
country_CH4_FAOSTAT[3,,,] <- country_EFCH4_FAOSTAT + country_MMCH4_FAOSTAT
country_CH4_2019T1 <- array(NA,c(3,243,16,length(year)))
country_CH4_2019T1[1,,,] <- country_EFCH4_2019T1
country_CH4_2019T1[2,,,] <- country_MMCH4_2019T1
country_CH4_2019T1[3,,,] <- country_EFCH4_2019T1 + country_MMCH4_2019T1
country_CH4_2019T1a <- array(NA,c(3,243,16,length(year)))
country_CH4_2019T1a[1,,,] <- country_EFCH4_2019T1a
country_CH4_2019T1a[2,,,] <- country_MMCH4_2019T1
country_CH4_2019T1a[3,,,] <- country_EFCH4_2019T1a + country_MMCH4_2019T1
country_CH4_2019MT <- array(NA,c(3,243,16,length(year)))
country_CH4_2019MT[1,,,] <- country_EFCH4_2019MT
country_CH4_2019MT[2,,,] <- country_MMCH4_2019MT
country_CH4_2019MT[3,,,] <- country_EFCH4_2019MT + country_MMCH4_2019MT
country_CH4_2019MT_draft <- array(NA,c(3,243,16,length(year)))
country_CH4_2019MT_draft[1,,,] <- country_EFCH4_2019MT_draft
country_CH4_2019MT_draft[2,,,] <- country_MMCH4_2019MT_draft
country_CH4_2019MT_draft[3,,,] <- country_EFCH4_2019MT_draft + country_MMCH4_2019MT_draft

##########################################
# 4.3.3 calculate regional livestock CH4 emissions
##########################################

regional_CH4_FAOSTAT <- array(NA,c(3,11,16,length(year)))
regional_CH4_2019T1 <- array(NA,c(3,11,16,length(year)))
regional_CH4_2019T1a <- array(NA,c(3,11,16,length(year)))
regional_CH4_2019MT <- array(NA,c(3,11,16,length(year)))
regional_CH4_2019MT_draft <- array(NA,c(3,11,16,length(year)))
for (k in 1:10) {
  if (k == 2) {
    regional_CH4_FAOSTAT[,k,,] <- country_CH4_FAOSTAT[,gleam.list[[k]],,]
    regional_CH4_2019T1[,k,,] <- country_CH4_2019T1[,gleam.list[[k]],,]
    regional_CH4_2019T1a[,k,,] <- country_CH4_2019T1a[,gleam.list[[k]],,]
    regional_CH4_2019MT[,k,,] <- country_CH4_2019MT[,gleam.list[[k]],,]
    regional_CH4_2019MT_draft[,k,,] <- country_CH4_2019MT_draft[,gleam.list[[k]],,]
  } else {
    regional_CH4_FAOSTAT[,k,,] <- apply(country_CH4_FAOSTAT[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    regional_CH4_2019T1[,k,,] <- apply(country_CH4_2019T1[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    regional_CH4_2019T1a[,k,,] <- apply(country_CH4_2019T1a[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    regional_CH4_2019MT[,k,,] <- apply(country_CH4_2019MT[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    regional_CH4_2019MT_draft[,k,,] <- apply(country_CH4_2019MT_draft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
  }
}
k <- 11 # global value
regional_CH4_FAOSTAT[,k,,] <- apply(country_CH4_FAOSTAT[,,,],c(1,3,4),sum,na.rm=TRUE)
regional_CH4_2019T1[,k,,] <- apply(country_CH4_2019T1[,,,],c(1,3,4),sum,na.rm=TRUE)
regional_CH4_2019T1a[,k,,] <- apply(country_CH4_2019T1a[,,,],c(1,3,4),sum,na.rm=TRUE)
regional_CH4_2019MT[,k,,] <- apply(country_CH4_2019MT[,,,],c(1,3,4),sum,na.rm=TRUE)
regional_CH4_2019MT_draft[,k,,] <- apply(country_CH4_2019MT_draft[,,,],c(1,3,4),sum,na.rm=TRUE)

##########################################
##########################################
# 4.4 calculate country/area and regional livestock CH4 emission intensity (EmisInt) per protein produced
##########################################
##########################################

##########################################
# 4.4.1 parameters on the protein content of livestock products
##########################################

# Cattle (Dairy+other), Buffaloes, Sheep, Goats, Pigs, Poultry
# From Table 9.1 of GLEAM v2.0 Documentation
# BVMEAT 158g/kg SheepMEAT 141g/kg GoatMEAT 134g/kg PGMEAT 131g/kg ALMILK 33g/kg PTEGGS 124g/kg PTMEAT 143g/kg
protein_BV <- 0.158
protein_SH <- 0.141
protein_GT <- 0.134
protein_PG <- 0.131
protein_MK <- 0.033
protein_EG <- 0.124
protein_PT <- 0.143

##########################################
# 4.4.2 calculate protein production for major livestock categories
##########################################

# we create aggregated category for pigs ("Swine, breeding" and "Swine, market") and poultry ("Chickens, broilers", "Chickens, layers", "Ducks", and "Turkeys")
country_protein <- array(NA,c(243,16+3,length(year)))
# Unit: tonne
#> MMCH4.Item.name
#[1] "Asses"              "Buffaloes"          "Camels"             "Cattle, dairy"      "Cattle, non-dairy"  "Chickens, broilers" "Chickens, layers"  
#[8] "Ducks"              "Goats"              "Horses"             "Llamas"             "Mules"              "Sheep"              "Swine, breeding"   
#[15] "Swine, market"      "Turkeys"
j_Item <- which(Production.Item.name == "Meat, ass")
country_protein[,1,] <- Production.mat[,j_Item,]*protein_GT
j_Item1 <- which(Production.Item.name == "Milk, whole fresh buffalo")
j_Item2 <- which(Production.Item.name == "Meat, buffalo")
country_protein[,2,] <- Production.mat[,j_Item1,]*protein_MK+Production.mat[,j_Item2,]*protein_BV
j_Item1 <- which(Production.Item.name == "Milk, whole fresh camel")
j_Item2 <- which(Production.Item.name == "Meat, camel")
country_protein[,3,] <- Production.mat[,j_Item1,]*protein_MK+Production.mat[,j_Item2,]*protein_BV
j_Item <- which(Production.Item.name == "Milk, whole fresh cow")
country_protein[,4,] <- Production.mat[,j_Item,]*protein_MK
j_Item <- which(Production.Item.name == "Meat, cattle")
country_protein[,5,] <- Production.mat[,j_Item,]*protein_BV
j_Item <- which(Production.Item.name == "Meat, chicken")
country_protein[,6,] <- Production.mat[,j_Item,]*protein_PT
j_Item <- which(Production.Item.name == "Eggs, hen, in shell")
country_protein[,7,] <- Production.mat[,j_Item,]*protein_EG
j_Item <- which(Production.Item.name == "Meat, duck")
country_protein[,8,] <- Production.mat[,j_Item,]*protein_PT
j_Item1 <- which(Production.Item.name == "Milk, whole fresh goat")
j_Item2 <- which(Production.Item.name == "Meat, goat")
country_protein[,9,] <- Production.mat[,j_Item1,]*protein_MK+Production.mat[,j_Item2,]*protein_GT
j_Item <- which(Production.Item.name == "Meat, horse")
country_protein[,10,] <- Production.mat[,j_Item,]*protein_BV
j_Item <- which(Production.Item.name == "Meat, other camelids")
country_protein[,11,] <- Production.mat[,j_Item,]*protein_GT
j_Item <- which(Production.Item.name == "Meat, mule")
country_protein[,12,] <- Production.mat[,j_Item,]*protein_GT
j_Item1 <- which(Production.Item.name == "Milk, whole fresh sheep")
j_Item2 <- which(Production.Item.name == "Meat, sheep")
country_protein[,13,] <- Production.mat[,j_Item1,]*protein_MK+Production.mat[,j_Item2,]*protein_SH
# No production for "Swine, breeding"
country_protein[,14,] <- 0.0
j_Item <- which(Production.Item.name == "Meat, pig")
country_protein[,15,] <- Production.mat[,j_Item,]*protein_PG
j_Item <- which(Production.Item.name == "Meat, turkey")
country_protein[,16,] <- Production.mat[,j_Item,]*protein_PT
# pigs ("Swine, breeding" and "Swine, market") 
country_protein[,17,] <- country_protein[,14,] + country_protein[,15,]
# poultry ("Chickens, broilers", "Chickens, layers", "Ducks", and "Turkeys")
country_protein[,18,] <- apply(country_protein[,c(6,7,8,16),],c(1,3),sum,na.rm=TRUE)
# Major protein producing livestock
country_protein[,19,] <- apply(country_protein[,c(2,4,5,9,13,17,18),],c(1,3),sum,na.rm=TRUE)

##########################################
# 4.4.3 calculate country/area livestock CH4 emission intensity (EmisInt) per protein produced
##########################################

# Note: emissions from draft animals should be excluded in calculating emission intensity per protein produced
# Here, we apply the same share of draft animal emissions from 2019MT to those from FAOSTAT (2006T1) and 2019T1
# 4.4.3.1 calculate share of emissions from draft animals
share_country_CH4_draft <- country_CH4_2019MT_draft/country_CH4_2019MT
share_country_CH4_draft[is.infinite(share_country_CH4_draft) | is.na(share_country_CH4_draft) | share_country_CH4_draft < 0.0] <- 0.0

# 4.4.3.2 calculate emissions from animals producing protein (i.e., excluding draft animals)
country_CH4_FAOSTAT_nodraft <- country_CH4_FAOSTAT * (1.0 - share_country_CH4_draft)
country_CH4_2019T1_nodraft <- country_CH4_2019T1 * (1.0 - share_country_CH4_draft)
country_CH4_2019MT_nodraft <- country_CH4_2019MT * (1.0 - share_country_CH4_draft)

# 4.4.3.3 calculate emissions for six periods (2000s and 2010s)
p5yr <- c("2000-2004","2005-2009","2010-2014","2014-2017","2000-2009","2010-2018")
p5yr.start <- c(1,6,11,15,1,11)
p5yr.end <- c(5,10,15,18,10,19)
p5yr_country_CH4_FAOSTAT_nodraft <- array(NA,c(3,243,19,length(p5yr)))
p5yr_country_CH4_2019T1_nodraft <- array(NA,c(3,243,19,length(p5yr)))
p5yr_country_CH4_2019MT_nodraft <- array(NA,c(3,243,19,length(p5yr)))
p5yr_country_protein <- array(NA,c(243,19,length(p5yr)))
for (p in 1:length(p5yr)) {
  p5yr_country_CH4_FAOSTAT_nodraft[,,1:16,p] <- apply(country_CH4_FAOSTAT_nodraft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_CH4_2019T1_nodraft[,,1:16,p] <- apply(country_CH4_2019T1_nodraft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_CH4_2019MT_nodraft[,,1:16,p] <- apply(country_CH4_2019MT_nodraft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_protein[,1:19,p] <- apply(country_protein[,1:19,(p5yr.start[p]):(p5yr.end[p])],c(1,2),mean,na.rm=TRUE)
}
# pigs
p5yr_country_CH4_FAOSTAT_nodraft[,,17,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_nodraft[,,17,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_nodraft[,,17,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
# poultry
p5yr_country_CH4_FAOSTAT_nodraft[,,18,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_nodraft[,,18,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_nodraft[,,18,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
# Major protein producing livestock
p5yr_country_CH4_FAOSTAT_nodraft[,,19,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,c(2,4,5,9,13,17,18),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_nodraft[,,19,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,c(2,4,5,9,13,17,18),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_nodraft[,,19,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,c(2,4,5,9,13,17,18),],c(1,2,4),sum,na.rm=TRUE) 

# 4.4.3.4 calculate emission intensity (EmisInt) per protein produced for six periods (2000s and 2010s)
# Gg CH4 *1e3 / tonnes protein = kg CH4 / kg protein
p5yr_country_EmisInt_FAOSTAT <- array(NA,c(3,243,19,length(p5yr)))
p5yr_country_EmisInt_2019T1 <- array(NA,c(3,243,19,length(p5yr)))
p5yr_country_EmisInt_2019MT <- array(NA,c(3,243,19,length(p5yr)))
for (qq in 1:3) {
  p5yr_country_EmisInt_FAOSTAT[qq,,1:19,] <- p5yr_country_CH4_FAOSTAT_nodraft[qq,,1:19,]/p5yr_country_protein[,1:19,]*1e3
  p5yr_country_EmisInt_2019T1[qq,,1:19,] <- p5yr_country_CH4_2019T1_nodraft[qq,,1:19,]/p5yr_country_protein[,1:19,]*1e3
  p5yr_country_EmisInt_2019MT[qq,,1:19,] <- p5yr_country_CH4_2019MT_nodraft[qq,,1:19,]/p5yr_country_protein[,1:19,]*1e3
}

##########################################
# 4.4.4 calculate regional livestock CH4 emission intensity (EmisInt) per protein produced
##########################################

# 4.4.4.1 calculate regional protein production for major livestock categories for six periods (2000s and 2010s)
p5yr_regional_protein <- array(NA,c(11,19,length(p5yr)))
for (k in 1:10) {
  if (k == 2) {
    p5yr_regional_protein[k,,] <- p5yr_country_protein[gleam.list[[k]],,]
  } else {
    p5yr_regional_protein[k,,] <- apply(p5yr_country_protein[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
  }
}
k <- 11 # global value
p5yr_regional_protein[k,,] <- apply(p5yr_country_protein[,,],c(2,3),sum,na.rm=TRUE)

# 4.4.4.2 calculate regional emissions from animals producing protein (i.e., excluding draft animals)
p5yr_regional_CH4_FAOSTAT_nodraft <- array(NA,c(3,11,19,length(p5yr)))
p5yr_regional_CH4_2019T1_nodraft <- array(NA,c(3,11,19,length(p5yr)))
p5yr_regional_CH4_2019MT_nodraft <- array(NA,c(3,11,19,length(p5yr)))
for (k in 1:10) {
  if (k == 2) {
    p5yr_regional_CH4_FAOSTAT_nodraft[,k,,] <- p5yr_country_CH4_FAOSTAT_nodraft[,gleam.list[[k]],,]
    p5yr_regional_CH4_2019T1_nodraft[,k,,] <- p5yr_country_CH4_2019T1_nodraft[,gleam.list[[k]],,]
    p5yr_regional_CH4_2019MT_nodraft[,k,,] <- p5yr_country_CH4_2019MT_nodraft[,gleam.list[[k]],,]
  } else {
    p5yr_regional_CH4_FAOSTAT_nodraft[,k,,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    p5yr_regional_CH4_2019T1_nodraft[,k,,] <- apply(p5yr_country_CH4_2019T1_nodraft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    p5yr_regional_CH4_2019MT_nodraft[,k,,] <- apply(p5yr_country_CH4_2019MT_nodraft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
  }
}
k <- 11 # global value
p5yr_regional_CH4_FAOSTAT_nodraft[,k,,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,,],c(1,3,4),sum,na.rm=TRUE)
p5yr_regional_CH4_2019T1_nodraft[,k,,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,,],c(1,3,4),sum,na.rm=TRUE)
p5yr_regional_CH4_2019MT_nodraft[,k,,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,,],c(1,3,4),sum,na.rm=TRUE)

# 4.4.4.3 calculate regional emission intensity (EmisInt) per protein produced
p5yr_regional_EmisInt_FAOSTAT <- array(NA,c(3,11,19,length(p5yr)))
p5yr_regional_EmisInt_2019T1 <- array(NA,c(3,11,19,length(p5yr)))
p5yr_regional_EmisInt_2019MT <- array(NA,c(3,11,19,length(p5yr)))
# Gg CH4 *1e3 / tonnes protein = kg CH4 / kg protein
for (qq in 1:3) {
  p5yr_regional_EmisInt_FAOSTAT[qq,,1:19,] <- p5yr_regional_CH4_FAOSTAT_nodraft[qq,,1:19,]/p5yr_regional_protein[,1:19,]*1e3
  p5yr_regional_EmisInt_2019T1[qq,,1:19,] <- p5yr_regional_CH4_2019T1_nodraft[qq,,1:19,]/p5yr_regional_protein[,1:19,]*1e3
  p5yr_regional_EmisInt_2019MT[qq,,1:19,] <- p5yr_regional_CH4_2019MT_nodraft[qq,,1:19,]/p5yr_regional_protein[,1:19,]*1e3
}

##########################################
# save output of livestock CH4 emissions and emission intensity per kg protein using the 2019 MT, 2019 T1, and 2006 T1 methods
##########################################
save.image(file=paste0(path_out,"Livestock_CH4_Emissions_and_EmissionIntensity_per_Protein_Produced.RData"))
# clear cache 
rm(list=ls())
gc()


##########################################
##########################################
# Part V Regression between livestock CH4 emission intensity per protein produced and GDP per capita at country/area and regional levels
##########################################
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

##########################################
##########################################
# 5.1 Load livestock CH4 emission intensity from FAOSTAT (2006T1), 2019T1, and 2019MT
##########################################
##########################################

# emission intensities and GDP per capita
load(file=paste0(path_out,"Livestock_CH4_Emissions_and_EmissionIntensity_per_Protein_Produced.RData"))

##########################################
##########################################
# 5.2 Re-calculate livestock CH4 emission intensity of each 5 year period from FAOSTAT (2006T1), 2019T1, and 2019MT
##########################################
##########################################

##########################################
# 5.2.1 calculate emissions for six periods (2000s and 2010s)
##########################################
p5yr <- c("2000-2004","2005-2009","2010-2014","2014-2017","2000-2009","2010-2017")
p5yr.start <- c(1,6,11,15,1,11)
p5yr.end <- c(5,10,15,18,10,18)
p5yr_country_CH4_FAOSTAT_nodraft <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_CH4_2019T1_nodraft <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_CH4_2019MT_nodraft <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_protein <- array(NA,c(243,18,length(p5yr)))
for (p in 1:length(p5yr)) {
  p5yr_country_CH4_FAOSTAT_nodraft[,,1:16,p] <- apply(country_CH4_FAOSTAT_nodraft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_CH4_2019T1_nodraft[,,1:16,p] <- apply(country_CH4_2019T1_nodraft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_CH4_2019MT_nodraft[,,1:16,p] <- apply(country_CH4_2019MT_nodraft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_protein[,1:18,p] <- apply(country_protein[,1:18,(p5yr.start[p]):(p5yr.end[p])],c(1,2),mean,na.rm=TRUE)
}
# pigs
p5yr_country_CH4_FAOSTAT_nodraft[,,17,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_nodraft[,,17,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_nodraft[,,17,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
# poultry
p5yr_country_CH4_FAOSTAT_nodraft[,,18,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_nodraft[,,18,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_nodraft[,,18,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 

##########################################
# 5.2.2 calculate emission intensity (EmisInt) per protein produced for six periods (2000s and 2010s)
##########################################

# Gg CH4 *1e3 / tonnes protein = kg CH4 / kg protein
p5yr_country_EmisInt_FAOSTAT <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_EmisInt_2019T1 <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_EmisInt_2019MT <- array(NA,c(3,243,18,length(p5yr)))
for (qq in 1:3) {
  p5yr_country_EmisInt_FAOSTAT[qq,,1:18,] <- p5yr_country_CH4_FAOSTAT_nodraft[qq,,1:18,]/p5yr_country_protein[,1:18,]*1e3
  p5yr_country_EmisInt_2019T1[qq,,1:18,] <- p5yr_country_CH4_2019T1_nodraft[qq,,1:18,]/p5yr_country_protein[,1:18,]*1e3
  p5yr_country_EmisInt_2019MT[qq,,1:18,] <- p5yr_country_CH4_2019MT_nodraft[qq,,1:18,]/p5yr_country_protein[,1:18,]*1e3
}

##########################################
# 5.2.3 calculate regional livestock CH4 emission intensity (EmisInt) per protein produced
##########################################

# 5.2.3.1 calculate regional protein production for major livestock categories for six periods (2000s and 2010s)
p5yr_regional_protein <- array(NA,c(11,16+2,length(p5yr)))
for (k in 1:10) {
  if (k == 2) {
    p5yr_regional_protein[k,,] <- p5yr_country_protein[gleam.list[[k]],,]
  } else {
    p5yr_regional_protein[k,,] <- apply(p5yr_country_protein[gleam.list[[k]],,],c(2,3),sum,na.rm=TRUE)  
  }
}
k <- 11 # global value
p5yr_regional_protein[k,,] <- apply(p5yr_country_protein[,,],c(2,3),sum,na.rm=TRUE)

# 5.2.3.2 calculate regional emissions from animals producing protein (i.e., excluding draft animals)
p5yr_regional_CH4_FAOSTAT_nodraft <- array(NA,c(3,11,18,length(p5yr)))
p5yr_regional_CH4_2019T1_nodraft <- array(NA,c(3,11,18,length(p5yr)))
p5yr_regional_CH4_2019MT_nodraft <- array(NA,c(3,11,18,length(p5yr)))
for (k in 1:10) {
  if (k == 2) {
    p5yr_regional_CH4_FAOSTAT_nodraft[,k,,] <- p5yr_country_CH4_FAOSTAT_nodraft[,gleam.list[[k]],,]
    p5yr_regional_CH4_2019T1_nodraft[,k,,] <- p5yr_country_CH4_2019T1_nodraft[,gleam.list[[k]],,]
    p5yr_regional_CH4_2019MT_nodraft[,k,,] <- p5yr_country_CH4_2019MT_nodraft[,gleam.list[[k]],,]
  } else {
    p5yr_regional_CH4_FAOSTAT_nodraft[,k,,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    p5yr_regional_CH4_2019T1_nodraft[,k,,] <- apply(p5yr_country_CH4_2019T1_nodraft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
    p5yr_regional_CH4_2019MT_nodraft[,k,,] <- apply(p5yr_country_CH4_2019MT_nodraft[,gleam.list[[k]],,],c(1,3,4),sum,na.rm=TRUE)  
  }
}
k <- 11 # global value
p5yr_regional_CH4_FAOSTAT_nodraft[,k,,] <- apply(p5yr_country_CH4_FAOSTAT_nodraft[,,,],c(1,3,4),sum,na.rm=TRUE)
p5yr_regional_CH4_2019T1_nodraft[,k,,] <- apply(p5yr_country_CH4_2019T1_nodraft[,,,],c(1,3,4),sum,na.rm=TRUE)
p5yr_regional_CH4_2019MT_nodraft[,k,,] <- apply(p5yr_country_CH4_2019MT_nodraft[,,,],c(1,3,4),sum,na.rm=TRUE)

# 5.2.3.3 calculate regional emission intensity (EmisInt) per protein produced
p5yr_regional_EmisInt_FAOSTAT <- array(NA,c(3,11,18,length(p5yr)))
p5yr_regional_EmisInt_2019T1 <- array(NA,c(3,11,18,length(p5yr)))
p5yr_regional_EmisInt_2019MT <- array(NA,c(3,11,18,length(p5yr)))
# Gg CH4 *1e3 / tonnes protein = kg CH4 / kg protein
for (qq in 1:3) {
  p5yr_regional_EmisInt_FAOSTAT[qq,,1:18,] <- p5yr_regional_CH4_FAOSTAT_nodraft[qq,,1:18,]/p5yr_regional_protein[,1:18,]*1e3
  p5yr_regional_EmisInt_2019T1[qq,,1:18,] <- p5yr_regional_CH4_2019T1_nodraft[qq,,1:18,]/p5yr_regional_protein[,1:18,]*1e3
  p5yr_regional_EmisInt_2019MT[qq,,1:18,] <- p5yr_regional_CH4_2019MT_nodraft[qq,,1:18,]/p5yr_regional_protein[,1:18,]*1e3
}

##########################################
# 5.2.4 find minimum (0.05-quantile) and maximum (0.95-quantile) EmisInt as limite 
##########################################

Dset <- c("2019MT","2019T1","FAOSTAT","EDGARv5")
for (w in 1:3) {
  min_tmp <- array(NA,c(18))
  qt05_tmp <- array(NA,c(18))
  max_tmp <- array(NA,c(18))
  qt95_tmp <- array(NA,c(18))
  for (m in c(2,4,5,9,13,17,18)) {
    # protein
    tmp_protein <- p5yr_country_protein[,m,4]
    # minimum of the a 5-year period around 2012 (2010-2014)
    tmp <- eval(parse(text=paste0("p5yr_country_EmisInt_",Dset[w])))[3,,m,3]
    tmp[is.infinite(tmp) | tmp <= 0.0] <- NA
    # only when protein production > 10 tonne, to avoid the unrealistic EmisInt from small countries/areas
    index <- which(tmp_protein > 100.0 & is.na(tmp_protein) == FALSE & is.finite(tmp_protein)) 
    min_tmp[m] <- min(tmp[index],na.rm=TRUE)
    qt05_tmp[m] <- quantile(tmp[index],probs=c(0.05),na.rm=TRUE)
    max_tmp[m] <- max(tmp[index],na.rm=TRUE)
    qt95_tmp[m] <- quantile(tmp[index],probs=c(0.95),na.rm=TRUE)
  }
  assign(paste0("min_country_protein100_EmisInt_",Dset[w]),min_tmp)
  assign(paste0("qt05_country_protein100_EmisInt_",Dset[w]),qt05_tmp)
  assign(paste0("max_country_protein100_EmisInt_",Dset[w]),max_tmp)
  assign(paste0("qt95_country_protein100_EmisInt_",Dset[w]),qt95_tmp)
}


##########################################
##########################################
# 5.3 Calculate GDP per capita over 5-yr periods
##########################################
##########################################

p5yr <- c("2000-2004","2005-2009","2010-2014","2014-2017","2000-2009","2010-2017")
p5yr.start <- c(1,6,11,15,1,11)
p5yr.end <- c(5,10,15,18,10,18)
p5yr_country_GDP <- array(NA,c(243,length(p5yr)))
p5yr_country_Population <- array(NA,c(243,length(p5yr)))
p5yr_country_GDPperCapita <- array(NA,c(243,length(p5yr)))
for (p in 1:length(p5yr)) {
  p5yr_country_GDP[,p] <- apply(GDP.mat[,1,(p5yr.start[p]):(p5yr.end[p])],c(1),mean,na.rm=TRUE)
  p5yr_country_Population[,p] <- apply(Population.mat[,1,(p5yr.start[p]):(p5yr.end[p])],c(1),mean,na.rm=TRUE)
  p5yr_country_GDPperCapita[,p] <- apply(GDP_per_capita.mat[,1,(p5yr.start[p]):(p5yr.end[p])],c(1),mean,na.rm=TRUE)
}

p5yr_regional_GDP <- array(NA,c(11,length(p5yr)))
p5yr_regional_Population <- array(NA,c(11,length(p5yr)))
p5yr_regional_GDPperCapita <- array(NA,c(11,length(p5yr)))
for (k in 1:10) {
  if (k == 2) {
    p5yr_regional_GDP[k,] <- p5yr_country_GDP[gleam.list[[k]],]
    p5yr_regional_Population[k,] <- p5yr_country_Population[gleam.list[[k]],]
  } else {
    p5yr_regional_GDP[k,] <- apply(p5yr_country_GDP[gleam.list[[k]],],c(2),sum,na.rm=TRUE)  
    p5yr_regional_Population[k,] <- apply(p5yr_country_Population[gleam.list[[k]],],c(2),sum,na.rm=TRUE)  
  }
}
p5yr_regional_GDPperCapita <- p5yr_regional_GDP/p5yr_regional_Population


##########################################
##########################################
# 5.4 Save Regression between 5-yr periods Emission intensities and GDP per capita for the period 2000-2017
##########################################
##########################################

spatial_cate_name <- c("Asses","Buffaloes","Camels","CattleDairy","CattleNonDairy",
                       "ChickensBroilers","ChickensLayers","Ducks","Goats","Horses",
                       "Llamas","Mules","Sheep","SwineBreeding","SwineMarket",
                       "Turkeys")
cate18 <- c(spatial_cate_name,"Pigs","Poultry")

# function as log(EmisInt) = a * log(GDPperCapita) + b
# a,b,R2,pvalue
regression_coeff_FAOSTAT <- array(NA,c(243,18,4))
regression_coeff_2019T1 <- array(NA,c(243,18,4))
regression_coeff_2019MT <- array(NA,c(243,18,4))
Dset <- c("2019MT","2019T1","FAOSTAT","EDGARv5")
for (w in 1:3) {
  regression_coeff <- array(0.0,c(243,18,4))
  for (m in c(2,4,5,9,13,17,18)) {
    cxx <- p5yr_country_GDPperCapita[,1:4]
    cyy <- eval(parse(text=paste0("p5yr_country_EmisInt_",Dset[w])))[3,,m,1:4]
    
    cxx[is.infinite(cxx) | cxx <= 0.0] <- NA
    cyy[is.infinite(cyy) | cyy <= 0.0] <- NA
    
    for (cc in 1:243) {
      xx <- cxx[cc,]
      yy <- cyy[cc,]
      xx[is.infinite(xx) | xx <= 0.0] <- NA
      yy[is.infinite(yy) | yy <= 0.0] <- NA
      index <- which(is.na(xx) == FALSE & is.na(yy) == FALSE)
      if (length(index) > 2) {
        # regression and record results
        # linear regression
        fit <- lm(yy~xx,na.action=na.omit)
        # get coefficients
        # a
        regression_coeff[cc,m,1] <- fit$coefficients[2]
        # b
        regression_coeff[cc,m,2] <- fit$coefficients[1]
        # R2
        regression_coeff[cc,m,3] <- summary(fit)$r.squared
        # pvalue
        regression_coeff[cc,m,4] <- summary(fit)$coefficients[2,4]
      }
      
    }
  }  
  #print(regression_coeff)
  assign(paste0("regression_coeff_",Dset[w]),regression_coeff)
}

##########################################
##########################################
# 5.5 Save Regression between 5-yr periods Emissions from draft animals and GDP per capita for each country/area during the period 2000-2017
##########################################
##########################################

##########################################
# 5.5.1 calculate emissions from draft animals only
##########################################
calc_country_CH4_FAOSTAT_draft <- array(NA,c(3,243,18,length(year)))
calc_country_CH4_2019T1_draft <- array(NA,c(3,243,18,length(year)))
calc_country_CH4_2019MT_draft <- array(NA,c(3,243,18,length(year)))

calc_country_CH4_FAOSTAT_draft[,,1:16,] <- country_CH4_FAOSTAT[,,1:16,] - country_CH4_FAOSTAT_nodraft[,,1:16,] 
calc_country_CH4_2019T1_draft[,,1:16,] <- country_CH4_2019T1[,,1:16,] - country_CH4_2019T1_nodraft[,,1:16,] 
calc_country_CH4_2019MT_draft[,,1:16,] <- country_CH4_2019MT[,,1:16,] - country_CH4_2019MT_nodraft[,,1:16,] 

##########################################
# 5.5.2 calculate emissions for six periods (2000s and 2010s)
##########################################

p5yr <- c("2000-2004","2005-2009","2010-2014","2014-2017","2000-2009","2010-2017")
p5yr.start <- c(1,6,11,15,1,11)
p5yr.end <- c(5,10,15,18,10,18)
p5yr_country_CH4_FAOSTAT_draft <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_CH4_2019T1_draft <- array(NA,c(3,243,18,length(p5yr)))
p5yr_country_CH4_2019MT_draft <- array(NA,c(3,243,18,length(p5yr)))
for (p in 1:length(p5yr)) {
  p5yr_country_CH4_FAOSTAT_draft[,,1:16,p] <- apply(calc_country_CH4_FAOSTAT_draft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_CH4_2019T1_draft[,,1:16,p] <- apply(calc_country_CH4_2019T1_draft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
  p5yr_country_CH4_2019MT_draft[,,1:16,p] <- apply(calc_country_CH4_2019MT_draft[,,1:16,(p5yr.start[p]):(p5yr.end[p])],c(1,2,3),mean,na.rm=TRUE)
}
# pigs
p5yr_country_CH4_FAOSTAT_draft[,,17,] <- apply(p5yr_country_CH4_FAOSTAT_draft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_draft[,,17,] <- apply(p5yr_country_CH4_2019T1_draft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_draft[,,17,] <- apply(p5yr_country_CH4_2019MT_draft[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
# poultry
p5yr_country_CH4_FAOSTAT_draft[,,18,] <- apply(p5yr_country_CH4_FAOSTAT_draft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019T1_draft[,,18,] <- apply(p5yr_country_CH4_2019T1_draft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
p5yr_country_CH4_2019MT_draft[,,18,] <- apply(p5yr_country_CH4_2019MT_draft[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 

# set nondraft (except "Buffaloes" and "CattleNonDairy") to 0
nondraft_cate <- c(1,3,4,6:18)

tmp <- p5yr_country_CH4_FAOSTAT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
p5yr_country_CH4_FAOSTAT_draft <- tmp

tmp <- p5yr_country_CH4_2019T1_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
p5yr_country_CH4_2019T1_draft <- tmp

tmp <- p5yr_country_CH4_2019MT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
p5yr_country_CH4_2019MT_draft <- tmp

##########################################
##########################################
# 5.5.3 get relationship between Emission from draft animals and GDP per capita for each country/area
##########################################
##########################################

# function as CH4_draft = a * GDPperCapita + b
# a,b,R2,pvalue
draft_regression_coeff_FAOSTAT <- array(NA,c(243,18,4))
draft_regression_coeff_2019T1 <- array(NA,c(243,18,4))
draft_regression_coeff_2019MT <- array(NA,c(243,18,4))
Dset <- c("2019MT","2019T1","FAOSTAT","EDGARv5")
for (w in 1:3) {
  tmp_draft <- eval(parse(text=paste0("p5yr_country_CH4_",Dset[w],"_draft")))[3,,,]
  # there is possibility that emission from draft animal = 0, need to set to NA for regression
  tmp_draft[is.na(tmp_draft) | is.infinite(tmp_draft) | tmp_draft <= 0.0] <- NA
  tmp_regression_coeff <- array(0.0,c(243,18,4))
  for (cc in 1:243) {
    for (m in c(2,5)) {
      # GDP has only 18 years from 2000 to 2017
      yy <- tmp_draft[cc,m,1:4]
      xx <- p5yr_country_GDPperCapita[cc,1:4]
      # only do regression when there is more than 1 
      index <- which(is.na(yy) == FALSE & yy > 0.0 & is.na(xx) == FALSE & xx > 0.0)
      if (length(index) >= 2) {
        # regression and record results
        # linear regression
        fit <- lm(yy~xx,na.action=na.omit)
        # get coefficients
        # a
        tmp_regression_coeff[cc,m,1] <- fit$coefficients[2]
        # b
        tmp_regression_coeff[cc,m,2] <- fit$coefficients[1]
        # R2
        tmp_regression_coeff[cc,m,3] <- summary(fit)$r.squared
        # pvalue
        tmp_regression_coeff[cc,m,4] <- summary(fit)$coefficients[2,4]
      }
    }
  }  
  assign(paste0("draft_regression_coeff_",Dset[w]),tmp_regression_coeff)
}

# Note that when the regression slope > 0 (i.e., emission from draft animal increase following increasing GDP per Capita)
# We should not apply the trend, but assume the constant emissions as that of 2018 

##########################################
# save output of regression coefficients between emission intensity from non-draft animal (or emission from draft animal) and GDP per capita
##########################################
save.image(file=paste0(path_out,"Livestock_CH4_Emissions_Regression_EmisInt_vs_GDPperCapita_p5yr.RData"))
# clear cache 
rm(list=ls())
gc()


##########################################
##########################################
# Part VI Project future livestock CH4 emissions using 2006T1 (FAOSTAT), 2019T1 and 2019MT methods
##########################################
##########################################

# Set root path
path_base <- dirname(rstudioapi::getSourceEditorContext()$path)
# Set data path
path_data <- paste0(path_base,"/Data/")
# Set path to save output
path_out <- paste0(path_base,"/Saved_RData/")
dir.create(path_out, showWarnings = FALSE)

# Load library
library(ncdf4)

##########################################
##########################################
# 6.1 Read GDP per capita and livestock projections from FAO, 2018 "The future of food and agriculture – Alternative pathways to 2050"
##########################################
##########################################
path_faodata <- paste0(path_data,"Source_FAODATA/")
csv.FAO2018var <- c("GDP_per_capita_Projection","Livestock_Projection")
for (mm in 1:2) { # variables
  if (mm == 1) {
    # 1.1 GDP
    csv.name <- paste0(path_faodata,"FOFA2050CountryData_Macro-indicators.csv")
  } else if (mm == 2) {
    # 1.2 Manure_Management_CH4
    csv.name <- paste0(path_faodata,"FOFA2050CountryData_Livestock.csv")
  }
  # read csv file
  df.data <- read.table(csv.name,header=TRUE,sep=",",colClasses = "character")
  dim(df.data);lapply(df.data, class)
  # assign data.frame
  assign(paste0(csv.FAO2018var[mm],".df.data"),df.data)
  # get levels of each column
  c.name <- colnames(df.data)
  for (k in 1:length(c.name)) {
    assign(paste0(csv.FAO2018var[mm],".",c.name[k],".name"),levels(factor(df.data[,k])))
  }
} # variables  

##########################################
# 6.2 Re-arrange data of GDP per capita and livestock projections from FAO, 2018
##########################################

# Read country/area name from world map
library(ncdf4)
path_map <- paste0(path_data,"Input_MAP/")
## get 243 country/area name and country id
ncfname <- paste(path_map,"regionname_world_country_0.5degree.nc",sep="")
ncin <- nc_open(ncfname)
regionid <- ncvar_get(ncin,"regionid")
idname <- ncvar_get(ncin,"idname")
nc_close(ncin)

# Read crosswalk table for mapping the countries from FAOSTAT to 243 countries/areas used in the map
# FAO country list is adapted from http://www.fao.org/countryprofiles/iso3list/en/
csv.name <- paste0(path_map,"map_regionid_FAO_List_of_countries_and_codes.csv")
df.crosswalk <- read.table(csv.name,header=TRUE,sep=",",colClasses = "character")
# convert from characters to numbers
df.crosswalk$GLEAM_Region <- as.numeric(df.crosswalk$GLEAM_Region)
df.crosswalk$Developed <- as.numeric(df.crosswalk$Developed)

# Re-arrange data from FAOSTAT to 243 countries/areas
for (mm in 1:2) {
  df.arranged <- eval(parse(text=paste0(csv.FAO2018var[mm],".df.data")))
  df.arranged$regionid <- 9999
  CountryCode.name <- eval(parse(text=paste0(csv.FAO2018var[mm],".CountryCode.name")))
  # check each country/area name fit and record countries/area in the map that have been allocated
  allocated.countryID <- array(NA,c(243))
  for (i in 1:length(idname)) {
    #print(df.crosswalk[i,])
    # country.name
    for (j in 1:length(CountryCode.name)) {
      if (CountryCode.name[j] == df.crosswalk$ISO3[i]) {
        index <- which(df.arranged$CountryCode == CountryCode.name[j])
        df.arranged$regionid[index] <- df.crosswalk$countryID[i]
        # countryID have been allocated with projections
        allocated.countryID[i] <- 1.0
      }
    }
  }
  
  # Countries/Areas that need combination
  # SRB Serbia as Serbia and Montenegro in the map
  index <- which(df.arranged$CountryCode == "SRB") 
  df.arranged$regionid[index] <- df.crosswalk$countryID[31]
  
  # detect the country/area that can not be allocated to the map of 243 countries/areas
  index <- which(df.arranged$regionid == 9999)
  df.left <- df.arranged[index,]
  FAO_rest_name <- levels(factor(df.left$CountryCode))
  print(csv.FAO2018var[mm]);print(FAO_rest_name)
  
  # allocate items to the matrix with 243 countries/areas
  # regionid,Item,Year
  mat.temp <- array(NA,c(243,length(eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))),
                          length(eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))),
                          length(eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))),
                          length(eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))),
                          length(eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name"))))))
  # loop data points
  for (i in 1:length(df.arranged[,1])) {
    if (i%%1000 == 0) { print(paste(csv.FAO2018var[mm],i)) }
    # loop country
    for (j_id in 1:243) {
      # if regionid 
      if (df.arranged$regionid[i] == j_id) {
        # loop Indicator
        for (j_Indicator in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name"))))) {
          if (df.arranged$Indicator[i] == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator]) {
            # loop Item
            for (j_Item in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name"))))) {
              if (df.arranged$Item[i] == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item]) {
                # loop Element
                for (j_Element in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name"))))) {
                  if (df.arranged$Element[i] == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element]) {
                    # loop Scenario
                    for (j_Scenario in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name"))))) {
                      if (df.arranged$Scenario[i] == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario]) {
                        # loop Year
                        for (j_Year in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name"))))) {
                          if (df.arranged$Year[i] == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]) {
                            mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[i])
                          }
                        }
                      }
                    }  
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  
  # allocate items to the matrix with the rest of the 243 countries/areas that was not allocated with value
  # loop country
  for (j_id in 1:243) {
    # loop Indicator
    for (j_Indicator in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name"))))) {
      # loop Item
      for (j_Item in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name"))))) {
        # loop Element
        for (j_Element in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name"))))) {
          # loop Scenario
          for (j_Scenario in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name"))))) {
            # loop Year
            for (j_Year in 1:length(eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name"))))) {
              if (is.na(mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year])) { # which have not been allocated
                # detect its GLEAM_region, Developed etc.
                # rest of "XEAP" "XECA" "XEU"  "XHIC" "XLAC" "XMNA" "XSAS" "XSSA"
                # East Asia and Pacific, Europe and Central Asia, European Union, 
                # High income countries, Latin America and Caribbean, Middle East and North America, 
                # SouthAsia, Sub-SaharanAfrica
                # region_abb <- c("NAM","RUS","WEU","EEU","NENA","ESA","OCE","SAS","LAC","SSA","Global")
                
                if ((df.crosswalk$GLEAM_Region[j_id] == 10) & df.crosswalk$Developed[j_id] == 0) { # 1. XSSA
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XSSA")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else if ((df.crosswalk$GLEAM_Region[j_id] == 8) & df.crosswalk$Developed[j_id] == 0) { # 2. XSAS
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XSAS")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else if ((df.crosswalk$GLEAM_Region[j_id] == 5) & df.crosswalk$Developed[j_id] == 0) { # 3. XMNA
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XMNA")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else if ((df.crosswalk$GLEAM_Region[j_id] == 9) & df.crosswalk$Developed[j_id] == 0) { # 4. XLAC
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XLAC")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else if ((df.crosswalk$GLEAM_Region[j_id] == 6 | df.crosswalk$GLEAM_Region[j_id] == 7) & df.crosswalk$Developed[j_id] == 0) { # 5. XEAP developing
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XEAP")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else if ((df.crosswalk$GLEAM_Region[j_id] <= 4) & df.crosswalk$Developed[j_id] == 0) { # 6. XECA developing
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XECA")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else if ((df.crosswalk$GLEAM_Region[j_id] == 3 | df.crosswalk$GLEAM_Region[j_id] == 4) & df.crosswalk$Developed[j_id] == 1) { # 7. XEU (all developed)
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XEU")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } else { # 8. XHIC developed countries/area
                  k_df <- which(df.arranged$Indicator == eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator] &
                                  df.arranged$Item == eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item] &
                                  df.arranged$Element == eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element] &
                                  df.arranged$Scenario == eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario] &
                                  df.arranged$Year == eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year] &
                                df.arranged$CountryCode == "XHIC")
                  if (length(k_df) == 1) {
                    mat.temp[j_id,j_Indicator,j_Item,j_Element,j_Scenario,j_Year] <- as.numeric(df.arranged$Value[k_df])
                  } else if (length(k_df) > 1) {
                    print(paste("error length = ",length(k_df),
                                df.crosswalk$countryID[j_id],df.crosswalk$countryname[j_id],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Indicator.name")))[j_Indicator],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Item.name")))[j_Item],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Element.name")))[j_Element],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Scenario.name")))[j_Scenario],
                                eval(parse(text=paste0(csv.FAO2018var[mm],".Year.name")))[j_Year]))
                  }
                } # choose GLEAM_region    
              }  
            }  
          }
        }
      }
    }
  }  

  assign(paste0(csv.FAO2018var[mm],".mat"),mat.temp)
}


##########################################
##########################################
# 6.3 calculate relative changes in livestock production using projections from FAO, 2018 The future of food and agriculture – Alternative pathways to 2050
##########################################
##########################################

#> dim(Livestock_Projection.mat)
#[1] 243   2   7   6   3   5
#> Livestock_Projection.Indicator.name
#[1] "Herd sizes"   "Productivity"
#> Livestock_Projection.Item.name
#[1] "Livestock units"      "Raising of buffaloes" "Raising of cattle"    "Raising of goats"     "Raising of pigs"     
#[6] "Raising of poultry"   "Raising of sheep"    
#> Livestock_Projection.Element.name
#[1] "Beef"                "Goat and sheep meat" "Pork"                "Poultry meat"        "Raw milk"           
#[6] "Total"              
#> Livestock_Projection.Scenario.name
#[1] "Business As Usual"     "Stratified Societies"  "Toward Sustainability"
#> Livestock_Projection.Year.name
#[1] "2012" "2030" "2035" "2040" "2050"

##########################################
# 6.3.1 parameters on the protein content of livestock products
##########################################

# Cattle (Dairy+other), Buffaloes, Sheep, Goats, Pigs, Poultry
# From Table 9.1 of GLEAM v2.0 Documentation
# BVMEAT 158g/kg SheepMEAT 141g/kg GoatMEAT 134g/kg PGMEAT 131g/kg ALMILK 33g/kg PTEGGS 124g/kg PTMEAT 143g/kg
protein_BV <- 0.158
protein_SH <- 0.141
protein_GT <- 0.134
protein_PG <- 0.131
protein_MK <- 0.033
protein_EG <- 0.124
protein_PT <- 0.143

##########################################
# 6.3.2 re-arrange projections on animal numbers and protein productivity
##########################################

# 243 country/area,3 Herd sizes/Productivity/Protein,7 Livestock_Projection.Item.name,2 Meat/Milk,3 Scenario,5 Year
Protein_Projection.mat <- array(NA,c(243,3,7,3,3,5))

# 6.3.2.1 re-arrange projections on animal numbers
# Herd sizes (both milking and meat animals) "Element" == "Total"
j_HerdSize <- 1
j_Productivity <- 2
j_Protein <- 3
j_Meat <- 1
j_Milk <- 2
j_MeatMilk <- 3
Protein_Projection.mat[1:243,j_HerdSize,1:7,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_HerdSize,1:7,6,1:3,1:5]
Protein_Projection.mat[1:243,j_HerdSize,1:7,j_Milk,1:3,1:5] <- Livestock_Projection.mat[1:243,j_HerdSize,1:7,6,1:3,1:5]

# 6.3.2.2 re-arrange projections on protein productivity
# Buffaloes Meat
j_Item <- which(Livestock_Projection.Item.name == "Raising of buffaloes")
j_Element <- which(Livestock_Projection.Element.name == "Beef")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_BV
# Buffaloes Milk
j_Item <- which(Livestock_Projection.Item.name == "Raising of buffaloes")
j_Element <- which(Livestock_Projection.Element.name == "Raw milk")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Milk,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_MK
# Cattle Meat
j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
j_Element <- which(Livestock_Projection.Element.name == "Beef")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_BV
# Cattle Milk
j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
j_Element <- which(Livestock_Projection.Element.name == "Raw milk")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Milk,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_MK
# Goats Meat
j_Item <- which(Livestock_Projection.Item.name == "Raising of goats")
j_Element <- which(Livestock_Projection.Element.name == "Goat and sheep meat")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_GT
# Goats Milk
j_Item <- which(Livestock_Projection.Item.name == "Raising of goats")
j_Element <- which(Livestock_Projection.Element.name == "Raw milk")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Milk,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_MK
# Sheep Meat
j_Item <- which(Livestock_Projection.Item.name == "Raising of sheep")
j_Element <- which(Livestock_Projection.Element.name == "Goat and sheep meat")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_SH
# Sheep Milk
j_Item <- which(Livestock_Projection.Item.name == "Raising of sheep")
j_Element <- which(Livestock_Projection.Element.name == "Raw milk")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Milk,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_MK
# Pigs Meat
j_Item <- which(Livestock_Projection.Item.name == "Raising of pigs")
j_Element <- which(Livestock_Projection.Element.name == "Pork")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_PG
# Poultry Meat
j_Item <- which(Livestock_Projection.Item.name == "Raising of poultry")
j_Element <- which(Livestock_Projection.Element.name == "Poultry meat")
Protein_Projection.mat[1:243,j_Productivity,j_Item,j_Meat,1:3,1:5] <- Livestock_Projection.mat[1:243,j_Productivity,j_Item,j_Element,1:3,1:5]*protein_PT

# 6.3.2.3 calculate projections on protein production
Protein_Projection.mat[1:243,j_Protein,1:7,1:2,1:3,1:5] <- Protein_Projection.mat[1:243,j_HerdSize,1:7,1:2,1:3,1:5]*Protein_Projection.mat[1:243,j_Productivity,1:7,1:2,1:3,1:5]
Protein_Projection.mat[is.na(Protein_Projection.mat) | is.infinite(Protein_Projection.mat)] <- 0.0
# Total protein (Meat + Milk)
Protein_Projection.mat[1:243,j_Protein,1:7,j_MeatMilk,1:3,1:5] <- Protein_Projection.mat[1:243,j_Protein,1:7,j_Meat,1:3,1:5] + Protein_Projection.mat[1:243,j_Protein,1:7,j_Milk,1:3,1:5]

##########################################                       
# 6.3.3 relative changes in protein production
##########################################

# 6.3.3.1 relative changes in protein production for 5 year from FAO, 2018
# 243 country/area,7 Livestock_Projection.Item.name,3 Meat/Milk/Meat+Milk,3 Scenario,5 Year
rel_Protein_Projection.mat <- array(NA,c(243,7,3,3,5))
for (i in 1:5) {
  # actual year divided by value of 2012 (2012 as 1)
  rel_Protein_Projection.mat[,,,,i] <- Protein_Projection.mat[,3,,,,i]/Protein_Projection.mat[,3,,,,1]
}
rel_Protein_Projection.mat[is.na(rel_Protein_Projection.mat) | is.infinite(rel_Protein_Projection.mat) | rel_Protein_Projection.mat < 0.0] <- 1.0

# 6.3.3.2 relative changes in protein production from 2012 to 2050
interp.year <- seq(2012,2050,by=1)
long_rel_Protein_Projection.mat <- array(NA,c(243,7,3,3,length(interp.year)))
long_rel_Protein_Projection.mat[,,,,1] <- rel_Protein_Projection.mat[,,,,1]
long_rel_Protein_Projection.mat[,,,,19] <- rel_Protein_Projection.mat[,,,,2]
long_rel_Protein_Projection.mat[,,,,24] <- rel_Protein_Projection.mat[,,,,3]
long_rel_Protein_Projection.mat[,,,,29] <- rel_Protein_Projection.mat[,,,,4]
long_rel_Protein_Projection.mat[,,,,39] <- rel_Protein_Projection.mat[,,,,5]
for (i in 1:(19-1-1)) {
  long_rel_Protein_Projection.mat[,,,,1+i] <- long_rel_Protein_Projection.mat[,,,,1]+
    (long_rel_Protein_Projection.mat[,,,,19]-long_rel_Protein_Projection.mat[,,,,1])/(19-1)*i
}
for (i in 1:(24-19-1)) {
  long_rel_Protein_Projection.mat[,,,,19+i] <- long_rel_Protein_Projection.mat[,,,,19]+
    (long_rel_Protein_Projection.mat[,,,,24]-long_rel_Protein_Projection.mat[,,,,19])/(24-19)*i
}
for (i in 1:(29-24-1)) {
  long_rel_Protein_Projection.mat[,,,,24+i] <- long_rel_Protein_Projection.mat[,,,,24]+
    (long_rel_Protein_Projection.mat[,,,,29]-long_rel_Protein_Projection.mat[,,,,24])/(29-24)*i
}
for (i in 1:(39-29-1)) {
  long_rel_Protein_Projection.mat[,,,,29+i] <- long_rel_Protein_Projection.mat[,,,,29]+
    (long_rel_Protein_Projection.mat[,,,,39]-long_rel_Protein_Projection.mat[,,,,29])/(39-29)*i
}


##########################################
##########################################
# 6.4 calculate relative changes in GDP per capita using projections from FAO, 2018 The future of food and agriculture – Alternative pathways to 2050
##########################################
##########################################

#> dim(GDP_per_capita_Projection.mat)
#[1] 243   2   1   2   3   5
#> GDP_per_capita_Projection.Indicator.name
#[1] "GDP per capita" "Population"    
#> GDP_per_capita_Projection.Item.name
#[1] "Total"
#> GDP_per_capita_Projection.Element.name
#[1] "GDP at market prices" "Total population"    
#> GDP_per_capita_Projection.Scenario.name
#[1] "Business As Usual"     "Stratified Societies"  "Toward Sustainability"
#> GDP_per_capita_Projection.Year.name
#[1] "2012" "2030" "2035" "2040" "2050"

##########################################
# 6.4.1 relative changes in GDP per capita for 5 year from FAO, 2018
##########################################

# 243 country/area,3 Scenario,5 Year
rel_GDP_per_capita_Projection.mat <- array(NA,c(243,3,5))
for (i in 1:5) {
  # actual year divided by value of 2012 (2012 as 1)
  rel_GDP_per_capita_Projection.mat[,,i] <- GDP_per_capita_Projection.mat[,1,1,1,,i]/GDP_per_capita_Projection.mat[,1,1,1,,1]
}
rel_GDP_per_capita_Projection.mat[is.na(rel_GDP_per_capita_Projection.mat) | is.infinite(rel_GDP_per_capita_Projection.mat) | rel_GDP_per_capita_Projection.mat < 0.0] <- 1.0

# 4.2 relative changes in GDP per capita from 2012 to 2050
interp.year <- seq(2012,2050,by=1)
long_rel_GDP_per_capita_Projection.mat <- array(NA,c(243,3,length(interp.year)))
long_rel_GDP_per_capita_Projection.mat[,,1] <- rel_GDP_per_capita_Projection.mat[,,1]
long_rel_GDP_per_capita_Projection.mat[,,19] <- rel_GDP_per_capita_Projection.mat[,,2]
long_rel_GDP_per_capita_Projection.mat[,,24] <- rel_GDP_per_capita_Projection.mat[,,3]
long_rel_GDP_per_capita_Projection.mat[,,29] <- rel_GDP_per_capita_Projection.mat[,,4]
long_rel_GDP_per_capita_Projection.mat[,,39] <- rel_GDP_per_capita_Projection.mat[,,5]
for (i in 1:(19-1-1)) {
  long_rel_GDP_per_capita_Projection.mat[,,1+i] <- long_rel_GDP_per_capita_Projection.mat[,,1]+
    (long_rel_GDP_per_capita_Projection.mat[,,19]-long_rel_GDP_per_capita_Projection.mat[,,1])/(19-1)*i
}
for (i in 1:(24-19-1)) {
  long_rel_GDP_per_capita_Projection.mat[,,19+i] <- long_rel_GDP_per_capita_Projection.mat[,,19]+
    (long_rel_GDP_per_capita_Projection.mat[,,24]-long_rel_GDP_per_capita_Projection.mat[,,19])/(24-19)*i
}
for (i in 1:(29-24-1)) {
  long_rel_GDP_per_capita_Projection.mat[,,24+i] <- long_rel_GDP_per_capita_Projection.mat[,,24]+
    (long_rel_GDP_per_capita_Projection.mat[,,29]-long_rel_GDP_per_capita_Projection.mat[,,24])/(29-24)*i
}
for (i in 1:(39-29-1)) {
  long_rel_GDP_per_capita_Projection.mat[,,29+i] <- long_rel_GDP_per_capita_Projection.mat[,,29]+
    (long_rel_GDP_per_capita_Projection.mat[,,39]-long_rel_GDP_per_capita_Projection.mat[,,29])/(39-29)*i
}

##########################################
# save data from FAO 2018 for projections of GDP, population, and livestock production
##########################################
save.image(file=paste0(path_out,"Relative_changes_Protein_and_GDPperCapita_from_FAO_projection.RData"))

##########################################
##########################################
# 6.5 Calculate changes in livestock CH4 emission intensities from FAOSTAT, 2019T1, and 2019MT respectively
##########################################
##########################################

##########################################
# 6.5.1 Load emission intensity per protein produced and the regression coefficients between annual emission intensities and GDP per Capita during the period 2000-2017
##########################################
load(file=paste0(path_out,"Livestock_CH4_Emissions_Regression_EmisInt_vs_GDPperCapita_p5yr.RData"))

##########################################
# 6.5.2 Get baseline (2012) Emission from protein-producing animals and draft animals, as well as emission intensities
##########################################

# 6.5.2.1 baseline (2012) Emission from all animals
baseline_country_CH4_FAOSTAT <- array(NA,c(3,243,18))
baseline_country_CH4_2019T1 <- array(NA,c(3,243,18))
baseline_country_CH4_2019MT <- array(NA,c(3,243,18))
p <- 13 # 2012 
baseline_country_CH4_FAOSTAT[,,1:16] <- country_CH4_FAOSTAT[,,1:16,p]
baseline_country_CH4_2019T1[,,1:16] <- country_CH4_2019T1[,,1:16,p]
baseline_country_CH4_2019MT[,,1:16] <- country_CH4_2019MT[,,1:16,p]
# pigs
baseline_country_CH4_FAOSTAT[,,17] <- apply(baseline_country_CH4_FAOSTAT[,,c(14,15)],c(1,2),sum,na.rm=TRUE) 
baseline_country_CH4_2019T1[,,17] <- apply(baseline_country_CH4_2019T1[,,c(14,15)],c(1,2),sum,na.rm=TRUE) 
baseline_country_CH4_2019MT[,,17] <- apply(baseline_country_CH4_2019MT[,,c(14,15)],c(1,2),sum,na.rm=TRUE) 
# poultry
baseline_country_CH4_FAOSTAT[,,18] <- apply(baseline_country_CH4_FAOSTAT[,,c(6,7,8,16)],c(1,2),sum,na.rm=TRUE) 
baseline_country_CH4_2019T1[,,18] <- apply(baseline_country_CH4_2019T1[,,c(6,7,8,16)],c(1,2),sum,na.rm=TRUE) 
baseline_country_CH4_2019MT[,,18] <- apply(baseline_country_CH4_2019MT[,,c(6,7,8,16)],c(1,2),sum,na.rm=TRUE) 

# 6.5.2.2 baseline (2012) Emission from protein-producing animals
baseline_country_CH4_FAOSTAT_nodraft <- array(NA,c(3,243,18))
baseline_country_CH4_2019T1_nodraft <- array(NA,c(3,243,18))
baseline_country_CH4_2019MT_nodraft <- array(NA,c(3,243,18))
baseline_country_CH4_FAOSTAT_nodraft <- baseline_country_CH4_FAOSTAT
baseline_country_CH4_2019T1_nodraft <- baseline_country_CH4_2019T1
baseline_country_CH4_2019MT_nodraft <- baseline_country_CH4_2019MT
# "Buffaloes" and "CattleNonDairy" only
draft_cate <- c(2,5)
p <- 13 # 2012 
baseline_country_CH4_FAOSTAT_nodraft[,,draft_cate] <- country_CH4_FAOSTAT_nodraft[,,draft_cate,p]
baseline_country_CH4_2019T1_nodraft[,,draft_cate] <- country_CH4_2019T1_nodraft[,,draft_cate,p]
baseline_country_CH4_2019MT_nodraft[,,draft_cate] <- country_CH4_2019MT_nodraft[,,draft_cate,p]

# 6.5.2.3 baseline (2012) Emission from draft animals
baseline_country_CH4_FAOSTAT_draft <- baseline_country_CH4_FAOSTAT - baseline_country_CH4_FAOSTAT_nodraft
baseline_country_CH4_2019T1_draft <- baseline_country_CH4_2019T1 - baseline_country_CH4_2019T1_nodraft
baseline_country_CH4_2019MT_draft <- baseline_country_CH4_2019MT - baseline_country_CH4_2019MT_nodraft

# 6.5.2.4 baseline (2000-2017) Emission Intensities
# Gg CH4 *1e3 / tonnes protein = kg CH4 / kg protein
baseline_country_EmisInt_FAOSTAT <- array(NA,c(3,243,18))
baseline_country_EmisInt_2019T1 <- array(NA,c(3,243,18))
baseline_country_EmisInt_2019MT <- array(NA,c(3,243,18))
p <- 3 # "2010-2014" 
baseline_country_EmisInt_FAOSTAT <- p5yr_country_EmisInt_FAOSTAT[,,,p]
baseline_country_EmisInt_2019T1 <- p5yr_country_EmisInt_2019T1[,,,p]
baseline_country_EmisInt_2019MT <- p5yr_country_EmisInt_2019MT[,,,p]

# 6.5.3 Calculate "real" GDP per capita using the GDP per capita calculated from FAOSTAT and the relative changes in GDP per capita from FAO, 2018
# This is to make sure there is no inconsistency between the two datasets
# 5.3.1 Get baseline (2012) GDP per capita from FAOSTAT
p <- 13 # 2012 
baseline_country_GDPperCapita <- GDP_per_capita.mat[,1,p]
# 5.3.2 Calculate "real" GDP per capita
interp.year <- seq(2012,2050,by=1)
calc_country_GDPperCapita <- array(NA,c(243,3,length(interp.year)))
for (s in 1:3) {
  for (i in 1:length(interp.year)) {
    calc_country_GDPperCapita[1:243,s,i] <- baseline_country_GDPperCapita[1:243]*long_rel_GDP_per_capita_Projection.mat[1:243,s,i]
  }
}

##########################################
# 6.5.3 Calculate "target" CH4 emission intensities for each year from 2012 to 2050
##########################################

# The trajectory of the "target" EmisInt at each country (Y) follows the changes in year (X), using the regression function Y=a*X+b
# where b is the EmisInt of 2012
# Country/Area,Scenario,Livestock category,Year
target_country_EmisInt_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
target_country_EmisInt_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
target_country_EmisInt_2019MT <- array(NA,c(243,3,18,length(interp.year)))

for (s in 1:3) {
  for (m in c(2,4,5,9,13,17,18)) {
    for (cc in 1:243) {
      for (i in 1:length(interp.year)) {
        target_country_EmisInt_FAOSTAT[cc,s,m,i] <- calc_country_GDPperCapita[cc,s,i] * regression_coeff_FAOSTAT[cc,m,1] + regression_coeff_FAOSTAT[cc,m,2]
        target_country_EmisInt_2019T1[cc,s,m,i] <- calc_country_GDPperCapita[cc,s,i] * regression_coeff_2019T1[cc,m,1] + regression_coeff_2019T1[cc,m,2]
        target_country_EmisInt_2019MT[cc,s,m,i] <- calc_country_GDPperCapita[cc,s,i] * regression_coeff_2019MT[cc,m,1] + regression_coeff_2019MT[cc,m,2]
      }
    }
  }
}

##########################################
# 6.5.4 Calculate "real" CH4 emission intensities for each year from 2012 to 2050
##########################################

# Given the fact that it is not realistic to reach a too low EmisInt, we used the 5% quantile of EmisInt during 2010-2014 as a minimum threshold (qt05_country_EmisInt_XXX)
# In addition, when the calculated Y is higher than the current EmisInt (i.e., more CH4 emission per kg protein production), we assume no improvement of EmisInt
# only allow decreasing EmisInt
calc_country_EmisInt_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
calc_country_EmisInt_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
calc_country_EmisInt_2019MT <- array(NA,c(243,3,18,length(interp.year)))
# allowing both decreasing and increasing EmisInt
decinc_country_EmisInt_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
decinc_country_EmisInt_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
decinc_country_EmisInt_2019MT <- array(NA,c(243,3,18,length(interp.year)))

# 6.5.4.1 baseline (2012) Emission Intensities 
for (s in 1:3) {
  calc_country_EmisInt_FAOSTAT[1:243,s,1:18,1] <- baseline_country_EmisInt_FAOSTAT[3,1:243,1:18]
  calc_country_EmisInt_2019T1[1:243,s,1:18,1] <- baseline_country_EmisInt_2019T1[3,1:243,1:18]
  calc_country_EmisInt_2019MT[1:243,s,1:18,1] <- baseline_country_EmisInt_2019MT[3,1:243,1:18]
  decinc_country_EmisInt_FAOSTAT[1:243,s,1:18,1] <- baseline_country_EmisInt_FAOSTAT[3,1:243,1:18]
  decinc_country_EmisInt_2019T1[1:243,s,1:18,1] <- baseline_country_EmisInt_2019T1[3,1:243,1:18]
  decinc_country_EmisInt_2019MT[1:243,s,1:18,1] <- baseline_country_EmisInt_2019MT[3,1:243,1:18]
}
# 6.5.4.2 Emission Intensities from 2013 to 2050
record_reach_qt05_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
record_reach_qt05_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
record_reach_qt05_2019MT <- array(NA,c(243,3,18,length(interp.year)))
record_reach_qt95_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
record_reach_qt95_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
record_reach_qt95_2019MT <- array(NA,c(243,3,18,length(interp.year)))

Dset <- c("2019MT","2019T1","FAOSTAT","EDGARv5")
for (s in 1:3) {
  for (m in c(2,4,5,9,13,17,18)) {
    for (cc in 1:243) {
      # record countries/areas that reach the qt05
      i <- 1
      tmp <- target_country_EmisInt_FAOSTAT[cc,s,m,i]
      EmisInt2012 <- calc_country_EmisInt_FAOSTAT[cc,s,m,1]
      if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
        if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_FAOSTAT[m]) {
          record_reach_qt05_FAOSTAT[cc,s,m,i] <- 1.0
        } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > qt95_country_protein100_EmisInt_FAOSTAT[m]) {
          record_reach_qt95_FAOSTAT[cc,s,m,i] <- 1.0
        }
      }  
      tmp <- target_country_EmisInt_2019T1[cc,s,m,i]
      EmisInt2012 <- calc_country_EmisInt_2019T1[cc,s,m,1]
      if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
        if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_2019T1[m]) {
          record_reach_qt05_2019T1[cc,s,m,i] <- 1.0
        } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > qt95_country_protein100_EmisInt_2019T1[m]) {
          record_reach_qt95_2019T1[cc,s,m,i] <- 1.0
        }
      }
      tmp <- target_country_EmisInt_2019MT[cc,s,m,i]
      EmisInt2012 <- calc_country_EmisInt_2019MT[cc,s,m,1]
      if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
        if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_2019MT[m]) {
          record_reach_qt05_2019MT[cc,s,m,i] <- 1.0
        } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > qt95_country_protein100_EmisInt_2019MT[m]) {
          record_reach_qt95_2019MT[cc,s,m,i] <- 1.0
        }
      }
      
      for (i in 2:length(interp.year)) {
        # FAOSTAT 
        tmp <- target_country_EmisInt_FAOSTAT[cc,s,m,i]
        EmisInt2012 <- calc_country_EmisInt_FAOSTAT[cc,s,m,1]
        # force EmisInt between the value of 2012 and the minimum EmisInt observed in any country/area
        if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
          if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > EmisInt2012) {
            tmp <- calc_country_EmisInt_FAOSTAT[cc,s,m,1]
          } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_FAOSTAT[m]) {
            tmp <- qt05_country_protein100_EmisInt_FAOSTAT[m] 
            # record the country that reach the qt05
            record_reach_qt05_FAOSTAT[cc,s,m,i] <- 1.0
          }
        }
        calc_country_EmisInt_FAOSTAT[cc,s,m,i] <- tmp
        
        tmp <- target_country_EmisInt_FAOSTAT[cc,s,m,i]
        EmisInt2012 <- calc_country_EmisInt_FAOSTAT[cc,s,m,1]
        # force EmisInt between the value of maximum and the minimum EmisInt observed in any country/area
        if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
          if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > qt95_country_protein100_EmisInt_FAOSTAT[m]) {
            tmp <- qt95_country_protein100_EmisInt_FAOSTAT[m]
            # record the country that reach the qt95
            record_reach_qt95_FAOSTAT[cc,s,m,i] <- 1.0
          } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_FAOSTAT[m]) {
            tmp <- qt05_country_protein100_EmisInt_FAOSTAT[m] 
            # record the country that reach the qt05
            record_reach_qt05_FAOSTAT[cc,s,m,i] <- 1.0
          }
        }
        decinc_country_EmisInt_FAOSTAT[cc,s,m,i] <- tmp
        
        
        # 2019T1 
        tmp <- target_country_EmisInt_2019T1[cc,s,m,i]
        EmisInt2012 <- calc_country_EmisInt_2019T1[cc,s,m,1]
        # force EmisInt between the value of 2012 and the minimum EmisInt observed in any country/area
        if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
          if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > EmisInt2012) {
            tmp <- calc_country_EmisInt_2019T1[cc,s,m,1]
          } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_2019T1[m]) {
            tmp <- qt05_country_protein100_EmisInt_2019T1[m] 
            # record the country that reach the qt05
            record_reach_qt05_2019T1[cc,s,m,i] <- 1.0
          }
        }
        calc_country_EmisInt_2019T1[cc,s,m,i] <- tmp
        
        tmp <- target_country_EmisInt_2019T1[cc,s,m,i]
        EmisInt2012 <- calc_country_EmisInt_2019T1[cc,s,m,1]
        # force EmisInt between the value of maximum and the minimum EmisInt observed in any country/area
        if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
          if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > qt95_country_protein100_EmisInt_2019T1[m]) {
            tmp <- qt95_country_protein100_EmisInt_2019T1[m]
            # record the country that reach the qt95
            record_reach_qt95_2019T1[cc,s,m,i] <- 1.0
          } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_2019T1[m]) {
            tmp <- qt05_country_protein100_EmisInt_2019T1[m] 
            # record the country that reach the qt05
            record_reach_qt05_2019T1[cc,s,m,i] <- 1.0
          }
        }
        decinc_country_EmisInt_2019T1[cc,s,m,i] <- tmp
        
        
        # 2019MT 
        tmp <- target_country_EmisInt_2019MT[cc,s,m,i]
        EmisInt2012 <- calc_country_EmisInt_2019MT[cc,s,m,1]
        # force EmisInt between the value of 2012 and the minimum EmisInt observed in any country/area
        if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
          if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > EmisInt2012) {
            tmp <- calc_country_EmisInt_2019MT[cc,s,m,1]
          } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_2019MT[m]) {
            tmp <- qt05_country_protein100_EmisInt_2019MT[m] 
            # record the country that reach the qt05
            record_reach_qt05_2019MT[cc,s,m,i] <- 1.0
          }
        }
        calc_country_EmisInt_2019MT[cc,s,m,i] <- tmp
        
        tmp <- target_country_EmisInt_2019MT[cc,s,m,i]
        EmisInt2012 <- calc_country_EmisInt_2019MT[cc,s,m,1]
        # force EmisInt between the value of maximum and the minimum EmisInt observed in any country/area
        if (is.finite(EmisInt2012) & is.na(EmisInt2012) == FALSE & EmisInt2012 > 0.0) {
          if (is.finite(tmp) & is.na(tmp) == FALSE & tmp > qt95_country_protein100_EmisInt_2019MT[m]) {
            tmp <- qt95_country_protein100_EmisInt_2019MT[m]
            # record the country that reach the qt95
            record_reach_qt95_2019MT[cc,s,m,i] <- 1.0
          } else if (is.finite(tmp) & is.na(tmp) == FALSE & tmp < qt05_country_protein100_EmisInt_2019MT[m]) {
            tmp <- qt05_country_protein100_EmisInt_2019MT[m] 
            # record the country that reach the qt05
            record_reach_qt05_2019MT[cc,s,m,i] <- 1.0
          }
        }
        decinc_country_EmisInt_2019MT[cc,s,m,i] <- tmp
        
      }
    }
  }
}  

##########################################
# 6.5.5 Calculate relative changes in CH4 emission intensities for each year from 2012 to 2050
##########################################

rel_country_EmisInt_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
rel_country_EmisInt_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
rel_country_EmisInt_2019MT <- array(NA,c(243,3,18,length(interp.year)))
rel_decinc_country_EmisInt_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
rel_decinc_country_EmisInt_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
rel_decinc_country_EmisInt_2019MT <- array(NA,c(243,3,18,length(interp.year)))
for (s in 1:3) {
  for (m in c(2,4,5,9,13,17,18)) {
    for (i in 1:length(interp.year)) {
      rel_country_EmisInt_FAOSTAT[1:243,s,m,i] <- calc_country_EmisInt_FAOSTAT[1:243,s,m,i]/calc_country_EmisInt_FAOSTAT[1:243,s,m,1]      
      rel_country_EmisInt_2019T1[1:243,s,m,i] <- calc_country_EmisInt_2019T1[1:243,s,m,i]/calc_country_EmisInt_2019T1[1:243,s,m,1]      
      rel_country_EmisInt_2019MT[1:243,s,m,i] <- calc_country_EmisInt_2019MT[1:243,s,m,i]/calc_country_EmisInt_2019MT[1:243,s,m,1]      
      rel_decinc_country_EmisInt_FAOSTAT[1:243,s,m,i] <- decinc_country_EmisInt_FAOSTAT[1:243,s,m,i]/decinc_country_EmisInt_FAOSTAT[1:243,s,m,1]      
      rel_decinc_country_EmisInt_2019T1[1:243,s,m,i] <- decinc_country_EmisInt_2019T1[1:243,s,m,i]/decinc_country_EmisInt_2019T1[1:243,s,m,1]      
      rel_decinc_country_EmisInt_2019MT[1:243,s,m,i] <- decinc_country_EmisInt_2019MT[1:243,s,m,i]/decinc_country_EmisInt_2019MT[1:243,s,m,1]      
    }
  }
}

##########################################
# 6.5.6 Reset relative changes > 1 to 1 
##########################################

# We assume no increase in the Emission intensities (i.e., either improved (lower EmisInt) or constant EmisInt ) 
rel_country_EmisInt_FAOSTAT[is.na(rel_country_EmisInt_FAOSTAT) | is.infinite(rel_country_EmisInt_FAOSTAT) | rel_country_EmisInt_FAOSTAT > 1.0] <- 1.0
rel_country_EmisInt_2019T1[is.na(rel_country_EmisInt_2019T1) | is.infinite(rel_country_EmisInt_2019T1) | rel_country_EmisInt_2019T1 > 1.0] <- 1.0
rel_country_EmisInt_2019MT[is.na(rel_country_EmisInt_2019MT) | is.infinite(rel_country_EmisInt_2019MT) | rel_country_EmisInt_2019MT > 1.0] <- 1.0
# We assume the Emission intensities can increase or decrease
rel_decinc_country_EmisInt_FAOSTAT[is.na(rel_decinc_country_EmisInt_FAOSTAT) | is.infinite(rel_decinc_country_EmisInt_FAOSTAT)] <- 1.0
rel_decinc_country_EmisInt_2019T1[is.na(rel_decinc_country_EmisInt_2019T1) | is.infinite(rel_decinc_country_EmisInt_2019T1)] <- 1.0
rel_decinc_country_EmisInt_2019MT[is.na(rel_decinc_country_EmisInt_2019MT) | is.infinite(rel_decinc_country_EmisInt_2019MT)] <- 1.0


##########################################
##########################################
# 6.6 Projections on the livestock CH4 emissions using changes in Emission intensities (decreasing only) depending on GDPperCapita variation in each country
##########################################
##########################################

##########################################
# 6.6.1 Projections on the major livestock categories
##########################################

# Country/Area,Scenario,Livestock category,Year
proj_country_CH4_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
proj_country_CH4_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
proj_country_CH4_2019MT <- array(NA,c(243,3,18,length(interp.year)))
#> Livestock_Projection.Item.name
#[1] "Livestock units"      "Raising of buffaloes" "Raising of cattle"    "Raising of goats"     "Raising of pigs"     
#[6] "Raising of poultry"   "Raising of sheep"    
# 243 country/area,7 Livestock_Projection.Item.name,2 Meat/Milk/Meat+Milk,3 Scenario,5 Year
#long_rel_Protein_Projection.mat <- array(NA,c(243,7,3,3,length(interp.year)))

for (m in c(2,4,5,9,13,17,18)) {
  # find the corresponding protein projection
  rel_country_Protein <- array(NA,c(243,3,length(interp.year)))
  if (m == 2) { # "Buffaloes" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of buffaloes")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 4) { # "CattleDairy" Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Milk,1:3,]
  } else if (m == 5) { # "CattleNonDairy" Meat
    j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } else if (m == 9) { # "Goats" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of goats")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 13) { # "Sheep" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of sheep")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 17) { # "Pigs" Meat
    j_Item <- which(Livestock_Projection.Item.name == "Raising of pigs")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } else if (m == 18) { # "Poultry" Meat 
    j_Item <- which(Livestock_Projection.Item.name == "Raising of poultry")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } 
  for (s in 1:3) {
    for (i in 1:length(interp.year)) {
      # baseline emission (2012 emission intensity * 2012 protein production ) * relative changes in protein production * relative changes in emission intensities 
      proj_country_CH4_FAOSTAT[1:243,s,m,i] <- baseline_country_CH4_FAOSTAT_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] * rel_country_EmisInt_FAOSTAT[1:243,s,m,i]    
      proj_country_CH4_2019T1[1:243,s,m,i] <- baseline_country_CH4_2019T1_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] * rel_country_EmisInt_2019T1[1:243,s,m,i]        
      proj_country_CH4_2019MT[1:243,s,m,i] <- baseline_country_CH4_2019MT_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] * rel_country_EmisInt_2019MT[1:243,s,m,i]         
    }
  }
}

##########################################
# 6.6.2 Projections on the other livestock categories
##########################################

# Emissions from other livestock is assumed to be the same as historical variation between 2012-2018, and be constant after 2018 
# Note: Pigs c(14,15) and Poultry c(6,7,8,16) have been calculated above thus kept as 0
other_cate <- c(1,3,10,11,12)

# Alternatively, we keep the 2012 value as projections
for (s in 1:3) {
  for (i in 1:length(interp.year)) {
    proj_country_CH4_FAOSTAT[1:243,s,other_cate,i] <- country_CH4_FAOSTAT[3,1:243,other_cate,13] 
    proj_country_CH4_2019T1[1:243,s,other_cate,i] <- country_CH4_2019T1[3,1:243,other_cate,13] 
    proj_country_CH4_2019MT[1:243,s,other_cate,i] <- country_CH4_2019MT[3,1:243,other_cate,13] 
  }  
}

# set NA and infinite value to 0 for further calculations
proj_country_CH4_FAOSTAT[is.na(proj_country_CH4_FAOSTAT) | is.infinite(proj_country_CH4_FAOSTAT) | proj_country_CH4_FAOSTAT < 0.0] <- 0.0
proj_country_CH4_2019T1[is.na(proj_country_CH4_2019T1) | is.infinite(proj_country_CH4_2019T1) | proj_country_CH4_2019T1 < 0.0] <- 0.0
proj_country_CH4_2019MT[is.na(proj_country_CH4_2019MT) | is.infinite(proj_country_CH4_2019MT) | proj_country_CH4_2019MT < 0.0] <- 0.0

##########################################
# 6.6.3 Projections on the draft animals
##########################################

# "Buffaloes" and "CattleNonDairy" only
draft_cate <- c(2,5)
# Emissions from draft animal is assumed to be the same as historical variation between 2012-2018
# After 2018, the trajectory of the Emissions from draft animal at each country (Y) follows the changes in GDP per capita (X), using the regression function Y=a*X+b
# The above trajectory only applies to the countries/areas that the regression slope < 0 (i.e., emission from draft animal decrease following increasing GDP per Capita) 
# For the countries/areas that the regression slope > 0 (i.e., emission from draft animal increase following increasing GDP per Capita), we assume the constant emissions as that of 2018 

# 6.6.3.1 calculate emissions from draft animals from 2012 to 2050
# Country/Area,Scenario,Livestock category,Year
proj_country_CH4_FAOSTAT_draft <- array(0.0,c(243,3,18,length(interp.year)))
proj_country_CH4_2019T1_draft <- array(0.0,c(243,3,18,length(interp.year)))
proj_country_CH4_2019MT_draft <- array(0.0,c(243,3,18,length(interp.year)))
for (m in c(2,5)) {
  for (cc in 1:243) {
    for (s in 1:3) {
      # we start using projections (GDP per capita dependent) since 2012
      # 2012 - 2050 projections calculated from GDP per Capita
      if (draft_regression_coeff_FAOSTAT[cc,m,1] < 0.0) {
        # After 2018, the trajectory of the Emissions from draft animal at each country (Y) follows the changes in GDP per capita (X), using the regression function Y=a*X+b
        # The above trajectory only applies to the countries/areas that the regression slope < 0 (i.e., emission from draft animal decrease following increasing GDP per Capita) 
        for (i in 1:length(interp.year)) {
          proj_country_CH4_FAOSTAT_draft[1:243,s,m,i] <- calc_country_GDPperCapita[1:243,s,i] * draft_regression_coeff_FAOSTAT[cc,m,1] + draft_regression_coeff_FAOSTAT[cc,m,2]
          proj_country_CH4_2019T1_draft[1:243,s,m,i] <- calc_country_GDPperCapita[1:243,s,i] * draft_regression_coeff_2019T1[cc,m,1] + draft_regression_coeff_2019T1[cc,m,2]
          proj_country_CH4_2019MT_draft[1:243,s,m,i] <- calc_country_GDPperCapita[1:243,s,i] * draft_regression_coeff_2019MT[cc,m,1] + draft_regression_coeff_2019MT[cc,m,2]
        }
      } else {
        # For the countries/areas that the regression slope > 0 (i.e., emission from draft animal increase following increasing GDP per Capita), we assume the constant emissions as that of 2018 
        for (i in 1:length(interp.year)) {
          proj_country_CH4_FAOSTAT_draft[1:243,s,m,i] <- calc_country_CH4_FAOSTAT_draft[3,1:243,m,13] 
          proj_country_CH4_2019T1_draft[1:243,s,m,i] <- calc_country_CH4_2019T1_draft[3,1:243,m,13]
          proj_country_CH4_2019MT_draft[1:243,s,m,i] <- calc_country_CH4_2019MT_draft[3,1:243,m,13]
        }
      }
      
    }
  }
}

# set nondraft (except "Buffaloes" and "CattleNonDairy"), NA, and < 0 values to 0
tmp <- proj_country_CH4_FAOSTAT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_country_CH4_FAOSTAT_draft <- tmp

tmp <- proj_country_CH4_2019T1_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_country_CH4_2019T1_draft <- tmp

tmp <- proj_country_CH4_2019MT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_country_CH4_2019MT_draft <- tmp

# 6.6.3.2 add emissions from draft animals to total emissions
# all values for draft animals need to be added to the values of non-draft animals calculated above
proj_country_CH4_FAOSTAT[1:243,,draft_cate,] <- proj_country_CH4_FAOSTAT[1:243,,draft_cate,] + proj_country_CH4_FAOSTAT_draft[1:243,,draft_cate,]
proj_country_CH4_2019T1[1:243,,draft_cate,] <- proj_country_CH4_2019T1[1:243,,draft_cate,] + proj_country_CH4_2019T1_draft[1:243,,draft_cate,]
proj_country_CH4_2019MT[1:243,,draft_cate,] <- proj_country_CH4_2019MT[1:243,,draft_cate,] + proj_country_CH4_2019MT_draft[1:243,,draft_cate,]


##########################################
##########################################
# 6.7 Projections on the livestock CH4 emissions using changes in Emission intensities (decreasing or increasing) depending on GDPperCapita variation in each country
##########################################
##########################################

##########################################
# 6.7.1 Projections on the major livestock categories
##########################################

# Country/Area,Scenario,Livestock category,Year
proj_decincEmisInt_country_CH4_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
proj_decincEmisInt_country_CH4_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
proj_decincEmisInt_country_CH4_2019MT <- array(NA,c(243,3,18,length(interp.year)))
#> Livestock_Projection.Item.name
#[1] "Livestock units"      "Raising of buffaloes" "Raising of cattle"    "Raising of goats"     "Raising of pigs"     
#[6] "Raising of poultry"   "Raising of sheep"    
# 243 country/area,7 Livestock_Projection.Item.name,2 Meat/Milk/Meat+Milk,3 Scenario,5 Year
#long_rel_Protein_Projection.mat <- array(NA,c(243,7,3,3,length(interp.year)))

for (m in c(2,4,5,9,13,17,18)) {
  # find the corresponding protein projection
  rel_country_Protein <- array(NA,c(243,3,length(interp.year)))
  if (m == 2) { # "Buffaloes" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of buffaloes")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 4) { # "CattleDairy" Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Milk,1:3,]
  } else if (m == 5) { # "CattleNonDairy" Meat
    j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } else if (m == 9) { # "Goats" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of goats")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 13) { # "Sheep" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of sheep")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 17) { # "Pigs" Meat
    j_Item <- which(Livestock_Projection.Item.name == "Raising of pigs")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } else if (m == 18) { # "Poultry" Meat 
    j_Item <- which(Livestock_Projection.Item.name == "Raising of poultry")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } 
  for (s in 1:3) {
    for (i in 1:length(interp.year)) {
      # baseline emission (2012 emission intensity * 2012 protein production ) * relative changes in protein production * relative changes in emission intensities 
      proj_decincEmisInt_country_CH4_FAOSTAT[1:243,s,m,i] <- baseline_country_CH4_FAOSTAT_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] * rel_decinc_country_EmisInt_FAOSTAT[1:243,s,m,i]    
      proj_decincEmisInt_country_CH4_2019T1[1:243,s,m,i] <- baseline_country_CH4_2019T1_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] * rel_decinc_country_EmisInt_2019T1[1:243,s,m,i]        
      proj_decincEmisInt_country_CH4_2019MT[1:243,s,m,i] <- baseline_country_CH4_2019MT_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] * rel_decinc_country_EmisInt_2019MT[1:243,s,m,i]         
    }
  }
}

##########################################
# 6.7.2 Projections on the other livestock categories
##########################################

# Emissions from other livestock is assumed to be the same as historical variation between 2012-2018, and be constant after 2018 
# Note: Pigs c(14,15) and Poultry c(6,7,8,16) have been calculated above thus kept as 0
other_cate <- c(1,3,10,11,12)

# Alternatively, we keep the 2012 value as projections
for (s in 1:3) {
  for (i in 1:length(interp.year)) {
    proj_decincEmisInt_country_CH4_FAOSTAT[1:243,s,other_cate,i] <- country_CH4_FAOSTAT[3,1:243,other_cate,13] 
    proj_decincEmisInt_country_CH4_2019T1[1:243,s,other_cate,i] <- country_CH4_2019T1[3,1:243,other_cate,13] 
    proj_decincEmisInt_country_CH4_2019MT[1:243,s,other_cate,i] <- country_CH4_2019MT[3,1:243,other_cate,13] 
  }  
}

# set NA and infinite value to 0 for further calculations
proj_decincEmisInt_country_CH4_FAOSTAT[is.na(proj_decincEmisInt_country_CH4_FAOSTAT) | is.infinite(proj_decincEmisInt_country_CH4_FAOSTAT) | proj_decincEmisInt_country_CH4_FAOSTAT < 0.0] <- 0.0
proj_decincEmisInt_country_CH4_2019T1[is.na(proj_decincEmisInt_country_CH4_2019T1) | is.infinite(proj_decincEmisInt_country_CH4_2019T1) | proj_decincEmisInt_country_CH4_2019T1 < 0.0] <- 0.0
proj_decincEmisInt_country_CH4_2019MT[is.na(proj_decincEmisInt_country_CH4_2019MT) | is.infinite(proj_decincEmisInt_country_CH4_2019MT) | proj_decincEmisInt_country_CH4_2019MT < 0.0] <- 0.0

##########################################
# 6.7.3 Projections on the draft animals
##########################################

# "Buffaloes" and "CattleNonDairy" only
draft_cate <- c(2,5)
# Emissions from draft animal is assumed to be the same as historical variation between 2012-2018
# After 2018, the trajectory of the Emissions from draft animal at each country (Y) follows the changes in GDP per capita (X), using the regression function Y=a*X+b
# The above trajectory only applies to the countries/areas that the regression slope < 0 (i.e., emission from draft animal decrease following increasing GDP per Capita) 
# For the countries/areas that the regression slope > 0 (i.e., emission from draft animal increase following increasing GDP per Capita), we assume the constant emissions as that of 2018 

# 6.7.3.1 calculate emissions from draft animals from 2012 to 2050
# Country/Area,Scenario,Livestock category,Year
proj_decincEmisInt_country_CH4_FAOSTAT_draft <- array(0.0,c(243,3,18,length(interp.year)))
proj_decincEmisInt_country_CH4_2019T1_draft <- array(0.0,c(243,3,18,length(interp.year)))
proj_decincEmisInt_country_CH4_2019MT_draft <- array(0.0,c(243,3,18,length(interp.year)))
for (m in c(2,5)) {
  for (cc in 1:243) {
    for (s in 1:3) {
      # we start using projections (GDP per capita dependent) since 2012
      # 2012 - 2050 projections calculated from GDP per Capita
      if (draft_regression_coeff_FAOSTAT[cc,m,1] < 0.0) {
        # After 2018, the trajectory of the Emissions from draft animal at each country (Y) follows the changes in GDP per capita (X), using the regression function Y=a*X+b
        # The above trajectory only applies to the countries/areas that the regression slope < 0 (i.e., emission from draft animal decrease following increasing GDP per Capita) 
        for (i in 1:length(interp.year)) {
          proj_decincEmisInt_country_CH4_FAOSTAT_draft[1:243,s,m,i] <- calc_country_GDPperCapita[1:243,s,i] * draft_regression_coeff_FAOSTAT[cc,m,1] + draft_regression_coeff_FAOSTAT[cc,m,2]
          proj_decincEmisInt_country_CH4_2019T1_draft[1:243,s,m,i] <- calc_country_GDPperCapita[1:243,s,i] * draft_regression_coeff_2019T1[cc,m,1] + draft_regression_coeff_2019T1[cc,m,2]
          proj_decincEmisInt_country_CH4_2019MT_draft[1:243,s,m,i] <- calc_country_GDPperCapita[1:243,s,i] * draft_regression_coeff_2019MT[cc,m,1] + draft_regression_coeff_2019MT[cc,m,2]
        }
      } else {
        # For the countries/areas that the regression slope > 0 (i.e., emission from draft animal increase following increasing GDP per Capita), we assume the constant emissions as that of 2018 
        for (i in 1:length(interp.year)) {
          proj_decincEmisInt_country_CH4_FAOSTAT_draft[1:243,s,m,i] <- calc_country_CH4_FAOSTAT_draft[3,1:243,m,13] 
          proj_decincEmisInt_country_CH4_2019T1_draft[1:243,s,m,i] <- calc_country_CH4_2019T1_draft[3,1:243,m,13]
          proj_decincEmisInt_country_CH4_2019MT_draft[1:243,s,m,i] <- calc_country_CH4_2019MT_draft[3,1:243,m,13]
        }
      }
      
    }
  }
}

# set nondraft (except "Buffaloes" and "CattleNonDairy"), NA, and < 0 values to 0
tmp <- proj_decincEmisInt_country_CH4_FAOSTAT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_decincEmisInt_country_CH4_FAOSTAT_draft <- tmp

tmp <- proj_decincEmisInt_country_CH4_2019T1_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_decincEmisInt_country_CH4_2019T1_draft <- tmp

tmp <- proj_decincEmisInt_country_CH4_2019MT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_decincEmisInt_country_CH4_2019MT_draft <- tmp

# 6.7.3.2 add emissions from draft animals to total emissions
# all values for draft animals need to be added to the values of non-draft animals calculated above
proj_decincEmisInt_country_CH4_FAOSTAT[1:243,,draft_cate,] <- proj_decincEmisInt_country_CH4_FAOSTAT[1:243,,draft_cate,] + proj_decincEmisInt_country_CH4_FAOSTAT_draft[1:243,,draft_cate,]
proj_decincEmisInt_country_CH4_2019T1[1:243,,draft_cate,] <- proj_decincEmisInt_country_CH4_2019T1[1:243,,draft_cate,] + proj_decincEmisInt_country_CH4_2019T1_draft[1:243,,draft_cate,]
proj_decincEmisInt_country_CH4_2019MT[1:243,,draft_cate,] <- proj_decincEmisInt_country_CH4_2019MT[1:243,,draft_cate,] + proj_decincEmisInt_country_CH4_2019MT_draft[1:243,,draft_cate,]


##########################################
##########################################
# 6.8 Projections on the livestock CH4 emissions using constant Emission intensities during the past 5 years (2014-2018)
##########################################
##########################################

##########################################
# 6.8.1 Projections on the major livestock categories
##########################################

# Country/Area,Scenario,Livestock category,Year
proj_consEmisInt_country_CH4_FAOSTAT <- array(NA,c(243,3,18,length(interp.year)))
proj_consEmisInt_country_CH4_2019T1 <- array(NA,c(243,3,18,length(interp.year)))
proj_consEmisInt_country_CH4_2019MT <- array(NA,c(243,3,18,length(interp.year)))
#> Livestock_Projection.Item.name
#[1] "Livestock units"      "Raising of buffaloes" "Raising of cattle"    "Raising of goats"     "Raising of pigs"     
#[6] "Raising of poultry"   "Raising of sheep"    
# 243 country/area,7 Livestock_Projection.Item.name,2 Meat/Milk/Meat+Milk,3 Scenario,5 Year
#long_rel_Protein_Projection.mat <- array(NA,c(243,7,3,3,length(interp.year)))

for (m in c(2,4,5,9,13,17,18)) {
  # find the corresponding protein projection
  rel_country_Protein <- array(NA,c(243,3,length(interp.year)))
  if (m == 2) { # "Buffaloes" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of buffaloes")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 4) { # "CattleDairy" Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Milk,1:3,]
  } else if (m == 5) { # "CattleNonDairy" Meat
    j_Item <- which(Livestock_Projection.Item.name == "Raising of cattle")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } else if (m == 9) { # "Goats" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of goats")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 13) { # "Sheep" Meat+Milk
    j_Item <- which(Livestock_Projection.Item.name == "Raising of sheep")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_MeatMilk,1:3,]
  } else if (m == 17) { # "Pigs" Meat
    j_Item <- which(Livestock_Projection.Item.name == "Raising of pigs")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } else if (m == 18) { # "Poultry" Meat 
    j_Item <- which(Livestock_Projection.Item.name == "Raising of poultry")
    rel_country_Protein[1:243,1:3,] <- long_rel_Protein_Projection.mat[1:243,j_Item,j_Meat,1:3,]
  } 
  for (s in 1:3) {
    for (i in 1:length(interp.year)) {
      # baseline emission (2012 emission intensity * 2012 protein production ) * relative changes in protein production * constant emission intensities 
      proj_consEmisInt_country_CH4_FAOSTAT[1:243,s,m,i] <- baseline_country_CH4_FAOSTAT_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i] 
      proj_consEmisInt_country_CH4_2019T1[1:243,s,m,i] <- baseline_country_CH4_2019T1_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i]      
      proj_consEmisInt_country_CH4_2019MT[1:243,s,m,i] <- baseline_country_CH4_2019MT_nodraft[3,1:243,m] * rel_country_Protein[1:243,s,i]       
    }
  }
}

##########################################
# 6.8.2 Projections on the other livestock categories
##########################################

# Emissions from other livestock is assumed to be the same as historical variation between 2012-2018, and be constant after 2018 
# Note: Pigs c(14,15) and Poultry c(6,7,8,16) have been calculated above thus kept as 0
other_cate <- c(1,3,10,11,12)

# Alternatively, we keep the 2012 value as projections
for (s in 1:3) {
  for (i in 1:length(interp.year)) {
    proj_consEmisInt_country_CH4_FAOSTAT[1:243,s,other_cate,i] <- country_CH4_FAOSTAT[3,1:243,other_cate,13] 
    proj_consEmisInt_country_CH4_2019T1[1:243,s,other_cate,i] <- country_CH4_2019T1[3,1:243,other_cate,13] 
    proj_consEmisInt_country_CH4_2019MT[1:243,s,other_cate,i] <- country_CH4_2019MT[3,1:243,other_cate,13] 
  }  
}

# set NA and infinite value to 0 for further calculations
proj_consEmisInt_country_CH4_FAOSTAT[is.na(proj_consEmisInt_country_CH4_FAOSTAT) | is.infinite(proj_consEmisInt_country_CH4_FAOSTAT) | proj_consEmisInt_country_CH4_FAOSTAT < 0.0] <- 0.0
proj_consEmisInt_country_CH4_2019T1[is.na(proj_consEmisInt_country_CH4_2019T1) | is.infinite(proj_consEmisInt_country_CH4_2019T1) | proj_consEmisInt_country_CH4_2019T1 < 0.0] <- 0.0
proj_consEmisInt_country_CH4_2019MT[is.na(proj_consEmisInt_country_CH4_2019MT) | is.infinite(proj_consEmisInt_country_CH4_2019MT) | proj_consEmisInt_country_CH4_2019MT < 0.0] <- 0.0

##########################################
# 6.8.3 Projections on the draft animals
##########################################

# "Buffaloes" and "CattleNonDairy" only
draft_cate <- c(2,5)
# Emissions from draft animal is assumed to be the same as historical variation between 2012-2018
# After 2018, constant emissions from draft animal is assumed

# 6.8.3.1 calculate emissions from draft animals from 2012 to 2050
# Country/Area,Scenario,Livestock category,Year
proj_consEmisInt_country_CH4_FAOSTAT_draft <- array(0.0,c(243,3,18,length(interp.year)))
proj_consEmisInt_country_CH4_2019T1_draft <- array(0.0,c(243,3,18,length(interp.year)))
proj_consEmisInt_country_CH4_2019MT_draft <- array(0.0,c(243,3,18,length(interp.year)))
for (m in c(2,5)) {
  for (cc in 1:243) {
    for (s in 1:3) {
      # we start use 2012 for all years
      # For the countries/areas that the regression slope > 0 (i.e., emission from draft animal increase following increasing GDP per Capita), we assume the constant emissions as that of 2018 
      for (i in 1:length(interp.year)) {
        proj_consEmisInt_country_CH4_FAOSTAT_draft[1:243,s,m,i] <- calc_country_CH4_FAOSTAT_draft[3,1:243,m,13] 
        proj_consEmisInt_country_CH4_2019T1_draft[1:243,s,m,i] <- calc_country_CH4_2019T1_draft[3,1:243,m,13]
        proj_consEmisInt_country_CH4_2019MT_draft[1:243,s,m,i] <- calc_country_CH4_2019MT_draft[3,1:243,m,13]
      }
    }
  }
}

# set nondraft (except "Buffaloes" and "CattleNonDairy"), NA, and < 0 values to 0
tmp <- proj_consEmisInt_country_CH4_FAOSTAT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_consEmisInt_country_CH4_FAOSTAT_draft <- tmp

tmp <- proj_consEmisInt_country_CH4_2019T1_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_consEmisInt_country_CH4_2019T1_draft <- tmp

tmp <- proj_consEmisInt_country_CH4_2019MT_draft
tmp[,,nondraft_cate,] <- 0.0
tmp[is.na(tmp) | is.infinite(tmp) | tmp < 0.0] <- 0.0
proj_consEmisInt_country_CH4_2019MT_draft <- tmp

# 6.8.3.2 add emissions from draft animals to total emissions
# all values for draft animals need to be added to the values of non-draft animals calculated above
proj_consEmisInt_country_CH4_FAOSTAT[1:243,,draft_cate,] <- proj_consEmisInt_country_CH4_FAOSTAT[1:243,,draft_cate,] + proj_consEmisInt_country_CH4_FAOSTAT_draft[1:243,,draft_cate,]
proj_consEmisInt_country_CH4_2019T1[1:243,,draft_cate,] <- proj_consEmisInt_country_CH4_2019T1[1:243,,draft_cate,] + proj_consEmisInt_country_CH4_2019T1_draft[1:243,,draft_cate,]
proj_consEmisInt_country_CH4_2019MT[1:243,,draft_cate,] <- proj_consEmisInt_country_CH4_2019MT[1:243,,draft_cate,] + proj_consEmisInt_country_CH4_2019MT_draft[1:243,,draft_cate,]


##########################################
##########################################
# 6.9 Calculate Regional Historical and Projections on the livestock CH4 emissions
##########################################
##########################################

##########################################
# 6.9.1 Regional historical emissions for all livestock category
##########################################
# First get country emissions for 18 livestock category
cate_country_CH4_FAOSTAT <- array(NA,c(3,243,18,19))
cate_country_CH4_2019T1 <- array(NA,c(3,243,18,19))
cate_country_CH4_2019MT <- array(NA,c(3,243,18,19))

cate_country_CH4_FAOSTAT[,,1:16,] <- country_CH4_FAOSTAT
cate_country_CH4_FAOSTAT[,,17,] <- apply(country_CH4_FAOSTAT[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
cate_country_CH4_FAOSTAT[,,18,] <- apply(country_CH4_FAOSTAT[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
cate_country_CH4_FAOSTAT[,,c(14,15),] <- 0.0
cate_country_CH4_FAOSTAT[,,c(6,7,8,16),] <- 0.0
cate_country_CH4_2019T1[,,1:16,] <- country_CH4_2019T1
cate_country_CH4_2019T1[,,17,] <- apply(country_CH4_2019T1[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
cate_country_CH4_2019T1[,,18,] <- apply(country_CH4_2019T1[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
cate_country_CH4_2019T1[,,c(14,15),] <- 0.0
cate_country_CH4_2019T1[,,c(6,7,8,16),] <- 0.0
cate_country_CH4_2019MT[,,1:16,] <- country_CH4_2019MT
cate_country_CH4_2019MT[,,17,] <- apply(country_CH4_2019MT[,,c(14,15),],c(1,2,4),sum,na.rm=TRUE) 
cate_country_CH4_2019MT[,,18,] <- apply(country_CH4_2019MT[,,c(6,7,8,16),],c(1,2,4),sum,na.rm=TRUE) 
cate_country_CH4_2019MT[,,c(14,15),] <- 0.0
cate_country_CH4_2019MT[,,c(6,7,8,16),] <- 0.0

# History/proj/proj_consEmisInt/proj_decincEmisInt,11,BAU/SS/TS,51
full.year <- seq(2000,2050,by=1)
comb_cate_regional_CH4_FAOSTAT <- array(NA,c(4,11,3,18,length(full.year)))
comb_cate_regional_CH4_2019T1 <- array(NA,c(4,11,3,18,length(full.year)))
comb_cate_regional_CH4_2019MT <- array(NA,c(4,11,3,18,length(full.year)))


for (k in 1:10) {
  if (k == 2) {
    comb_cate_regional_CH4_FAOSTAT[1,k,1,1:18,1:length(year)] <- cate_country_CH4_FAOSTAT[3,gleam.list[[k]],1:18,1:length(year)]/1e3
    comb_cate_regional_CH4_FAOSTAT[2,k,1:3,1:18,13:length(full.year)] <- proj_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    comb_cate_regional_CH4_FAOSTAT[3,k,1:3,1:18,13:length(full.year)] <- proj_consEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    comb_cate_regional_CH4_FAOSTAT[4,k,1:3,1:18,13:length(full.year)] <- proj_decincEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    
    comb_cate_regional_CH4_2019T1[1,k,1,1:18,1:length(year)] <- cate_country_CH4_2019T1[3,gleam.list[[k]],1:18,1:length(year)]/1e3
    comb_cate_regional_CH4_2019T1[2,k,1:3,1:18,13:length(full.year)] <- proj_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    comb_cate_regional_CH4_2019T1[3,k,1:3,1:18,13:length(full.year)] <- proj_consEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    comb_cate_regional_CH4_2019T1[4,k,1:3,1:18,13:length(full.year)] <- proj_decincEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    
    comb_cate_regional_CH4_2019MT[1,k,1,1:18,1:length(year)] <- cate_country_CH4_2019MT[3,gleam.list[[k]],1:18,1:length(year)]/1e3
    comb_cate_regional_CH4_2019MT[2,k,1:3,1:18,13:length(full.year)] <- proj_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    comb_cate_regional_CH4_2019MT[3,k,1:3,1:18,13:length(full.year)] <- proj_consEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    comb_cate_regional_CH4_2019MT[4,k,1:3,1:18,13:length(full.year)] <- proj_decincEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)]/1e3
    
  } else {
    comb_cate_regional_CH4_FAOSTAT[1,k,1,1:18,1:length(year)] <- apply(cate_country_CH4_FAOSTAT[3,gleam.list[[k]],1:18,1:length(year)],c(2,3),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_FAOSTAT[2,k,1:3,1:18,13:length(full.year)] <- apply(proj_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_FAOSTAT[3,k,1:3,1:18,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_FAOSTAT[4,k,1:3,1:18,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    
    comb_cate_regional_CH4_2019T1[1,k,1,1:18,1:length(year)] <- apply(cate_country_CH4_2019T1[3,gleam.list[[k]],1:18,1:length(year)],c(2,3),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_2019T1[2,k,1:3,1:18,13:length(full.year)] <- apply(proj_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_2019T1[3,k,1:3,1:18,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_2019T1[4,k,1:3,1:18,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    
    comb_cate_regional_CH4_2019MT[1,k,1,1:18,1:length(year)] <- apply(cate_country_CH4_2019MT[3,gleam.list[[k]],1:18,1:length(year)],c(2,3),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_2019MT[2,k,1:3,1:18,13:length(full.year)] <- apply(proj_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_2019MT[3,k,1:3,1:18,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    comb_cate_regional_CH4_2019MT[4,k,1:3,1:18,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
    
  }
}
k <- 11
comb_cate_regional_CH4_FAOSTAT[1,k,1,1:18,1:length(year)] <- apply(cate_country_CH4_FAOSTAT[3,1:243,1:18,1:length(year)],c(2,3),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_FAOSTAT[2,k,1:3,1:18,13:length(full.year)] <- apply(proj_country_CH4_FAOSTAT[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_FAOSTAT[3,k,1:3,1:18,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_FAOSTAT[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_FAOSTAT[4,k,1:3,1:18,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_FAOSTAT[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3

comb_cate_regional_CH4_2019T1[1,k,1,1:18,1:length(year)] <- apply(cate_country_CH4_2019T1[3,1:243,1:18,1:length(year)],c(2,3),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_2019T1[2,k,1:3,1:18,13:length(full.year)] <- apply(proj_country_CH4_2019T1[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_2019T1[3,k,1:3,1:18,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019T1[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_2019T1[4,k,1:3,1:18,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019T1[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3

comb_cate_regional_CH4_2019MT[1,k,1,1:18,1:length(year)] <- apply(cate_country_CH4_2019MT[3,1:243,1:18,1:length(year)],c(2,3),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_2019MT[2,k,1:3,1:18,13:length(full.year)] <- apply(proj_country_CH4_2019MT[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_2019MT[3,k,1:3,1:18,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019MT[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3
comb_cate_regional_CH4_2019MT[4,k,1:3,1:18,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019MT[1:243,1:3,1:18,1:length(interp.year)],c(2,3,4),sum,na.rm=TRUE)/1e3


comb_cate_regional_CH4_FAOSTAT[comb_cate_regional_CH4_FAOSTAT == 0.0] <- NA
comb_cate_regional_CH4_2019T1[comb_cate_regional_CH4_2019T1 == 0.0] <- NA
comb_cate_regional_CH4_2019MT[comb_cate_regional_CH4_2019MT == 0.0] <- NA
# fill historical value
for (s in 1:3) {
  comb_cate_regional_CH4_FAOSTAT[2,,s,,1:12] <- comb_cate_regional_CH4_FAOSTAT[1,,1,,1:12]
  comb_cate_regional_CH4_2019T1[2,,s,,1:12] <- comb_cate_regional_CH4_2019T1[1,,1,,1:12]
  comb_cate_regional_CH4_2019MT[2,,s,,1:12] <- comb_cate_regional_CH4_2019MT[1,,1,,1:12]
  comb_cate_regional_CH4_FAOSTAT[3,,s,,1:12] <- comb_cate_regional_CH4_FAOSTAT[1,,1,,1:12]
  comb_cate_regional_CH4_2019T1[3,,s,,1:12] <- comb_cate_regional_CH4_2019T1[1,,1,,1:12]
  comb_cate_regional_CH4_2019MT[3,,s,,1:12] <- comb_cate_regional_CH4_2019MT[1,,1,,1:12]
  comb_cate_regional_CH4_FAOSTAT[4,,s,,1:12] <- comb_cate_regional_CH4_FAOSTAT[1,,1,,1:12]
  comb_cate_regional_CH4_2019T1[4,,s,,1:12] <- comb_cate_regional_CH4_2019T1[1,,1,,1:12]
  comb_cate_regional_CH4_2019MT[4,,s,,1:12] <- comb_cate_regional_CH4_2019MT[1,,1,,1:12]
}

##########################################
# 6.9.2 Regional historical emissions for all livestock category
##########################################

# History/proj/proj_consEmisInt/proj_decincEmisInt,11,BAU/SS/TS,51
full.year <- seq(2000,2050,by=1)
comb_regional_CH4_FAOSTAT <- array(NA,c(4,11,3,length(full.year)))
comb_regional_CH4_2019T1 <- array(NA,c(4,11,3,length(full.year)))
comb_regional_CH4_2019MT <- array(NA,c(4,11,3,length(full.year)))

for (k in 1:10) {
  if (k == 2) {
    comb_regional_CH4_FAOSTAT[1,k,1,1:length(year)] <- apply(country_CH4_FAOSTAT[3,gleam.list[[k]],1:16,1:length(year)],c(2),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_FAOSTAT[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_FAOSTAT[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_FAOSTAT[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    
    comb_regional_CH4_2019T1[1,k,1,1:length(year)] <- apply(country_CH4_2019T1[3,gleam.list[[k]],1:16,1:length(year)],c(2),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019T1[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019T1[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019T1[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    
    comb_regional_CH4_2019MT[1,k,1,1:length(year)] <- apply(country_CH4_2019MT[3,gleam.list[[k]],1:16,1:length(year)],c(2),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019MT[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019MT[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019MT[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(1,3),sum,na.rm=TRUE)/1e3
    
  } else {
    comb_regional_CH4_FAOSTAT[1,k,1,1:length(year)] <- apply(country_CH4_FAOSTAT[3,gleam.list[[k]],1:16,1:length(year)],c(3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_FAOSTAT[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_FAOSTAT[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_FAOSTAT[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_FAOSTAT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    
    comb_regional_CH4_2019T1[1,k,1,1:length(year)] <- apply(country_CH4_2019T1[3,gleam.list[[k]],1:16,1:length(year)],c(3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019T1[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019T1[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019T1[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019T1[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    
    comb_regional_CH4_2019MT[1,k,1,1:length(year)] <- apply(country_CH4_2019MT[3,gleam.list[[k]],1:16,1:length(year)],c(3),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019MT[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019MT[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    comb_regional_CH4_2019MT[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019MT[gleam.list[[k]],1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
    
  }
}
k <- 11
comb_regional_CH4_FAOSTAT[1,k,1,1:length(year)] <- apply(country_CH4_FAOSTAT[3,1:243,1:16,1:length(year)],c(3),sum,na.rm=TRUE)/1e3
comb_regional_CH4_FAOSTAT[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_FAOSTAT[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
comb_regional_CH4_FAOSTAT[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_FAOSTAT[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
comb_regional_CH4_FAOSTAT[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_FAOSTAT[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3

comb_regional_CH4_2019T1[1,k,1,1:length(year)] <- apply(country_CH4_2019T1[3,1:243,1:16,1:length(year)],c(3),sum,na.rm=TRUE)/1e3
comb_regional_CH4_2019T1[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_2019T1[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
comb_regional_CH4_2019T1[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019T1[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
comb_regional_CH4_2019T1[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019T1[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3

comb_regional_CH4_2019MT[1,k,1,1:length(year)] <- apply(country_CH4_2019MT[3,1:243,1:16,1:length(year)],c(3),sum,na.rm=TRUE)/1e3
comb_regional_CH4_2019MT[2,k,1:3,13:length(full.year)] <- apply(proj_country_CH4_2019MT[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
comb_regional_CH4_2019MT[3,k,1:3,13:length(full.year)] <- apply(proj_consEmisInt_country_CH4_2019MT[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3
comb_regional_CH4_2019MT[4,k,1:3,13:length(full.year)] <- apply(proj_decincEmisInt_country_CH4_2019MT[1:243,1:3,1:18,1:length(interp.year)],c(2,4),sum,na.rm=TRUE)/1e3


comb_regional_CH4_FAOSTAT[comb_regional_CH4_FAOSTAT == 0.0] <- NA
comb_regional_CH4_2019T1[comb_regional_CH4_2019T1 == 0.0] <- NA
comb_regional_CH4_2019MT[comb_regional_CH4_2019MT == 0.0] <- NA

##########################################
# save output of projections on livestock emissions using 2006T1 (FAOSTAT), 2019T1 and 2019MT methods
##########################################
save.image(file=paste0(path_out,"Livestock_CH4_Emissions_projections_p5yr.RData"))
# clear cache 
rm(list=ls())
gc()

