#! /usr/bin/env python

from lab.reports import arithmetic_mean, geometric_mean, percentile_50, percentile_75, Attribute

from tools.my_experiment import MyExperiment
from tools.my_reports import FlexibleAttribute

import ms_parser_attributes
import sys
sys.path.insert(1, '../merge-strategies')
import merge_strategies_attributes
import miasm_attributes
import symmetries_attributes
sys.path.insert(1, '../merge-and-shrink-jair')
import old_mas_attributes

import math
import os.path

# Set the following variables for the experiment
REPO_NAME = 'downward/dissertation'
REVISIONS = [
    'a7b231dd8d46', # dissertation branch (all except olr)
]

CONFIGS = {}

# Create the experiment. Add parsers, fetchers or reports...
exp = MyExperiment(
    repo_name=REPO_NAME,
    configs=CONFIGS,
    revisions=REVISIONS,
    #test_suite=['depot:pfile1', 'depot:pfile2', 'gripper:prob01.pddl', 'gripper:prob02.pddl']
)
# for attributes
exp.add_search_parser('../ms-parser.py', ms_parser_attributes.get())
exp.add_search_parser('../merge-strategies/merge-strategies-parser.py', merge_strategies_attributes.get())
exp.add_search_parser('../merge-strategies/miasm-parser.py', miasm_attributes.get())
exp.add_search_parser('../merge-strategies/symmetries-parser.py', symmetries_attributes.get())
exp.add_search_parser('../merge-and-shrink-jair/old-mas-parser.py', old_mas_attributes.get())

def remove_revision(run):
    algorithm = run['algorithm']
    for rev in REVISIONS:
        algorithm = algorithm.replace('%s-' % rev, '')
    run['algorithm'] = algorithm
    return run

exp.add_fetcher(os.path.expanduser('~/experiments/dissertation/nlr-b50k-label-reduction-eval/'),filter=[remove_revision])
exp.add_fetcher(os.path.expanduser('~/experiments/dissertation/nlr-b50k-symm-eval/'),filter=[remove_revision])

flexible_attributes = [
    FlexibleAttribute('coverage', 'Coverage', None, absolute=True, min_wins=False, functions=[sum]),
    FlexibleAttribute('expansions_until_last_jump', 'Exp', float('inf'), absolute=False, min_wins=True, functions=[percentile_50, percentile_75]),
    FlexibleAttribute('search_time', 'Search time', None, absolute=False, min_wins=True, functions=[geometric_mean]),
    FlexibleAttribute('total_time', 'Total time', None, absolute=False, min_wins=True, functions=[geometric_mean]),
    #FlexibleAttribute('search_out_of_memory', 'Search oom', None, absolute=True, min_wins=True, functions=[sum]),
    #FlexibleAttribute('search_out_of_time', 'Search oot', None, absolute=True, min_wins=True, functions=[sum]),

    FlexibleAttribute('ms_abstraction_constructed', '\# constr', False, absolute=True, min_wins=False, functions=[sum]),
    #FlexibleAttribute('ms_atomic_construction_time', 'Atomic constr time', None, absolute=False, min_wins=True, functions=[arithmetic_mean]),
    FlexibleAttribute('ms_construction_time', 'Constr time', None, absolute=False, min_wins=True, functions=[arithmetic_mean]),
    FlexibleAttribute('ms_out_of_memory', 'Constr oom', None, absolute=True, min_wins=True, functions=[sum]),
    FlexibleAttribute('ms_out_of_time', 'Constr oot', None, absolute=True, min_wins=True, functions=[sum]),
    FlexibleAttribute('perfect_heuristic', 'Perfect h', None, absolute=True, min_wins=False),
    ##FlexibleAttribute('ms_tiebreaking_iterations', 'Tiebreak it', 0, absolute=True, min_wins=True, functions=[sum]),
    #FlexibleAttribute('ms_max_int_abs_size', 'MITSS', None, absolute=False, min_wins=True, functions=[geometric_mean]),
    FlexibleAttribute('ms_linear_order', 'Linear tree', None, absolute=True, min_wins=True, functions=[sum]),
    FlexibleAttribute('fallback_only', 'Fallback', None, absolute=True, min_wins=True, functions=[sum]),
    FlexibleAttribute('bliss_total_calls', 'Bliss comp', None, absolute=True, min_wins=False),
    # Ideally, we would like to make the following to non-absolute, but the
    # baseline configs do not have a default value for these attributes.
    FlexibleAttribute('bliss_time_average', 'B time avg (am)', None, absolute=True, min_wins=True, functions=[arithmetic_mean]),
    FlexibleAttribute('bliss_time_average', 'B time avg (gm)', None, absolute=True, min_wins=True, functions=[geometric_mean]),
    FlexibleAttribute('bliss_memory_out', 'Bliss oom', None, absolute=True, min_wins=True),
    FlexibleAttribute('bliss_timeout', 'Bliss oot', None, absolute=True, min_wins=True),
    FlexibleAttribute('merging_for_symmetries_attempts', 'Symm attempts', None, absolute=True, min_wins=False, functions=[sum]),
    FlexibleAttribute('merging_for_symmetries_fail_shrinking', 'Symm fail shrink', None, absolute=True, min_wins=True, functions=[sum]),
    FlexibleAttribute('merging_for_symmetries_fail_pruning', 'Symm fail prune', None, absolute=True, min_wins=True, functions=[sum]),
    FlexibleAttribute('merging_for_symmetries_fail_any', 'Symm fail any', None, absolute=True, min_wins=True, functions=[sum]),
]


####### Fix problem of not started runs ##########

import os
from lab.reports.filter import FilterReport

missing_olr_tasks = ['gripper:prob02.pddl', 'pipesworld-notankage:p43-net5-b24-g3.pddl', 'satellite:p33-HC-pfile13.pddl', 'sokoban-opt08-strips:p05.pddl'] # olr
missing_nlr_tasks = ['barman-opt14-strips:p638-2.pddl', 'blocks:probBLOCKS-8-2.pddl', 'elevators-opt11-strips:p09.pddl', 'freecell:p16.pddl', 'ged-opt14-strips:d-2-3.pddl', 'ged-opt14-strips:d-8-9.pddl', 'gripper:prob02.pddl', 'miconic:s21-1.pddl', 'miconic:s6-2.pddl', 'parking-opt14-strips:p_12_7-02.pddl', 'parking-opt14-strips:p_16_9-04.pddl', 'pathways-noneg:p07.pddl', 'psr-small:p10-s17-n2-l2-f30.pddl', 'satellite:p23-HC-pfile3.pddl', 'satellite:p33-HC-pfile13.pddl', 'tetris-opt14-strips:p03-4.pddl', 'transport-opt08-strips:p23.pddl', 'visitall-opt14-strips:p-1-12.pddl'] # nlr
missing_sota_tasks = ['airport:p10-airport3-p1.pddl', 'barman-opt11-strips:pfile04-015.pddl', 'blocks:probBLOCKS-14-1.pddl', 'driverlog:p04.pddl', 'elevators-opt08-strips:p27.pddl', 'elevators-opt11-strips:p05.pddl', 'elevators-opt11-strips:p15.pddl', 'floortile-opt11-strips:opt-p08-015.pddl', 'grid:prob04.pddl', 'hiking-opt14-strips:ptesting-2-4-6.pddl', 'logistics00:probLOGISTICS-13-1.pddl', 'logistics98:prob09.pddl', 'miconic:s22-2.pddl', 'miconic:s26-4.pddl', 'miconic:s28-0.pddl', 'movie:prob13.pddl', 'nomystery-opt11-strips:p16.pddl', 'openstacks-opt08-strips:p20.pddl', 'openstacks-opt14-strips:p20_2.pddl', 'openstacks-strips:p05.pddl', 'parcprinter-08-strips:p07.pddl', 'pegsol-08-strips:p18.pddl', 'pegsol-opt11-strips:p07.pddl', 'pipesworld-notankage:p11-net2-b10-g2.pddl', 'psr-small:p25-s40-n3-l4-f10.pddl', 'rovers:p09.pddl', 'satellite:p19-pfile19.pddl', 'satellite:p21-HC-pfile1.pddl', 'satellite:p33-HC-pfile13.pddl', 'scanalyzer-opt11-strips:p03.pddl', 'sokoban-opt11-strips:p16.pddl', 'sokoban-opt11-strips:p18.pddl', 'tidybot-opt11-strips:p15.pddl', 'tpp:p11.pddl', 'transport-opt08-strips:p08.pddl', 'transport-opt14-strips:p13.pddl', 'zenotravel:p09.pddl'] # sota

missing_olr_algorithms = ['olr-b50k-cggl', 'olr-b50k-cggl-nolr', 'olr-b50k-gcgl', 'olr-b50k-gcgl-nolr', 'olr-b50k-l', 'olr-b50k-l-nolr', 'olr-b50k-rl', 'olr-b50k-rl-nolr', 'olr-f50k-cggl', 'olr-f50k-cggl-nolr', 'olr-f50k-gcgl', 'olr-f50k-gcgl-nolr', 'olr-f50k-l', 'olr-f50k-l-nolr', 'olr-f50k-rl', 'olr-f50k-rl-nolr', 'olr-ginf-cggl', 'olr-ginf-cggl-nolr', 'olr-ginf-gcgl', 'olr-ginf-gcgl-nolr', 'olr-ginf-l', 'olr-ginf-l-nolr', 'olr-ginf-rl', 'olr-ginf-rl-nolr'] # olr
missing_nlr_algorithms = ['nlr-b50k-cggl-2ts', 'nlr-b50k-cggl-fprnd', 'nlr-b50k-cggl-nolr', 'nlr-b50k-cggl-oncernd', 'nlr-b50k-dfp-2ts', 'nlr-b50k-dfp-fprnd', 'nlr-b50k-dfp-nolr', 'nlr-b50k-dfp-oncernd', 'nlr-b50k-gcgl-2ts', 'nlr-b50k-gcgl-fprnd', 'nlr-b50k-gcgl-nolr', 'nlr-b50k-gcgl-oncernd', 'nlr-b50k-l-2ts', 'nlr-b50k-l-fprnd', 'nlr-b50k-l-nolr', 'nlr-b50k-l-oncernd', 'nlr-b50k-miasm-2ts', 'nlr-b50k-miasm-fprnd', 'nlr-b50k-miasm-nolr', 'nlr-b50k-miasm-oncernd', 'nlr-b50k-rl-2ts', 'nlr-b50k-rl-fprnd', 'nlr-b50k-rl-nolr', 'nlr-b50k-rl-oncernd', 'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd', 'nlr-b50k-symmb60cggl-smallest-nonlinear-fprnd', 'nlr-b50k-symmb60dfp-largest-nonlinear-fprnd', 'nlr-b50k-symmb60dfp-smallest-nonlinear-fprnd', 'nlr-b50k-symmb60gcgl-largest-nonlinear-fprnd', 'nlr-b50k-symmb60gcgl-smallest-nonlinear-fprnd', 'nlr-b50k-symmb60l-largest-nonlinear-fprnd', 'nlr-b50k-symmb60l-smallest-nonlinear-fprnd', 'nlr-b50k-symmb60miasm-largest-nonlinear-fprnd', 'nlr-b50k-symmb60miasm-smallest-nonlinear-fprnd', 'nlr-b50k-symmb60rl-largest-nonlinear-fprnd', 'nlr-b50k-symmb60rl-smallest-nonlinear-fprnd', 'nlr-b50k-symmi1cggl-largest-nonlinear-fprnd', 'nlr-b50k-symmi1cggl-smallest-nonlinear-fprnd', 'nlr-b50k-symmi1dfp-largest-nonlinear-fprnd', 'nlr-b50k-symmi1dfp-smallest-nonlinear-fprnd', 'nlr-b50k-symmi1gcgl-largest-nonlinear-fprnd', 'nlr-b50k-symmi1gcgl-smallest-nonlinear-fprnd', 'nlr-b50k-symmi1l-largest-nonlinear-fprnd', 'nlr-b50k-symmi1l-smallest-nonlinear-fprnd', 'nlr-b50k-symmi1miasm-largest-nonlinear-fprnd', 'nlr-b50k-symmi1miasm-smallest-nonlinear-fprnd', 'nlr-b50k-symmi1rl-largest-nonlinear-fprnd', 'nlr-b50k-symmi1rl-smallest-nonlinear-fprnd', 'nlr-f50k-cggl-2ts', 'nlr-f50k-cggl-fprnd', 'nlr-f50k-cggl-nolr', 'nlr-f50k-cggl-oncernd', 'nlr-f50k-dfp-2ts', 'nlr-f50k-dfp-fprnd', 'nlr-f50k-dfp-nolr', 'nlr-f50k-dfp-oncernd', 'nlr-f50k-gcgl-2ts', 'nlr-f50k-gcgl-fprnd', 'nlr-f50k-gcgl-nolr', 'nlr-f50k-gcgl-oncernd', 'nlr-f50k-l-2ts', 'nlr-f50k-l-fprnd', 'nlr-f50k-l-nolr', 'nlr-f50k-l-oncernd', 'nlr-f50k-miasm-2ts', 'nlr-f50k-miasm-fprnd', 'nlr-f50k-miasm-nolr', 'nlr-f50k-miasm-oncernd', 'nlr-f50k-rl-2ts', 'nlr-f50k-rl-fprnd', 'nlr-f50k-rl-nolr', 'nlr-f50k-rl-oncernd', 'nlr-f50k-symmb60cggl-largest-nonlinear-fprnd', 'nlr-f50k-symmb60dfp-largest-nonlinear-fprnd', 'nlr-f50k-symmb60gcgl-largest-nonlinear-fprnd', 'nlr-f50k-symmb60l-largest-nonlinear-fprnd', 'nlr-f50k-symmb60miasm-largest-nonlinear-fprnd', 'nlr-f50k-symmb60rl-largest-nonlinear-fprnd', 'nlr-ginf-cggl-2ts', 'nlr-ginf-cggl-fprnd', 'nlr-ginf-cggl-nolr', 'nlr-ginf-cggl-oncernd', 'nlr-ginf-dfp-2ts', 'nlr-ginf-dfp-fprnd', 'nlr-ginf-dfp-nolr', 'nlr-ginf-dfp-oncernd', 'nlr-ginf-gcgl-2ts', 'nlr-ginf-gcgl-fprnd', 'nlr-ginf-gcgl-nolr', 'nlr-ginf-gcgl-oncernd', 'nlr-ginf-l-2ts', 'nlr-ginf-l-fprnd', 'nlr-ginf-l-nolr', 'nlr-ginf-l-oncernd', 'nlr-ginf-miasm-2ts', 'nlr-ginf-miasm-fprnd', 'nlr-ginf-miasm-nolr', 'nlr-ginf-miasm-oncernd', 'nlr-ginf-rl-2ts', 'nlr-ginf-rl-fprnd', 'nlr-ginf-rl-nolr', 'nlr-ginf-rl-oncernd', 'nlr-ginf-symmb60cggl-largest-nonlinear-fprnd', 'nlr-ginf-symmb60dfp-largest-nonlinear-fprnd', 'nlr-ginf-symmb60gcgl-largest-nonlinear-fprnd', 'nlr-ginf-symmb60l-largest-nonlinear-fprnd', 'nlr-ginf-symmb60miasm-largest-nonlinear-fprnd', 'nlr-ginf-symmb60rl-largest-nonlinear-fprnd'] # nlr
missing_sota_algorithms = ['b50k-cggl-fprnd-full-reg', 'b50k-dfp-allrnd-fprnd-full-reg', 'b50k-dfp-l-nto-abp-fprnd-full-reg', 'b50k-dfp-l-nto-pba-fprnd-full-reg', 'b50k-dfp-l-otn-abp-fprnd-full-reg', 'b50k-dfp-l-otn-pba-fprnd-full-reg', 'b50k-dfp-l-rnd-abp-fprnd-full-reg', 'b50k-dfp-l-rnd-pba-fprnd-full-reg', 'b50k-dfp-rl-nto-abp-fprnd-full-reg', 'b50k-dfp-rl-nto-pba-fprnd-full-reg', 'b50k-dfp-rl-otn-abp-fprnd-full-reg', 'b50k-dfp-rl-otn-pba-fprnd-full-reg', 'b50k-dfp-rl-rnd-abp-fprnd-full-reg', 'b50k-dfp-rl-rnd-pba-fprnd-full-reg', 'b50k-dfp-rnd-nto-abp-fprnd-full-reg', 'b50k-dfp-rnd-nto-pba-fprnd-full-reg', 'b50k-dfp-rnd-otn-abp-fprnd-full-reg', 'b50k-dfp-rnd-otn-pba-fprnd-full-reg', 'b50k-dfp-rnd-rnd-abp-fprnd-full-reg', 'b50k-dfp-rnd-rnd-pba-fprnd-full-reg', 'b50k-gcgl-fprnd-full-reg', 'b50k-l-fprnd-full-reg', 'b50k-miasm-dfp-l-nto-abp-fprnd-full-reg', 'b50k-miasm-dfp-rl-nto-pba-fprnd-full-reg', 'b50k-rl-fprnd-full-reg', 'b50k-sccs-dec-cggl-full-reg', 'b50k-sccs-dec-dfp-allrnd-fprnd-full-reg', 'b50k-sccs-dec-dfp-l-nto-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-l-nto-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-l-otn-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-l-otn-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-dec-dfp-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-dec-dfp-rnd-rnd-pba-fprnd-full-reg', 'b50k-sccs-dec-gcgl-full-reg', 'b50k-sccs-dec-l-full-reg', 'b50k-sccs-dec-rl-full-reg', 'b50k-sccs-dec-sbmiasm-allrnd-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-l-nto-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-l-nto-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-l-otn-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-l-otn-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-dec-sbmiasm-rnd-rnd-pba-fprnd-full-reg', 'b50k-sccs-inc-cggl-full-reg', 'b50k-sccs-inc-dfp-allrnd-fprnd-full-reg', 'b50k-sccs-inc-dfp-l-nto-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-l-nto-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-l-otn-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-l-otn-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-inc-dfp-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-inc-dfp-rnd-rnd-pba-fprnd-full-reg', 'b50k-sccs-inc-gcgl-full-reg', 'b50k-sccs-inc-l-full-reg', 'b50k-sccs-inc-rl-full-reg', 'b50k-sccs-revtop-cggl-full-reg', 'b50k-sccs-revtop-dfp-allrnd-fprnd-full-reg', 'b50k-sccs-revtop-dfp-l-nto-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-l-nto-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-l-otn-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-l-otn-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-revtop-dfp-rnd-rnd-pba-fprnd-full-reg', 'b50k-sccs-revtop-gcgl-full-reg', 'b50k-sccs-revtop-l-full-reg', 'b50k-sccs-revtop-rl-full-reg', 'b50k-sccs-revtop-sbmiasm-allrnd-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-l-nto-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-l-nto-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-l-otn-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-l-otn-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-revtop-sbmiasm-rnd-rnd-pba-fprnd-full-reg', 'b50k-sccs-top-cggl-full-reg', 'b50k-sccs-top-dfp-allrnd-fprnd-full-reg', 'b50k-sccs-top-dfp-l-nto-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-l-nto-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-l-otn-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-l-otn-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-top-dfp-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-top-dfp-rnd-rnd-pba-fprnd-full-reg', 'b50k-sccs-top-gcgl-full-reg', 'b50k-sccs-top-l-full-reg', 'b50k-sccs-top-rl-full-reg', 'b50k-sccs-top-sbmiasm-allrnd-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-l-nto-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-l-nto-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-l-otn-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-l-otn-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-l-rnd-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-l-rnd-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rl-nto-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rl-nto-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rl-otn-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rl-otn-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rl-rnd-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rl-rnd-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rnd-nto-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rnd-nto-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rnd-otn-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rnd-otn-pba-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rnd-rnd-abp-fprnd-full-reg', 'b50k-sccs-top-sbmiasm-rnd-rnd-pba-fprnd-full-reg', 'b50k-slone5-dfp-rl-nto-pba-fprnd-pirr-reg', 'b50k-symmb60cggl-largest-secondary-fprnd-full-reg', 'b50k-symmb60cggl-smallest-secondary-fprnd-full-reg', 'b50k-symmb60dfp-l-nto-abp-full-reg', 'b50k-symmb60dfp-l-nto-abp-largest-secondary-fprnd-full-reg', 'b50k-symmb60dfp-l-nto-abp-smallest-secondary-fprnd-full-reg', 'b50k-symmb60dfp-rl-nto-pba-full-reg', 'b50k-symmb60dfp-rl-nto-pba-largest-secondary-fprnd-full-reg', 'b50k-symmb60dfp-rl-nto-pba-smallest-secondary-fprnd-full-reg', 'b50k-symmb60gcgl-largest-secondary-fprnd-full-reg', 'b50k-symmb60gcgl-smallest-secondary-fprnd-full-reg', 'b50k-symmb60l-largest-secondary-fprnd-full-reg', 'b50k-symmb60l-smallest-secondary-fprnd-full-reg', 'b50k-symmb60rl-largest-secondary-fprnd-full-reg', 'b50k-symmb60rl-smallest-secondary-fprnd-full-reg', 'b50k-symmb60sbmiasm-rl-nto-pba-largest-secondary-fprnd-full-reg', 'b50k-symmb60sbmiasm-rl-nto-pba-smallest-secondary-fprnd-full-reg', 'b50k-symmi1cggl-largest-secondary-fprnd-full-reg', 'b50k-symmi1cggl-smallest-secondary-fprnd-full-reg', 'b50k-symmi1dfp-l-nto-abp-largest-secondary-fprnd-full-reg', 'b50k-symmi1dfp-l-nto-abp-smallest-secondary-fprnd-full-reg', 'b50k-symmi1dfp-rl-nto-pba-largest-secondary-fprnd-full-reg', 'b50k-symmi1dfp-rl-nto-pba-smallest-secondary-fprnd-full-reg', 'b50k-symmi1gcgl-largest-secondary-fprnd-full-reg', 'b50k-symmi1gcgl-smallest-secondary-fprnd-full-reg', 'b50k-symmi1l-largest-secondary-fprnd-full-reg', 'b50k-symmi1l-smallest-secondary-fprnd-full-reg', 'b50k-symmi1rl-largest-secondary-fprnd-full-reg', 'b50k-symmi1rl-smallest-secondary-fprnd-full-reg', 'b50k-symmi1sbmiasm-rl-nto-pba-largest-secondary-fprnd-full-reg', 'b50k-symmi1sbmiasm-rl-nto-pba-smallest-secondary-fprnd-full-reg', 'f50k-cggl-fprnd-full-reg', 'f50k-dfp-allrnd-fprnd-full-reg', 'f50k-dfp-l-nto-abp-fprnd-full-reg', 'f50k-dfp-l-nto-pba-fprnd-full-reg', 'f50k-dfp-l-otn-abp-fprnd-full-reg', 'f50k-dfp-l-otn-pba-fprnd-full-reg', 'f50k-dfp-l-rnd-abp-fprnd-full-reg', 'f50k-dfp-l-rnd-pba-fprnd-full-reg', 'f50k-dfp-rl-nto-abp-fprnd-full-reg', 'f50k-dfp-rl-nto-pba-fprnd-full-reg', 'f50k-dfp-rl-otn-abp-fprnd-full-reg', 'f50k-dfp-rl-otn-pba-fprnd-full-reg', 'f50k-dfp-rl-rnd-abp-fprnd-full-reg', 'f50k-dfp-rl-rnd-pba-fprnd-full-reg', 'f50k-dfp-rnd-nto-abp-fprnd-full-reg', 'f50k-dfp-rnd-nto-pba-fprnd-full-reg', 'f50k-dfp-rnd-otn-abp-fprnd-full-reg', 'f50k-dfp-rnd-otn-pba-fprnd-full-reg', 'f50k-dfp-rnd-rnd-abp-fprnd-full-reg', 'f50k-dfp-rnd-rnd-pba-fprnd-full-reg', 'f50k-gcgl-fprnd-full-reg', 'f50k-l-fprnd-full-reg', 'f50k-miasm-dfp-l-nto-abp-fprnd-full-reg', 'f50k-miasm-dfp-rl-nto-pba-fprnd-full-reg', 'f50k-rl-fprnd-full-reg', 'f50k-symmb60cggl-fprnd-full-reg', 'f50k-symmb60dfp-l-nto-abp-fprnd-full-reg', 'f50k-symmb60dfp-rl-nto-pba-fprnd-full-reg', 'f50k-symmb60gcgl-fprnd-full-reg', 'f50k-symmb60l-fprnd-full-reg', 'f50k-symmb60miasm-dfp-l-nto-abp-fprnd-full-reg', 'f50k-symmb60miasm-dfp-rl-nto-pba-fprnd-full-reg', 'f50k-symmb60rl-fprnd-full-reg', 'ginf-cggl-fprnd-full-reg', 'ginf-dfp-allrnd-fprnd-full-reg', 'ginf-dfp-l-nto-abp-fprnd-full-reg', 'ginf-dfp-l-nto-pba-fprnd-full-reg', 'ginf-dfp-l-otn-abp-fprnd-full-reg', 'ginf-dfp-l-otn-pba-fprnd-full-reg', 'ginf-dfp-l-rnd-abp-fprnd-full-reg', 'ginf-dfp-l-rnd-pba-fprnd-full-reg', 'ginf-dfp-rl-nto-abp-fprnd-full-reg', 'ginf-dfp-rl-nto-pba-fprnd-full-reg', 'ginf-dfp-rl-otn-abp-fprnd-full-reg', 'ginf-dfp-rl-otn-pba-fprnd-full-reg', 'ginf-dfp-rl-rnd-abp-fprnd-full-reg', 'ginf-dfp-rl-rnd-pba-fprnd-full-reg', 'ginf-dfp-rnd-nto-abp-fprnd-full-reg', 'ginf-dfp-rnd-nto-pba-fprnd-full-reg', 'ginf-dfp-rnd-otn-abp-fprnd-full-reg', 'ginf-dfp-rnd-otn-pba-fprnd-full-reg', 'ginf-dfp-rnd-rnd-abp-fprnd-full-reg', 'ginf-dfp-rnd-rnd-pba-fprnd-full-reg', 'ginf-gcgl-fprnd-full-reg', 'ginf-l-fprnd-full-reg', 'ginf-miasm-dfp-l-nto-abp-fprnd-full-reg', 'ginf-miasm-dfp-rl-nto-pba-fprnd-full-reg', 'ginf-rl-fprnd-full-reg', 'ginf-symmb60cggl-fprnd-full-reg', 'ginf-symmb60dfp-l-nto-abp-fprnd-full-reg', 'ginf-symmb60dfp-rl-nto-pba-fprnd-full-reg', 'ginf-symmb60gcgl-fprnd-full-reg', 'ginf-symmb60l-fprnd-full-reg', 'ginf-symmb60miasm-dfp-l-nto-abp-fprnd-full-reg', 'ginf-symmb60miasm-dfp-rl-nto-pba-fprnd-full-reg', 'ginf-symmb60rl-fprnd-full-reg', 'symmb60cggl-b50k-full-reg', 'symmb60gcgl-b50k-full-reg', 'symmb60l-b50k-full-reg', 'symmb60miasm-dfp-l-nto-abp-b50k-full-reg', 'symmb60miasm-dfp-rl-nto-pba-b50k-full-reg', 'symmb60rl-b50k-full-reg'] # sota

def remove_faulty_runs(run):
    algorithm = run['algorithm']
    domain_task = '{}:{}'.format(run['domain'], run['problem'])
    if (algorithm in missing_olr_algorithms and domain_task in missing_olr_tasks) or (algorithm in missing_nlr_algorithms and domain_task in missing_nlr_tasks) or (algorithm in missing_sota_algorithms and domain_task in missing_sota_tasks):
        #print "excluding run {} {}".format(algorithm, domain_task)
        return False
    return True

def remove_fixed_runs_revision(run):
    algorithm = run['algorithm']
    algorithm = algorithm.replace('81ab07c70f65-', '')
    run['algorithm'] = algorithm
    return run

exp.add_report(FilterReport(filter=[remove_faulty_runs]),outfile=os.path.join(exp.eval_dir, 'fixed-properties'))
exp.add_fetcher(os.path.expanduser('~/experiments/dissertation/olr-missing-runs-eval'),filter=[remove_fixed_runs_revision])
exp.add_fetcher(os.path.expanduser('~/experiments/dissertation/nlr-missing-runs-eval'),filter=[remove_fixed_runs_revision])
exp.add_fetcher(os.path.expanduser('~/experiments/dissertation/sota-missing-runs-eval'),filter=[remove_fixed_runs_revision])


###### HTML tables ###################################################

#exp.add_absolute_report(
    #attributes=['bliss_time_average', 'bliss_time_median'],publish=False,
    #filter_algorithm=[
        #'nlr-b50k-cggl-fprnd',
        #'nlr-b50k-symmi1cggl-largest-nonlinear-fprnd',
        #'nlr-b50k-symmi1cggl-smallest-nonlinear-fprnd',
        #'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
        #'nlr-b50k-symmb60cggl-smallest-nonlinear-fprnd',
    #]
#)


####### Paper tables #################################################

#### Factored symmetries ####

exp.add_comparison_report(
    algorithm_pairs=[
        ('nlr-b50k-cggl-fprnd', 'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd'),
        ('nlr-b50k-rl-fprnd', 'nlr-b50k-symmb60rl-largest-nonlinear-fprnd'),
        ('nlr-b50k-dfp-fprnd', 'nlr-b50k-symmb60dfp-largest-nonlinear-fprnd'),
        ('nlr-b50k-miasm-fprnd', 'nlr-b50k-symmb60miasm-largest-nonlinear-fprnd'),
    ],
    format='html',
    name='base-symm',
    #attributes=['coverage', 'expansions_until_last_jump'],
    publish=False,
)

exp.add_flexible_aggregation_report(
    algorithm_rows=[
        # first row
        [
            [
                'nlr-b50k-cggl-fprnd',
                'nlr-b50k-symmi1cggl-largest-nonlinear-fprnd',
                'nlr-b50k-symmi1cggl-smallest-nonlinear-fprnd',
                'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
                'nlr-b50k-symmb60cggl-smallest-nonlinear-fprnd',
            ],
            [
                'nlr-b50k-rl-fprnd',
                'nlr-b50k-symmi1rl-largest-nonlinear-fprnd',
                'nlr-b50k-symmi1rl-smallest-nonlinear-fprnd',
                'nlr-b50k-symmb60rl-largest-nonlinear-fprnd',
                'nlr-b50k-symmb60rl-smallest-nonlinear-fprnd',
            ],
        ],
        # second row
        [
            [
                'nlr-b50k-dfp-fprnd',
                'nlr-b50k-symmi1dfp-largest-nonlinear-fprnd',
                'nlr-b50k-symmi1dfp-smallest-nonlinear-fprnd',
                'nlr-b50k-symmb60dfp-largest-nonlinear-fprnd',
                'nlr-b50k-symmb60dfp-smallest-nonlinear-fprnd',
            ],
            [
                'nlr-b50k-miasm-fprnd',
                'nlr-b50k-symmi1miasm-largest-nonlinear-fprnd',
                'nlr-b50k-symmi1miasm-smallest-nonlinear-fprnd',
                'nlr-b50k-symmb60miasm-largest-nonlinear-fprnd',
                'nlr-b50k-symmb60miasm-smallest-nonlinear-fprnd',
            ],
        ]
    ],
    flexible_attributes=flexible_attributes,
    single_aggregation=False,
    name='symm',
)

exp.add_domains_with_different_values_of_attribute_report(
    attributes=['coverage'],
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    name='cggl-base-symm',
)

# result of the previous report
# mystery: both solve 13, but on different tasks
domains_different_expansions = [
    'airport',
    'elevators-opt08-strips',
    'elevators-opt11-strips',
    'gripper',
    'logistics98',
    'miconic',
    #'mystery',
    'parking-opt11-strips',
    'parking-opt14-strips',
    'pipesworld-notankage',
    'pipesworld-tankage',
    'satellite',
    'sokoban-opt08-strips',
    'tetris-opt14-strips',
    'tidybot-opt11-strips',
    'trucks-strips',
    'visitall-opt11-strips',
    'visitall-opt14-strips',
    'woodworking-opt08-strips',
    'woodworking-opt11-strips',
    'zenotravel',
]

exp.add_absolute_report(
    attributes=['coverage'],
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    filter_domain=domains_different_expansions,
    name='cggl-base-symm-different-domains-coverage',
    format='tex',
    publish=False,
)

exp.add_scatter_plot_report(
    attributes=['expansions_until_last_jump'],
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    name='cggl-base-symm-expansions',
    format='tex',
)

def get_domain(run1, run2):
    return run1['domain']

exp.add_scatter_plot_report(
    attributes=['expansions_until_last_jump'],
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    filter_domain=domains_different_expansions,
    name='cggl-base-symm-different-domains-expansions',
    get_category=get_domain,
    format='tex',
)

exp.add_scatter_plot_report(
    attributes=['total_time'],
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    filter_domain=domains_different_expansions,
    name='cggl-base-symm-total-time',
    format='tex',
)

exp.add_absolute_report(
    #attributes=['coverage'],
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    #filter_domain=domains_different_expansions,
    name='cggl-base-symm',
    format='tex',
    publish=False,
)

exp.add_tasks_with_specific_value_of_attribute_report(
    specific_value=True,
    attributes=['coverage'],
    filter_algorithm=[
        'nlr-b50k-symmb60cggl-largest-nonlinear-fprnd',
    ],
    name='cggl-symm-solved-tasks',
)

tasks_solved_by_cggl_symm = [
'airport:p01-airport1-p1.pddl',
'airport:p02-airport1-p1.pddl',
'airport:p03-airport1-p2.pddl',
'airport:p04-airport2-p1.pddl',
'airport:p05-airport2-p1.pddl',
'airport:p06-airport2-p2.pddl',
'airport:p07-airport2-p2.pddl',
'airport:p08-airport2-p3.pddl',
'airport:p09-airport2-p4.pddl',
'airport:p10-airport3-p1.pddl',
'airport:p11-airport3-p1.pddl',
'airport:p12-airport3-p2.pddl',
'airport:p13-airport3-p2.pddl',
'airport:p14-airport3-p3.pddl',
'airport:p15-airport3-p3.pddl',
'barman-opt11-strips:pfile01-001.pddl',
'barman-opt11-strips:pfile01-002.pddl',
'barman-opt11-strips:pfile01-003.pddl',
'barman-opt11-strips:pfile01-004.pddl',
'blocks:probBLOCKS-10-0.pddl',
'blocks:probBLOCKS-10-1.pddl',
'blocks:probBLOCKS-10-2.pddl',
'blocks:probBLOCKS-11-0.pddl',
'blocks:probBLOCKS-11-2.pddl',
'blocks:probBLOCKS-12-1.pddl',
'blocks:probBLOCKS-4-0.pddl',
'blocks:probBLOCKS-4-1.pddl',
'blocks:probBLOCKS-4-2.pddl',
'blocks:probBLOCKS-5-0.pddl',
'blocks:probBLOCKS-5-1.pddl',
'blocks:probBLOCKS-5-2.pddl',
'blocks:probBLOCKS-6-0.pddl',
'blocks:probBLOCKS-6-1.pddl',
'blocks:probBLOCKS-6-2.pddl',
'blocks:probBLOCKS-7-0.pddl',
'blocks:probBLOCKS-7-1.pddl',
'blocks:probBLOCKS-7-2.pddl',
'blocks:probBLOCKS-8-0.pddl',
'blocks:probBLOCKS-8-1.pddl',
'blocks:probBLOCKS-8-2.pddl',
'blocks:probBLOCKS-9-0.pddl',
'blocks:probBLOCKS-9-1.pddl',
'blocks:probBLOCKS-9-2.pddl',
'depot:p01.pddl',
'depot:p02.pddl',
'depot:p03.pddl',
'depot:p04.pddl',
'depot:p07.pddl',
'depot:p10.pddl',
'depot:p13.pddl',
'driverlog:p01.pddl',
'driverlog:p02.pddl',
'driverlog:p03.pddl',
'driverlog:p04.pddl',
'driverlog:p05.pddl',
'driverlog:p06.pddl',
'driverlog:p07.pddl',
'driverlog:p08.pddl',
'driverlog:p09.pddl',
'driverlog:p10.pddl',
'driverlog:p11.pddl',
'driverlog:p13.pddl',
'elevators-opt08-strips:p01.pddl',
'elevators-opt08-strips:p02.pddl',
'elevators-opt08-strips:p03.pddl',
'elevators-opt08-strips:p04.pddl',
'elevators-opt08-strips:p05.pddl',
'elevators-opt08-strips:p06.pddl',
'elevators-opt08-strips:p11.pddl',
'elevators-opt08-strips:p12.pddl',
'elevators-opt08-strips:p13.pddl',
'elevators-opt08-strips:p14.pddl',
'elevators-opt08-strips:p15.pddl',
'elevators-opt08-strips:p21.pddl',
'elevators-opt08-strips:p22.pddl',
'elevators-opt11-strips:p01.pddl',
'elevators-opt11-strips:p02.pddl',
'elevators-opt11-strips:p03.pddl',
'elevators-opt11-strips:p04.pddl',
'elevators-opt11-strips:p05.pddl',
'elevators-opt11-strips:p06.pddl',
'elevators-opt11-strips:p08.pddl',
'elevators-opt11-strips:p09.pddl',
'elevators-opt11-strips:p10.pddl',
'elevators-opt11-strips:p12.pddl',
'elevators-opt11-strips:p16.pddl',
'floortile-opt11-strips:opt-p01-001.pddl',
'floortile-opt11-strips:opt-p01-002.pddl',
'freecell:p01.pddl',
'freecell:p02.pddl',
'freecell:p03.pddl',
'freecell:p04.pddl',
'freecell:p05.pddl',
'freecell:probfreecell-2-1.pddl',
'freecell:probfreecell-2-2.pddl',
'freecell:probfreecell-2-3.pddl',
'freecell:probfreecell-2-4.pddl',
'freecell:probfreecell-2-5.pddl',
'freecell:probfreecell-3-1.pddl',
'freecell:probfreecell-3-2.pddl',
'freecell:probfreecell-3-3.pddl',
'freecell:probfreecell-3-4.pddl',
'freecell:probfreecell-3-5.pddl',
'freecell:probfreecell-4-2.pddl',
'freecell:probfreecell-4-3.pddl',
'freecell:probfreecell-4-4.pddl',
'freecell:probfreecell-4-5.pddl',
'ged-opt14-strips:d-1-2.pddl',
'ged-opt14-strips:d-1-3.pddl',
'ged-opt14-strips:d-1-4.pddl',
'ged-opt14-strips:d-1-8.pddl',
'ged-opt14-strips:d-2-1.pddl',
'ged-opt14-strips:d-2-3.pddl',
'ged-opt14-strips:d-2-4.pddl',
'ged-opt14-strips:d-3-1.pddl',
'ged-opt14-strips:d-3-2.pddl',
'ged-opt14-strips:d-3-4.pddl',
'ged-opt14-strips:d-4-2.pddl',
'ged-opt14-strips:d-4-3.pddl',
'ged-opt14-strips:d-7-5.pddl',
'ged-opt14-strips:d-7-6.pddl',
'ged-opt14-strips:d-8-1.pddl',
'grid:prob01.pddl',
'grid:prob02.pddl',
'gripper:prob01.pddl',
'gripper:prob02.pddl',
'gripper:prob03.pddl',
'gripper:prob04.pddl',
'gripper:prob05.pddl',
'gripper:prob06.pddl',
'gripper:prob07.pddl',
'gripper:prob08.pddl',
'gripper:prob09.pddl',
'gripper:prob10.pddl',
'gripper:prob11.pddl',
'gripper:prob12.pddl',
'gripper:prob13.pddl',
'gripper:prob14.pddl',
'gripper:prob15.pddl',
'gripper:prob16.pddl',
'gripper:prob17.pddl',
'gripper:prob18.pddl',
'hiking-opt14-strips:ptesting-1-2-3.pddl',
'hiking-opt14-strips:ptesting-1-2-4.pddl',
'hiking-opt14-strips:ptesting-1-2-5.pddl',
'hiking-opt14-strips:ptesting-1-2-7.pddl',
'hiking-opt14-strips:ptesting-1-2-8.pddl',
'hiking-opt14-strips:ptesting-2-2-3.pddl',
'hiking-opt14-strips:ptesting-2-2-4.pddl',
'hiking-opt14-strips:ptesting-2-2-5.pddl',
'hiking-opt14-strips:ptesting-2-3-4.pddl',
'hiking-opt14-strips:ptesting-2-3-5.pddl',
'hiking-opt14-strips:ptesting-2-4-3.pddl',
'hiking-opt14-strips:ptesting-2-4-4.pddl',
'logistics00:probLOGISTICS-10-0.pddl',
'logistics00:probLOGISTICS-10-1.pddl',
'logistics00:probLOGISTICS-11-0.pddl',
'logistics00:probLOGISTICS-12-0.pddl',
'logistics00:probLOGISTICS-4-0.pddl',
'logistics00:probLOGISTICS-4-1.pddl',
'logistics00:probLOGISTICS-4-2.pddl',
'logistics00:probLOGISTICS-5-0.pddl',
'logistics00:probLOGISTICS-5-1.pddl',
'logistics00:probLOGISTICS-5-2.pddl',
'logistics00:probLOGISTICS-6-0.pddl',
'logistics00:probLOGISTICS-6-1.pddl',
'logistics00:probLOGISTICS-6-2.pddl',
'logistics00:probLOGISTICS-6-9.pddl',
'logistics00:probLOGISTICS-7-0.pddl',
'logistics00:probLOGISTICS-7-1.pddl',
'logistics00:probLOGISTICS-8-0.pddl',
'logistics00:probLOGISTICS-8-1.pddl',
'logistics00:probLOGISTICS-9-0.pddl',
'logistics00:probLOGISTICS-9-1.pddl',
'logistics98:prob01.pddl',
'logistics98:prob05.pddl',
'logistics98:prob31.pddl',
'logistics98:prob32.pddl',
'miconic:s1-0.pddl',
'miconic:s1-1.pddl',
'miconic:s1-2.pddl',
'miconic:s1-3.pddl',
'miconic:s1-4.pddl',
'miconic:s10-0.pddl',
'miconic:s10-1.pddl',
'miconic:s10-2.pddl',
'miconic:s10-3.pddl',
'miconic:s10-4.pddl',
'miconic:s11-0.pddl',
'miconic:s11-1.pddl',
'miconic:s11-2.pddl',
'miconic:s11-3.pddl',
'miconic:s11-4.pddl',
'miconic:s12-0.pddl',
'miconic:s12-1.pddl',
'miconic:s12-2.pddl',
'miconic:s12-3.pddl',
'miconic:s12-4.pddl',
'miconic:s13-0.pddl',
'miconic:s13-1.pddl',
'miconic:s13-2.pddl',
'miconic:s13-3.pddl',
'miconic:s13-4.pddl',
'miconic:s14-0.pddl',
'miconic:s14-1.pddl',
'miconic:s14-2.pddl',
'miconic:s14-3.pddl',
'miconic:s14-4.pddl',
'miconic:s15-1.pddl',
'miconic:s15-3.pddl',
'miconic:s15-4.pddl',
'miconic:s16-1.pddl',
'miconic:s16-3.pddl',
'miconic:s16-4.pddl',
'miconic:s17-4.pddl',
'miconic:s2-0.pddl',
'miconic:s2-1.pddl',
'miconic:s2-2.pddl',
'miconic:s2-3.pddl',
'miconic:s2-4.pddl',
'miconic:s3-0.pddl',
'miconic:s3-1.pddl',
'miconic:s3-2.pddl',
'miconic:s3-3.pddl',
'miconic:s3-4.pddl',
'miconic:s4-0.pddl',
'miconic:s4-1.pddl',
'miconic:s4-2.pddl',
'miconic:s4-3.pddl',
'miconic:s4-4.pddl',
'miconic:s5-0.pddl',
'miconic:s5-1.pddl',
'miconic:s5-2.pddl',
'miconic:s5-3.pddl',
'miconic:s5-4.pddl',
'miconic:s6-0.pddl',
'miconic:s6-1.pddl',
'miconic:s6-2.pddl',
'miconic:s6-3.pddl',
'miconic:s6-4.pddl',
'miconic:s7-0.pddl',
'miconic:s7-1.pddl',
'miconic:s7-2.pddl',
'miconic:s7-3.pddl',
'miconic:s7-4.pddl',
'miconic:s8-0.pddl',
'miconic:s8-1.pddl',
'miconic:s8-2.pddl',
'miconic:s8-3.pddl',
'miconic:s8-4.pddl',
'miconic:s9-0.pddl',
'miconic:s9-1.pddl',
'miconic:s9-2.pddl',
'miconic:s9-3.pddl',
'miconic:s9-4.pddl',
'movie:prob01.pddl',
'movie:prob02.pddl',
'movie:prob03.pddl',
'movie:prob04.pddl',
'movie:prob05.pddl',
'movie:prob06.pddl',
'movie:prob07.pddl',
'movie:prob08.pddl',
'movie:prob09.pddl',
'movie:prob10.pddl',
'movie:prob11.pddl',
'movie:prob12.pddl',
'movie:prob13.pddl',
'movie:prob14.pddl',
'movie:prob15.pddl',
'movie:prob16.pddl',
'movie:prob17.pddl',
'movie:prob18.pddl',
'movie:prob19.pddl',
'movie:prob20.pddl',
'movie:prob21.pddl',
'movie:prob22.pddl',
'movie:prob23.pddl',
'movie:prob24.pddl',
'movie:prob25.pddl',
'movie:prob26.pddl',
'movie:prob27.pddl',
'movie:prob28.pddl',
'movie:prob29.pddl',
'movie:prob30.pddl',
'mprime:prob01.pddl',
'mprime:prob02.pddl',
'mprime:prob03.pddl',
'mprime:prob04.pddl',
'mprime:prob05.pddl',
'mprime:prob07.pddl',
'mprime:prob08.pddl',
'mprime:prob09.pddl',
'mprime:prob11.pddl',
'mprime:prob12.pddl',
'mprime:prob16.pddl',
'mprime:prob25.pddl',
'mprime:prob26.pddl',
'mprime:prob27.pddl',
'mprime:prob28.pddl',
'mprime:prob29.pddl',
'mprime:prob31.pddl',
'mprime:prob32.pddl',
'mprime:prob34.pddl',
'mprime:prob35.pddl',
'mystery:prob01.pddl',
'mystery:prob02.pddl',
'mystery:prob03.pddl',
'mystery:prob06.pddl',
'mystery:prob09.pddl',
'mystery:prob11.pddl',
'mystery:prob20.pddl',
'mystery:prob25.pddl',
'mystery:prob26.pddl',
'mystery:prob27.pddl',
'mystery:prob28.pddl',
'mystery:prob29.pddl',
'mystery:prob30.pddl',
'nomystery-opt11-strips:p01.pddl',
'nomystery-opt11-strips:p02.pddl',
'nomystery-opt11-strips:p03.pddl',
'nomystery-opt11-strips:p04.pddl',
'nomystery-opt11-strips:p05.pddl',
'nomystery-opt11-strips:p06.pddl',
'nomystery-opt11-strips:p07.pddl',
'nomystery-opt11-strips:p08.pddl',
'nomystery-opt11-strips:p10.pddl',
'nomystery-opt11-strips:p11.pddl',
'nomystery-opt11-strips:p12.pddl',
'nomystery-opt11-strips:p13.pddl',
'nomystery-opt11-strips:p14.pddl',
'nomystery-opt11-strips:p15.pddl',
'nomystery-opt11-strips:p16.pddl',
'nomystery-opt11-strips:p17.pddl',
'nomystery-opt11-strips:p18.pddl',
'nomystery-opt11-strips:p20.pddl',
'openstacks-opt08-strips:p01.pddl',
'openstacks-opt08-strips:p02.pddl',
'openstacks-opt08-strips:p03.pddl',
'openstacks-opt08-strips:p04.pddl',
'openstacks-opt08-strips:p05.pddl',
'openstacks-opt08-strips:p06.pddl',
'openstacks-opt08-strips:p07.pddl',
'openstacks-opt08-strips:p08.pddl',
'openstacks-opt08-strips:p09.pddl',
'openstacks-opt08-strips:p10.pddl',
'openstacks-opt08-strips:p11.pddl',
'openstacks-opt08-strips:p12.pddl',
'openstacks-opt08-strips:p13.pddl',
'openstacks-opt08-strips:p14.pddl',
'openstacks-opt08-strips:p15.pddl',
'openstacks-opt08-strips:p16.pddl',
'openstacks-opt08-strips:p17.pddl',
'openstacks-opt08-strips:p18.pddl',
'openstacks-opt08-strips:p19.pddl',
'openstacks-opt08-strips:p22.pddl',
'openstacks-opt11-strips:p01.pddl',
'openstacks-opt11-strips:p02.pddl',
'openstacks-opt11-strips:p03.pddl',
'openstacks-opt11-strips:p04.pddl',
'openstacks-opt11-strips:p05.pddl',
'openstacks-opt11-strips:p06.pddl',
'openstacks-opt11-strips:p07.pddl',
'openstacks-opt11-strips:p08.pddl',
'openstacks-opt11-strips:p09.pddl',
'openstacks-opt11-strips:p10.pddl',
'openstacks-opt11-strips:p11.pddl',
'openstacks-opt11-strips:p12.pddl',
'openstacks-opt11-strips:p13.pddl',
'openstacks-opt11-strips:p14.pddl',
'openstacks-opt11-strips:p15.pddl',
'openstacks-opt14-strips:p20_1.pddl',
'openstacks-opt14-strips:p20_2.pddl',
'openstacks-opt14-strips:p20_3.pddl',
'openstacks-strips:p01.pddl',
'openstacks-strips:p02.pddl',
'openstacks-strips:p03.pddl',
'openstacks-strips:p04.pddl',
'openstacks-strips:p05.pddl',
'openstacks-strips:p06.pddl',
'openstacks-strips:p07.pddl',
'parcprinter-08-strips:p01.pddl',
'parcprinter-08-strips:p02.pddl',
'parcprinter-08-strips:p03.pddl',
'parcprinter-08-strips:p04.pddl',
'parcprinter-08-strips:p05.pddl',
'parcprinter-08-strips:p06.pddl',
'parcprinter-08-strips:p11.pddl',
'parcprinter-08-strips:p12.pddl',
'parcprinter-08-strips:p13.pddl',
'parcprinter-08-strips:p14.pddl',
'parcprinter-08-strips:p21.pddl',
'parcprinter-08-strips:p22.pddl',
'parcprinter-08-strips:p23.pddl',
'parcprinter-08-strips:p24.pddl',
'parcprinter-08-strips:p25.pddl',
'parcprinter-08-strips:p26.pddl',
'parcprinter-opt11-strips:p01.pddl',
'parcprinter-opt11-strips:p02.pddl',
'parcprinter-opt11-strips:p03.pddl',
'parcprinter-opt11-strips:p04.pddl',
'parcprinter-opt11-strips:p05.pddl',
'parcprinter-opt11-strips:p06.pddl',
'parcprinter-opt11-strips:p07.pddl',
'parcprinter-opt11-strips:p08.pddl',
'parcprinter-opt11-strips:p09.pddl',
'parcprinter-opt11-strips:p10.pddl',
'parcprinter-opt11-strips:p11.pddl',
'parcprinter-opt11-strips:p12.pddl',
'parking-opt11-strips:pfile03-011.pddl',
'parking-opt11-strips:pfile03-012.pddl',
'parking-opt11-strips:pfile04-013.pddl',
'parking-opt11-strips:pfile04-015.pddl',
'parking-opt14-strips:p_12_7-01.pddl',
'parking-opt14-strips:p_12_7-02.pddl',
'parking-opt14-strips:p_12_7-03.pddl',
'parking-opt14-strips:p_12_7-04.pddl',
'parking-opt14-strips:p_14_8-02.pddl',
'parking-opt14-strips:p_14_8-03.pddl',
'pathways-noneg:p01.pddl',
'pathways-noneg:p02.pddl',
'pathways-noneg:p03.pddl',
'pathways-noneg:p04.pddl',
'pegsol-08-strips:p01.pddl',
'pegsol-08-strips:p02.pddl',
'pegsol-08-strips:p03.pddl',
'pegsol-08-strips:p04.pddl',
'pegsol-08-strips:p05.pddl',
'pegsol-08-strips:p06.pddl',
'pegsol-08-strips:p07.pddl',
'pegsol-08-strips:p08.pddl',
'pegsol-08-strips:p09.pddl',
'pegsol-08-strips:p10.pddl',
'pegsol-08-strips:p11.pddl',
'pegsol-08-strips:p12.pddl',
'pegsol-08-strips:p13.pddl',
'pegsol-08-strips:p14.pddl',
'pegsol-08-strips:p15.pddl',
'pegsol-08-strips:p16.pddl',
'pegsol-08-strips:p17.pddl',
'pegsol-08-strips:p18.pddl',
'pegsol-08-strips:p19.pddl',
'pegsol-08-strips:p20.pddl',
'pegsol-08-strips:p21.pddl',
'pegsol-08-strips:p22.pddl',
'pegsol-08-strips:p23.pddl',
'pegsol-08-strips:p24.pddl',
'pegsol-08-strips:p25.pddl',
'pegsol-08-strips:p26.pddl',
'pegsol-08-strips:p27.pddl',
'pegsol-opt11-strips:p01.pddl',
'pegsol-opt11-strips:p02.pddl',
'pegsol-opt11-strips:p03.pddl',
'pegsol-opt11-strips:p04.pddl',
'pegsol-opt11-strips:p05.pddl',
'pegsol-opt11-strips:p06.pddl',
'pegsol-opt11-strips:p07.pddl',
'pegsol-opt11-strips:p08.pddl',
'pegsol-opt11-strips:p09.pddl',
'pegsol-opt11-strips:p10.pddl',
'pegsol-opt11-strips:p11.pddl',
'pegsol-opt11-strips:p12.pddl',
'pegsol-opt11-strips:p13.pddl',
'pegsol-opt11-strips:p14.pddl',
'pegsol-opt11-strips:p15.pddl',
'pegsol-opt11-strips:p16.pddl',
'pegsol-opt11-strips:p17.pddl',
'pipesworld-notankage:p01-net1-b6-g2.pddl',
'pipesworld-notankage:p02-net1-b6-g4.pddl',
'pipesworld-notankage:p03-net1-b8-g3.pddl',
'pipesworld-notankage:p04-net1-b8-g5.pddl',
'pipesworld-notankage:p05-net1-b10-g4.pddl',
'pipesworld-notankage:p06-net1-b10-g6.pddl',
'pipesworld-notankage:p07-net1-b12-g5.pddl',
'pipesworld-notankage:p08-net1-b12-g7.pddl',
'pipesworld-notankage:p09-net1-b14-g6.pddl',
'pipesworld-notankage:p10-net1-b14-g8.pddl',
'pipesworld-notankage:p11-net2-b10-g2.pddl',
'pipesworld-notankage:p12-net2-b10-g4.pddl',
'pipesworld-notankage:p13-net2-b12-g3.pddl',
'pipesworld-notankage:p15-net2-b14-g4.pddl',
'pipesworld-notankage:p21-net3-b12-g2.pddl',
'pipesworld-tankage:p01-net1-b6-g2-t50.pddl',
'pipesworld-tankage:p02-net1-b6-g4-t50.pddl',
'pipesworld-tankage:p03-net1-b8-g3-t80.pddl',
'pipesworld-tankage:p04-net1-b8-g5-t80.pddl',
'pipesworld-tankage:p05-net1-b10-g4-t50.pddl',
'pipesworld-tankage:p06-net1-b10-g6-t50.pddl',
'pipesworld-tankage:p07-net1-b12-g5-t80.pddl',
'pipesworld-tankage:p08-net1-b12-g7-t80.pddl',
'pipesworld-tankage:p11-net2-b10-g2-t30.pddl',
'pipesworld-tankage:p13-net2-b12-g3-t70.pddl',
'pipesworld-tankage:p15-net2-b14-g4-t30.pddl',
'pipesworld-tankage:p17-net2-b16-g5-t20.pddl',
'pipesworld-tankage:p21-net3-b12-g2-t60.pddl',
'pipesworld-tankage:p31-net4-b14-g3-t20.pddl',
'psr-small:p01-s2-n1-l2-f50.pddl',
'psr-small:p02-s5-n1-l3-f30.pddl',
'psr-small:p03-s7-n1-l3-f70.pddl',
'psr-small:p04-s8-n1-l4-f10.pddl',
'psr-small:p05-s9-n1-l4-f30.pddl',
'psr-small:p06-s10-n1-l4-f50.pddl',
'psr-small:p07-s11-n1-l4-f70.pddl',
'psr-small:p08-s12-n1-l5-f10.pddl',
'psr-small:p09-s13-n1-l5-f30.pddl',
'psr-small:p10-s17-n2-l2-f30.pddl',
'psr-small:p11-s18-n2-l2-f50.pddl',
'psr-small:p12-s21-n2-l3-f30.pddl',
'psr-small:p13-s22-n2-l3-f50.pddl',
'psr-small:p14-s23-n2-l3-f70.pddl',
'psr-small:p15-s24-n2-l4-f10.pddl',
'psr-small:p16-s29-n2-l5-f30.pddl',
'psr-small:p17-s30-n2-l5-f50.pddl',
'psr-small:p18-s31-n2-l5-f70.pddl',
'psr-small:p19-s33-n3-l2-f30.pddl',
'psr-small:p20-s34-n3-l2-f50.pddl',
'psr-small:p21-s35-n3-l2-f70.pddl',
'psr-small:p22-s37-n3-l3-f30.pddl',
'psr-small:p23-s38-n3-l3-f50.pddl',
'psr-small:p24-s39-n3-l3-f70.pddl',
'psr-small:p25-s40-n3-l4-f10.pddl',
'psr-small:p26-s41-n3-l4-f30.pddl',
'psr-small:p27-s42-n3-l4-f50.pddl',
'psr-small:p28-s43-n3-l4-f70.pddl',
'psr-small:p29-s45-n3-l5-f30.pddl',
'psr-small:p30-s46-n3-l5-f50.pddl',
'psr-small:p31-s49-n4-l2-f30.pddl',
'psr-small:p32-s50-n4-l2-f50.pddl',
'psr-small:p33-s51-n4-l2-f70.pddl',
'psr-small:p34-s55-n4-l3-f70.pddl',
'psr-small:p35-s57-n5-l2-f30.pddl',
'psr-small:p36-s65-n6-l2-f30.pddl',
'psr-small:p37-s67-n6-l2-f70.pddl',
'psr-small:p38-s78-n3-l3-f50.pddl',
'psr-small:p39-s79-n3-l3-f70.pddl',
'psr-small:p40-s80-n3-l4-f10.pddl',
'psr-small:p41-s81-n3-l4-f30.pddl',
'psr-small:p42-s82-n3-l4-f50.pddl',
'psr-small:p43-s83-n3-l4-f70.pddl',
'psr-small:p44-s89-n4-l2-f30.pddl',
'psr-small:p45-s94-n4-l3-f50.pddl',
'psr-small:p46-s97-n5-l2-f30.pddl',
'psr-small:p47-s98-n5-l2-f50.pddl',
'psr-small:p48-s101-n5-l3-f30.pddl',
'psr-small:p49-s105-n6-l2-f30.pddl',
'psr-small:p50-s107-n6-l2-f70.pddl',
'rovers:p01.pddl',
'rovers:p02.pddl',
'rovers:p03.pddl',
'rovers:p04.pddl',
'rovers:p05.pddl',
'rovers:p06.pddl',
'rovers:p07.pddl',
'rovers:p12.pddl',
'satellite:p01-pfile1.pddl',
'satellite:p02-pfile2.pddl',
'satellite:p03-pfile3.pddl',
'satellite:p04-pfile4.pddl',
'satellite:p05-pfile5.pddl',
'satellite:p06-pfile6.pddl',
'scanalyzer-08-strips:p01.pddl',
'scanalyzer-08-strips:p02.pddl',
'scanalyzer-08-strips:p03.pddl',
'scanalyzer-08-strips:p04.pddl',
'scanalyzer-08-strips:p05.pddl',
'scanalyzer-08-strips:p06.pddl',
'scanalyzer-08-strips:p22.pddl',
'scanalyzer-08-strips:p23.pddl',
'scanalyzer-08-strips:p24.pddl',
'scanalyzer-08-strips:p25.pddl',
'scanalyzer-08-strips:p26.pddl',
'scanalyzer-08-strips:p27.pddl',
'scanalyzer-opt11-strips:p01.pddl',
'scanalyzer-opt11-strips:p02.pddl',
'scanalyzer-opt11-strips:p03.pddl',
'scanalyzer-opt11-strips:p04.pddl',
'scanalyzer-opt11-strips:p06.pddl',
'scanalyzer-opt11-strips:p11.pddl',
'scanalyzer-opt11-strips:p12.pddl',
'scanalyzer-opt11-strips:p13.pddl',
'scanalyzer-opt11-strips:p14.pddl',
'sokoban-opt08-strips:p01.pddl',
'sokoban-opt08-strips:p02.pddl',
'sokoban-opt08-strips:p03.pddl',
'sokoban-opt08-strips:p04.pddl',
'sokoban-opt08-strips:p05.pddl',
'sokoban-opt08-strips:p06.pddl',
'sokoban-opt08-strips:p07.pddl',
'sokoban-opt08-strips:p08.pddl',
'sokoban-opt08-strips:p09.pddl',
'sokoban-opt08-strips:p10.pddl',
'sokoban-opt08-strips:p11.pddl',
'sokoban-opt08-strips:p12.pddl',
'sokoban-opt08-strips:p13.pddl',
'sokoban-opt08-strips:p14.pddl',
'sokoban-opt08-strips:p15.pddl',
'sokoban-opt08-strips:p16.pddl',
'sokoban-opt08-strips:p17.pddl',
'sokoban-opt08-strips:p18.pddl',
'sokoban-opt08-strips:p19.pddl',
'sokoban-opt08-strips:p20.pddl',
'sokoban-opt08-strips:p21.pddl',
'sokoban-opt08-strips:p22.pddl',
'sokoban-opt08-strips:p23.pddl',
'sokoban-opt08-strips:p24.pddl',
'sokoban-opt08-strips:p25.pddl',
'sokoban-opt08-strips:p26.pddl',
'sokoban-opt08-strips:p27.pddl',
'sokoban-opt08-strips:p28.pddl',
'sokoban-opt08-strips:p29.pddl',
'sokoban-opt08-strips:p30.pddl',
'sokoban-opt11-strips:p01.pddl',
'sokoban-opt11-strips:p02.pddl',
'sokoban-opt11-strips:p03.pddl',
'sokoban-opt11-strips:p04.pddl',
'sokoban-opt11-strips:p05.pddl',
'sokoban-opt11-strips:p06.pddl',
'sokoban-opt11-strips:p07.pddl',
'sokoban-opt11-strips:p08.pddl',
'sokoban-opt11-strips:p09.pddl',
'sokoban-opt11-strips:p10.pddl',
'sokoban-opt11-strips:p11.pddl',
'sokoban-opt11-strips:p12.pddl',
'sokoban-opt11-strips:p13.pddl',
'sokoban-opt11-strips:p14.pddl',
'sokoban-opt11-strips:p15.pddl',
'sokoban-opt11-strips:p16.pddl',
'sokoban-opt11-strips:p17.pddl',
'sokoban-opt11-strips:p18.pddl',
'sokoban-opt11-strips:p19.pddl',
'sokoban-opt11-strips:p20.pddl',
'storage:p01.pddl',
'storage:p02.pddl',
'storage:p03.pddl',
'storage:p04.pddl',
'storage:p05.pddl',
'storage:p06.pddl',
'storage:p07.pddl',
'storage:p08.pddl',
'storage:p09.pddl',
'storage:p10.pddl',
'storage:p11.pddl',
'storage:p12.pddl',
'storage:p13.pddl',
'storage:p14.pddl',
'storage:p15.pddl',
'tetris-opt14-strips:p02-4.pddl',
'tpp:p01.pddl',
'tpp:p02.pddl',
'tpp:p03.pddl',
'tpp:p04.pddl',
'tpp:p05.pddl',
'tpp:p06.pddl',
'transport-opt08-strips:p01.pddl',
'transport-opt08-strips:p02.pddl',
'transport-opt08-strips:p03.pddl',
'transport-opt08-strips:p04.pddl',
'transport-opt08-strips:p11.pddl',
'transport-opt08-strips:p12.pddl',
'transport-opt08-strips:p13.pddl',
'transport-opt08-strips:p21.pddl',
'transport-opt08-strips:p22.pddl',
'transport-opt08-strips:p23.pddl',
'transport-opt08-strips:p24.pddl',
'transport-opt11-strips:p01.pddl',
'transport-opt11-strips:p02.pddl',
'transport-opt11-strips:p03.pddl',
'transport-opt11-strips:p04.pddl',
'transport-opt11-strips:p05.pddl',
'transport-opt11-strips:p06.pddl',
'transport-opt14-strips:p01.pddl',
'transport-opt14-strips:p02.pddl',
'transport-opt14-strips:p03.pddl',
'transport-opt14-strips:p07.pddl',
'transport-opt14-strips:p13.pddl',
'transport-opt14-strips:p14.pddl',
'trucks-strips:p01.pddl',
'trucks-strips:p02.pddl',
'trucks-strips:p03.pddl',
'trucks-strips:p04.pddl',
'trucks-strips:p05.pddl',
'trucks-strips:p06.pddl',
'trucks-strips:p07.pddl',
'trucks-strips:p08.pddl',
'visitall-opt11-strips:problem02-full.pddl',
'visitall-opt11-strips:problem02-half.pddl',
'visitall-opt11-strips:problem03-full.pddl',
'visitall-opt11-strips:problem03-half.pddl',
'visitall-opt11-strips:problem04-full.pddl',
'visitall-opt11-strips:problem04-half.pddl',
'visitall-opt11-strips:problem05-full.pddl',
'visitall-opt11-strips:problem05-half.pddl',
'visitall-opt11-strips:problem06-full.pddl',
'visitall-opt11-strips:problem06-half.pddl',
'visitall-opt14-strips:p-05-5.pddl',
'visitall-opt14-strips:p-05-6.pddl',
'visitall-opt14-strips:p-05-7.pddl',
'visitall-opt14-strips:p-1-13.pddl',
'visitall-opt14-strips:p-1-5.pddl',
'visitall-opt14-strips:p-1-6.pddl',
'woodworking-opt08-strips:p01.pddl',
'woodworking-opt08-strips:p02.pddl',
'woodworking-opt08-strips:p03.pddl',
'woodworking-opt08-strips:p11.pddl',
'woodworking-opt08-strips:p12.pddl',
'woodworking-opt08-strips:p13.pddl',
'woodworking-opt08-strips:p14.pddl',
'woodworking-opt08-strips:p15.pddl',
'woodworking-opt08-strips:p21.pddl',
'woodworking-opt08-strips:p22.pddl',
'woodworking-opt08-strips:p23.pddl',
'woodworking-opt08-strips:p24.pddl',
'woodworking-opt11-strips:p01.pddl',
'woodworking-opt11-strips:p02.pddl',
'woodworking-opt11-strips:p03.pddl',
'woodworking-opt11-strips:p04.pddl',
'woodworking-opt11-strips:p05.pddl',
'woodworking-opt11-strips:p07.pddl',
'woodworking-opt11-strips:p10.pddl',
'zenotravel:p01.pddl',
'zenotravel:p02.pddl',
'zenotravel:p03.pddl',
'zenotravel:p04.pddl',
'zenotravel:p05.pddl',
'zenotravel:p06.pddl',
'zenotravel:p07.pddl',
'zenotravel:p08.pddl',
'zenotravel:p09.pddl',
'zenotravel:p11.pddl',
]

def filter_tasks_solved_by_cggl_symm(run):
    domain = run['domain']
    task = run['problem']
    if '{}:{}'.format(domain, task) in tasks_solved_by_cggl_symm:
        return True
    return False

exp.add_absolute_report(
    filter_algorithm=[
        'nlr-b50k-cggl-fprnd',
    ],
    filter=filter_tasks_solved_by_cggl_symm,
    name='cggl-base-on-tasks-solved-by-symm',
    format='tex',
    publish=False,
)

exp.run_steps()
