function [relL2onlineErrT, relL2onlineErrS, relL2offlineErrT, relL2offlineErrS, speedups, relL2onlineErr, simTimeROM, relL2onlineErr_POD, simTimeROM_POD] = simulateWildlandFire(initialCondition, plotsToBeCreated, modeOfWhatToDo, n_modes, nPODModes, numberOfRunsForAveragingRunTime, betaTraining, betaTest, nPODModesAreaI, nPODModesAreaII, windVelocity)
% simulateWildlandFire - simulates the wildland fire full-order model, 
% creates a reduced-order model, and simulates the reduced-order model
%
% inputs:
% - initialCondition: string determining which initial condition is used; 
% options are 'Gaussian' (initial condition used in section 5.3) and 
% 'separatedWaves' (initial condition used in section 5.2)
% - plotsToBeCreated: array of strings determining which plots should be
% carried out; possible options: 'temperatureSnapshots', 
% 'supplyMassFractionSnapshots', 'leftGoingWave', 'modes', 
% 'nonlinearitySnapshots', 'parameterSampling', 'temperatureZoom', 
% 'coefficientComparison', 'singleSnapshots'
% - modeOfWhatToDo: either a scalar or a vector with 2 entries; if the
% first entry is 1, then no shifted POD offline or online phase is carried
% out, but only the FOM is simulated; if it is 2, then additionally the
% shifted POD offline phase is carried out; if it is 3, then additionally
% the shifted POD online phase is carried out; the second entry (if there 
% is a second entry) determines whether also a POD offline (2) and/or 
% online phase (3) is carried out
% - n_modes: number of transformed modes per traveling wave and per
% physical variable; optional argument with default value 1
% - nPODModes: number of POD modes; optional argument; is only used if 
% modeOfWhatToDo is an array with 2 entries and the second entry is larger
% than 1
% - numberOfRunsForAveragingRunTime: integer specifying how often the ROM
% simulations have to be repeated to get an average run time; optional
% argument; default value is 1
% - betaTraining: values for the Arrhenius coefficient used in the offline 
% phase; optional parameter; default value: 558.49
% - betaTest: values for the Arrhenius coefficient used in the online 
% phase; optional parameter; no test runs are performed if this input is
% not provided
% - nPODModesAreaI: number of POD modes per physical variable used in area 
% (i); optional parameter; default value is 14
% - nPODModesAreaII: number of POD modes per physical variable used in area 
% (ii); optional parameter; default value is 2
% - windVelocity: advection velocity; optional parameter; default value is
% 0
%
% outputs:
% - relL2onlineErrT: vector containing the relative shifted POD online 
% errors for the temperature for each parameter value
% - relL2onlineErrS: vector containing the relative shifted POD online 
% errors for the supply mass fraction for each parameter value
% - relL2offlineErrT: vector containing the relative shifted POD offline 
% errors for the temperature for each parameter value
% - relL2offlineErrS: vector containing the relative shifted POD offline 
% errors for the supply mass fraction for each parameter value
% - speedups: vector containing the speedup of the shifted-POD-ROM for each 
% parameter value
% - relL2onlineErr: vector containing the total relative shifted POD online 
% errors for  each parameter value
% - simTimeROM: vector containing the run time of the shifted-POD-ROM for 
% each parameter value
% - relL2onlineErr_POD: vector containing the total relative POD online 
% errors for  each parameter value
% - simTimeROM_POD: vector containing the run time of the POD ROM for 
% each parameter value
%
% dependencies:
% - compute_truth_solution_struct_array (in LIB/SIMULATION)
% - create_discretization_struct (in LIB/HELPER)
% - create_fom_struct (in LIB/HELPER)
% - create_offline_struct (in LIB/HELPER)
% - create_rom_struct (in LIB/HELPER)
% - plotExtrapolatedVsProjectedCoefficients (in LIB/PLOT)
% - plotLeftGoingSeparatedWave (in LIB/PLOT)
% - POD_offline (in LIB/POD)
% - pod_online_struct_array (in LIB/POD)
% - PODROMPreProcessing (in LIB/HYPERREDUCTION)
% - shifted_pod_offline_struct_array (in LIB/MODE_DECOMPOSITION)
% - relative_L2_norm (in LIB/ERROR)
% - shifted_pod_online_struct_array (in LIB/SPOD_ONLINE)
% - sPODROMPreProcessing (in LIB/HYPERREDUCTION)
% - subspace_decomposition_by_simpleDomainDivision (in
% LIB/MODE_DECOMPOSITION)
% - subspaceDecompositionSPODswitch (in LIB/MODE_DECOMPOSITION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% add needed directories to the MATLAB path
addpath('LIB/ERROR')
addpath('LIB/HELPER')
addpath('LIB/HYPERREDUCTION')
addpath('LIB/INITIAL_CONDITION')
addpath('LIB/INTERPOLATION')
addpath('LIB/MODE_DECOMPOSITION')
addpath('LIB/PLOT')
addpath('LIB/POD')
addpath('LIB/SHIFT')
addpath('LIB/SIMULATION')
addpath('LIB/SPOD_ONLINE')

%--------------------------------------------------------------------------
%% Settings
%--------------------------------------------------------------------------

% Arrhenius coefficient for the offline phase
betaDefaultValue = 5.5849e02 ;
if(nargin<7)
    beta_values.training = betaDefaultValue ; 
else
    if(isempty(betaTraining))
        beta_values.training = betaDefaultValue ; 
    else
        beta_values.training = betaTraining ;
    end
end

% Arrhenius coefficient for the online phase
if(nargin>7)
    beta_values.testing = betaTest ;
end

% wind velocity
if(nargin<11)
	v = 0 ;
else
	v = windVelocity ;
end

% discretization settings
discretization_default.space.n_steps = 3000; % number of steps in space
discretization_default.time.n_steps = 600; % number of steps in time
discretization_default.order = 6; % order of the finite difference scheme (options are 2, 4, and 6)

% number of modes: 2 entries for two traveling waves
if(nargin>3)
    offline_default.n_modes = n_modes*[1 1] ;
else
    offline_default.n_modes = [1 1] ;
end

switch initialCondition
    case 'separatedWaves'
        beta_values.separatedWavesIC = 5.5849e02; % Arrhenius coefficient used for creating the initial condition
        % method used for mode decomposition
        offline_default.method = @subspace_decomposition_by_simpleDomainDivision ;
        % offset used for the path post processing (linear interpolation is
        % performed for all time indices after this one)
        offline_default.shiftPostProcessing.offset = 0;
    case 'Gaussian'
        % method used for mode decomposition
        offline_default.method = @subspaceDecompositionSPODswitch;
        % offset used for the path post processing (linear interpolation is
        % performed for all time indices after this one)
        offline_default.shiftPostProcessing.offset = 9;
    otherwise
        warning('Unknown initial condition')
end

% boolean useSwitchedROM determines whether the time interval is split or
% not
if strcmp(func2str(offline_default.method),'subspaceDecompositionSPODswitch')
    useSwitchedROM = true ;
else
    useSwitchedROM = false ;
end

if(useSwitchedROM)
    offline_default.switchTime = 100; % time between areas (i) and (ii)
    % number of POD modes in area (i); the first argument is for the 
    % temperature, the second for the supply mass fraction
    nPODModesBeforeSwitchDefault = 14 ;
    if(nargin>8)
        offline_default.nPODModesBeforeSwitch = nPODModesAreaI*[1,1] ;
    else
        offline_default.nPODModesBeforeSwitch = nPODModesBeforeSwitchDefault*[1,1] ;
    end
    % number of POD modes in area (ii)
    nPODModesAfterSwitchDefault = 2 ;
    if(nargin>9)
        offline_default.nPODModesAfterSwitch = nPODModesAreaII*[1,1] ;
    else
        offline_default.nPODModesAfterSwitch = nPODModesAfterSwitchDefault*[1,1] ;
    end
    % shifted POD modes after switching are computed only for some part
    % of the snapshot data. The first 'shiftedPODstart' percent of the
    % data is ignored for computing the modes.
    % NOTE: This has only an effect if the corresponding
    % nPODModesAfterSwitch is bigger zero.
    offline_default.shiftedPODstart = [0.35 0.2] ;

    % the coefficients in area (ii-a) are determined via extrapolation of
    % the coefficients determined in area (ii-b); the degree of the
    % corresponding extrapolation polynomial is given by the following
    % value
    offline_default.extrapolationPolOrder = 2;
end

% hyperreduction settings
if(modeOfWhatToDo(1)>2)
    if(useSwitchedROM)
        % number of DEIM modes used in area (i)
        hyper.nPODModesBeforeSwitch = 2*offline_default.nPODModesBeforeSwitch(2) ;
        % number of DEIM modes used in area (ii)
        hyper.nPODModesAfterSwitch = 2*offline_default.nPODModesAfterSwitch(1) ;
    end
    switch initialCondition
        case 'Gaussian'
            % number of transformed DEIM modes
            hyper.n_modes = round(2*offline_default.n_modes) ;
            % if samplingRate is x, then every x-th spatial point is used
            % for sampling the paths
            hyper.sample.samplingRate = 1 ;
            % samplingRange specifies in which interval the path should be
            % sampled (specified in terms of percentage of the space
            % interval)
            hyper.sample.samplingRange = [0 0.5] ; 
        case 'separatedWaves'
            % number of transformed DEIM modes
            hyper.n_modes = 2*offline_default.n_modes ;
            % if samplingRate is x, then every x-th spatial point is used
            % for sampling the paths
            hyper.sample.samplingRate = 20 ;
            % samplingRange specifies in which interval the path should be
            % sampled (specified in terms of percentage of the space
            % interval)
            hyper.sample.samplingRange = [0 0.3] ; 
        otherwise
            error('Unknown initial condition.')
    end
end

if(nargin<=5)
    numberOfRunsForAveragingRunTime = 1 ;
end

%--------------------------------------------------------------------------
%% Further settings (do not change these values)
%--------------------------------------------------------------------------

% physical parameters that cannot be changed with this implementation
alpha   = 1.8793e02 ; % coefficient in temperature equation
gammaS  = 1.6250e-01 ; % coefficient in supply mass fraction equation

n_vars = 2 ; % number of physical variables

%--------------------------------------------------------------------------
%% Set up structs
%--------------------------------------------------------------------------

if(strcmp(initialCondition,'separatedWaves'))
    % the initial condition separatedWaves is computed from the Gaussian
    % initial condition. To ensure the same initial condition for all
    % parameter samples, this is performed with
    % beta_values.separatedWavesIC
    initialCondition = 'Gaussian' ;
    fprintf('compute initial condition for separated waves case\n');
    fom = create_fom_struct(initialCondition, 'parameter.beta', beta_values.separatedWavesIC, 'parameter.alpha', alpha, 'parameter.gammaS', gammaS, 'parameter.v', v);
    discretization = create_discretization_struct(fom, discretization_default, initialCondition);
    snapshotMatrixTemp = compute_truth_solution_struct_array(fom, discretization); % simulate wildland fire with Gaussian initial condition
    % separated Waves initial condition is retrieved by the simulation with 
    % Gaussian initial condition by taking the solution after a third of 
    % the time interval:
    separatedWaveIndex = round(discretization_default.time.n_steps/3) ; 
     % separated waves initial condition for the temperature:
    T0 = snapshotMatrixTemp{1}(1:discretization_default.space.n_steps,separatedWaveIndex) ;
     % separated waves initial condition for the supply mass fraction:
    S0 = snapshotMatrixTemp{1}(discretization_default.space.n_steps+1:end,separatedWaveIndex) ;
    if ~exist('DATA', 'dir')
       mkdir('DATA')
    end
    save('DATA/ICSeparatedWaves.mat', 'T0', 'S0') % save the initial condition; it is loaded later
    initialCondition = 'separatedWaves' ;
    clear snapshotMatrixTemp T0 S0
end

% FOM parameters
fom = create_fom_struct(initialCondition, 'parameter.beta', beta_values.training, 'parameter.alpha', alpha, 'parameter.gammaS', gammaS, 'parameter.v', v);

% Discretization parameters
discretization = create_discretization_struct(fom, discretization_default, initialCondition);

% Offline phase parameters
offline = create_offline_struct(offline_default);

% ROM parameters
rom = create_rom_struct(fom, 'parameter.beta', beta_values.training, 'parameter.alpha', alpha, 'parameter.gammaS', gammaS, 'parameter.v', v);

nSteps = discretization(1).space.n_steps ; % number of spatial steps
nParams = length(beta_values.training) ; % number of parameter values for the offline phase

%-------------------------------------------------------------------------------
%% Simulation of the Full-Order Model
%-------------------------------------------------------------------------------
    
fprintf(['Generate snapshots of the full-order model with ', initialCondition, ' initial condition\n']);
fprintf('=============================================================\n');
[snapshotMatrix, computation_time_fom, FSnapshots, A] = compute_truth_solution_struct_array(fom, discretization);

%-------------------------------------------------------------------------------
%% Mode Decomposition (Offline Phase) via shifted POD
%-------------------------------------------------------------------------------

if(modeOfWhatToDo(1)>1)
    fprintf('\nConstruct shifted POD modes for the snapshots with %d shifted modes per frame and variable\n', offline_default.n_modes(1));
    fprintf('=============================================================\n');
    if(useSwitchedROM)
        fprintf('* use %d POD modes before switching for the temperature\n',offline.nPODModesBeforeSwitch(1)); 
        fprintf('* use %d POD modes before switching for the supply mass fraction\n',offline.nPODModesBeforeSwitch(2)); 
        fprintf('* use %d POD modes after switching for the temperature \n',offline.nPODModesAfterSwitch(1)); 
        fprintf('* use %d POD modes after switching for the supply mass fraction \n',offline.nPODModesAfterSwitch(2)); 
        fprintf('* switching time: \t%f\n',offline.switchTime);
        [modes, redStateMatrixOffline, offline, ~, sPODapproximation, PODModes] = shifted_pod_offline_struct_array(snapshotMatrix, fom, discretization, offline) ;
    else
        [modes, redStateMatrixOffline, offline, ~, sPODapproximation] = shifted_pod_offline_struct_array(snapshotMatrix, fom, discretization, offline);
    end
    n_frames = offline.n_frames ; % number of traveling waves
    % paths determined in the offline phase
    pathsOffline = cell(1,nParams);
    % offline coefficient of first shifted POD mode for the temperature for
    % the left-going wave
    coefficientOf1stTModeOffline = cell(1,nParams);
    % offline coefficient of first shifted POD mode for the supply mass 
    % fraction for the left-going wave
    coefficientOf1stSModeOffline = cell(1,nParams);
    for k=1:nParams
        pathsOffline{k} = redStateMatrixOffline{k}{end,1}(end-n_frames+1:end,:) ;
        coefficientOf1stTModeOffline{k} = redStateMatrixOffline{k}{end,1}(1,:) ;
        coefficientOf1stSModeOffline{k} = redStateMatrixOffline{k}{end,1}(offline_default.n_modes(1)+1,:) ;
    end
    fprintf('\n')
    fprintf('maximal (relative) deviation in the first offline coefficient of the temperature: %e\n', max(abs(coefficientOf1stTModeOffline{1}-mean(coefficientOf1stTModeOffline{1})))/mean(coefficientOf1stTModeOffline{1}))
    fprintf('maximal (relative) deviation in the first offline coefficient of the supply mass fraction: %e\n', max(abs(coefficientOf1stSModeOffline{1}-mean(coefficientOf1stSModeOffline{1})))/mean(coefficientOf1stSModeOffline{1}))
    fprintf('\n')
    % normalize modes 
    nModes = size(modes, 2) ; % number of modes
    for i=1:nModes
        modes(:,i) = modes(:,i)/norm(modes(:,i)) ; % normalize the modes
        for j=1:length(beta_values.training)
            % adjust the coefficients accordingly
            redStateMatrixOffline{j}{end}(i,:) = redStateMatrixOffline{j}{end}(i,:)*norm(modes(:,i)) ;
        end
    end
    % compute relative L2 errors
    relL2offlineErrT = zeros(1, nParams); % relative L2 error for the temperature
    relL2offlineErrS = zeros(1, nParams); % relative L2 error for the supply mass fraction
    for k=1:nParams
        relL2offlineErrT(k) = relative_L2_norm(snapshotMatrix{k}(1:nSteps,:), snapshotMatrix{k}(1:nSteps,:)-sPODapproximation{k}(1:nSteps,:));
        relL2offlineErrS(k) = relative_L2_norm(snapshotMatrix{k}(nSteps+1:2*nSteps,:), snapshotMatrix{k}(nSteps+1:2*nSteps,:)-sPODapproximation{k}(nSteps+1:2*nSteps,:)); 
    end
else
    relL2offlineErrT = NaN ;
    relL2offlineErrS = NaN ;
end

%-------------------------------------------------------------------------------
%% shifted-POD-ROM simulations
%-------------------------------------------------------------------------------

if(modeOfWhatToDo(1)>2)
    % intialize paths used for decomposing the nonlinearity snapshots (use
    % same paths as for the state)
    FShifts = cell(nParams, 1) ;
    for i=1:nParams
        if(useSwitchedROM)
            FShifts{i} = NaN(size(FSnapshots{i},2), 2) ; 
            FShifts{i}(offline.switchTimeIndex+1:end,:) = redStateMatrixOffline{i}{2}(end-1:end,:)' ;
        else 
            FShifts{i} = redStateMatrixOffline{i}{1}(end-1:end,:)' ;
        end
    end
    fprintf('\n -> construct shifted POD modes for the nonlinearity\n');
    fprintf('-------------------------------------------------------------\n');
    % perform mode decomposition for the nonlinearity snapshots
    if(useSwitchedROM)
        % auxiliary variables
        tempNModesFBeforeSwitch = offline.nPODModesBeforeSwitch ;
        tempNModesFAfterSwitch = offline.nPODModesAfterSwitch ;
        % set numbers of DEIM modes for areas (i) and (ii)
        offline.nPODModesBeforeSwitch = hyper.nPODModesBeforeSwitch ;
        offline.nPODModesAfterSwitch = hyper.nPODModesAfterSwitch ;
        [FModes, ~, ~, ~, ~, FPODModes] = shifted_pod_offline_struct_array(FSnapshots, fom, discretization, offline, FShifts) ;
        % reset numbers of POD modes for areas (i) and (ii)
        offline.nPODModesBeforeSwitch = tempNModesFBeforeSwitch ;
        offline.nPODModesAfterSwitch = tempNModesFAfterSwitch ;
    else
        [FModes, ~, ~, ~] = shifted_pod_offline_struct_array(FSnapshots, fom, discretization, offline, FShifts);
    end
    % determine active subspace based on the paths determined in the
    % offline phase
    pathsOfflineMat = cell2mat(pathsOffline); % put offline paths into a matrix
    % compute SVD of path matrix
    [U, ~, ~] = svd(pathsOfflineMat) ;
    % scale leading left singular vector such that U(2,1) entry is 1
    U_scaled = U(:,1) / U(2,1);
    % active subspace is defined as p1=U_scaled(1)*p2
    offline.active_subspace_factor = U_scaled(1);
    fprintf('\nDetermined active subspace is given by p1=a*p2 with a=%.2f', offline.active_subspace_factor)
    fprintf('\n\n')
    % precompute coefficient matrices
    if(useSwitchedROM)
        PODROMMatrices = PODROMPreProcessing(A, gammaS/alpha, PODModes{1}, FPODModes{1}, offline.nPODModesBeforeSwitch) ;
        sampledMatrices = sPODROMPreProcessing(modes, discretization(1), hyper, FModes, offline, A, gammaS/alpha, PODModes{2}, FPODModes{2}) ;
    else
        sampledMatrices = sPODROMPreProcessing(modes, discretization(1), hyper, FModes, offline, A, gammaS/alpha) ;
    end
    % Solve ROM for the training parameters
    fprintf('\nSolve ROM for the training parameters\n');
    fprintf('=============================================================\n');
    simTimeROMVec = NaN(numberOfRunsForAveragingRunTime, nParams) ;
    for i=1:numberOfRunsForAveragingRunTime
        if(useSwitchedROM)
            [sPODSol, ~, simTimeROMVec(i,:)] = shifted_pod_online_struct_array(rom, discretization, offline, modes, sampledMatrices, hyper, PODROMMatrices, PODModes, FPODModes) ;
        else
            [sPODSol, ~, simTimeROMVec(i,:)] = shifted_pod_online_struct_array(rom, discretization, offline, modes, sampledMatrices, hyper) ;
        end
    end
    simTimeROM = median(simTimeROMVec, 1) ;
    % compute relative online errors
    relL2onlineErr = zeros(1, nParams); % total relative L2 error
    relL2onlineErrT = zeros(1, nParams); % relative L2 error for the temperature
    relL2onlineErrS = zeros(1, nParams); % relative L2 error for the supply mass fraction
    speedups = zeros(1, nParams) ; % speedup for each parameter value
    fprintf('\n');
    for k=1:nParams
        relL2onlineErr(k) = relative_L2_norm(snapshotMatrix{k}, snapshotMatrix{k}-sPODSol{k});
        relL2onlineErrT(k) = relative_L2_norm(snapshotMatrix{k}(1:nSteps,:), snapshotMatrix{k}(1:nSteps,:)-sPODSol{k}(1:nSteps,:));
        relL2onlineErrS(k) = relative_L2_norm(snapshotMatrix{k}(nSteps+1:2*nSteps,:), snapshotMatrix{k}(nSteps+1:2*nSteps,:)-sPODSol{k}(nSteps+1:2*nSteps,:));
        speedups(k) = computation_time_fom(k)/simTimeROM(k) ;
        fprintf('Parameter beta = %f\n',beta_values.training(k));
        fprintf(' -- Total relative online error measured in L2 norm: \t\t\t%e\n',relL2onlineErr(k));
        fprintf(' -- Relative online error of physical variable #1 measured in L2 norm: \t%e\n',relL2onlineErrT(k));
        fprintf(' -- Relative online error of physical variable #2 measured in L2 norm: \t%e\n\n',relL2onlineErrS(k));
        fprintf(' -- Computation time ROM:\t%f (sec)\n',simTimeROM(k));
        fprintf('\n')
    end
    if(nargin>7)
        if(~isempty(betaTest))
            % perform additional runs for test parameter values
            fprintf('Solve FOM for the testing parameters\n');
            fprintf('=============================================================\n');
            % structs for the parameter sampling
            fomSampling = create_fom_struct(initialCondition,'parameter.beta',beta_values.testing, 'parameter.alpha', alpha, 'parameter.gammaS', gammaS, 'parameter.v', v);
            discretizationSampling = create_discretization_struct(fomSampling,discretization_default,initialCondition);
            romSampling = create_rom_struct(fomSampling,'parameter.beta', beta_values.testing, 'parameter.alpha', alpha, 'parameter.gammaS', gammaS, 'parameter.v', v);
            % solve FOM for the test parameter values
            [snapshotMatrixTesting, computation_time_fom_testing, ~, ~] = compute_truth_solution_struct_array(fomSampling, discretizationSampling);
            % solve ROM for the test parameter values
            fprintf('\nSolve ROM for the testing parameters\n');
            fprintf('=============================================================\n');
            if(useSwitchedROM)
                [sPODSolTesting, ~, simTimeROMTesting] = shifted_pod_online_struct_array(romSampling, discretizationSampling, offline, modes, sampledMatrices, hyper, PODROMMatrices, PODModes, FPODModes) ;
            else
                [sPODSolTesting, ~, simTimeROMTesting] = shifted_pod_online_struct_array(romSampling, discretizationSampling, offline, modes, sampledMatrices, hyper) ;
            end
            % compute relative L2 online errors
            fprintf('\nCompute errors for the testing parameters\n');
            fprintf('=============================================================\n');
            n_params_testing = length(beta_values.testing); % number of parameter test values
            relL2onlineErrTesting = NaN(1,n_params_testing); % total relative online errors
            relL2onlineErrTTesting = NaN(1,n_params_testing); % relative online errors for the temperature
            relL2onlineErrSTesting = NaN(1,n_params_testing); % relative online errors for the supply mass fraction
            speedups = computation_time_fom_testing./simTimeROMTesting;
            for k=1:n_params_testing
                relL2onlineErrTesting(k) = relative_L2_norm(snapshotMatrixTesting{k}, snapshotMatrixTesting{k}-sPODSolTesting{k});
                relL2onlineErrTTesting(k) = relative_L2_norm(snapshotMatrixTesting{k}(1:nSteps,:), snapshotMatrixTesting{k}(1:nSteps,:)-sPODSolTesting{k}(1:nSteps,:));
                relL2onlineErrSTesting(k) = relative_L2_norm(snapshotMatrixTesting{k}(nSteps+1:2*nSteps,:), snapshotMatrixTesting{k}(nSteps+1:2*nSteps,:)-sPODSolTesting{k}(nSteps+1:2*nSteps,:));
            end
        end
    end
else
    simTimeROM = NaN ;
    relL2onlineErr = NaN ;
    relL2onlineErrT = NaN ;
    relL2onlineErrS = NaN ;
    speedups = NaN ;
end

%-------------------------------------------------------------------------------
%% Mode Decomposition (Offline Phase) via POD
%-------------------------------------------------------------------------------

if(length(modeOfWhatToDo)>1)
    if(modeOfWhatToDo(2)>1)
        fprintf('\nConstruct POD modes for the snapshots with %d modes per variable\n', nPODModes);
        fprintf('=============================================================\n');
        nPODModesVec = nPODModes*[1 1] ;
        [modes_POD, ~, ~] = POD_offline(snapshotMatrix, nPODModesVec, discretization);
    end
end

%-------------------------------------------------------------------------------
%% POD ROM simulations
%-------------------------------------------------------------------------------

if(length(modeOfWhatToDo)>1)
    if(modeOfWhatToDo(2)>2)
        fprintf('\n -> construct POD modes for the nonlinearity\n');
        fprintf('-------------------------------------------------------------\n');
        % perform mode decomposition for the nonlinearity snapshots; number
        % of DEIM modes is taken as the double of the number of POD modes
        [FModes_POD, ~, ~] = POD_offline(FSnapshots, 2*nPODModesVec(1), discretization) ;
        % precompute coefficient matrices
        PODROMMatrices = PODROMPreProcessing(A, gammaS/alpha, modes_POD, FModes_POD, nPODModesVec) ;
        % Solve ROM for the training parameters
        fprintf('\nSolve POD ROM for the training parameters\n');
        fprintf('=============================================================\n');
        simTimeROMVec = NaN(1,numberOfRunsForAveragingRunTime) ;
        for i=1:numberOfRunsForAveragingRunTime
            [PODSol, ~, simTimeROMVec(i)] = pod_online_struct_array(rom, discretization, modes_POD, PODROMMatrices, FModes_POD) ;
        end
        simTimeROM_POD = median(simTimeROMVec) ;
        % compute relative online errors
        relL2onlineErr_POD = zeros(1, nParams); % total relative L2 error
        for k=1:nParams
            relL2onlineErr_POD(k) = relative_L2_norm(snapshotMatrix{k}, snapshotMatrix{k}-PODSol{k});
        end
    end
end

%-------------------------------------------------------------------------------
%% Plots
%-------------------------------------------------------------------------------
    
% prepare Data
temperatureFOM{1} = snapshotMatrix{1}(1:nSteps,:)' ;
smfFOM{1} = snapshotMatrix{1}(nSteps+1:end,:)' ;

% Plot of the FOM - Temperature
if(contains(plotsToBeCreated,'temperatureSnapshots'))
    figure() ;
    plotSpacePoints = (1:1:nSteps)*discretization(1).space.dx ; % space points used for the plot
    plotTimePoints = (0:1:discretization(1).time.n_steps)*discretization(1).time.dt ; % time points used for the plot
    pcolor(plotSpacePoints, plotTimePoints,temperatureFOM{1}) ;
    shading flat; 
    colorbar ;
    title(sprintf('FOM solution: temperature, beta=%f',beta_values.training(1)));
    xlabel('x[m]')
    ylabel('t[s]')
end
    
% Plot of the FOM - Supply Mass Fraction (smf)
if(contains(plotsToBeCreated,'supplyMassFractionSnapshots'))
    figure() ;
    plotSpacePoints = (1:1:nSteps)*discretization(1).space.dx ; % space points used for the plot
    plotTimePoints = (0:1:discretization(1).time.n_steps)*discretization(1).time.dt ; % time points used for the plot
    pcolor(plotSpacePoints, plotTimePoints,smfFOM{1}) ;
    shading flat; 
    colorbar ;
    title(sprintf('FOM solution: smf, beta=%f',beta_values.training(1)));
    xlabel('x[m]')
    ylabel('t[s]')
end

% Plots of left-going separated wave
if(contains(plotsToBeCreated,'leftGoingWave')&&(modeOfWhatToDo(1)>1))
    plotLeftGoingSeparatedWave(discretization, offline, snapshotMatrix{1}(1:nSteps,:), pathsOffline{1}')
end

% Plots of transformed modes for the left-going wave
if(contains(plotsToBeCreated,'modes')&&(modeOfWhatToDo(1)>1))
    % sort modes per physical variable
    modesPerVar = cell(n_vars,1) ;
    nModes = offline.n_modes ; % number of modes
    if(size(nModes,1)==1) % this case means that the same number of modes is used for the temperature and for the supply mass fraction
        n_modes_per_var_per_frame = nModes / n_vars ; % number of modes per physical variable and per traveling wave
        helper = [0 cumsum(n_modes_per_var_per_frame)] ;
        for i=1:n_vars
            modesPerVar{i,1} = modes((i-1)*nSteps+1:i*nSteps,1 + helper(i):1 + helper(i+1) - 1);
        end
        clear n_modes_per_var_per_frame helper fStart
    else
        for j=1:n_vars
            start2 = sum(nModes(1:j-1,1)) ;
            currentModeIndices = start2+(1:nModes(j,1)) ;
            modesPerVar{j,1} = modesData((j-1)*nSteps+1:j*nSteps,currentModeIndices) ;
        end
        clear start1 start2 currentModeIndices
    end
    everyNthPoint = 4 ; % this number allows to plot only (number of space points)/everyNthPoint points (roughly)
    plotSpacePoints = (1:everyNthPoint:nSteps)*discretization(1).space.dx ; % space points used for the plot
    % plot the modes for each physical variable separately
    for i=1:n_vars
        figure() ;
        plot(plotSpacePoints, modesPerVar{i,1}(1:everyNthPoint:end,:));
        title(sprintf('Modes var%d, frame %d',i,j)) ;
        xlabel('x[m]')
    end
end

% Plots of nonlinearity snapshots
if(contains(plotsToBeCreated,'nonlinearitySnapshots'))
    figure() ;
    plotSpacePoints = (1:1:nSteps)*discretization(1).space.dx ; % space points used for the plot
    plotTimePoints = (0:1:discretization(1).time.n_steps)*discretization(1).time.dt ; % time points used for the plot
    pcolor(plotSpacePoints, plotTimePoints, FSnapshots{1}') ;
    shading flat; 
    colorbar ;
    title(sprintf('FOM nonlinearity, beta=%f',beta_values.training(1)));
    xlabel('x[m]')
    ylabel('t[s]')
    set(gca,'ColorScale','log')
    caxis([0.0012, 117.9972])
end

% Plot of the relative L2 online errors for the parameter test values
if(contains(plotsToBeCreated,'parameterSampling')&&(modeOfWhatToDo(1)>1)&&nargin>7)
    figure();
    semilogy(beta_values.testing,relL2onlineErrTTesting); 
    hold on
    semilogy(beta_values.testing,relL2onlineErrSTesting);
    legend('temperature','supply mass fraction');
    title('Relative Error for the testing paramters');
    xlabel('Arrhenius coefficient \beta [K]')
    ylabel('relative online error')
    % produce a table with the relative errors and speedups
    fprintf('\n')
    fprintf('--------------------------------------------------------------\n');
    fprintf('- Summary of testing parameters ------------------------------\n');
    fprintf('--------------------------------------------------------------\n');
    fprintf('\n')
    quantity = {'relative error'; 'speedup'} ;
    minimumValue = [min(relL2onlineErrTesting); min(speedups)] ;
    meanValue = [mean(relL2onlineErrTesting); mean(speedups)] ;
    maximumValue = [max(relL2onlineErrTesting); max(speedups)] ;
    disp(table(quantity, minimumValue, meanValue, maximumValue))
end

% Plot of single FOM and ROM approximation snapshots for the worst-case parameter value
if(contains(plotsToBeCreated,'singleSnapshots')&&(modeOfWhatToDo(1)>1)&&nargin>7)
	[~, worstCaseInd] = max(relL2onlineErrTTesting); % determine parameter value which results in the biggest ROM error
	figure()
	secondsToPlot = [200, 800, 1400, 2000]; % time points of the snapshots to be plotted
	legendEntries = cell(length(secondsToPlot),1);
	plotIndices = round(secondsToPlot/discretization(1).time.length*discretization(1).time.n_steps); % time indices of the snapshots to be plotted
	% fill legend entries
    for k=1:length(secondsToPlot)
        legendEntries{k} = sprintf('t=%d s',secondsToPlot(k));
    end
    everyNthPoint = 4 ; % this number allows to plot only (number of space points)/everyNthPoint points (roughly)
    plotSpacePoints = (1:everyNthPoint:nSteps)*discretization(1).space.dx ; % space points used for the plot
	plotIndicesSpace = 1:everyNthPoint:discretization(1).space.n_steps; % space indices used for the plot
	plot(plotSpacePoints,snapshotMatrixTesting{worstCaseInd}(plotIndicesSpace,plotIndices));
	hold on
	set(gca,'ColorOrderIndex',1) % ensure that the same colors are used for the ROM snapshots as for the FOM snapshots
	plot(plotSpacePoints,sPODSolTesting{worstCaseInd}(plotIndicesSpace,plotIndices),'d');
	title(sprintf('testing: worst case error - temperature: FOM vs ROM (beta=%d K)',beta_values.testing(worstCaseInd)));
	legend(legendEntries);
	tmpYlim = ylim;
	ylim([0 tmpYlim(2)]);
    xlabel('x[m]')
    ylabel('temperature [K]')
end

% Plot of the FOM - Temperature (zoom into the initial region)
if(contains(plotsToBeCreated,'temperatureZoom'))
    figure() ;
    plotSpacePoints = (1:1:nSteps)*discretization(1).space.dx ; % space points used for the plot
    plotTimePoints = (0:1:discretization(1).time.n_steps)*discretization(1).time.dt ; % time points used for the plot
    pcolor(plotSpacePoints, plotTimePoints,temperatureFOM{1}) ;
    shading flat; 
    colorbar ;
    title(sprintf('FOM solution: temperature, beta=%f',beta_values.training(1)));
    xlabel('x[m]')
    ylabel('t[s]')
    yline(offline_default.switchTime, 'w-.', 'LineWidth', 2)
    yline(offline_default.switchTime+offline_default.shiftedPODstart(1)*(discretization(1).time.length-offline_default.switchTime), 'w--', 'LineWidth', 2)
	temperatureZoomICtime = 150;
	midpoint = floor(nSteps/2);
	offset = floor(0.15*midpoint);
	spaceIndices = (midpoint-offset):(midpoint+offset);
	timeIndices = 1:ceil(temperatureZoomICtime/discretization(1).time.length*discretization(1).time.n_steps);
	figure() ;
	f = pcolor(plotSpacePoints(spaceIndices), plotTimePoints(timeIndices), temperatureFOM{1}(timeIndices,spaceIndices)) ;
	shading flat; 
	colorbar ;
	title(sprintf('FOM solution: temperature, beta=%f',beta_values.training(1)));
    xlabel('x[m]')
    ylabel('t[s]')
    yline(offline_default.switchTime, 'w-.', 'LineWidth', 2)
end

% Plot of the comparison of extrapolated vs. projected coefficients in area
% (ii)
if(contains(plotsToBeCreated,'coefficientComparison')&&(modeOfWhatToDo(1)>1)&&strcmp(initialCondition,'Gaussian'))
    plotExtrapolatedVsProjectedCoefficients(discretization(1), offline, snapshotMatrix{1}(1:nSteps,:), pathsOffline{1}')  
end

fprintf('\n')