%% Efficient Wildland Fire Simulation via Nonlinear Model Order Reduction
% This is the main file to compute the numerical examples for the paper.
%
% Reference:
%   F.Black, P. Schulze, B. Unger: Efficient Wildland Fire Simulation via Nonlinear Model Order Reduction (2021)
%
% Authors:
%   Felix Black, black@math.tu-berlin.de
%   Philipp Schulze, pschulze@math.tu-berlin.de
%   Benjamin Unger, benjamin.unger@simtech.uni-stuttgart.de
%
% Version:
%   1.0 (July 26, 2021)
% 

restoredefaultpath
clear
close all
clc

%--------------------------------------------------------------------------
%% Setup
%--------------------------------------------------------------------------

% The following variable 'dataToCreate' determines which figure or table of 
% the paper should be reproduced. The options are:

% - 'snapshotWaveEqVsWildFire': creates analytical solution of the wave 
% equation and simulates the wildland fire model; produces Figure 1

% - 'snapshotSeparatedWaves': simulates the wildland fire model with the
% separated waves initial condition; produces Figure 2

% - 'modesForSeparatedWaves': simulates the wildland fire model with the
% separated waves initial condition and computes a mode decomposition based 
% on two modes per traveling wave and per physical variable ; produces
% Figure 3

% - 'nonlinearitySnapshots': simulates the wildland fire model with
% separated waves initial condition and afterwards with Gaussian initial
% condition; produces Figure 4

% - 'PODvsSPODSeparatedWaves': simulates the wildland fire model with
% separated waves initial condition, computes mode decompositions via POD
% and shifted POD for different mode numbers, simulates POD- and
% shifted-POD-ROM; produces Figure 5

% - 'parameterSamplingSeparatedWaves': simulates the wildland fire model
% with separates waves initial condition and 3 different values for the
% Arrhenius coefficient; afterwards, the snapshots are used for determining
% shifted POD modes and the shifted-POD-ROM is simulated for many different
% test values for the Arrhenius coefficient; produces Figure 6

% - 'temperatureSnapshotsZoom': simulates the wildland fire model with
% Gaussian initial condition and plots the temperature snapshots while
% zooming into the ignition region at the beginning of the time interval;
% produces Figure 7

% - 'comparisonOfExtrapolatedAndProjectedCoefficients': simulates the
% wildland fire model with Gaussian initial condition, constructs a mode
% decomposition via shifted POD, and compares the approaches using
% extrapolated and projected shifted POD coefficients; produces Figure 8

% - 'tryDifferentPODModeNumbersInAreasIandII': simulated the wildland fire
% model with Gaussian initial condition and constructs a mode decomposition
% with different POD mode numbers in area (i) and (ii); produces Figure 9

% - 'PODvsSPODGaussian': simulates the wildland fire model with
% Gaussian initial condition, computes mode decompositions via POD
% and shifted POD for different mode numbers, simulates POD- and
% shifted-POD-ROM; produces Figure 10

% - 'parameterSamplingGaussian': simulates the wildland fire model
% with Gaussian initial condition and 3 different values for the
% Arrhenius coefficient; afterwards, the snapshots are used for determining
% shifted POD modes and the shifted-POD-ROM is simulated for many different
% test values for the Arrhenius coefficient; produces Figures 11, 12, and 
% Table 4

% - 'approximationWithWind': simulates the wildland fire model
% with Gaussian initial condition and 3 different values for the
% Arrhenius coefficient and with non-zero wind velocity; afterwards, 
% the snapshots are used for determining shifted POD modes and the 
% shifted-POD-ROM is simulated for many different test values for the 
% Arrhenius coefficient; produces Figure 13

% - 'errorsSeparatedWavesForDifferentModeNumbers': simulates the wildland 
% fire model with the separated waves initial condition; afterwards
% computes mode decompositions and ROM simulations for different mode
% numbers; produces Table 3

dataToCreate = 'snapshotWaveEqVsWildFire' ;

%% Calculations 

switch dataToCreate
    case 'approximationWithWind'
    	plotsToBeCreated = ['parameterSampling', 'singleSnapshots'] ;
    	modeOfWhatToDo = 3 ; % simulate the FOM, perform a mode decomposition of the snapshot data, simulate the ROM
        n_modes = 3 ; % number of transformed modes per traveling wave and per physical variable
    	betaTrainingValues = 540:20:580 ; % values for the Arrhenius coefficient used in the offline phase
    	betaTestValues = 540:0.5:580 ; % values for the Arrhenius coefficient used in the online phase
    	windVelocity = 0.02 ;
    	simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo, n_modes, 0, 1, betaTrainingValues, betaTestValues, 14, 2, 0.02) ;
    case 'comparisonOfExtrapolatedAndProjectedCoefficients'
        plotsToBeCreated = 'coefficientComparison' ;
        modeOfWhatToDo = 2 ; % simulate the FOM and perform a mode decomposition of the snapshot data
        n_modes = 3 ; % number of transformed modes per traveling wave and per physical variable
        simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo, n_modes) ;
    case 'errorsSeparatedWavesForDifferentModeNumbers'
        plotsToBeCreated = '' ;
        modeOfWhatToDo = 3 ; % simulate the FOM, perform a mode decomposition of the snapshot data, simulate the ROM
        modeNumbers = 1:6 ; % numbers of transformed modes per traveling wave and per physical variable
        nModeNumbers = length(modeNumbers) ;
        % initialization
        relL2onlineErr_T = NaN(nModeNumbers, 1) ; % relative online errors for the temperature
        relL2onlineErr_S = NaN(nModeNumbers, 1) ; % relative online errors for the supply mass fraction
        relL2offlineErr_T = NaN(nModeNumbers, 1) ; % relative offline errors for the temperature
        relL2offlineErr_S = NaN(nModeNumbers, 1) ; % relative offline errors for the supply mass fraction
        speedup = NaN(nModeNumbers, 1) ;
        DOF = cell(nModeNumbers, 1) ;
        for i=1:nModeNumbers
            [relL2onlineErr_T(i), relL2onlineErr_S(i), relL2offlineErr_T(i), relL2offlineErr_S(i), speedup(i)] = simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo, modeNumbers(i)) ;
            DOF{i} = [num2str(4*modeNumbers(i)),'+2'] ;
        end
        disp(table(DOF, relL2offlineErr_T, relL2offlineErr_S, relL2onlineErr_T, relL2onlineErr_S, speedup))
    case 'modesForSeparatedWaves'
        plotsToBeCreated = 'modes' ;
        modeOfWhatToDo = 2 ; % simulate the FOM and perform a mode decomposition of the snapshot data
        n_modes = 2 ; % number of transformed modes per traveling wave and per physical variable
        simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo, n_modes) ;
    case 'nonlinearitySnapshots'
        plotsToBeCreated = 'nonlinearitySnapshots' ;
        modeOfWhatToDo = 1 ; % only simulate the FOM
        simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo) ;
        simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo) ;
    case 'parameterSamplingGaussian'
        plotsToBeCreated = ['parameterSampling', 'singleSnapshots'] ;
    	modeOfWhatToDo = 3 ; % simulate the FOM, perform a mode decomposition of the snapshot data, simulate the ROM
    	n_modes = 3 ; % number of transformed modes per traveling wave and per physical variable
    	betaTrainingValues = 540:20:580 ; % values for the Arrhenius coefficient used in the offline phase
    	betaTestValues = 540:0.5:580 ; % values for the Arrhenius coefficient used in the online phase
    	simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo, n_modes, 0, 1, betaTrainingValues, betaTestValues) ;
    case 'parameterSamplingSeparatedWaves'
        plotsToBeCreated = 'parameterSampling' ;
        modeOfWhatToDo = 3 ; % simulate the FOM, perform a mode decomposition of the snapshot data, simulate the ROM
        n_modes = 3 ; % number of transformed modes per traveling wave and per physical variable
        betaTrainingValues = 540:20:580 ; % values for the Arrhenius coefficient used in the offline phase
        betaTestValues = 540:0.5:580 ; % values for the Arrhenius coefficient used in the online phase
        simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo, n_modes, 0, 1, betaTrainingValues, betaTestValues) ;
    case 'PODvsSPODGaussian'
        plotsToBeCreated = '' ;
        numberOfRunsForAveragingRunTime = 50 ;
        modeOfWhatToDo = [1 3] ; % simulates the FOM, computed mode decomposition based on POD, simulates POD ROM
        modeNumbers_POD = 30:10:170 ; % considered POD mode numbers
        nModeNumbers_POD = length(modeNumbers_POD) ;
        % initialization
        relL2onlineErrors_POD = NaN(nModeNumbers_POD, 1) ; % relative online errors for POD
        simTimes_POD = NaN(nModeNumbers_POD, 1) ; % run times for POD ROM
        for i=1:nModeNumbers_POD
            [~, ~, ~, ~, ~, ~, ~, relL2onlineErrors_POD(i), simTimes_POD(i)] = simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo, 0, modeNumbers_POD(i), numberOfRunsForAveragingRunTime) ;
        end
        modeOfWhatToDo = 3 ; % simulate the FOM, perform a mode decomposition of the snapshot data, simulate the ROM
        modeNumbers_sPOD = 1:2:5 ; % numbers of transformed modes per traveling wave and per physical variable
        PODModeNumbersAreaI = 14 ; % numbers of POD modes per physical variable used in area (i)
        PODModeNumbersAreaII = [1 3] ; % numbers of POD modes per physical variable used in area (ii)
        nModeNumbers_sPOD = length(modeNumbers_sPOD) ;
        nPODModeNumbersAreaI = length(PODModeNumbersAreaI) ;
        nPODModeNumbersAreaII = length(PODModeNumbersAreaII) ;
        % initialization
        relL2onlineErrors_sPOD = NaN(nModeNumbers_sPOD, nPODModeNumbersAreaI, nPODModeNumbersAreaII) ; % relative online errors for shifted POD
        simTimes_sPOD = NaN(nModeNumbers_sPOD, nPODModeNumbersAreaI, nPODModeNumbersAreaII) ; % run times for shifted-POD-ROM
        for i=1:nModeNumbers_sPOD
            for j=1:nPODModeNumbersAreaI
                for k=1:nPODModeNumbersAreaII
                    [~, ~, ~, ~, ~, relL2onlineErrors_sPOD(i,j,k), simTimes_sPOD(i,j,k)] = simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo, modeNumbers_sPOD(i), 0, numberOfRunsForAveragingRunTime, [], [], PODModeNumbersAreaI(j), PODModeNumbersAreaII(k)) ;
                end
            end
        end
        figure()
        semilogy(simTimes_POD, relL2onlineErrors_POD, 'o')
        hold on
        semilogy(simTimes_sPOD(:), relL2onlineErrors_sPOD(:), '+')
        title('Comparison between POD-DEIM-ROM and sPOD-sDEIM-ROM')
        xlabel('run time')
        ylabel('relative L_2 error')
        legend('POD-DEIM','sPOD-sDEIM')
        for k=1:nModeNumbers_POD
            text(simTimes_POD(k)+0.0025,relL2onlineErrors_POD(k)*0.85,num2str(2*modeNumbers_POD(k)))
        end
        for i=1:nModeNumbers_sPOD
            for j=1:nPODModeNumbersAreaI
                for k=1:nPODModeNumbersAreaII
                    text(simTimes_sPOD(i,j,k)+0.002,relL2onlineErrors_sPOD(i,j,k)*0.9,[num2str(2*PODModeNumbersAreaI(j)),'-(',num2str(4*modeNumbers_sPOD(i)),'+',num2str(2*PODModeNumbersAreaII(k)),'+2)'])
                end
            end
        end
    case 'PODvsSPODSeparatedWaves'
        plotsToBeCreated = '' ;
        numberOfRunsForAveragingRunTime = 50 ;
        modeOfWhatToDo = [1 3] ; % simulates the FOM, computed mode decomposition based on POD, simulates POD ROM
        modeNumbers_POD = 30:10:100 ; % considered POD mode numbers
        nModeNumbers_POD = length(modeNumbers_POD) ;
        % initialization
        relL2onlineErrors_POD = NaN(nModeNumbers_POD, 1) ; % relative online errors for POD
        simTimes_POD = NaN(nModeNumbers_POD, 1) ; % run times for POD ROM
        for i=1:nModeNumbers_POD
            [~, ~, ~, ~, ~, ~, ~, relL2onlineErrors_POD(i), simTimes_POD(i)] = simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo, 0, modeNumbers_POD(i), numberOfRunsForAveragingRunTime) ;
        end
        modeOfWhatToDo = 3 ; % simulate the FOM, perform a mode decomposition of the snapshot data, simulate the ROM
        modeNumbers_sPOD = 1:8 ; % numbers of transformed modes per traveling wave and per physical variable
        nModeNumbers_sPOD = length(modeNumbers_sPOD) ;
        % initialization
        relL2onlineErrors_sPOD = NaN(nModeNumbers_sPOD, 1) ; % relative online errors for shifted POD
        simTimes_sPOD = NaN(nModeNumbers_sPOD, 1) ; % run times for shifted-POD-ROM
        for i=1:nModeNumbers_sPOD
            [~, ~, ~, ~, ~, relL2onlineErrors_sPOD(i), simTimes_sPOD(i)] = simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo, modeNumbers_sPOD(i), 0, numberOfRunsForAveragingRunTime) ;
        end
        figure()
        semilogy(simTimes_POD, relL2onlineErrors_POD, 'o')
        hold on
        semilogy(simTimes_sPOD, relL2onlineErrors_sPOD, '+')
        title('Comparison between POD-DEIM-ROM and sPOD-sDEIM-ROM')
        xlabel('run time')
        ylabel('relative L_2 error')
        legend('POD-DEIM','sPOD-sDEIM')
        for k=1:nModeNumbers_POD
            text(simTimes_POD(k)+0.0025,relL2onlineErrors_POD(k)*0.85,num2str(2*modeNumbers_POD(k)))
        end
        for k=1:nModeNumbers_sPOD
            text(simTimes_sPOD(k)+0.002,relL2onlineErrors_sPOD(k)*0.9,[num2str(4*modeNumbers_sPOD(k)),'+2'])
        end
    case 'snapshotSeparatedWaves'
        plotsToBeCreated = ['temperatureSnapshots','leftGoingWave'] ;
        modeOfWhatToDo = 2 ; % simulate the FOM and perform a mode decomposition of the snapshot data
        simulateWildlandFire('separatedWaves', plotsToBeCreated, modeOfWhatToDo) ;
    case 'snapshotWaveEqVsWildFire'
        createAndPlotWaveEquationSnapshots() % create plot for wave equation
        plotsToBeCreated = ['supplyMassFractionSnapshots','temperatureSnapshots'] ;
        modeOfWhatToDo = 1 ; % only simulate the FOM
        simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo) ;
    case 'temperatureSnapshotsZoom'
      	plotsToBeCreated = 'temperatureZoom' ;
    	modeOfWhatToDo = 1 ; % only simulate the FOM
    	simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo) ;
    case 'tryDifferentPODModeNumbersInAreasIandII'
        plotsToBeCreated = '' ;
        modeOfWhatToDo = 2 ; % simulate the FOM and perform a mode decomposition of the snapshot data
        n_modes = 3 ; % number of transformed modes per traveling wave and per physical variable
        PODModeNumbersAreaI = 2:2:20 ; % numbers of POD modes per physical variable used in area (i)
        PODModeNumbersAreaII = 1:2:5 ; % numbers of POD modes per physical variable used in area (ii)
        nPODModeNumbersAreaI = length(PODModeNumbersAreaI) ;
        nPODModeNumbersAreaII = length(PODModeNumbersAreaII) ;
        relL2offlineErr_T = NaN(nPODModeNumbersAreaI, nPODModeNumbersAreaII) ; % relative offline errors for the temperature
        relL2offlineErr_S = NaN(nPODModeNumbersAreaI, nPODModeNumbersAreaII) ; % relative offline errors for the supply mass fraction
        for i=1:nPODModeNumbersAreaI
            for j=1:nPODModeNumbersAreaII
                [~, ~, relL2offlineErr_T(i,j), relL2offlineErr_S(i,j)] = simulateWildlandFire('Gaussian', plotsToBeCreated, modeOfWhatToDo, n_modes, 0, 1, [], [], PODModeNumbersAreaI(i), PODModeNumbersAreaII(j)) ;
            end
        end
        % create plot for temperature errors
        figure
        xlabel('Number of POD modes before the switch')
        ylabel('Relative offline error in T')
        Legend = cell(nPODModeNumbersAreaII, 1) ;
        hold on
        for i=1:nPODModeNumbersAreaII
            semilogy(2*PODModeNumbersAreaI, relL2offlineErr_T(:,i), '+')
            Legend{i} = [num2str(2*PODModeNumbersAreaII(i)),' POD modes after the switch'] ;
        end
        legend(Legend)
        xlim([min(2*PODModeNumbersAreaI)-1,max(2*PODModeNumbersAreaI)+1])
        set(gca, 'YScale', 'log')
        % create plot for supply mass fraction errors
        figure
        xlabel('Number of POD modes before the switch')
        ylabel('Relative offline error in S')
        Legend = cell(nPODModeNumbersAreaII, 1) ;
        hold on
        for i=1:nPODModeNumbersAreaII
            semilogy(2*PODModeNumbersAreaI, relL2offlineErr_S(:,i), '+')
            Legend{i} = [num2str(2*PODModeNumbersAreaII(i)),' POD modes after the switch'] ;
        end
        legend(Legend)
        xlim([min(2*PODModeNumbersAreaI)-1,max(2*PODModeNumbersAreaI)+1])
        set(gca, 'YScale', 'log')
    otherwise
        error('Unknown data.')
end