function rhs = wildlandFireROMRHS_hyperreduced(t, current, offline, params, sampledMatrices, hyper, PODModes, FPODModes)
% wildlandFireROMRHS_hyperreduced - gives the right-hand side of the
% shifted-POD-shifted-DEIM ROM
%
% inputs:
% - t: dummy input to fit to the ode45 syntax
% - current: current value of the ROM state
% - offline: struct containing settings for the mode decomposition
% - params: struct containing the values of some of the PDE coefficients
% - sampledMatrices: struct containing the sampled path-dependent
% coefficient matrices
% - hyper: struct containing hyperreduction parameters
% - PODModes: POD modes for area (ii); optional
% - FPODModes: DEIM modes for area (ii); optional
%
% outputs:
% - rhs: current value of the ROM right-hand side
%
% dependencies:
% - evaluateWildlandFireNonlinearity (in LIB/SIMULATION)
% - findIntervalAndGiveInterpolationWeight_1D (in LIB/INTERPOLATION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% initializations and set-up

if(nargin>6) % if POD modes are provided 
    % total number of POD modes
    nPODModes = size(PODModes, 2) ;
    % number of DEIM modes
    nFPODModes = size(FPODModes, 2) ;
    % number of POD modes for each physical variable
    nPODModesVec = offline.nPODModesAfterSwitch ;
else % if no POD modes are provided
    % set corresponding mode numbers to zero
    nPODModes = 0 ;
    nFPODModes = 0 ;
    nPODModesVec = [0 0] ;
end

% mode numbers
nModes = offline.n_modes ; % number of shifted POD modes for each traveling wave
nNonlinearityModes = hyper.n_modes ; % number of shifted DEIM modes for each traveling wave
totalNShiftedModes = sum(nModes) ; % total number of shifted POD modes
totalNShiftedNonlinearityModes = sum(nNonlinearityModes) ; % total number of shifted DEIM modes
totalNModes = totalNShiftedModes+nPODModes ; % total number of POD and shifted POD modes
totalNNonlinearityModes = totalNShiftedNonlinearityModes+nFPODModes ; % total number of DEIM and shifted DEIM modes

 % number of traveling waves (or equivalently number of paths)
nFrames = length(current)-totalNModes ;

% current coefficients
alpha = current(1:totalNModes) ;

% auxiliary variable for handling the mode number vector
helper = cumsum([0,nModes]) ;

% current path values
p1 = current(end-1) ;
p2 = current(end) ;
p = [p1; p2] ;

%% load sampled coefficient matrices

% path samples from the offline phase
samplingPoints = sampledMatrices.samplingPoints ;
% find in which path sample interval the current path p(2) is and determine
% the corresponding weights for linear interpolation
[intervalIdx, weight] = findIntervalAndGiveInterpolationWeight_1D(samplingPoints, p(2)) ;
% nearest neighbor interpolation is used in the following and its 
% interpolation weights can be obtained by rounding the linear 
% interpolation weights
weight = round(weight) ;

% assemble the block matrix containing the coefficients for each traveling
% wave on the block diagonal
Dalpha = [alpha(helper(1)+1:helper(2)),zeros(nModes(1),1); zeros(nModes(2),1),alpha(helper(2)+1:helper(3))] ;

% load the sampled ROM mass matrix and obtain the current one via
% interpolation

% load the blocks of the sampled mass matrix
M11Sampled = sampledMatrices.M11 ;
M12Sampled = sampledMatrices.M12 ;
M22Sampled = sampledMatrices.M22 ;
% obtain the current mass matrix blocks via nearest neighbor interpolation
M11 = weight*M11Sampled(:,:,intervalIdx)+(1-weight)*M11Sampled(:,:,intervalIdx+1) ;
M12 = (weight*M12Sampled(:,:,intervalIdx)+(1-weight)*M12Sampled(:,:,intervalIdx+1))*Dalpha ;
M22 = Dalpha'*(weight*M22Sampled(:,:,intervalIdx)+(1-weight)*M22Sampled(:,:,intervalIdx+1))*Dalpha ;
% assemble the mass matrix based on its blocks
M = [M11,M12; M12',M22] ;

% load the sampled ROM Ar matrix and obtain the current one via
% interpolation

% load the blocks of the sampled Ar matrix
A1Sampled = sampledMatrices.A1 ;
A2Sampled = sampledMatrices.A2 ;
% obtain the current Ar matrix blocks via nearest neighbor interpolation
A1 = weight*A1Sampled(:,:,intervalIdx)+(1-weight)*A1Sampled(:,:,intervalIdx+1) ;
A2 = Dalpha'*(weight*A2Sampled(:,:,intervalIdx)+(1-weight)*A2Sampled(:,:,intervalIdx+1)) ;
% assemble the Ar matrix based on its blocks
Ar = [A1,zeros(totalNModes,nFrames); A2,zeros(nFrames)] ;

% load the samplings of the coefficient matrices which are needed to 
% evaluate the nonlinear part of the ROM and obtain the current one via 
% interpolation

% get the sampled matrices from the sampledMatrices struct
PTPhiSampled = sampledMatrices.PTPhiSampled ;
F1PTPsiInvSampled = sampledMatrices.F1PTPsiInverse ;
F2PTPsiInvSampled = sampledMatrices.F2PTPsiInverse ;
% obtain the current matrices via nearest neighbor interpolation
PTPhi = weight*PTPhiSampled(:,:,intervalIdx)+(1-weight)*PTPhiSampled(:,:,intervalIdx+1) ;
F1PTPsiInv = weight*F1PTPsiInvSampled(:,:,intervalIdx)+(1-weight)*F1PTPsiInvSampled(:,:,intervalIdx+1) ;
F2PTPsiInv = Dalpha'*(weight*F2PTPsiInvSampled(:,:,intervalIdx)+(1-weight)*F2PTPsiInvSampled(:,:,intervalIdx+1)) ;
% assemble the leading matrix of the ROM nonlinearity based on its blocks
FPTPsiInverse = [F1PTPsiInv; F2PTPsiInv] ;

%% evaluate the ROM nonlinearity and compute the ROM right-hand side

% get current temperature and supply mass fraction approximations evaluated 
% at the DEIM points

% initialization
Tred = zeros(totalNNonlinearityModes, 1) ; % temperature approximation
Sred = zeros(totalNNonlinearityModes, 1) ; % supply mass fraction approximation
% add the contribution from the shifted POD modes
for i=1:nFrames
    currentModeIndices_T = helper(i)+(1:nModes(i)/2) ;
    currentModeIndices_S = helper(i)+((nModes(i)/2+1):nModes(i)) ;
    Tred = Tred+PTPhi(:,currentModeIndices_T)*alpha(currentModeIndices_T) ;
    Sred = Sred+PTPhi(:,currentModeIndices_S)*alpha(currentModeIndices_S) ;
end
% add the contributions from the POD modes
currentModeIndices_T = totalNShiftedModes+(1:nPODModesVec(1)) ;
currentModeIndices_S = totalNShiftedModes+(nPODModesVec(1)+1:nPODModes) ;
Tred = Tred+PTPhi(:,currentModeIndices_T)*alpha(currentModeIndices_T) ;
Sred = Sred+PTPhi(:,currentModeIndices_S)*alpha(currentModeIndices_S) ;

% evalute the reduced nonlinearity
tempRHS = evaluateWildlandFireNonlinearity([Tred;Sred], params.beta) ;

% tempRHS consist of two column vectors stacked on top of each other who
% differ only by a constant factor; since the pre-factor of the first half
% (alpha) is bigger than the pre-factor of the second half, the DEIM
% algorithm automatically only selects rows in the first half; these
% selected rows are in the following written into the variable betaRhs
betaRhs = tempRHS(1:end/2) ; 

% compute the right-hand side of the ROM
rhs = M\(Ar*current+FPTPsiInverse*betaRhs) ;