function [sPODSol, redState, simTime] = shifted_pod_online_struct_array(model, discretization, offline, modes, sampledMatrices, hyper, varargin)
% shifted_pod_online_struct_array - computes the shifted POD ROM solution
% for all parameter values specified in the struct 'model'
%
% inputs:
% - model: struct array containing ROM settings like the Arrhenius 
% coefficient to be used
% - discretization: struct array containing discretization parameters
% - offline: struct containing parameters for the mode decomposition
% - modes: transformed modes determined in the offline phase
% - sampledMatrices: struct containing the sampled path-dependent
% coefficient matrices
% - hyper: struct containing hyperreduction parameters
% - varargin: this function allows 3 additional input arguments which
% only apply when the time interval is split:
%       - varargin{1}: struct array containing the precomputed coefficient
%       matrices of the POD-DEIM-ROM for area (i)
%       - varargin{2}: POD modes for area (i)
%       - varargin{3}: DEIM modes for area (i)
%
% outputs:
% - sPODSol: cell array containing the approximation based on the ROM for
% each parameter value
% - redState: cell array containing the coefficients and the paths for each
% parameter value
% - simTime: vector containing the run time for each parameter value
%
% dependencies:
% - shifted_pod_online_ode45 (in LIB/SPOD_ONLINE)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% number of ROM simulations to be performed (coincides with the number of
% parameter values)
n_simulations = length(model);

% initialization of output variables
sPODSol = cell(1, n_simulations);
redState = cell(1, n_simulations);
simTime = zeros(1, n_simulations);

% simulate ROM for each parameter value
for i = 1:n_simulations
    % simulate ROM for current parameter value
    switch nargin
        case 6
            [sPODSol{i}, redState{i}, simTime(i)] = shifted_pod_online_ode45(modes, discretization(i), offline, model(i).parameter, model(i).initial_condition, sampledMatrices, hyper) ;
        case 9
            [sPODSol{i}, redState{i}, simTime(i)] = shifted_pod_online_ode45(modes, discretization(i), offline, model(i).parameter, model(i).initial_condition, sampledMatrices, hyper, varargin{1}, varargin{2}, varargin{3}) ;
        otherwise
            error('The number of input arguments for the function shifted_pod_online_struct_array needs to be 7 or 10.')
    end
end