function [sPODSol, redState, simTime] = shifted_pod_online_ode45(modes, discretization, offline, params, initial_condition, sampledMatrices, hyper, varargin)
% shifted_pod_online_ode45 - computes the shifted POD ROM solution
% for a given parameter value; ode45 is used for the time integration
%
% inputs:
% - modes: transformed modes determined in the offline phase
% - discretization: struct containing discretization parameters
% - offline: struct containing settings for the mode decomposition
% - params: struct containing the values of some of the PDE coefficients
% - initial_condition: cell array containing function handles for the
% initial values of the temperature and of the supply mass fraction
% - sampledMatrices: struct array containing the sampled path-dependent
% coefficient matrices
% - hyper: struct containing hyperreduction parameters
% - varargin: this function allows 3 additional input arguments which
% only apply when the time interval is split:
%       - varargin{1}: struct array containing the precomputed coefficient
%       matrices of the POD-DEIM-ROM for area (i)
%       - varargin{2}: POD modes
%       - varargin{3}: DEIM modes
%
% outputs:
% - sPODSol: matrix containing the approximation based on the ROM
% - redState: matrix containing the coefficients and the paths
% - simTime: run time
%
% dependencies:
% - trackMaximumNegativeSlope (in LIB/SHIFT)
% - trackMaximumSlope (in LIB/SHIFT)
% - wildlandFirePODROMRHS_hyperreduced (in LIB/POD)
% - wildlandFireROMRHS_hyperreduced (in LIB/SPOD_ONLINE)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% initializations and set-up

% boolean useSwitchedROM determines whether the time interval is split or
% not; this is the case if and only if POD modes are provided
useSwitchedROM = false ;

if(nargin==10) % get additional input arguments if there are any
    useSwitchedROM = true ;
    PODROMMatrices = varargin{1} ;
    PODModes = varargin{2} ;
    FPODModes = varargin{3} ;
    % number of POD modes in area (ii)
    nPODModesAfterSwitch = size(PODModes{2}, 2) ;
    % time point between areas (i) and (ii)
    switchTime = offline.switchTime ;
end

% the wildland fire model has two physical variables: temperature and
% supply mass fraction
n_vars = 2;

% get settings from the offline struct
n_modes = offline.n_modes; % number of shifted modes for each traveling wave
nFrames = offline.n_frames; % number of traveling waves
shift_matrix_generator = offline.shift_matrix_generator; % function for creating the shift matrices
n_total_modes = sum(n_modes); % total number of shifted modes

% get settings from the discretization struct
length_space = discretization.space.length; % length of computational domain
n_steps_space_per_var = size(modes, 1) / n_vars; % number of steps in space
dx = length_space / n_steps_space_per_var; % spatial mesh width
space = dx * (1:n_steps_space_per_var); % vector of discret space points

beta = params.beta ; % Arrhenius coefficient

% helper vector for solution build up later in this method
offset_vector = [0, cumsum(n_modes)];
n_rows = n_vars * n_steps_space_per_var; % number of FOM unknowns per time step

%% Initial Condition

% evaluate initial values at discret space points
u_0 = [initial_condition{1}(space, length_space), initial_condition{2}(space, length_space)];

% initial condition for the ROM
if(useSwitchedROM)
    % in the first time interval (area (i)), only POD modes are used
    allModes = PODModes{1} ;
    % initial condition is obtained via orthogonal projection of the FOM
    % initial value
    redState = allModes'*u_0' ;
else
    % ROM state consists of coefficients and paths
    redState = zeros(n_total_modes + nFrames, 1);
    % in this case there are only transformed modes
    allModes = modes ;
    % initial condition is obtained via orthogonal projection of the FOM
    % initila value
    redState(1:end-nFrames) = (allModes' * allModes) \ (allModes' * u_0');
    % the initial condition for the paths is zero
end


%% Time Integration

opts = odeset('Refine', 1); % ode45 options

fprintf('- shifted POD ROM simulation (beta = %f) ...', beta);

if(useSwitchedROM) % if time interval is split into the areas (i) and (ii)
    % vector of discrete time points for area (i)
    tsize1 = 0:discretization.time.dt:switchTime ;
    tic
    % right-hand side of the ROM for area (i)
    f = @(t,y)wildlandFirePODROMRHS_hyperreduced(t,y,params,PODROMMatrices,FPODModes{1},offline.nPODModesBeforeSwitch) ; 
    % time integration for the first time interval (area (i))
    [timeSteps1, redState1] = ode45(f, tsize1, redState, opts) ;
    % compute the current approximation of the FOM state
    intermediateFOMState = PODModes{1}*redState1(end,:)' ;
    % initialize the ROM initial condition for area (ii)
    newIC = NaN(n_total_modes+nPODModesAfterSwitch+nFrames, 1) ;
    % determine space index of left-going wave at current time point
    shift1 = trackMaximumSlope(intermediateFOMState(1:n_steps_space_per_var)) ;
    % determine space index of right-going wave at current time point
    shift2 = trackMaximumNegativeSlope(intermediateFOMState(1:n_steps_space_per_var)) ;
    % determine space index of left-going wave at initial time point
    shift01 = trackMaximumSlope(u_0(1:n_steps_space_per_var)') ;
    % determine space index of right-going wave at initial time point
    shift02 = trackMaximumNegativeSlope(u_0(1:n_steps_space_per_var)') ;
    % the initial paths for area (ii) are given by the difference between
    % the wave positions at the current time point and the wave positions
    % at the initial time points
    newIC(end-nFrames+1:end) = dx*[shift1-shift01;shift2-shift02] ; 
    % create shift matrices based on the current path values
    shiftMatrices1 = shift_matrix_generator(discretization, -newIC(end-1)) ;
    shiftMatrices2 = shift_matrix_generator(discretization, -newIC(end)) ;
    % shift matrices are duplicated on the block diagonal to be able to shift
    % vectors containing more than one physical variable
    shiftMatrix1 = kron(eye(nFrames), shiftMatrices1{1}) ;
    shiftMatrix2 = kron(eye(nFrames), shiftMatrices2{1}) ;
    % determine the current shifted modes
    shiftedModes = [shiftMatrix1*modes(:,1:end/2) shiftMatrix2*modes(:,end/2+1:end)] ;
    % add the POD modes to collect all modes
    allModes = [shiftedModes PODModes{2}] ;
    % determine the initial condition of the coefficients via orthogonal
    % projection of the current approximate FOM state onto the span of all
    % modes
    newIC(1:end-nFrames) = (allModes' * allModes) \ (allModes' * intermediateFOMState); 
    % vector of discrete time points for area (ii)
    tsize2 = timeSteps1(end):discretization.time.dt:discretization.time.length ;
    % right-hand side of the ROM for area (ii)
    f = @(t,y)wildlandFireROMRHS_hyperreduced(t,y,offline,params,sampledMatrices,hyper,PODModes{2},FPODModes{2}) ;
    % time integration for the second time interval (area (ii))
    [~, redState2] = ode45(f, tsize2, newIC, opts) ;
else % if time interval is not split
    % vector of discrete time points
    tsize = discretization.time.dt * (0:1:discretization.time.n_steps);
    tic
    % right-hand side of the ROM
    f = @(t,y)wildlandFireROMRHS_hyperreduced(t,y,offline,params,sampledMatrices,hyper) ;
    % time integration
    [~, redStateTmp] = ode45(f, tsize, redState, opts) ;
end

simTime = toc; % run time for the time integration

fprintf('... done (%f sec)\n',simTime);

%% Build up of FOM state approximation based on the ROM solution

if(useSwitchedROM) % if time interval is split into the areas (i) and (ii)
    % assemble reduced state as cell array (one entry for area (i), one
    % for area (ii))
    redState = cell(2, 1) ;
    redState{1} = redState1' ;
    redState{2} = redState2' ;
    % total number of time steps
    nTimeSteps = [size(redState{1},2) size(redState{2},2)] ;
    % initialize approximation
    sPODSol = zeros(n_rows, sum(nTimeSteps)-1);
    sPODSol(:,1:nTimeSteps(1)) = PODModes{1}*redState{1} ; % approximation in area (i)
    % part of the approximation in area (ii) corresponding to the
    % contribution of the POD modes
    sPODSol(:,nTimeSteps(1)+1:end) = PODModes{2}*redState{2}(n_total_modes+(1:nPODModesAfterSwitch),2:end) ;
    % part of the approximation in area (ii) corresponding to the
    % contribution of the shifted POD modes
    for i=2:nTimeSteps(2)
        for j=1:nFrames
            current_shift = redState{2}(end-nFrames+j, i); % current path value
            % corresponding shift matrix
            current_shift_matrices = shift_matrix_generator(discretization, -current_shift);
            current_shift_matrix = kron(eye(nFrames), current_shift_matrices{1});
            lower_index = offset_vector(j) + 1;
            upper_index = offset_vector(j + 1);
            % current shifted POD modes
            shiftedModes = current_shift_matrix*modes(:,lower_index:upper_index) ;
            sPODSol(:,nTimeSteps(1)+i-1) = sPODSol(:,nTimeSteps(1)+i-1)+shiftedModes*redState{2}(lower_index:upper_index,i) ;
        end
    end
else % if time interval is not split
    % assemble reduced state as cell array with one entry (since time
    % interval is considered as a whole)
    redState = cell(1) ;
    redState{1} = redStateTmp' ;
    % number of time steps
    nTimeSteps = size(redState{1}, 2) ;
    % initialize approximation
    sPODSol = zeros(n_rows, nTimeSteps) ;
    for i=1:nTimeSteps
        for j=1:nFrames
            current_shift = redState{1}(end-nFrames+j, i); % current path value
            % corresponding shift matrix
            current_shift_matrices = shift_matrix_generator(discretization, -current_shift);
            current_shift_matrix = kron(eye(nFrames), current_shift_matrices{1});
            lower_index = offset_vector(j) + 1;
            upper_index = offset_vector(j + 1);
            % current shifted POD modes
            shiftedModes = current_shift_matrix*modes(:,lower_index:upper_index) ;
            sPODSol(:,i) = sPODSol(:,i)+shiftedModes*redState{1}(lower_index:upper_index,i) ;
        end
    end
end