function D = createGeneralFDMmatrix(order, n_steps_space, dx)
% createGeneralFDMmatrix - creates a finite difference matrix for the first
% spatial derivative; the boundary is treated via one-sided finite
% differences such that this function is independent of any boundary 
% conditions
%
% inputs:
% - order: order of the finite difference scheme
% - n_steps_space: number of steps in space
% - dx: spatial mesh width
%
% outputs:
% - D: finite difference matrix for the 1st derivative
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% column vectors of all ones
enm2 = ones(n_steps_space-2, 1);
enm4 = ones(n_steps_space-4, 1) ;
enm6 = ones(n_steps_space-6, 1) ;

% column vectors of all zeros
z4 = zeros(4, 1) ;
z6 = zeros(6, 1) ;
znm2 = 0*enm2 ;

% determine the diagonal entries 'diagonals_D' and the corresponding
% diagonal indices 'indices' based on the specified order
switch order
    case 2
        diagonals_D = 0.5*[[-enm2;-2;0],[-2;znm2;2],[0;2;enm2]] ;
        indices = [-1 0 1];
    case 4
        diagonals_D = (1/12)*[[enm4;z4],[-6;-8*enm4;-6;-12;0],[-12;znm2;12],[0;12;6;8*enm4;6],[z4;-enm4]] ;
        indices = [-2 -1 0 1 2];
    case 6
        diag3 = [-enm6; z6] ;
        diag2 = [5; 9*enm6; 5; z4] ;
        diag1 = [-30; -40; -45*enm6; -40; -30; -60; 0] ;
        diag0 = [-60; znm2; 60] ;
        diagonals_D = (1/60)*[diag3,diag2,diag1,diag0,-flipud(diag1),-flipud(diag2),-flipud(diag3)] ;
        indices = [-3 -2 -1 0 1 2 3];
    otherwise
        error("Order of accuracy %i is not supported.", order);
end

% assemble the ouput matrix
D = spdiags(diagonals_D, indices, n_steps_space, n_steps_space);
D = D * (1/dx);