function nonlinearity = evaluateWildlandFireNonlinearity(u, beta, alpha, gammaS)
% evaluateWildlandFireNonlinearity - evaluates the nonlinear part of the 
% wildland fire model
%
% inputs:
% - u: current value of the state ([temperature; supply mass fraction])
% - beta: Arrhenius coefficient
% - alpha: temperature rise per second
% - gammaS: pre-exponential factor
%
% outputs:
% - nonlinearity: vector containing the nonlinear part of the wildfire
% model
%
% dependencies:
% - evaluateWildlandFireNonlinearityHelper (in LIB/HELPER)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

T0 = 0 ; % temperature of the environment
gamma = 4.8372e-05 ; % heat transfer coefficient

if(nargin<=2)
    alpha = 1.8793e02 ; % temperature rise per second
    gammaS = 1.6250e-01 ; % pre-exponential factor
end

Nxi = round(0.5*length(u)) ; % number of steps in space
T = u(1:Nxi) ; % current temperature
S = u(Nxi+1:end) ; % current supply mass fraction

% compute reaction rate
reactionRate = S.*evaluateWildlandFireNonlinearityHelper(T, beta) ; 

% nonlinear part of right-hand side of wildfire model
nonlinearity = [alpha*(reactionRate+gamma*T0); -gammaS*reactionRate] ;