function [D, DD] = create_finite_difference_matrix(order, n_steps_space, dx)
% create_finite_difference_matrix - returns finite difference matrices D 
% and DD for the first and second spatial derivative.
%
% inputs:
% - order: The order of the scheme; allowed values: 2, 4, 6
% - n_steps_space: The number of spatial steps.
% - dx: spatial step size.
%
% outputs:
% - D: finite difference matrix for the 1st derivative
% - DD: finite difference matrix for the 2nd derivative
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% diagonals_D: diagonal entries occurring in D
% diagonals_DD: diagonal entries occurring in DD
% indices: indices of the corresponding diagonals
switch order
    case 2
        diagonals_D = ones(n_steps_space, 5) .* [(1/2) -(1/2) 0 (1/2) (-1/2)];
        diagonals_DD = ones(n_steps_space, 5) .* [1 1 -2 1 1];
        indices = [-(n_steps_space-1) -1 0 1 n_steps_space-1];

    case 4
        diagonals_D = ones(n_steps_space, 9) .* [(2/3) -(1/12) (1/12) -(2/3) 0 (2/3) -(1/12) (1/12) -(2/3)];
        diagonals_DD = ones(n_steps_space, 9) .* [(4/3) -(1/12) -(1/12) (4/3) -(5/2) (4/3) -(1/12) -(1/12) (4/3)];
        indices = [-(n_steps_space-1) -(n_steps_space-2) -2 -1 0 1 2 n_steps_space-2 n_steps_space-1];

    case 6
        diagonals_D = ones(n_steps_space, 13) .* [(3/4) -(3/20) (1/60) -(1/60) (3/20) -(3/4) 0 (3/4) -(3/20) (1/60) -(1/60) (3/20) -(3/4)];
        diagonals_DD = ones(n_steps_space, 13) .* [(3/2) -(3/20) (1/90) (1/90) -(3/20) (3/2) -(49/18) (3/2) -(3/20) (1/90) (1/90) -(3/20) (3/2)];
        indices = [-(n_steps_space-1) -(n_steps_space-2) -(n_steps_space-3) -3 -2 -1 0 1 2 3 n_steps_space-3 n_steps_space-2 n_steps_space-1];

    otherwise
        error("Order of accuracy %i is not supported.", order);

end

% assemble the matrices
D = spdiags(diagonals_D, indices, n_steps_space, n_steps_space);
DD = spdiags(diagonals_DD, indices, n_steps_space, n_steps_space);
D = D * (1/dx);
DD = DD * (1/(dx^2));