function [snapshot_matrix, computation_time, nonlinearitySnapshots, wildlandFireMatrix] = compute_truth_solution_struct_array(fom, discretization)
% compute_truth_solution_struct_array - computes the full order solution
% for every set of parameters contained in the structure arrays fom and
% discretization. Before simulating, it is checked if the simulation
% results are already present in the DATA folder. If not, the simulation
% results are automatically saved after the simulation.
%
% inputs:
% - fom: struct array containing parameters for the FOM
% - discretization: struct array containing discretization parameters
%
% outputs:
% - snapshot_matrix: cell array containing snapshot matrix of FOM solution
% for each parameter value
% - computation_time: array containing run time for each parameter value
% - nonlinearitySnapshots: cell array containing snapshot matrix of FOM 
% nonlinearity for each parameter value
% - wildlandFireMatrix: matrix representing the linear part of the wildland
% fire model
%
% dependencies:
% - create_finite_difference_matrix (in LIB/SIMULATION)
% - compute_truth_solution_ode45 (in LIB/SIMULATION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------


    %% setup

    % folders specifying the location of the FOM data
    DATAfolder = 'DATA';
    FOMfolder = 'FOM';
    FOMdataFolder = sprintf('%s/%s',DATAfolder,FOMfolder);
    % the overview file contains a list specifying which FOM simulations are
    % already stored in the FOM folder
    overviewFile = sprintf('%s/computedFOMdataOverview.mat',FOMdataFolder);

    if ~exist(DATAfolder,'dir')
        mkdir(DATAfolder) ;
    end
    if ~exist(FOMdataFolder,'dir')
        mkdir(FOMdataFolder) ;
    end

    % initialization
    snapshot_matrix = cell(1,length(fom));
    nonlinearitySnapshots = cell(1,length(fom));
    computation_time = NaN(1, length(fom));

    % get information which FOM data are already available
    precomputedData = loadComputedFOMoverview(overviewFile,FOMdataFolder);

    % create finite difference matrices for 1st and 2nd derivative
    [D, DD] = create_finite_difference_matrix(discretization(1).order, discretization(1).space.n_steps, discretization(1).space.dx);

    %% computations

    for i=1:length(fom)
        % create struct containing parameters of the current FOM
        info = struct();
        info.initial_condition_name = fom(i).initial_condition_name;
        info.beta = fom(i).parameter.beta; % Arrhenius coefficient
        info.v = fom(i).parameter.v; % wind velocity
        info.spaceOrder = discretization(i).order; % order of finite difference scheme
        info.boundary_condition = discretization(i).boundary_condition;
        info.spaceDim = discretization(i).space.n_steps;
        info.timeDim = discretization(i).time.n_steps;

        % check if data for the current FOM is already available
        [precomputed,fileName] = isPrecomputed(precomputedData,info);
        if precomputed % if FOM data is already available
            fprintf('Use precomputed data (IC = %s, dim = %d, spaceOrder = %d, beta = %f, v = %f).\n',info.initial_condition_name,info.spaceDim,info.spaceOrder,info.beta,info.v);
            % load already computed FOM data
            data = load(fileName); 
            snapshotMatrix = data.snapshotMatrix;
            computationTime = data.computationTime;
            nonlinSnapshots = data.nonlinSnapshots;
            wildlandFireMatrix = data.wildlandFireMatrix;
        else
            % simulate current FOM
            [snapshotMatrix, computationTime, nonlinSnapshots, wildlandFireMatrix] = compute_truth_solution_ode45(fom(i), discretization(i), D, DD) ;

            % save FOM data to a file
            fileName = sprintf('%s/FOM-n%d-%d-%02d%02d%03d.mat',FOMdataFolder,info.spaceDim,yyyymmdd(datetime),hour(datetime),minute(datetime),floor(second(datetime)*10));
            save(fileName,'info','snapshotMatrix','computationTime','nonlinSnapshots','wildlandFireMatrix');
            fprintf(' -> save computed data in the file %s\n\n',fileName);

            % update precompute data cell
            dataFOM = struct('info',info,'fileName',fileName);
            precomputedData{end+1} = dataFOM;
        end
        % write results for current FOM in output arrays 
        snapshot_matrix{i} = snapshotMatrix;
        computation_time(i) = computationTime;
        nonlinearitySnapshots{i} = nonlinSnapshots;
    end

    % update overview file
    overview = precomputedData;
    save(overviewFile,'overview');
end

function overview = loadComputedFOMoverview(overviewFile,folder)
    if ~exist(overviewFile,'file')
        % create overview file in case it does not already exist
        overview = cell(1);
        overviewIndex = 1; % counter for the available data sets
        % load existing mat files
        data = dir(folder); % list with files in the specified folder
        for i=1:length(data) % loop over the available files
            fileName = data(i).name; % current file name
            if length(fileName)>5 % consider only file names with more than five characters
                if strcmp(fileName(1:5),'FOM-n') % consider only file names which fit to the pattern
                    FOMdata = load(sprintf('%s/%s',folder,fileName),'info'); % load data from current file
                    dataFOM = struct('info',FOMdata.info,'fileName',sprintf('%s/%s',folder,fileName)); % create struct for current file
                    overview{overviewIndex} = dataFOM; % add struct to overview
                    overviewIndex = overviewIndex + 1;
                end
            end
        end
    else % if overview already exists
        % load overview file
        data = load(overviewFile); 
        overview = data.overview;
    end
end

function [bol,fName] = isPrecomputed(overviewData,FOMinfo)
    % check if data for FOM specified by FOMinfo is already available
    
    % initialization
    bol = false; % output boolean (specifies if data is available or not)
    fName = NaN; % file name of FOM data (if available)
    for i=1:length(overviewData) % go through the overview file
        if isfield(overviewData{i},'info')
            if isequaln(overviewData{i}.info,FOMinfo)
                % if info structs match: set boolean to true and get the
                % file name
                bol = true;
                fName = overviewData{i}.fileName;
                break;
            end
        end
    end
end
