function [snapshot_matrix, computation_time, nonlinearitySnapshots, wildlandFireMatrix] = compute_truth_solution_ode45(fom, discretization, D, DD)
% compute_truth_solution_ode45 - computes the full order solution based on 
% Matlab's ode45 solver based on the parameters in the
% structures fom and discretization.
%
% inputs:
% - fom: struct containing parameters for the FOM
% - discretization: struct containing discretization parameters
% - D: finite difference matrix for the 1st derivative
% - DD: finite difference matrix for the 2nd derivative 
%
% outputs:
% - snapshot_matrix: snapshot matrix of FOM solution
% - computation_time: run time
% - nonlinearitySnapshots: snapshot matrix of FOM nonlinearity
% - wildlandFireMatrix: matrix representing the linear part of the wildland
% fire model
%
% dependencies:
% - evaluate_wildlandFire_right_hand_side (in LIB/SIMULATION)
% - evaluateWildlandFireNonlinearity (in LIB/SIMULATION)
% - wildlandFireLinearPart_matrix (in LIB/SIMULATION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% destructure the input structures

% fom
initial_condition = fom.initial_condition;

% discretization
dx = discretization.space.dx; % spatial mesh width
n_steps_space = discretization.space.n_steps; % number of points in space
L = discretization.space.length; % length of computational domain

% array of spatial points
space = dx * (1:n_steps_space);

% read the problem parameters from the fom struct
u_0 = [initial_condition{1}(space, L), initial_condition{2}(space, L)]; % initial condition
% PDE coefficients
beta = fom.parameter.beta;
alpha = fom.parameter.alpha;
gammaS = fom.parameter.gammaS;
v = fom.parameter.v;

% initialize snapshot matrix
snapshot_matrix = u_0';

% Time integration

% array of discrete time points
tsize = discretization.time.dt * (0:1:discretization.time.n_steps);

opts = odeset('Refine', 1); % ode45 options

fprintf('Starting computation of full order model for wildland fire model with solver ode45\n and parameters beta = %f\n on time domain [0,%f]\n and spatial domain [0,%f] discretized with %i grid points  ...', beta, discretization.time.length, discretization.space.length, discretization.space.n_steps);

tic;
% get the matrix for the linear part of the wildfire model
wildlandFireMatrix = wildlandFireLinearPart_matrix(D, DD, v) ;
% perform the time integration
[time_steps, snapshot_matrix] = ode45(@(t, u) evaluate_wildlandFire_right_hand_side(t, u, wildlandFireMatrix, beta, alpha, gammaS), tsize, snapshot_matrix, opts);
computation_time = toc; % get run time

fprintf('... done!\n');

% Since ode solver solutions are time by space instead of space by time, transpose to make it space by time.
snapshot_matrix = snapshot_matrix';

if(nargout>2) % if required, output the nonlinearity snapshots
    nTimeSteps = length(time_steps) ; % number of time steps
    % initialization of nonlinearity snapshot matrix
    nonlinearitySnapshots = NaN(size(snapshot_matrix,1)/2, nTimeSteps) ;
    for i=1:nTimeSteps
        % compute current nonlinearity snapshot based on the current state
        % snapshot
        tempRhs = evaluateWildlandFireNonlinearity(snapshot_matrix(:,i), beta) ;
        % since 1st and 2nd half of nonlinearity only differ by a constant
        % factor, we only need to save the 1st half
        nonlinearitySnapshots(:,i) = tempRhs(1:end/2,:) ;
    end
end