function shifted_matrix = shift_matrix(shift_matrices, matrix)
% shift_matrix - shifts each column of the snapshot matrix 'matrix' 
% by multiplication from the left by the shift_matrices.
%
% inputs:
% - shift_matrices: cell array containing shift matrices for each time step
% - matrix: snapshot matrix to be shifted
% - negative_velocity: boolean specifying if negative shift should be
% applied; optional argument; default value: false
%
% outputs:
% - shifted_matrix: the resulting shifted snapshot matrix; saved as a 3D
% array, where the second dimension corresponds to the number of physical
% variables
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% dimensions of the snapshot matrix
size_of_matrix = size(matrix);

if(numel(size_of_matrix) == 2) % if 'matrix' is a 2D array...
    % ... reshape it to make it a 3D array to comply with the further 
    % structure of the program.
    matrix = reshape(matrix, size_of_matrix(1), 1, size_of_matrix(2));
    size_of_matrix = size(matrix);
end

n_vars = size_of_matrix(end-1); % number of physical variables (corresponds to number of PDEs on the infinite-dimensional level)
n_steps_time = size_of_matrix(end); % number of steps in time

% initialization of shifted snapshot matrix
shifted_matrix = NaN(size_of_matrix); 

if(isvector(matrix)) % case of only one snapshot
    shifted_matrix = shift_matrices{1}*matrix ; % perform shift
    return
end

% Loop over the time steps ...
for i = 1:n_steps_time
    % ... and the physical variables ...
    for j = 1:n_vars
        % ... and shift the corresponding matrix (tensor) entry by multiplying it with the correct shift matrix.
        shifted_matrix(:,j,i) = shift_matrices{i} * matrix(:,j,i); 
    end
end