function shiftMatrices = shiftMatrixGenerator_constExtrapolation(discretization, shift)
% shiftMatrixGenerator_constExtrapolation - constructs shift matrices where
% the function to be shifted is continued over the boundaries via a
% constant function
%
% inputs:
% - discretization: struct array containing discretization parameters
% - shift: vector containing paths for each time step
%
% outputs:
% - shiftMatrices: cell array containing shift matrix for each time step
%
% dependencies:
% - lagrange (in LIB/INTERPOLATION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% number of steps in space
Nxi = discretization.space.n_steps; 
dxi = discretization.space.dx; % spatial mesh width
nCs = length(shift) ; % number of time steps

% initialization
shiftMatrices = cell(nCs, 1) ;
diagonalIndexVec = NaN(1, 4) ; % vector containing the indices of the non-zero diagonals of the shift matrix

for i=1:nCs % loop over time steps
    shiftLength = shift(i) ; % current path value 
    % current indices of non-zero diagonals
    diagonalIndexVec(2) = floor(shiftLength/dxi) ;
    diagonalIndexVec(1) = diagonalIndexVec(2)-1 ;
    diagonalIndexVec(3) = diagonalIndexVec(2)+1 ;  
    diagonalIndexVec(4) = diagonalIndexVec(2)+2 ;
    % weight used for the interpolation
    dxiWeight = shiftLength/dxi-diagonalIndexVec(2) ;
    % computed Lagrange interpolation coefficients
    Lagrange = lagrange(dxiWeight) ;
    % entries of non-zero diagonals
    entries = repmat(Lagrange', Nxi, 1) ;
    % first, set up the shift matrix ignoring the constant extrapolation
    T = spdiags(entries, diagonalIndexVec, Nxi, Nxi) ;
    % take care of the constant extrapolation by adding and modifying 
    % entries
    T(1:min(-diagonalIndexVec(4)+1,Nxi),1) = 1 ;
    if((diagonalIndexVec(4)<=1)&&(diagonalIndexVec(4)>=-(Nxi-2)))
        T(-diagonalIndexVec(4)+2,1) = sum(Lagrange(1:3)) ;
    end
    if((diagonalIndexVec(4)<=2)&&(diagonalIndexVec(4)>=-(Nxi-3)))
        T(-diagonalIndexVec(4)+3,1) = sum(Lagrange(1:2)) ;
    end
    T(max(end-diagonalIndexVec(1),1):end,end) = 1 ;
    if((diagonalIndexVec(1)>=-1)&&(diagonalIndexVec(1)<=Nxi-2))
        T(end-diagonalIndexVec(1)-1,end) = sum(Lagrange(2:4)) ;
    end
    if((diagonalIndexVec(1)>=-2)&&(diagonalIndexVec(1)<=Nxi-3))
        T(end-diagonalIndexVec(1)-2,end) = sum(Lagrange(3:4)) ;
    end
    % write current shift matrix into the cell array
    shiftMatrices{i} = T ; 
end