function shift = determineTemperatureShift(snapshotData, dx, offset)
%determineTemperatureShift - compute two shifts for the temperature data
%via front tracking and possible (piecewise) linear approximation
%afterwards
%
% inputs:
% - snapshotData: snapshot matrix of FOM solution
% - dx: spatial mesh width
% - offset: offset used for the path post processing (linear interpolation 
% is performed for all time indices after this one)
%
% output:
% - shift: matrix containing paths for each traveling wave and for each
% time step
%
% dependencies:
% - trackMaximumSlope.m (in LIB/SHIFT)
% - trackMaximumNegativeSlope.m (in LIB/SHIFT)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

nTimesteps = size(snapshotData,2); % number of snapshots
% determine path of left-going wave via tracking the maximal spatial
% derivative
shift1 = trackMaximumSlope(snapshotData) ;
% determine path of right-going wave via tracking the minimal spatial
% derivative
shift2 = trackMaximumNegativeSlope(snapshotData) ;
% initial shift is subtracted to achieve a homogeneous initial condition
% for the paths
shift = dx*[shift1'-shift1(1),shift2'-shift2(1)] ;
% auxiliary variable
tmpShift = shift;

% replace parts of the paths via linear interpolation to make them smoother
offset = offset+1;
shiftNew = [interp1([offset,nTimesteps],[tmpShift(offset,1),tmpShift(end,1)],offset:nTimesteps)',interp1([offset,nTimesteps],[tmpShift(offset,2),tmpShift(end,2)],offset:nTimesteps)'];
if offset > 1
    shift = [tmpShift(1:offset-1,:); shiftNew];
else
    shift = shiftNew;
end