function rhs = wildlandFirePODROMRHS_hyperreduced(t, current, params, PODROMMatrices, FPODModes, nPODModesVec)
% wildlandFirePODROMRHS_hyperreduced - gives the right-hand side of the
% POD-DEIM ROM
%
% inputs:
% - t: dummy input to fit to the ode45 syntax
% - current: current value of the ROM state
% - params: struct containing the values of some of the PDE coefficients
% - PODROMMatrices: struct containing the pre-computed ROM coefficient 
% matrices
% - FPODModes: DEIM modes
% - nPODModesVec: number of POD modes for each physical variable
%
% outputs:
% - rhs: current value of the ROM right-hand side
%
% dependencies:
% - evaluateWildlandFireNonlinearity (in LIB/SIMULATION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% number of DEIM modes
nFPODModes = size(FPODModes, 2) ;
% auxiliary variable
rPODHelper = cumsum(nPODModesVec) ;
        
% reduced A matrix (corresponding to linear part of the FOM rhs)
Ar = PODROMMatrices.PODAMatrix ;

% PTPhi contains those entries of the POD modes which are needed to
% evaluate the hyperreduced nonlinearity in the ROM
PTPhi = PODROMMatrices.PTPhi ;
% PODDEIMMatrix is the leading matrix of the nonlinear term in the ROM
PODDEIMMatrix = PODROMMatrices.PODDEIMMatrix ;

% get current temperature and supply mass fraction approximations evaluated 
% at the DEIM points
Tred = zeros(nFPODModes, 1) ;
Sred = zeros(nFPODModes, 1) ;
modeIndices_T = 1:rPODHelper(1) ;
modeIndices_S = rPODHelper(1)+1:rPODHelper(2) ;
Tred = Tred+PTPhi(:,modeIndices_T)*current(modeIndices_T) ;
Sred = Sred+PTPhi(:,modeIndices_S)*current(modeIndices_S) ;

% evalute the reduced nonlinearity
tempRHS = evaluateWildlandFireNonlinearity([Tred;Sred], params.beta) ;

% tempRHS consist of two column vectors stacked on top of each other who
% differ only by a constant factor; since the pre-factor of the first half
% (alpha) is bigger than the pre-factor of the second half, the DEIM
% algorithm automatically only selects rows in the first half; these
% selected rows are in the following written into the variable betaRhs
betaRhs = tempRHS(1:end/2) ;

% compute the right-hand side of the ROM
rhs = Ar*current+PODDEIMMatrix*betaRhs ;