function [pod_solution, reduced_state_matrix, computation_time] = pod_online_ode45(modes, discretization, parameter, initial_condition, PODROMMatrices, FModes_POD)
% pod_online_ode45 - computes the POD ROM solution for a given parameter 
% value; ode45 is used for the time integration
%
% inputs:
% - modes: transformed modes determined in the offline phase
% - discretization: struct containing discretization parameters
% - parameter: struct containing the values of some of the PDE coefficients
% - initial_condition: cell array containing function handles for the
% - PODROMMatrices: struct array containing the precomputed coefficient
% matrices of the POD-DEIM-ROM
% - FModes_POD: DEIM modes
%
% outputs:
% - pod_solution: matrix containing the approximation based on the ROM
% - reduced_state_matrix: matrix containing the coefficients
% - computation_time: run time
%
% dependencies:
% - wildlandFirePODROMRHS_hyperreduced (in LIB/POD)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% initializations and set-up

% the wildland fire model has two physical variables: temperature and
% supply mass fraction
n_vars = 2;

n_total_modes = size(modes,2) ; % total number of POD modes
nPODModesPerVariable = 0.5*n_total_modes*[1,1] ; % number of POD modes per physical variable

% get settings from the discretization struct
length_space = discretization.space.length; % length of computational domain
n_steps_space_per_var = size(modes, 1) / n_vars; % number of steps in space
dx = length_space / n_steps_space_per_var; % spatial mesh width
space = dx * (1:n_steps_space_per_var); % vector of discret space points

beta = parameter.beta ; % Arrhenius coefficient

%% Initial Condition

% evaluate initial values at discret space points
u_0 = [initial_condition{1}(space, length_space), initial_condition{2}(space, length_space)];

% initial condition is obtained via orthogonal projection of the FOM
% initial value
redState = modes'*u_0' ;

%% Time Integration

opts = odeset('Refine', 1); % ode45 options

fprintf('- POD ROM simulation (beta = %f) ...', beta);

% vector of discrete time points
tsize = discretization.time.dt * (0:1:discretization.time.n_steps);
tic
% right-hand side of the ROM
f = @(t,y)wildlandFirePODROMRHS_hyperreduced(t,y,parameter,PODROMMatrices,FModes_POD,nPODModesPerVariable) ; 

% time integration
[~, redState] = ode45(f, tsize, redState, opts) ;

computation_time = toc; % run time for the time integration

fprintf('... done (%f sec)\n',computation_time);

%% Build up of FOM state approximation based on the ROM solution

% transpose redState to obtain a matrix with snapshots in the columns
reduced_state_matrix = redState' ;

% compute approximation
pod_solution = modes*reduced_state_matrix ;