function [mode_matrix, coefficient_matrix, relative_error] = POD_offline(snapshot_matrix, nModes, discretization)
% POD_offline_struct_array - performs mode decomposition via POD for each 
% parameter value
%
% inputs:
% - snapshot_matrix: cell array containing snapshot matrix of FOM solution
% for each parameter value
% - nModes: vector containing number of POD modes for each physical
% variable
% - discretization: struct array containing discretization parameters
%
% outputs:
% - mode_matrix: matrix containing the determined modes
% - coefficient_matrix: cell array containing the coefficients for each
% parameter value
% - relative_error: vector containing the relative approximation error for 
% each parameter value
%
% dependencies:
% - relative_L2_norm (in LIB/ERROR)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% initialization

n_param_samples = length(snapshot_matrix); % number of parameter values

n_steps_space = size(snapshot_matrix{1}, 1); % number of steps in space times number of physical variables
nStepsPerVar = discretization(1).space.n_steps; % number of steps in space
n_vars = n_steps_space / nStepsPerVar ; % number of physical variables

% number of steps in time for each parameter value
n_timesteps = NaN(1,n_param_samples); 
for i=1:n_param_samples
    n_timesteps(i) = discretization(i).time.n_steps ;
end

% initializations
modesPerVariable = cell(1, n_vars) ; % POD modes for each physical Variable
coeffsPerVariable = cell(1, n_vars) ; % POD coefficients for each physical Variable
qsTilde = NaN(n_steps_space, sum(n_timesteps+1)) ; % POD approximation of the snapshot matrix
% initialization of POD approximation for each parameter value
PODapproximation = cell(1,n_param_samples);

%% mode calculation

% total number of modes (counting modes for all physical variables)
totalNModes = sum(nModes) ;

% Perform the mode decomposition for each physical variable separately
for i=1:n_vars
    % assemble combined snapshot matrix for the parameters
    snapshotMatrixVar = NaN(nStepsPerVar,sum(n_timesteps+1));
    currentIndex = 1;
    for j=1:n_param_samples
        snapshotMatrixVar(:,currentIndex:sum(n_timesteps(1:j)+1)) = snapshot_matrix{j}((i-1)*nStepsPerVar+1:i*nStepsPerVar,:);
        currentIndex = sum(n_timesteps(1:j)+1)+1;
    end
    % perform the mode decomposition
    [modesPerVariable{i}, singular_value_matrix, right_singular_vector_matrix] = svds(snapshotMatrixVar, nModes(i)) ;
    coeffsPerVariable{i} = singular_value_matrix*right_singular_vector_matrix' ;
    % transfer values from cell array qsTildeTemp to matrix qsTilde
    qsTilde((i-1)*nStepsPerVar+1:i*nStepsPerVar,:) = modesPerVariable{i}*coeffsPerVariable{i} ;
end

% compute relative Errors
relative_error = NaN(1,n_param_samples); % total relative error per parameter value
currentIndex = 1;
for j=1:n_param_samples % loop over parameter samples
    PODapproximation{j} = qsTilde(:,currentIndex:sum(n_timesteps(1:j)+1)); % current POD approximation
    currentIndex = 1+sum(n_timesteps(1:j)+1);
    % compute relative L2 error for whole snapshot matrix
    relative_error(j) = relative_L2_norm(snapshot_matrix{j}, PODapproximation{j}-snapshot_matrix{j});
end

% collect the modes of all physical variables in one mode matrix
mode_matrix = zeros(n_steps_space, sum(nModes));
helper = [0, cumsum(nModes)];
for i = 1:n_vars
    mode_matrix((i-1)*nStepsPerVar+1:i*nStepsPerVar,helper(i)+1:helper(i+1)) = modesPerVariable{i};
end

% collect coefficients and paths for each parameter value
redStateMatrixTmp = NaN(totalNModes, sum(n_timesteps+1)) ;
for j=1:n_vars
    % coefficients for the transformed modes for each physical variable
    redStateMatrixTmp(helper(j)+1:helper(j+1),:) = coeffsPerVariable{j} ;
end
% transfer entries from matrix redStateMatrixTmp to cell array 
% coefficient_matrix
coefficient_matrix = cell(n_param_samples, 1) ;
currentIndex = 1;
for j=1:n_param_samples
    coefficient_matrix{j} = redStateMatrixTmp(:,currentIndex:sum(n_timesteps(1:j)+1)) ;
    currentIndex = 1+sum(n_timesteps(1:j)+1);
end