function plotLeftGoingSeparatedWave(discretization, offline, snapshotMatrix, shift)  
% plotLeftGoingSeparatedWave - plots only the left-going wave both in the
% lab frame and in the co-moving reference frame
%
% inputs:
% - discretization: struct containing the discretization parameters
% - offline: struct containing the parameters for the mode decomposition in
% the offline phase
% - snapshotMatrix: matrix containing snapshots of either temperature or
% supply mass fraction
% - shift: matrix containing the path values for each traveling wave
%
% dependencies:
% - init_shift_matrix (in LIB/SHIFT)
% - shift_matrix (in LIB/SHIFT)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% setup
n_steps_space = size(snapshotMatrix, 1); % number of steps in space
space_midpoint_ind = floor(n_steps_space/2) ; % index corresponding to middle of the computational domain
n_steps_time = size(snapshotMatrix, 2); % number of steps in time

shift_matrix_generator = offline.shift_matrix_generator; % function used for constructing the shift matrices
% construct the shift matrices for each time step
shiftedPODSystem = init_shift_matrix(discretization, shift_matrix_generator, shift);

%% plot left-going wave

% roughly split the snapshot matrix in the middle and fill the right half
% with zeros
frameSnapshotMatrix = zeros(size(snapshotMatrix));
frameSnapshotMatrix(1:space_midpoint_ind,:) = snapshotMatrix(1:space_midpoint_ind,:) ;

% plot the snapshots of the left-going wave in the lab frame
figure()
plotSpacePoints = (1:1:discretization.space.n_steps)*discretization.space.dx ; % space points used for the plot
plotTimePoints = (0:1:discretization.time.n_steps)*discretization.time.dt ; % time points used for the plot
f = pcolor(plotSpacePoints, plotTimePoints, frameSnapshotMatrix');
set(f,'EdgeColor','none');
colorbar
title('Left-going wave in lab frame');
xlabel('x[m]')
ylabel('t[s]')

% shift the snapshots of the left-going wave into the co-moving reference
% frame
shiftedFrameSnapshotMatrix = shift_matrix(shiftedPODSystem.shift_matrices{1,1}, frameSnapshotMatrix); 
% the output of shift_matrix is a 3D array and thus we have to reshape 
% before plotting
shiftedFrameSnapshotMatrix = reshape(shiftedFrameSnapshotMatrix,n_steps_space,n_steps_time) ;

% plot the snapshots of the left-going wave in the co-moving reference 
% frame
figure()
f = pcolor(plotSpacePoints, plotTimePoints, shiftedFrameSnapshotMatrix') ;
colorbar;
set(f,'EdgeColor','none');
title('Left-going wave in co-moving frame');
xlabel('x-p_1(t)[m]')
ylabel('t[s]')