function plotExtrapolatedVsProjectedCoefficients(discretization, offline, snapshotMatrix, shift)  
% plotExtrapolatedVsProjectedCoefficients - plots the coefficients
% corresponding to the transformed modes and compares the ones obtained via
% extrapolation and the ones obtained via orthogonal projection 
%
% inputs:
% - discretization: struct containing the discretization parameters
% - offline: struct containing the parameters for the mode decomposition in
% the offline phase
% - snapshotMatrix: matrix containing snapshots of either temperature or
% supply mass fraction
% - shift: matrix containing the path values for each traveling wave
%
% dependencies:
% - init_shift_matrix (in LIB/SHIFT)
% - shift_matrix (in LIB/SHIFT)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% setup

% general settings
n_steps_space = size(snapshotMatrix, 1) ; % number of rows of snapshot matrix
space_midpoint_ind = floor(n_steps_space/2) ; % index corresponding to middle of the computational domain
nTemperatureModes = 0.5*offline.n_modes ; % number of transformed modes for the temperature
n_params = length(offline.nTimestepsPerParam) ; % number of parameter values

shift_matrix_generator = offline.shift_matrix_generator ; % function used for creating the shift matrices

% method-specific options
switchTimeIndex = offline.switchTimeIndex ; % time index between areas (i) and (ii)

extrapolationPolOrder = offline.extrapolationPolOrder; % degree of the extrapolation polynomial
shiftedPODstart = offline.shiftedPODstart; % time point between areas (ii-a) and (ii-b)

% number of time steps in area (ii)
nTimestepsAfterSwitch   = (offline.nTimestepsPerParam+1-switchTimeIndex);
% first time index of area (ii-b)
firstSPODIndex          = floor(nTimestepsAfterSwitch*shiftedPODstart)+1;

% initialization and precomputations
paramTimeStepIndices = [0 offline.nTimestepsPerParam+1]; % add the zero to make computations easier later on
% time indices corresponding to area (i) (initialized as if there would be 
% only one parameter sample)
beforeSwitchIdx = 1:switchTimeIndex; 
% time indices corresponding to area (ii) (initialized as if there would be 
% only one parameter sample)
afterSwitchIdx = (switchTimeIndex+1):offline.nTimestepsPerParam(1)+1;
% indices corresponding to area (ii-b); these indices are relative to the 
% snapshot data after the switch
sPODindices = firstSPODIndex(1):nTimestepsAfterSwitch(1);
% sPODindicesParam is as sPODindices, but values are stored in a cell array
sPODindicesParam = cell(1,n_params);
sPODindicesParam{1} = sPODindices;
% add indices for the other parameter values
if n_params>1
    for k=2:n_params
        beforeSwitchIdx = [beforeSwitchIdx (1:switchTimeIndex)+sum(paramTimeStepIndices(1:k))];
        afterSwitchIdx = [afterSwitchIdx ((switchTimeIndex+1):(offline.nTimestepsPerParam(k)+1))+sum(paramTimeStepIndices(1:k))];
        sPODindicesParam{k} = (firstSPODIndex(k):nTimestepsAfterSwitch(k)) + sum(nTimestepsAfterSwitch(1:(k-1)));
        sPODindices = [sPODindices sPODindicesParam{k}];
    end
end

% snapshot data in area (ii)
dataAfterSwitch = snapshotMatrix(:,afterSwitchIdx);

% construct the shift matrices for each time step in area (ii)
shiftedPODSystem = init_shift_matrix(discretization, shift_matrix_generator, shift) ;


%% initialize mode and coefficients variables
modes = cell(1,1) ;
coefficients = cell(1,1); % coefficients obtained via extrapolation
coefficients2 = cell(1,1); % coefficients obtained via projection

%% compute approximation after switch per Frame

% roughly split the snapshot matrix in the middle and fill the right half
% with zeros
frameSnapshotMatrix = zeros(size(dataAfterSwitch));
frameSnapshotMatrix(1:space_midpoint_ind,:) = dataAfterSwitch(1:space_midpoint_ind,:) ;

% shift the snapshots of the into the co-moving reference frame
shiftedFrameSnapshotMatrix = shift_matrix(shiftedPODSystem.shift_matrices{1,1}, frameSnapshotMatrix); 
% the output of shift_matrix is a 3D array and thus we have to reshape 
% before plotting
shiftedFrameSnapshotMatrix = reshape(shiftedFrameSnapshotMatrix,n_steps_space,[]) ;

% compute modes via SVD of the snapshots in area (ii-b)
[modes{1},S,V] = svds(shiftedFrameSnapshotMatrix(:,sPODindices),nTemperatureModes(1));

% compute extrapolated coefficients.
% in area (ii-b), the coefficients are obtained from the SVD
% and in area (ii-a) the coefficients are obtained by
% extrapolating the coefficients determined for area (ii-b)
extrapolatedCoefficients = zeros(nTemperatureModes(1),size(shiftedFrameSnapshotMatrix,2)); % temporary variable for the coefficients
extrapolatedCoefficients(:,sPODindices) = S*V'; % coefficients in area (ii-b)
for k=1:n_params
    % time indices where coefficients are to be determined via
    % extrapolation
    if k>1
        extrapolationIdx = sPODindicesParam{k-1}(end):(sPODindicesParam{k}(1)-1);
    else
        extrapolationIdx = 1:(sPODindicesParam{k}(1)-1);
    end
    for j=1:nTemperatureModes(1) % loop over coefficients corresponding to left-going traveling wave
        % fit a polynomial based on the determined coefficient
        % values in area (ii-b)
        p = polyfit(sPODindicesParam{k},extrapolatedCoefficients(j,sPODindicesParam{k}),extrapolationPolOrder);
        extrapolatedCoefficients(j,extrapolationIdx) = polyval(p,extrapolationIdx); 
    end
end
coefficients{1} = extrapolatedCoefficients;

% compute projected coefficients
coefficients2{1} = modes{1}'*shiftedFrameSnapshotMatrix;

% create plot for comparing the two sets of coefficients

figure();
% time indices for area (ii)
timeLine = linspace(offline.switchTime,discretization(1).time.length,length(coefficients{1}));
% time between areas (ii-a) and (ii-b)
sPODstartTime = offline.switchTime+(discretization(1).time.length-offline.switchTime)*offline.shiftedPODstart(1);

% plot coefficients corresponding to first mode
subplot(2,1,1); 
plot(timeLine,coefficients{1}(1,:)'); % extrapolated coefficients
hold on;
set(gca,'ColorOrderIndex',1);
plot(timeLine,coefficients2{1}(1,:)','--'); % projected coefficients
tempYlim = ylim;
plot([sPODstartTime sPODstartTime],tempYlim,'k--'); % vertical line for indicating the border between areas (ii-a) and (ii-b)
xlim([timeLine(1),timeLine(end)]);
ylim(tempYlim);
hold off;
legend('first mode');
title('Comparison of projected (--) and extrapolated coefficients (-)')

% plot coefficients corresponding to second to last mode
subplot(2,1,2);
hold on;
set(gca,'ColorOrderIndex',2);
plot(timeLine,coefficients{1}(2:end,:)'); % extrapolated coefficients
set(gca,'ColorOrderIndex',2);
plot(timeLine,coefficients2{1}(2:end,:)','--'); % projected coefficients
tempYlim = ylim;
plot([sPODstartTime sPODstartTime],tempYlim,'k--'); % vertical line for indicating the border between areas (ii-a) and (ii-b)
xlim([timeLine(1),timeLine(end)]);
ylim(tempYlim);
xlabel('t[s]')