function [shiftedPODSystem, snapshotMatrixApprox] = subspace_decomposition_by_simpleDomainDivision(discretization, offline, snapshotMatrix, shift, options)  
% subspace_decomposition_by_simpleDomainDivision - computes a mode
% decomposition using shifted POD based on a simple domain division in the
% middle of the computational domain
%
% inputs:
% - discretization: struct array containing discretization parameters
% - offline: struct containing parameters for the mode decomposition
% - snapshotMatrix: snapshot matrix of FOM solution
% - shift: matrix containing path values for each time step and each
% traveling wave
% - options: dummy input to fit to the call in the parent function 
% 'shifted_pod_offline'
%
% outputs:
% - shiftedPODSystem: struct containing results of the mode decomposition
% such as the modes and the corresponding coefficients
% - snapshotMatrixApprox: approximation of the snapshot matrix
%
% dependencies:
% - init_shift_matrix (in LIB/SHIFT)
% - shift_matrix (in LIB/SHIFT)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------


%% setup

n_steps_space = size(snapshotMatrix, 1); % number of rows of snapshot matrix
space_midpoint_ind = floor(n_steps_space/2) ; % index corresponding to middle of the computational domain
n_steps_time = size(snapshotMatrix, 2); % number of snapshots
n_frames = 2 ; % number of traveling waves
n_modes = offline.n_modes; % number of transformed modes

shift_matrix_generator = offline.shift_matrix_generator; % function used for creating the shift matrices
% construct the shift matrices for each time step
shiftedPODSystem = init_shift_matrix(discretization, shift_matrix_generator, shift);
% construct the shift matrices shifting in the opposite direction for each time step
backShift = init_shift_matrix(discretization, shift_matrix_generator, -shift);
    
%% initialize result variables

modes = cell(n_frames,1); % modes per traveling wave
coefficients = cell(n_frames,1); % corresponding coefficients

% coefficients are also stored in shiftedPODSystem struct
shiftedPODSystem.alpha = NaN(sum(n_modes), n_steps_time) ;

% approximation of the snapshot matrix
snapshotMatrixApprox = zeros(size(snapshotMatrix));

%% compute approximation per Frame

for i=1:n_frames
    % roughly split the snapshot matrix in the middle and fill one half
    % with zeros
    frameSnapshotMatrix = zeros(size(snapshotMatrix));
    frameSnapshotMatrix((i-1)*space_midpoint_ind+1:i*space_midpoint_ind,:) = snapshotMatrix((i-1)*space_midpoint_ind+1:i*space_midpoint_ind,:) ;
    
    % shift the snapshots of the into the co-moving reference frame
    shiftedFrameSnapshotMatrix = shift_matrix(shiftedPODSystem.shift_matrices{i,1}, frameSnapshotMatrix);
    % the output of shift_matrix is a 3D array and thus we have to reshape 
    % before plotting
    shiftedFrameSnapshotMatrix = reshape(shiftedFrameSnapshotMatrix,n_steps_space,n_steps_time) ;
 
    % compute modes and coefficients via SVD
    [modes{i},~,~] = svds(shiftedFrameSnapshotMatrix,n_modes(i));
    coefficients{i} = modes{i}'*shiftedFrameSnapshotMatrix;
    % transfer coefficients into the shiftedPODSystem struct
    shiftedPODSystem.alpha(sum(n_modes(1:i-1))+1:sum(n_modes(1:i)),:) = coefficients{i} ;
    % approximation of the current traveling wave in the co-moving frame
    shiftedFrameApprox = modes{i}*coefficients{i} ;
    
    % shift approximation back into the lab frame
    frameApprox = shift_matrix(backShift.shift_matrices{i,1}, shiftedFrameApprox);
    frameApprox = reshape(frameApprox,n_steps_space,n_steps_time) ;
    
    % update the approximation of the complete snapshot matrix
    snapshotMatrixApprox = snapshotMatrixApprox + frameApprox ;
end

% transfer the determined modes into the shiftedPODSystem array
shiftedPODSystem.U = modes;