function [intervalIdx, alpha] = findIntervalAndGiveInterpolationWeight_1D(xPoints, xStar)
% findIntervalAndGiveInterpolationWeight_1D - given the sorted array of 
% sample points 'xPoints' in a 1D domain and the point 'xStar', this 
% function searches for the sub-interval where xStar is located in and
% gives the corresponding interpolation weight for a linear interpolant
% within this sub-interval
%
% inputs:
% - xPoints: sorted array of points in a 1D domain
% - xStar: query point, where the interpolant is to be evaluated
%
% outputs:
% - intervalIdx: index of the interval where xStar is located
% - alpha: interpolation weight based on linear interpolation
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% number of sample points
nPoints = length(xPoints) ;

% find the interval where xStar is located in
intervalIdx = find(xStar>=xPoints, 1, 'last') ;

if(isempty(intervalIdx)) % case when xStar is smaller than smallest value in xPoints
    warning('Constant Extrapolation is used instead of Interpolation.')
    % use constant extrapolation based on xPoints(1)
    intervalIdx = 1 ;
    alpha = 1 ;
else
    if(intervalIdx==nPoints) % case when xStar is bigger than biggest value in xPoints
        warning('Constant Extrapolation is used instead of Interpolation.')
        % use constant extrapolation based on xPoints(end)
        intervalIdx = nPoints-1 ;
        alpha = 0 ;
    else % case when xStar lies within the interval [xPoints(1) xPoints(end))
        % compute interpolation weight based on linear interpolation
        alpha = (xPoints(intervalIdx+1)-xStar)/(xPoints(intervalIdx+1)-xPoints(intervalIdx)) ;
    end
end