function sampledMatrices = sPODROMPreProcessing(modes, discretization, hyperreduction, FModes, offline, A, gammaSByAlpha, PODModes, FPODModes)
% sPODROMPreProcessing - samples the path-dependent coefficient matrices 
% occurring in the shifted-POD-shifted-DEIM ROM
%
% inputs:
% - modes: transformed modes determined in the offline phase
% - discretization: struct array containing discretization parameters
% - hyperreduction: struct containing hyperreduction parameters
% - FModes: transformed DEIM modes determined in the offline phase
% - offline: struct containing parameters for the mode decomposition
% - A: matrix for the linear part of the wildland fire model
% - gammaSByAlpha: ratio of pre-exponential factor and temperature rise per
% second
% - PODModes: matrix containing the POD modes; optional input argument only
% needed when there are POD modes in addition to the transformed modes
% - FPODModes: matrix containing the DEIM modes; optional input argument 
% only needed when there are POD modes in addition to the transformed modes
%
% outputs:
% - sampledMatrices: struct array containing the sampled path-dependent
% coefficient matrices
%
% dependencies:
% - createGeneralFDMmatrix (in LIB/SPOD_ONLINE)
% - iPOM (in LIB/HYPERREDUCTION)
% - QDEIM (in LIB/HYPERREDUCTION)
% - wIPOM (in LIB/HYPERREDUCTION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

%% initialization and set-up

% the following variables are introduced first:
% usePODModes: boolean which determines if there are POD modes in addition to the transformed modes
% mPOD: number of DEIM modes
% rPOD: total number of POD modes
if(nargin>7) % if POD and DEIM modes are provided
    usePODModes = true ;
    mPOD = size(FPODModes, 2) ;
    rPOD = size(PODModes, 2) ;
    % number of POD modes for each physical variable
    rPODVec = offline.nPODModesAfterSwitch ;
else
    usePODModes = false ;
    mPOD = 0 ;
    rPOD = 0 ;
    % if no POD or DEIM modes are provided, introduce them as empty arrays
    PODModes = [] ;
    FPODModes = [] ;
end

Nx = discretization.space.n_steps ; % number of steps in space
dx = discretization.space.dx ; % spatial mesh width
nModes = offline.n_modes ; % number of transformed modes
nNonlinearityModes = hyperreduction.n_modes ; % number of transformed DEIM modes

% function used for creating the shift matrices
shift_matrix_generator = offline.shift_matrix_generator ;
nRows = size(modes, 1) ; % number of entries per mode
nVars = nRows/Nx ; % number of physical variables

nFrames = size(nModes, 2) ; % number of traveling waves (needs to be 2)
m = size(FModes, 2) ; % number of transformed DEIM modes

% auxiliary variables
helper = cumsum([0,nModes]) ;

rSPOD = size(modes, 2) ; % number of transformed modes

% number of transformed modes for the 1st traveling wave
nModesInFirstFrame = nModes(1) ;
% corresponding mode indices
indices1stFrame = 1:nModesInFirstFrame ;
% number of transformed modes for the 2nd traveling wave
nModesInSecondFrame = nModes(2) ;
% corresponding mode indices
indices2ndFrame = nModesInFirstFrame+(1:nModesInSecondFrame) ;
% mode indices corresponding to the POD modes
indicesOfPODModes = nModesInFirstFrame+nModesInSecondFrame+(1:rPOD) ;
% transformed modes for the 1st traveling wave
modes1stFrame = modes(:,indices1stFrame) ;
% transformed modes for the 2nd traveling wave
modes2ndFrame = modes(:,indices2ndFrame) ;

% number of transformed DEIM modes for the 1st traveling wave
nNonlinearityModesInFirstFrame = nNonlinearityModes(1) ;
% corresponding mode indices
indicesF1stFrame = 1:nNonlinearityModesInFirstFrame ;
% number of transformed DEIM modes for the 2nd traveling wave
nNonlinearityModesInSecondFrame = nNonlinearityModes(2) ;
% corresponding mode indices
indicesF2ndFrame = nNonlinearityModesInFirstFrame+(1:nNonlinearityModesInSecondFrame) ;
% transformed DEIM modes for the 1st traveling wave
FModes1stFrame = FModes(:,indicesF1stFrame) ;
% transformed DEIM modes for the 2nd traveling wave
FModes2ndFrame = FModes(:,indicesF2ndFrame) ;
% mode indices corresponding to the DEIM modes
indicesOfFPODModes = nNonlinearityModesInFirstFrame+nNonlinearityModesInSecondFrame+(1:mPOD) ;

% DEIM modes only describes the upper half of the nonlinearity snapshots; 
% to get modes for the complete nonlinearity, we need to duplicate the 
% modes and premultiply the lower half by the constant -gammaSByAlpha
FModes1stFrameKron = kron([1;-gammaSByAlpha], FModes1stFrame) ;
FModes2ndFrameKron = kron([1;-gammaSByAlpha], FModes2ndFrame) ;
FPODModesKron = kron([1;-gammaSByAlpha], FPODModes) ;

%% determine DEIM rows

fprintf(' -> perform DEIM\t ...');
tic;
% sampling rate for the paths
samplingRate = hyperreduction.sample.samplingRate ; 
% NxForSampling specified how many spatial grid points are within the
% sampling range for the paths
NxForSampling = round(Nx*hyperreduction.sample.samplingRange) ;
% sampling points for the paths
samplingPoints = NxForSampling(1)*dx:(dx*samplingRate):NxForSampling(2)*dx ;
% number of sampling points for the paths
nSamplingPoints = length(samplingPoints) ;
% create shift matrices for each path sample
shiftMatrices = shift_matrix_generator(discretization, -samplingPoints) ;
% create shift matrices which shift in the opposite direction for each path
% sample (for the case v~=0, the factor offline.active_subspace_factor
% takes care of the fact that the symmetry p1=-p2 does no longer hold; in
% the case v=0, this factor is just -1)
shiftBackMatrices = shift_matrix_generator(discretization, -offline.active_subspace_factor*samplingPoints) ;
% store the sampling points in output struct
sampledMatrices.samplingPoints = samplingPoints ;

% shift matrices are duplicated on the block diagonal to be able to shift
% vectors containing more than one physical variable
shiftMatricesBlock = cell(nSamplingPoints, 1) ;
shiftBackMatricesBlock = cell(nSamplingPoints, 1) ;
for i=1:nSamplingPoints
    shiftMatricesBlock{i} = kron(speye(nVars), shiftMatrices{i}) ;
    shiftBackMatricesBlock{i} = kron(speye(nVars), shiftBackMatrices{i}) ;
end

% determine the DEIM rows for each path sample
SDEIMSampled = NaN(m+mPOD, nSamplingPoints) ;
for i=1:nSamplingPoints
    % shift DEIM modes into 1st co-moving reference frame
    shiftedFModesIn1stFrame = shiftBackMatrices{i}*FModes1stFrame ;
    % shift DEIM modes into 2nd co-moving reference frame
    shiftedFModesIn2ndFrame = shiftMatrices{i}*FModes2ndFrame ;
    % compute DEIM points using the QDEIM algorithm
    SDEIMSampled(:,i) = sort(QDEIM([shiftedFModesIn1stFrame,shiftedFModesIn2ndFrame,FPODModes])) ;
end
DEIMtime = toc; % run time for determining the DEIM rows
fprintf(' done! (%f sec)\n',DEIMtime);

%% sample the path-dependent coefficient matrices originating from the linear part of the FOM

tic;
fprintf(' -> perform sampling (%d points) \t ...',nSamplingPoints);
% order of the finite difference scheme
order = discretization.order ;
% assemble a finite difference matrix for the first space derivative
D = createGeneralFDMmatrix(order, Nx, dx) ;
% finite difference matrix is duplicated on the block diagonal to be able 
% to compute the derivative of functions containing more than one physical 
% variable
Dblock = kron(speye(nVars), D) ;
% compute the negative of the 1st derivative of the transformed modes
DModes = -Dblock*modes ;
% negative derivatives of modes for the 1st traveling wave
DModes1stFrame = DModes(:,indices1stFrame) ;
% negative derivatives of modes for the 2nd traveling wave
DModes2ndFrame = DModes(:,indices2ndFrame) ;

% create shift matrices with zero extrapolation for shifting the
% derivatives of the modes
shiftMatricesD = shiftMatrixGenerator_constZeroExtrapolation(discretization, -samplingPoints) ;
shiftBackMatricesD = shiftMatrixGenerator_constZeroExtrapolation(discretization, -offline.active_subspace_factor*samplingPoints) ;
shiftMatricesDBlock = cell(nSamplingPoints, 1) ;
shiftBackMatricesDBlock = cell(nSamplingPoints, 1) ;
for i=1:nSamplingPoints
    shiftMatricesDBlock{i} = kron(speye(nVars), shiftMatricesD{i}) ;
    shiftBackMatricesDBlock{i} = kron(speye(nVars), shiftBackMatricesD{i}) ;
end

% sample the ROM mass matrix
PODMassMatrix = PODModes'*PODModes ; % mass matrix corresponding to the POD modes
% initialization of sampled mass matrices
M11Sampled = NaN(rSPOD+rPOD, rSPOD+rPOD, nSamplingPoints) ; % 1-1-block of the ROM mass matrix
M12Sampled = NaN(rSPOD+rPOD, rSPOD, nSamplingPoints) ; % 1-2-block of the ROM mass matrix
M22Sampled = NaN(rSPOD, rSPOD, nSamplingPoints) ; % 2-2-block of the ROM mass matrix
% beginning of the sampling
for k=1:nSamplingPoints
    % current shift matrix for 1st traveling wave
    shiftMatrixk = shiftBackMatricesBlock{k} ;
    % current shift matrix for 2nd traveling wave
    shiftMatrixl = shiftMatricesBlock{k} ;
    % current shift matrix with zero extrapolation for 1st traveling wave
    shiftMatrixkD = shiftBackMatricesDBlock{k} ;
    % current shift matrix with zero extrapolation for 2nd traveling wave
    shiftMatrixlD = shiftMatricesDBlock{k} ;
    % compute inner products of shifted modes
    M11Sampled(indices1stFrame,indices1stFrame,k) = iPOM(shiftMatrixk, shiftMatrixk, modes1stFrame, modes1stFrame) ;
    M11Sampled(indices1stFrame,indices2ndFrame,k) = iPOM(shiftMatrixk, shiftMatrixl, modes1stFrame, modes2ndFrame) ;
    M11Sampled(indices2ndFrame,indices1stFrame,k) = M11Sampled(indices1stFrame,indices2ndFrame,k)' ; % exploit symmetry of inner product
    M11Sampled(indices2ndFrame,indices2ndFrame,k) = iPOM(shiftMatrixl, shiftMatrixl, modes2ndFrame, modes2ndFrame) ;
    % compute inner products of shifted modes and their negative derivatives
    M12Sampled(indices1stFrame,indices1stFrame,k) = iPOM(shiftMatrixk, shiftMatrixkD, modes1stFrame, DModes1stFrame) ;
    M12Sampled(indices1stFrame,indices2ndFrame,k) = iPOM(shiftMatrixk, shiftMatrixlD, modes1stFrame, DModes2ndFrame) ;
    M12Sampled(indices2ndFrame,indices1stFrame,k) = iPOM(shiftMatrixl, shiftMatrixkD, modes2ndFrame, DModes1stFrame) ;
    M12Sampled(indices2ndFrame,indices2ndFrame,k) = iPOM(shiftMatrixl, shiftMatrixlD, modes2ndFrame, DModes2ndFrame) ;
    % compute inner products of negative derivatives of shifted modes
    M22Sampled(indices1stFrame,indices1stFrame,k) = iPOM(shiftMatrixkD, shiftMatrixkD, DModes1stFrame, DModes1stFrame) ;
    M22Sampled(indices1stFrame,indices2ndFrame,k) = iPOM(shiftMatrixkD, shiftMatrixlD, DModes1stFrame, DModes2ndFrame) ;
    M22Sampled(indices2ndFrame,indices1stFrame,k) = M22Sampled(indices1stFrame,indices2ndFrame,k)' ; % exploit symmetry of inner product
    M22Sampled(indices2ndFrame,indices2ndFrame,k) = iPOM(shiftMatrixlD, shiftMatrixlD, DModes2ndFrame, DModes2ndFrame) ;
    if(usePODModes)
        % compute inner products of shifted modes and POD modes
        M11Sampled(indices1stFrame,indicesOfPODModes,k) = iPOM(shiftMatrixk, speye(nRows), modes1stFrame, PODModes) ;
        M11Sampled(indices2ndFrame,indicesOfPODModes,k) = iPOM(shiftMatrixl, speye(nRows), modes2ndFrame, PODModes) ;
        M11Sampled(indicesOfPODModes,indices1stFrame,k) = M11Sampled(indices1stFrame,indicesOfPODModes,k)' ; % exploit symmetry of inner product
        M11Sampled(indicesOfPODModes,indices2ndFrame,k) = M11Sampled(indices2ndFrame,indicesOfPODModes,k)' ; % exploit symmetry of inner product
        % inner products of POD modes
        M11Sampled(indicesOfPODModes,indicesOfPODModes,k) = PODMassMatrix ;
        % compute inner products of POD modes and negative derivatives of shifted
        % modes
        M12Sampled(indicesOfPODModes,indices1stFrame,k) = iPOM(speye(nRows), shiftMatrixkD, PODModes, DModes1stFrame) ;
        M12Sampled(indicesOfPODModes,indices2ndFrame,k) = iPOM(speye(nRows), shiftMatrixlD, PODModes, DModes2ndFrame) ;
    end
end
% store the sampled mass matrices in the output struct
sampledMatrices.M11 = M11Sampled ;
sampledMatrices.M12 = M12Sampled ;
sampledMatrices.M22 = M22Sampled ;

% sample the ROM matrix corresponding to the linear term of the FOM
if(usePODModes)
     % ROM matrix corresponding to the POD modes
    PODAMatrix = PODModes'*A*PODModes ;
end
% initialization of sampled matrices
A1Sampled = NaN(rSPOD+rPOD, rSPOD+rPOD, nSamplingPoints) ; % 1st block of the ROM matrix
A2Sampled = NaN(rSPOD, rSPOD+rPOD, nSamplingPoints) ; % 2nd block of the ROM matrix
% beginning of the sampling
for k=1:nSamplingPoints
    % current shift matrix for 1st traveling wave
    shiftMatrixk = shiftBackMatricesBlock{k} ;
    % current shift matrix for 2nd traveling wave
    shiftMatrixl = shiftMatricesBlock{k} ;
    % current shift matrix with zero extrapolation for 1st traveling wave
    shiftMatrixkD = shiftBackMatricesDBlock{k} ;
    % current shift matrix with zero extrapolation for 2nd traveling wave
    shiftMatrixlD = shiftMatricesDBlock{k} ;
    % compute weighted inner products of shifted modes
    A1Sampled(indices1stFrame,indices1stFrame,k) = wIPOM(shiftMatrixk, shiftMatrixk, modes1stFrame, modes1stFrame, A) ;
    A1Sampled(indices1stFrame,indices2ndFrame,k) = wIPOM(shiftMatrixk, shiftMatrixl, modes1stFrame, modes2ndFrame, A) ;
    A1Sampled(indices2ndFrame,indices1stFrame,k) = wIPOM(shiftMatrixl, shiftMatrixk, modes2ndFrame, modes1stFrame, A) ;
    A1Sampled(indices2ndFrame,indices2ndFrame,k) = wIPOM(shiftMatrixl, shiftMatrixl, modes2ndFrame, modes2ndFrame, A) ;
    % compute weighted inner products of shifted modes and their
    % negative derivatives
    A2Sampled(indices1stFrame,indices1stFrame,k) = wIPOM(shiftMatrixkD, shiftMatrixk, DModes1stFrame, modes1stFrame, A) ;
    A2Sampled(indices1stFrame,indices2ndFrame,k) = wIPOM(shiftMatrixkD, shiftMatrixl, DModes1stFrame, modes2ndFrame, A) ;
    A2Sampled(indices2ndFrame,indices1stFrame,k) = wIPOM(shiftMatrixlD, shiftMatrixk, DModes2ndFrame, modes1stFrame, A) ;
    A2Sampled(indices2ndFrame,indices2ndFrame,k) = wIPOM(shiftMatrixlD, shiftMatrixl, DModes2ndFrame, modes2ndFrame, A) ;
    if(usePODModes) 
        % compute weighted inner products of shifted modes and POD modes
        A1Sampled(indices1stFrame,indicesOfPODModes,k) = wIPOM(shiftMatrixk, speye(nRows), modes1stFrame, PODModes, A) ;
        A1Sampled(indices2ndFrame,indicesOfPODModes,k) = wIPOM(shiftMatrixl, speye(nRows), modes2ndFrame, PODModes, A) ;
        % compute weighted inner products of POD modes and shifted modes
        A1Sampled(indicesOfPODModes,indices1stFrame,k) = wIPOM(speye(nRows), shiftMatrixk, PODModes, modes1stFrame, A) ;
        A1Sampled(indicesOfPODModes,indices2ndFrame,k) = wIPOM(speye(nRows), shiftMatrixl, PODModes, modes2ndFrame, A) ;    
        % weighted inner products of POD modes
        A1Sampled(indicesOfPODModes,indicesOfPODModes,k) = PODAMatrix ;
        % compute weighted inner products of negative derivatives of shifted modes and POD modes
        A2Sampled(indices1stFrame,indicesOfPODModes,k) = wIPOM(shiftMatrixkD, speye(nRows), DModes1stFrame, PODModes, A) ;
        A2Sampled(indices2ndFrame,indicesOfPODModes,k) = wIPOM(shiftMatrixlD, speye(nRows), DModes2ndFrame, PODModes, A) ;
    end
end
% store the sampled matrices in the output struct
sampledMatrices.A1 = A1Sampled ;
sampledMatrices.A2 = A2Sampled ;
    
%% sample coefficient matrices originating from the shifted DEIM method

% determine leading matrix of the nonlinear term in the ROM
PODFMatrix = PODModes'*FPODModesKron ; % inner products of POD modes and DEIM modes
% initialization of sampled coefficient matrices
F1PTPsiInverseSampled = NaN(rSPOD+rPOD, m+mPOD, nSamplingPoints) ; % 1st block of the leading matrix of the nonlinear term in the ROM
F2PTPsiInverseSampled = NaN(rSPOD, m+mPOD, nSamplingPoints) ; % 2nd block of the leading matrix of the nonlinear term in the ROM
 % auxiliary variables
F1Temp = NaN(rSPOD+rPOD, m+mPOD) ;
F2Temp = NaN(rSPOD, m+mPOD) ;
PTPsiTemp = NaN(m+mPOD, m+mPOD) ;
% sampling begins
for k=1:nSamplingPoints
    % current shift matrix for 1st traveling wave
    shiftMatrixk = shiftBackMatricesBlock{k} ;
    % current shift matrix for 2nd traveling wave
    shiftMatrixl = shiftMatricesBlock{k} ;
    % current shift matrix with zero extrapolation for 1st traveling wave
    shiftMatrixkD = shiftBackMatricesDBlock{k} ;
    % current shift matrix with zero extrapolation for 2nd traveling wave
    shiftMatrixlD = shiftMatricesDBlock{k} ;
    % inner products of shifted POD modes and shifted DEIM modes
    F1Temp(indices1stFrame,indicesF1stFrame) = iPOM(shiftMatrixk, shiftMatrixk, modes1stFrame, FModes1stFrameKron) ;
    F1Temp(indices1stFrame,indicesF2ndFrame) = iPOM(shiftMatrixk, shiftMatrixl, modes1stFrame, FModes2ndFrameKron) ;
    F1Temp(indices2ndFrame,indicesF1stFrame) = iPOM(shiftMatrixl, shiftMatrixk, modes2ndFrame, FModes1stFrameKron) ;
    F1Temp(indices2ndFrame,indicesF2ndFrame) = iPOM(shiftMatrixl, shiftMatrixl, modes2ndFrame, FModes2ndFrameKron) ;
    % inner products of negative derivatives of shifted POD modes and shifted DEIM modes
    F2Temp(indices1stFrame,indicesF1stFrame) = iPOM(shiftMatrixkD, shiftMatrixk, DModes1stFrame, FModes1stFrameKron) ;
    F2Temp(indices1stFrame,indicesF2ndFrame) = iPOM(shiftMatrixkD, shiftMatrixl, DModes1stFrame, FModes2ndFrameKron) ;
    F2Temp(indices2ndFrame,indicesF1stFrame) = iPOM(shiftMatrixlD, shiftMatrixk, DModes2ndFrame, FModes1stFrameKron) ;
    F2Temp(indices2ndFrame,indicesF2ndFrame) = iPOM(shiftMatrixlD, shiftMatrixl, DModes2ndFrame, FModes2ndFrameKron) ;
    % shifted DEIM modes for 1st traveling wave
    ShiftedFModes1stFrame = shiftMatrixk*FModes1stFrameKron ;
    % shifted DEIM modes for 2nd traveling wave
    ShiftedFModes2ndFrame = shiftMatrixl*FModes2ndFrameKron ;
    % shifted DEIM modes evaluated at the determined DEIM rows
    PTPsiTemp(:,indicesF1stFrame) = ShiftedFModes1stFrame(SDEIMSampled(:,k),:) ;
    PTPsiTemp(:,indicesF2ndFrame) = ShiftedFModes2ndFrame(SDEIMSampled(:,k),:) ;
    if(usePODModes)
        % inner products of shifted POD modes and DEIM modes
        F1Temp(indices1stFrame,indicesOfFPODModes) = iPOM(shiftMatrixk, speye(nRows), modes1stFrame, FPODModesKron) ;
        F1Temp(indices2ndFrame,indicesOfFPODModes) = iPOM(shiftMatrixl, speye(nRows), modes2ndFrame, FPODModesKron) ;
        % inner products of POD modes and shifted DEIM modes
        F1Temp(indicesOfPODModes,indicesF1stFrame) = iPOM(speye(nRows), shiftMatrixk, PODModes, FModes1stFrameKron) ;
        F1Temp(indicesOfPODModes,indicesF2ndFrame) = iPOM(speye(nRows), shiftMatrixl, PODModes, FModes2ndFrameKron) ;
        % inner products of POD modes and DEIM modes
        F1Temp(indicesOfPODModes,indicesOfFPODModes) = PODFMatrix ;
        % inner products of negative derivatives of shifted POD modes and DEIM modes
        F2Temp(indices1stFrame,indicesOfFPODModes) = iPOM(shiftMatrixkD, speye(nRows), DModes1stFrame, FPODModesKron) ;
        F2Temp(indices2ndFrame,indicesOfFPODModes) = iPOM(shiftMatrixlD, speye(nRows), DModes2ndFrame, FPODModesKron) ;
        % DEIM modes evaluated at the determined DEIM rows
        PTPsiTemp(:,m+1:end) = FPODModes(SDEIMSampled(:,k),:) ;
    end
    % compute leading matrix of the nonlinear term in the ROM
    F1PTPsiInverseSampled(:,:,k) = F1Temp/PTPsiTemp ;
    F2PTPsiInverseSampled(:,:,k) = F2Temp/PTPsiTemp ;
end
% store determined samples in output struct
sampledMatrices.F1PTPsiInverse = F1PTPsiInverseSampled ;
sampledMatrices.F2PTPsiInverse = F2PTPsiInverseSampled ;

% compute the matrix PTPhiSampled which contains those entries of the 
% shifted POD modes and POD modeswhich are needed to evaluate the 
% hyperreduced nonlinearity in the ROM
PTPhiSampled = NaN(m+mPOD, rSPOD+rPOD, nSamplingPoints) ; % initialization
SDEIMSampledmod = mod(SDEIMSampled-1,Nx)+1 ; % indices of the spatial points corresponding to the determined DEIM rows
currentShiftMatrices = cell(2, 1) ; % auxiliary variable
% sampling begins
for k=1:nSamplingPoints
    % current shift matrix for 1st traveling wave
    currentShiftMatrices{1} = shiftBackMatricesBlock{k} ;
    % current shift matrix for 2nd traveling wave
    currentShiftMatrices{2} = shiftMatricesBlock{k} ;
    % determine PTPhiSampled for the shifted POD modes of each traveling wave
    for i=1:nFrames
        % mode indices corresponding to the temperature shifted POD modes
        TIndices = helper(i)+(1:nModes(i)/2) ;
        % mode indices corresponding to the supply mass fraction shifted POD modes
        SIndices = helper(i)+(nModes(i)/2+1:nModes(i)) ;
        % shifted modes of current traveling wave
        currentShiftedModes = currentShiftMatrices{i}*modes(:,[TIndices,SIndices]) ;
        PTPhiSampled(:,TIndices,k) = currentShiftedModes(SDEIMSampledmod(:,k),1:end/2) ;
        PTPhiSampled(:,SIndices,k) = currentShiftedModes(Nx+SDEIMSampledmod(:,k),end/2+1:end) ;
    end
    % determine PTPhiSampled for the POD modes
    if(usePODModes)
        % mode indices corresponding to the temperature POD modes
        TIndices = rSPOD+(1:rPODVec(1)) ;
        % mode indices corresponding to the supply mass fraction POD modes
        SIndices = rSPOD+(rPODVec(1)+1:rPOD) ;
        PTPhiSampled(:,TIndices,k) = PODModes(SDEIMSampledmod(:,k),TIndices-rSPOD) ;
        PTPhiSampled(:,SIndices,k) = PODModes(Nx+SDEIMSampledmod(:,k),SIndices-rSPOD) ;
    end
end
% store sampled data in output struct
sampledMatrices.PTPhiSampled = PTPhiSampled ;

samplingTime = toc; % run time of sampling
fprintf(' done! (%f sec)\n',samplingTime);