function PODROMMatrices = PODROMPreProcessing(A, gammaSByAlpha, PODModes, FPODModes, rPODVec)
% PODROMPreProcessing - precomputes ROM matrices needed for the POD-DEIM
% ROM
%
% inputs:
% - A: matrix for the linear part of the wildland fire model
% - gammaSByAlpha: ratio of pre-exponential factor and temperature rise per
% second
% - PODModes: matrix containing the POD modes
% - FPODModes: matrix containing the DEIM modes
% - rPODVec: number of POD modes for each physical variable
%
% outputs:
% - PODROMMatrices: struct array containing the precomputed coefficient
% matrices of the POD-DEIM-ROM
%
% dependencies:
% - QDEIM (in LIB/HYPERREDUCTION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 28, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

rPODSum = sum(rPODVec) ; % total number of POD modes
mPOD = size(FPODModes, 2) ; % number of DEIM modes

% number of steps in space (here, we use that there are two physical
% variables)
Nx = size(PODModes,1)/2 ;

% FPODModes only describes the upper half of the nonlinearity snapshots; to
% get modes for the complete nonlinearity, we need to duplicate the modes
% and premultiply the lower half by the constant -gammaSByAlpha
FPODModesKron = kron([1;-gammaSByAlpha], FPODModes) ;

fprintf(' -> perform DEIM\t ...');
tic;
% perform QDEIM algorithm to obtain the DEIM row indices
SDEIM = sort(QDEIM(FPODModes)) ;
DEIMtime = toc; % run time for QDEIM algorithm
fprintf(' done! (%f sec)\n',DEIMtime);

% reduced A matrix
PODROMMatrices.PODAMatrix = PODModes'*A*PODModes ;

% precomputed matrices for DEIM part

rPODHelper = cumsum(rPODVec) ; % auxiliary variable
% product of POD modes and DEIM modes
PODFMatrix = PODModes'*FPODModesKron ;
% sampled DEIM modes
PTPsiTemp = FPODModes(SDEIM,:) ;
% PODDEIMMatrix is the leading matrix of the nonlinear term in the ROM
PODROMMatrices.PODDEIMMatrix = PODFMatrix/PTPsiTemp ;

% PTPhi contains those entries of the POD modes which are needed to
% evaluate the hyperreduced nonlinearity in the ROM
PTPhi = NaN(mPOD, rPODSum) ;
TIndices = 1:rPODHelper(1) ; % indices corresponding to the temperature
SIndices = rPODHelper(1)+1:rPODHelper(2) ; % indices corresponding to the supply mass fraction
SDEIMmod = mod(SDEIM-1,Nx)+1 ; % indices of the spatial points corresponding to the determined DEIM rows
PTPhi(:,TIndices) = PODModes(SDEIMmod,TIndices) ;
PTPhi(:,SIndices) = PODModes(Nx+SDEIMmod,SIndices) ;
PODROMMatrices.PTPhi = PTPhi ;