function fom = create_fom_struct(initialCondition, varargin)
% create_fom_struct - creates a struct containing the parameters of the
% full-order model
%
% inputs:
% - initialCondition: string determining which initial condition is used; 
% options are 'Gaussian' (initial condition used in section 5.3) and 
% 'separatedWaves' (initial condition used in section 5.2)
% - varargin: an even number of optional input arguments for specifying
% some of the coefficients in the PDE model; optional inputs are paired;
% first input denotes name of the parameter (options are 'parameter.beta',
% 'parameter.alpha', 'parameter.gammaS', 'parameter.v'); second input 
% denotes value of the parameter
%
% outputs:
% - rom: struct containing settins for the full-order models
%
% dependencies:
% - gaussian_pulse (in LIB/INITIAL_CONDITION)
% - loadInitialTemperature (in LIB/INITIAL_CONDITION)
% - loadInitialSpeciesFraction (in LIB/INITIAL_CONDITION)
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% admissible parameter names in varargin
admissible_names = {'parameter.beta', 'parameter.alpha', 'parameter.gammaS', 'parameter.v'};

% Get the names from varargin
names = varargin(1:2:end);
% Get the corresponding values from varargin
values = varargin(2:2:end);

% Get the number of elements in the values 
n_elements = zeros(1, length(values));
for i = 1:length(values)
    n_elements(i) = length(values{i});
end

%% Process the name-value pairs into a struct array

% Compute the number of structs in the fom struct array
n_structs = 1;
for i = 1:length(values)
    current_value = values{i};
    n_structs = n_structs * length(current_value);
end

% Initialization of struct array
for i = 1:n_structs
    fom(i) = struct();
end

switch length(n_elements)
    case 1
        indices = 1:n_structs;
    case 2
        [ind1, ind2] = ind2sub(n_elements, (1:n_structs));
        indices = [ind1; ind2];
    case 3
        [ind1, ind2, ind3] = ind2sub(n_elements, (1:n_structs));
        indices = [ind1; ind2; ind3];
    case 4
        [ind1, ind2, ind3, ind4] = ind2sub(n_elements, (1:n_structs));
        indices = [ind1; ind2; ind3; ind4];
    case 5
        [ind1, ind2, ind3, ind4, ind5] = ind2sub(n_elements, (1:n_structs));
        indices = [ind1; ind2; ind3; ind4; ind5];
end

% Write values to the structs
for i = 1:length(names)
    field_name = names{i};
    field_values = values{i};
    for j = 1:n_structs
        switch field_name
            case 'parameter.alpha'
                fom(j).parameter.alpha = field_values(indices(i,j));
            case 'parameter.beta'
                fom(j).parameter.beta = field_values(indices(i,j));
            case 'parameter.gammaS'
                fom(j).parameter.gammaS = field_values(indices(i,j));
            case 'parameter.v'
                fom(j).parameter.v = field_values(indices(i,j));
        end
    end
end

% specify the initial condition based on the input initialCondition
for j = 1:n_structs
    switch initialCondition
        case 'Gaussian'
            fom(j).initial_condition_name = 'Gaussian';
            fom(j).initial_condition{1} = @(x,L)1200*gaussian_pulse(x,L); % temperature initial condition
            fom(j).initial_condition{2} = @(x,L) ones(1, length(x)); % supply mass fraction initial condition
        case 'separatedWaves'
            fom(j).initial_condition_name = 'separatedWaves';
            fom(j).initial_condition{1} = @(x,L) loadInitialTemperature(x,L) ; % temperature initial condition
            fom(j).initial_condition{2} = @(x,L) loadInitialSpeciesFraction(x,L) ; % supply mass fraction initial condition
        otherwise
            error('Unknown Intitial Condition')
    end
    fom(j).n_vars = 2; % number of physical variables (temperature, supply mass fraction)
end

% Set default values if there are no optional parameters
default_names = setdiff(admissible_names, names);

for i = 1:length(default_names)
    field_name = default_names{i};
    for j = 1:n_structs
        switch field_name
            case 'parameter.alpha'
                fom(j).parameter.alpha = 1.8793e02 ;
            case 'parameter.beta'
                fom(j).parameter.beta = 5.5849e02;
            case 'parameter.gammaS'
                fom(j).parameter.gammaS = 1.6250e-01 ;
            case 'parameter.v'
                fom(j).parameter.v = 0 ;
        end
    end
end