function discretization = create_discretization_struct(fom_rom, discretization_default, initialCondition)
% create_fom_struct - helper function that creates a struct array 
% discretization with discretization parameters for each struct that is 
% contained in the struct array fom_rom.
%
% inputs:
% - fom_rom: array of structs
% - discretization_default: struct array containing optional fields;
% options are 'order', 'onlineOrder', 'space.n_steps', 'time.n_steps'
% - initialCondition: string determining which initial condition is used; 
% options are 'Gaussian' (initial condition used in section 5.3) and 
% 'separatedWaves' (initial condition used in section 5.2)
%
% outputs:
% - discretization: struct array containing the discretization parameters
%
%--------------------------------------------------------------------------
% version 1.0 (July 27, 2021)
% authors:
% - Felix Black (TU Berlin), black@math.tu-berlin.de
% - Philipp Schulze (TU Berlin), pschulze@math.tu-berlin.de
% - Benjamin Unger (U Stuttgart), benjamin.unger@simtech.uni-stuttgart.de
%--------------------------------------------------------------------------

% default values
default_space_n_steps = 1000; % number of steps in space
default_time_n_steps = 600; % number of steps in time
default_order = 6; % order of the finite difference scheme used for the FOM
default_onlineOrder = 6; % order of the finite difference scheme used for the ROM

% initialization of discretization struct array
n_structs = length(fom_rom);
for i = 1:n_structs
    discretization(i) = struct();
end

% specifying the values within the discretization struct array
for i = 1:n_structs
    % order of the finite difference scheme used for the FOM
    if isfield(discretization_default,'order')
        discretization(i).order = discretization_default.order;
    else
        discretization(i).order = default_order;
    end
    % order of the finite difference scheme used for the ROM
    if isfield(discretization_default,'onlineOrder')
        discretization(i).onlineOrder = discretization_default.onlineOrder;
    else
        discretization(i).onlineOrder = default_onlineOrder;
    end
    discretization(i).boundary_condition = 'periodic'; % used boundary conditions
    % number of steps in space
    if and(isfield(discretization_default,'space'),isfield(discretization_default.space,'n_steps'))
        discretization(i).space.n_steps = discretization_default.space.n_steps;
    else
        discretization(i).space.n_steps = default_space_n_steps;
    end
    discretization(i).space.length = 1000; % length of computational domain
    discretization(i).space.dx = discretization(i).space.length /discretization(i).space.n_steps; % spatial mesh width
    % length of time interval
    switch initialCondition
        case 'Gaussian'
            discretization(i).time.length = 2100; 
        case 'separatedWaves'
            discretization(i).time.length = 1400;
        otherwise
            error('Unknown Intitial Condition')
    end
    % number of steps in time
    if and(isfield(discretization_default,'time'),isfield(discretization_default.time,'n_steps'))
        discretization(i).time.n_steps = discretization_default.time.n_steps;
    else
        discretization(i).time.n_steps = default_time_n_steps;
    end
    discretization(i).time.dt = discretization(i).time.length / discretization(i).time.n_steps; % temporal grid size
end