#!/usr/bin/env python3
# -*- coding: utf8 -*-
"""
"""

# Author: Tommi A. Pirinen <tommi.pirinen@uit.no> 2020

#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

import argparse
from io import StringIO
from sys import exit, stderr

import yaml
from lxml import etree

from corpustools import ccat, errormarkup


def make_errormarkup(sent):
    para = etree.Element('p')
    para.text = sent
    errormarkup.add_error_markup(para)

    return para


def main():
    ap = argparse.ArgumentParser(
        description="Make neural corpus from yaml gramchecker file")

    ap.add_argument("--quiet",
                    "-q",
                    action="store_false",
                    dest="verbose",
                    default=False,
                    help="do not print output to stdout while processing")
    ap.add_argument("--verbose",
                    "-v",
                    action="store_true",
                    default=False,
                    help="print each step to stdout while processing")
    ap.add_argument("--version", "-V", action="version")
    ap.add_argument("--input",
                    "-i",
                    action="append",
                    required=True,
                    dest='infilenames',
                    metavar="IFILE",
                    help="read corpus from IFILEs")
    ap.add_argument("--output",
                    "-o",
                    action="store",
                    required=True,
                    dest='outfilename',
                    metavar="OFILE",
                    help="write resulting data to src-OFILE and tgt-OFILE")
    args = ap.parse_args()
    src = open('src-' + args.outfilename, 'w')
    tgt = open('tgt-' + args.outfilename, 'w')
    error_printer = ccat.XMLPrinter()
    correct_printer = ccat.XMLPrinter(correction=True)

    sents = 0
    for yaml_filename in args.infilenames:
        if args.verbose:
            print("Reading from", yaml_filename)
        f = open(yaml_filename, 'r')
        corpus = yaml.load(f, Loader=yaml.FullLoader)
        for sent in corpus['Tests']:
            para = make_errormarkup(sent)
            error_printer.collect_text(para, 'sme', src)
            correct_printer.collect_text(para, 'sme', tgt)

        print()
    exit()


if __name__ == "__main__":
    main()
