#include <math.h>
#include <stdio.h>
#include <string.h>
#include <uncertain.h>

typedef enum
{
    kBenchmarkModesCustomMonteCarlo = 1 << 0,
} BenchmarkModes;

double calcPosterior_bySampling(double prior, double likelihood, const int n_samples);
double calcPosterior(double prior, double likelihood, double marginal_likelihood);
float calcM(double Phi, float alpha);
float calcTheta(double Phi);
double bayesianQPEupdate(double Phi, float M, float theta, BenchmarkModes benchmarkMode);

int main(int argc, char *argv[])
{
    BenchmarkModes mode = 0;
    float E_vals[] = {0, 1};
    float alpha = 0;
    float mu_0 = M_PI_2;
    float sigma_0 = 0.5;
    int QPEsteps = 5;
    double Phi[QPEsteps + 1];
    float M[QPEsteps + 1];
    float theta[QPEsteps + 1];
    double FinalPhi;

    /*
     *  Get mode from arguments.
     */
    if ((argc == 2) && (strcmp("1", argv[1]) == 0))
    {
        mode |= kBenchmarkModesCustomMonteCarlo;
    }
    if (!(mode && kBenchmarkModesCustomMonteCarlo))
    {
        printf("Uncertain Propagation mode.\n");
    }
    else
    {
        printf("Custom Monte Carlo mode.\n");
    }

    M[0] = 1 / pow(sigma_0, alpha);
    theta[0] = mu_0 - sigma_0;
    // Define Phi Prior
    Phi[0] = libUncertainDoubleUniformDist(-M_PI, M_PI);

    printf("Phi (iter 0): %lf\n", Phi[0]);
    libUncertainDoublePrint(Phi[0]);
    for (size_t i = 0; i < QPEsteps; i++)
    {
        Phi[i + 1] = bayesianQPEupdate(Phi[i], M[i], theta[i], mode);
        M[i + 1] = calcM(Phi[i + 1], alpha);
        theta[i + 1] = calcTheta(Phi[i + 1]);
        printf("Phi (iter %d): %lf\n", i + 1, Phi[i + 1]);
        libUncertainDoublePrint(Phi[i + 1]);
    }
    FinalPhi = Phi[QPEsteps];
    return 0;
}

float calcM(double Phi, float alpha)
{
    return 1 / pow(sqrt(libUncertainDoubleNthMoment(Phi, 2)), alpha);
}

float calcTheta(double Phi)
{
    return libUncertainDoubleNthMoment(Phi, 1) - sqrt(libUncertainDoubleNthMoment(Phi, 2));
}

double bayesianQPEupdate(double Phi, float M, float theta, BenchmarkModes benchmarkMode)
{
    double E0_given_phi;
    double E1_given_phi;
    double Phi_given_E0;
    double Phi_given_E1;
    double Phi_given_E01;
    //--------- using UPP - math.h cos ---------
    if (!(benchmarkMode && kBenchmarkModesCustomMonteCarlo))
    {
        E0_given_phi = ((1 + cos(M * (Phi - theta))) / 2);
        E1_given_phi = ((1 - cos(M * (Phi - theta))) / 2);
    }
    else
    {
        //---------- using MC ----------------------
        const int n_samples = 3000;
        double E0_given_phi_samples[n_samples];
        double E1_given_phi_samples[n_samples];
        for (size_t i = 0; i < n_samples; i++)
        {
            E0_given_phi_samples[i] = ((1 + cos(M * (libUncertainDoubleSample(Phi) - theta))) / 2);
            E1_given_phi_samples[i] = ((1 - cos(M * (libUncertainDoubleSample(Phi) - theta))) / 2);
        }
        E0_given_phi = libUncertainDoubleDistFromSamples(E0_given_phi_samples, n_samples);
        E1_given_phi = libUncertainDoubleDistFromSamples(E1_given_phi_samples, n_samples);
    }

    Phi_given_E0 = E0_given_phi * Phi; //proportional to
    Phi_given_E1 = E1_given_phi * Phi; //proportional to

    Phi_given_E01 = libUncertainDoubleMixture(Phi_given_E0, Phi_given_E1, 0.5);
    return Phi_given_E01;
}
double calcPosterior(double prior, double likelihood, double marginal_likelihood)
{
    return (prior * likelihood) / marginal_likelihood;
}
double calcPosterior_bySampling(double prior, double likelihood, const int n_samples)
{
    double sample_array[n_samples];
    int i = 0;
    for (i = 0; i < n_samples; i++)
    {
        sample_array[i] = libUncertainDoubleSample(prior) * libUncertainDoubleSample(likelihood);
        ;
    }
    return libUncertainDoubleDistFromSamples(sample_array, n_samples);
}