/*
	Authored 2021, Orestis Kaparounakis.

	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

	*	Redistributions of source code must retain the above
		copyright notice, this list of conditions and the following
		disclaimer.

	*	Redistributions in binary form must reproduce the above
		copyright notice, this list of conditions and the following
		disclaimer in the documentation and/or other materials
		provided with the distribution.

	*	Neither the name of the author nor the names of its
		contributors may be used to endorse or promote products
		derived from this software without specific prior written
		permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <uncertain.h>

#include "fptaylor-real2float.h"

enum {
	kBenchmarkFunctionCount 	= 11, 
	kBenchmarkMaxArgumentCount 	= 6,
};

double	loadDoubleDistFromPath(double *  var, const char *  filename);
float	loadFloatDistFromPath(float *  var, const char *  filename);
void	usage();

int
main(int argc, char *  argv[])
{
	
	double	ex[kBenchmarkFunctionCount];
	double	in[kBenchmarkFunctionCount][kBenchmarkMaxArgumentCount];
	/*
	 *	Can't use this because it triggers the correlation mechanism.
	 */
	// double	in[kBenchmarkMaxArgumentCount];

	if (argc != 1)
	{
		usage();
		exit(EXIT_FAILURE);
	}

	/*
	 *	Load samples into distributional double variables and call
	 *	benchmark functions.
	 */
	loadDoubleDistFromPath(&in[0][0], "input-ex0-x.txt");
	ex[0] = ex0(in[0][0]);
	printf("ex0 = %lf\n", ex[0]);
	libUncertainDoublePrint(ex[0]);

	loadDoubleDistFromPath(&in[1][0], "input-ex1-x.txt");
	loadDoubleDistFromPath(&in[1][1], "input-ex1-r.txt");
	loadDoubleDistFromPath(&in[1][2], "input-ex1-lat.txt");
	loadDoubleDistFromPath(&in[1][3], "input-ex1-lon.txt");
	ex[1] = ex1(in[1][0], in[1][1], in[1][2], in[1][3]);
	printf("ex1 = %lf\n", ex[1]);
	libUncertainDoublePrint(ex[1]);

	loadDoubleDistFromPath(&in[2][1], "input-ex2-lat1.txt");
	loadDoubleDistFromPath(&in[2][1], "input-ex2-lat2.txt");
	loadDoubleDistFromPath(&in[2][2], "input-ex2-lon1.txt");
	loadDoubleDistFromPath(&in[2][3], "input-ex2-lon2.txt");
	ex[2] = ex2(in[2][0], in[2][1], in[2][2], in[2][3]);
	printf("ex2 = %lf\n", ex[2]);
	libUncertainDoublePrint(ex[2]);

	loadDoubleDistFromPath(&in[3][0], "input-ex3-x1.txt");
	loadDoubleDistFromPath(&in[3][1], "input-ex3-x2.txt");
	loadDoubleDistFromPath(&in[3][2], "input-ex3-x3.txt");
	loadDoubleDistFromPath(&in[3][3], "input-ex3-x4.txt");
	loadDoubleDistFromPath(&in[3][4], "input-ex3-x5.txt");
	loadDoubleDistFromPath(&in[3][5], "input-ex3-x6.txt");
	ex[3] = ex3(in[3][0], in[3][1], in[3][2], in[3][3], in[3][4], in[3][5]);
	printf("ex3 = %lf\n", ex[3]);
	libUncertainDoublePrint(ex[3]);

	loadDoubleDistFromPath(&in[4][0], "input-ex4-x1.txt");
	loadDoubleDistFromPath(&in[4][1], "input-ex4-x2.txt");
	ex[4] = ex4(in[4][0], in[4][1]);
	printf("ex4 = %lf\n", ex[4]);
	libUncertainDoublePrint(ex[4]);

	loadDoubleDistFromPath(&in[5][0], "input-ex5-x1.txt");
	loadDoubleDistFromPath(&in[5][1], "input-ex5-x2.txt");
	ex[5] = ex5(in[5][0], in[5][1]);
	printf("ex5 = %lf\n", ex[5]);
	libUncertainDoublePrint(ex[5]);

	loadDoubleDistFromPath(&in[6][0], "input-ex6-x1.txt");
	loadDoubleDistFromPath(&in[6][1], "input-ex6-x2.txt");
	loadDoubleDistFromPath(&in[6][2], "input-ex6-x3.txt");
	ex[6] = ex6(in[6][0], in[6][1], in[6][2]);
	printf("ex6 = %lf\n", ex[6]);
	libUncertainDoublePrint(ex[6]);

	loadDoubleDistFromPath(&in[7][0], "input-ex7-x1.txt");
	loadDoubleDistFromPath(&in[7][1], "input-ex7-x2.txt");
	loadDoubleDistFromPath(&in[7][2], "input-ex7-x3.txt");
	loadDoubleDistFromPath(&in[7][3], "input-ex7-x4.txt");
	loadDoubleDistFromPath(&in[7][4], "input-ex7-x5.txt");
	loadDoubleDistFromPath(&in[7][5], "input-ex7-x6.txt");
	ex[7] = ex7(in[7][0], in[7][1], in[7][2], in[7][3], in[7][4], in[7][5]);
	printf("ex7 = %lf\n", ex[7]);
	libUncertainDoublePrint(ex[7]);

	loadDoubleDistFromPath(&in[8][0], "input-ex8-x1.txt");
	loadDoubleDistFromPath(&in[8][1], "input-ex8-x2.txt");
	loadDoubleDistFromPath(&in[8][2], "input-ex8-x3.txt");
	loadDoubleDistFromPath(&in[8][3], "input-ex8-x4.txt");
	loadDoubleDistFromPath(&in[8][4], "input-ex8-x5.txt");
	loadDoubleDistFromPath(&in[8][5], "input-ex8-x6.txt");
	ex[8] = ex8(in[8][0], in[8][1], in[8][2], in[8][3], in[8][4], in[8][5]);
	printf("ex8 = %lf\n", ex[8]);
	libUncertainDoublePrint(ex[8]);

	loadDoubleDistFromPath(&in[9][0], "input-ex9-x1.txt");
	loadDoubleDistFromPath(&in[9][1], "input-ex9-x2.txt");
	loadDoubleDistFromPath(&in[9][2], "input-ex9-x3.txt");
	loadDoubleDistFromPath(&in[9][3], "input-ex9-x4.txt");
	ex[9] = ex9(in[9][0], in[9][1], in[9][2], in[9][3]);
	printf("ex9 = %lf\n", ex[9]);
	libUncertainDoublePrint(ex[9]);


	loadDoubleDistFromPath(&in[10][0], "input-ex10-x1.txt");
	loadDoubleDistFromPath(&in[10][1], "input-ex10-x2.txt");
	loadDoubleDistFromPath(&in[10][2], "input-ex10-x3.txt");
	loadDoubleDistFromPath(&in[10][3], "input-ex10-x4.txt");
	loadDoubleDistFromPath(&in[10][4], "input-ex10-x5.txt");
	loadDoubleDistFromPath(&in[10][5], "input-ex10-x6.txt");
	ex[10] = ex10(in[10][0], in[10][1], in[10][2], in[10][3], in[10][4], in[10][5]);
	printf("ex10 = %lf\n", ex[10]);
	libUncertainDoublePrint(ex[10]);

	return 0;
}

double
loadDoubleDistFromPath(double *  var, const char *  filename)
{
	FILE *		inp;
	int		sampleCount;
	double *	samples;
	double		returnValue;

	inp = fopen(filename, "r");
	if (inp == NULL)
	{
		/*
		 *	`fprintf(stderr,...)` may suffer from emulator issue #92.
		 */
		fprintf(stdout, "Error: could not open input file: %s\n", filename);
		exit(EXIT_FAILURE);
	}
	
	if (fscanf(inp, "%d\n", &sampleCount) != 1)
	{
		fprintf(stderr, "Error: could not read number of samples at start of input file.\n");
		exit(EXIT_FAILURE);
	}

	printf("Number of samples from %s: %d\n", filename, sampleCount);

	samples = malloc(sampleCount * sizeof(double));
	if (samples == NULL)
	{
		fprintf(stderr, "Error: could not allocate memory for samples.\n");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < sampleCount; i++)
	{
		fscanf(inp, "%lf\n", &samples[i]);
	}

	if (var == NULL )
	{
		returnValue = libUncertainDoubleDistFromSamples(samples, sampleCount);
	}
	else
	{
		(*var) = libUncertainDoubleDistFromSamples(samples, sampleCount);
	}

	if (fclose(inp) != 0)
	{
		fprintf(stderr, "Warning: could not close input file.\n");
	}
	free(samples);

	return returnValue;
}

float
loadFloatDistFromPath(float *  var, const char *  filename)
{
	FILE *		inp;
	int		sampleCount;
	float *	samples;
	float		returnValue;

	inp = fopen(filename, "r");
	if (inp == NULL)
	{
		/*
		 *	`fprintf(stderr,...)` may suffer from emulator issue #92.
		 */
		fprintf(stdout, "Error: could not open input file: %s\n", filename);
		exit(EXIT_FAILURE);
	}
	fscanf(inp, "%d\n", &sampleCount);

	printf("Number of samples from %s: %d\n", filename, sampleCount);

	samples = malloc(sampleCount * sizeof(float));
	if (samples == NULL)
	{
		fprintf(stderr, "Error: could not allocate memory for samples.\n");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < sampleCount; i++)
	{
		fscanf(inp, "%f\n", &samples[i]);
	}

	if (var == NULL )
	{
		returnValue = libUncertainFloatDistFromSamples(samples, sampleCount);
	}
	else
	{
		(*var) = libUncertainFloatDistFromSamples(samples, sampleCount);
	}

	if (fclose(inp) != 0)
	{
		fprintf(stderr, "Warning: could not close input file.\n");
	}
	free(samples);

	return returnValue;
}

void
usage(void)
{
	fprintf(stderr, "Usage: main\nInputs loaded from working directory.\nSee run.m for more information.\n");
}
