/*
	Authored 2021, Phillip Stanley-Marbell.

	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

	*	Redistributions of source code must retain the above
		copyright notice, this list of conditions and the following
		disclaimer.

	*	Redistributions in binary form must reproduce the above
		copyright notice, this list of conditions and the following
		disclaimer in the documentation and/or other materials
		provided with the distribution.

	*	Neither the name of the author nor the names of its
		contributors may be used to endorse or promote products
		derived from this software without specific prior written
		permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdint.h>
#include <inttypes.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <uncertain.h>

enum
{
	kBrownHamModelConstantNumberSamplesInDistributionFiles	= 1000,
} BrownHamModelConstant;

static void	usage(void);
static void	loadDoubleDistFromPath(const char *  filename, int sampleCount, double *  returnValue);


int
main(int argc, char *  argv[])
{
	double	sigmaC, G, M, Rs, b, gamma, phi;

	if (argc != 1)
	{
		usage();
	}

	loadDoubleDistFromPath("../inputs/G.csv", kBrownHamModelConstantNumberSamplesInDistributionFiles, &G);
	loadDoubleDistFromPath("../inputs/M.csv", kBrownHamModelConstantNumberSamplesInDistributionFiles, &M);
	loadDoubleDistFromPath("../inputs/Rs.csv", kBrownHamModelConstantNumberSamplesInDistributionFiles, &Rs);
	loadDoubleDistFromPath("../inputs/b.csv", kBrownHamModelConstantNumberSamplesInDistributionFiles, &b);
	loadDoubleDistFromPath("../inputs/gamma.csv", kBrownHamModelConstantNumberSamplesInDistributionFiles, &gamma);
	loadDoubleDistFromPath("../inputs/phi.csv", kBrownHamModelConstantNumberSamplesInDistributionFiles, &phi);

	sigmaC = ((M*gamma)/(2.0*b))*(sqrt((8.0*gamma*phi*Rs)/(M_PI*G*pow(b, 2))) - phi);
	libUncertainDoublePrint(sigmaC);

	return 0;
}

void
loadDoubleDistFromPath(const char *  filename, int sampleCount, double *  returnValue)
{
	FILE *		fp;
	double *	samples;

	fp = fopen(filename, "r");
	if (fp == NULL)
	{
		perror("Error: could not open input file");
		exit(EXIT_FAILURE);
	}

	printf("Number of samples from %s: %d\n", filename, sampleCount);

	samples = malloc(sampleCount * sizeof(double));
	if (samples == NULL)
	{
		perror("Error: could not allocate memory for samples");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < sampleCount; i++)
	{
		fscanf(fp, "%lf\n", &samples[i]);
	}

	*returnValue = libUncertainDoubleDistFromSamples(samples, sampleCount);
	
	if (fclose(fp) != 0)
	{
		perror("Warning: could not close input file");
	}
	free(samples);

	return;
}

void
usage(void)
{
	fprintf(stderr, "Usage: Brown-Ham-model\n");

	exit(EXIT_FAILURE);
}
