/*
	Authored 2020, James Timothy Meech.
	Authored 2021, Orestis Kaparounakis.

	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

	*	Redistributions of source code must retain the above
		copyright notice, this list of conditions and the following
		disclaimer.

	*	Redistributions in binary form must reproduce the above
		copyright notice, this list of conditions and the following
		disclaimer in the documentation and/or other materials
		provided with the distribution.

	*	Neither the name of the author nor the names of its
		contributors may be used to endorse or promote products
		derived from this software without specific prior written
		permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <uncertain.h>

enum EDoubleMultiplyMode {
	kDirectComputation = 1 << 0,
} DoubleMultiplyMode;

int
main(int argc, char *  argv[])
{
	int32_t	mode = 0;
	
	char	filenameBuffer[100];
	double *	samplesA;
	double *	samplesB;
	int	numberOfSamplesA;
	int	numberOfSamplesB;
	double	A = 0;
	double	B = 0;
	double	C = 0;
	FILE *	inp;	

	if (argc < 2)
	{
		printf("Please provide input file name.\n");
		exit(EXIT_FAILURE);
	}

	/*
	 *	Get mode from arguments.
	 */
	if (argc > 2 && strcmp("1", argv[1]) == 0)
	{
		mode |= kDirectComputation;
	}

	/*
	 *	Get input from arguments.
	 */ 
	strcpy(filenameBuffer, argv[argc-1]);
	inp = fopen(filenameBuffer, "r");
	if (inp == NULL)
	{
		perror("error: could not open input file");
		exit(EXIT_FAILURE);
	}

	/*
	 *	Load samples for variable A.
	 */
	fscanf(inp, "%d\n", &numberOfSamplesA);
	printf("Number of samples for A: %d\n", numberOfSamplesA);
	samplesA = malloc(numberOfSamplesA * sizeof(double));
	if (samplesA == NULL)
	{
		perror("error: could not allocate memory for samplesA");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < numberOfSamplesA; i++)
	{
		fscanf(inp, "%lf\n", &samplesA[i]);
	}

	/*
	 *	Load samples for variable B.
	 */
	fscanf(inp, "%d\n", &numberOfSamplesB);
	printf("Number of samples for B: %d\n", numberOfSamplesB);
	samplesB = malloc(numberOfSamplesB * sizeof(double));
	if (samplesB == NULL)
	{
		perror("error: could not allocate memory for samplesB");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < numberOfSamplesB; i++)
	{
		fscanf(inp, "%lf\n", &samplesB[i]);
	}

	fclose(inp);

	printf("\n");

	if (!(mode && kDirectComputation))
	{
		/*
		 *	Load samples into distributional double variables.
		 */
		A = libUncertainDoubleDistFromSamples(samplesA, numberOfSamplesA);
		B = libUncertainDoubleDistFromSamples(samplesB, numberOfSamplesB);

		printf("A = %lf\n", A);
		libUncertainDoublePrint(A);

		printf("B = %lf\n", B);
		libUncertainDoublePrint(B);

		/*
		 *	Perform a conventional-syntax multiplication.
		 */
		C = A * B;

		printf("C = A * B = %lf\n", C);
		libUncertainDoublePrint(C);
	}
	else
	{
		/*
		 *	Direct computation of the product A*B.
		 */
		printf("Direct Computation\n");

		double conv[numberOfSamplesA][numberOfSamplesB];
		printf("Computing: \n");
		double trueMean = 0;
		double temp = 0;
		printf("|..........|\r|");
		for (size_t i = 0; i < numberOfSamplesA; i++)
		{
			if (i % 100 == 0)
			{
				printf("#");
				fflush(stdout);				
			}
			
			for (size_t j = 0; j < numberOfSamplesB; j++)
			{
				conv[i][j] = samplesA[i]*samplesB[j];
				temp += conv[i][j];
			}
		}
		printf("|\n");
		trueMean = temp / (numberOfSamplesA*numberOfSamplesB);
		printf("C (authoritative mean) = %lf\n", trueMean);
		
		C = libUncertainDoubleDistFromSamples(&conv[0][0], numberOfSamplesA*numberOfSamplesB);
		printf("C = %lf\n", C);
		libUncertainDoublePrint(C);
	}

	free(samplesA);
	free(samplesB);

	return 0;
}
