/*
	Authored 2020, James Timothy Meech.
	Authored 2021, Orestis Kaparounakis.

	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

	*	Redistributions of source code must retain the above
		copyright notice, this list of conditions and the following
		disclaimer.

	*	Redistributions in binary form must reproduce the above
		copyright notice, this list of conditions and the following
		disclaimer in the documentation and/or other materials
		provided with the distribution.

	*	Neither the name of the author nor the names of its
		contributors may be used to endorse or promote products
		derived from this software without specific prior written
		permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <uncertain.h>

typedef enum {
	kBenchmarkModesDirectComputation = 1 << 0,
} BenchmarkModes;

typedef enum {
	kBenchmarkDefaultIterations 		= 10,
	kBenchmarkDefaultTracingGranularity 	= 1,
} BenchmarkDefaults;

double 		loadDoubleDistFromPath(double *  var, const char *  filename);
static void	usage(void);


int
main(int argc, char *  argv[])
{
	BenchmarkModes	mode = 0;

	double		A, C;

	if (argc > 3)
	{
		usage();
	}

	/*
	 *	Get mode from arguments.
	 */
	if (argc > 1 && strcmp("1", argv[1]) == 0)
	{
		mode |= kBenchmarkModesDirectComputation;
	}
	
	if (!(mode && kBenchmarkModesDirectComputation))
	{
		loadDoubleDistFromPath(&A, "input-A.txt");
		printf("A = %lf\n", A);
		libUncertainDoublePrint(A);

		C = A * A;
		printf("C = A * A = %lf\n", C);
		libUncertainDoublePrint(C);
	}
	else
	{
		FILE *		inp = fopen("input-A.txt", "r");
		double *	samplesA;
		int		numberOfSamplesA;
		double *	result;

		printf("Explicit computation of A * A.\n");

		if (inp == NULL)
		{
			perror("error: could not open input file");
			exit(EXIT_FAILURE);
		}

		fscanf(inp, "%d\n", &numberOfSamplesA);
		printf("Number of samples for A: %d\n", numberOfSamplesA);
		samplesA = malloc(numberOfSamplesA * sizeof(double));
		if (samplesA == NULL)
		{
			perror("error: could not allocate memory for samplesA");
			exit(EXIT_FAILURE);
		}
		for (int i = 0; i < numberOfSamplesA; i++)
		{
			fscanf(inp, "%lf\n", &samplesA[i]);
		}

		result = malloc(numberOfSamplesA * sizeof(double));
		if (result == NULL)
		{
			perror("error: could not allocate memory for samplesA");
			exit(EXIT_FAILURE);
		}
		for (size_t i = 0; i < numberOfSamplesA; i++)
		{
			result[i] = samplesA[i]*samplesA[i];
		}

		A = libUncertainDoubleDistFromSamples(samplesA, numberOfSamplesA);
		C = libUncertainDoubleDistFromSamples(result, numberOfSamplesA);
		printf("C = A * A = %lf\n", C);
		libUncertainDoublePrint(C);

		free(result);
		free(samplesA);
	}

	return 0;
}

double
loadDoubleDistFromPath(double *  var, const char *  filename)
{
	FILE *		inp;
	int		sampleCount;
	double *	samples;
	double		returnValue;

	inp = fopen(filename, "r");
	if (inp == NULL)
	{
		/*
		 *	`fprintf(stderr,...)` may suffer from emulator issue #92.
		 */
		fprintf(stderr, "Error: could not open input file: %s\n", filename);
		exit(EXIT_FAILURE);
	}
	fscanf(inp, "%d\n", &sampleCount);

	printf("Number of samples from %s: %d\n", filename, sampleCount);

	samples = malloc(sampleCount * sizeof(double));
	if (samples == NULL)
	{
		fprintf(stderr, "Error: could not allocate memory for samples.\n");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < sampleCount; i++)
	{
		fscanf(inp, "%lf\n", &samples[i]);
	}

	if (var == NULL )
	{
		returnValue = libUncertainDoubleDistFromSamples(samples, sampleCount);
	}
	else
	{
		(*var) = libUncertainDoubleDistFromSamples(samples, sampleCount);
	}

	if (fclose(inp) != 0)
	{
		fprintf(stderr, "Warning: could not close input file.\n");
	}
	free(samples);

	return returnValue;
}

void
usage(void)
{
	fprintf(stderr, "Usage: double-multiply-autocorrelation [<1> (explicit computation)]\nSee run.m file.\n");
	exit(EXIT_FAILURE);
}
