/*
	Authored 2020, James Timothy Meech.
	Authored 2021, Phillip Stanley-Marbell.

	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

	*	Redistributions of source code must retain the above
		copyright notice, this list of conditions and the following
		disclaimer.

	*	Redistributions in binary form must reproduce the above
		copyright notice, this list of conditions and the following
		disclaimer in the documentation and/or other materials
		provided with the distribution.

	*	Neither the name of the author nor the names of its
		contributors may be used to endorse or promote products
		derived from this software without specific prior written
		permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <uncertain.h>

typedef enum
{
	kBenchmarkModesExplicitComputation	= (1 << 0),
} BenchmarkModes;

static	void	usage(void);



int
main(int argc, char *argv[])
{
	FILE *			fpA;
	FILE *			fpB;
	double *			samplesA;
	double *			samplesB;
	double			A, B, C;
	int			numberOfSamples;
	BenchmarkModes		mode = 0;


	if (argc < 3)
	{
		usage();
	}

	/*
	 *	Get mode from arguments.
	 */
	if ((argc == 4) && !strcmp("1", argv[3]))
	{
		mode |= kBenchmarkModesExplicitComputation;
	}

	fpA = fopen(argv[1], "r");
	if (fpA == NULL)
	{
		fprintf(stderr, "Opening input samples for A failed...\n");
		return -1;
	}

	fscanf(fpA, "%d\n", &numberOfSamples);
	if (numberOfSamples <= 0)
	{
		fprintf(stderr, "Illegal number of samples (%d) specified in samples file ...\n", numberOfSamples);
		usage();
	}

	samplesA = malloc(numberOfSamples*sizeof(double));
	if (samplesA == NULL)
	{
		fprintf(stderr, "alloca() failed for samples array...\n");
		usage();
	}

	for(int i = 0; i <= numberOfSamples; i++)
	{
		fscanf(fpA, "%lf\n", &samplesA[i]);
	}

	fpB = fopen(argv[2], "r");
	if (fpB == NULL)
	{
		fprintf(stderr, "Opening input samples for B failed...\n");
		return -1;
	}

	fscanf(fpB, "%d\n", &numberOfSamples);
	if (numberOfSamples <= 0)
	{
		fprintf(stderr, "Illegal number of samples (%d) specified in samples file ...\n", numberOfSamples);
		usage();
	}

	samplesB = malloc(numberOfSamples*sizeof(double));
	if (samplesB == NULL)
	{
		fprintf(stderr, "alloca() failed for samples array...\n");
		usage();
	}

	for (int i = 0; i <= numberOfSamples; i++)
	{
		fscanf(fpB, "%lf\n", &samplesB[i]);
	}

	fclose(fpA);
	fclose(fpB);

	if (mode & kBenchmarkModesExplicitComputation)
	{
		double	convolutionMethodResults[numberOfSamples][numberOfSamples];
		double	convolutionMethodMean = 0;
		double	convolutionMethodTotal = 0;

		/*
		 *	Explicit computation of the quotient A/B.
		 */
		printf("Explicit computation of A/B...\n");

		for (size_t i = 0; i < numberOfSamples; i++)
		{
			for (size_t j = 0; j < numberOfSamples; j++)
			{
				convolutionMethodResults[i][j] = samplesA[i] / samplesB[j];
				convolutionMethodTotal += convolutionMethodResults[i][j];
			}
		}

		convolutionMethodMean = convolutionMethodTotal / (numberOfSamples*numberOfSamples);
		printf("C (explictly-calculated convolution mean) = %lf\n", convolutionMethodMean);
		
		C = libUncertainDoubleDistFromSamples(&convolutionMethodResults[0][0], numberOfSamples*numberOfSamples);
		printf("C = %lf\n", C);
		libUncertainDoublePrint(C);
	}
	else
	{
		/*
		 *	Implicit computation of the quotient A/B.
		 */
		printf("Implicit computation of A/B...\n");

		A = libUncertainDoubleDistFromSamples(samplesA, numberOfSamples);
		printf("A = %lf\n\n", A);
		libUncertainDoublePrint(A);

		B = libUncertainDoubleDistFromSamples(samplesB, numberOfSamples);
		printf("B = %lf\n\n", B);
		libUncertainDoublePrint(B);

		C = A/B;
		printf("C = %lf\n\n", C);
		libUncertainDoublePrint(C);
	}


	return 0;
}

void
usage(void)
{
	fprintf(stderr, "Usage: double-divide <samples file for A> <samples file for B> <mode: 1 for explicit computation, 0 for implicit uncertainty tracking>\n");

	exit(EXIT_FAILURE);
}
