/*
	Authored 2020, James Timothy Meech.
	Authored 2021, Orestis Kaparounakis.

	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

	*	Redistributions of source code must retain the above
		copyright notice, this list of conditions and the following
		disclaimer.

	*	Redistributions in binary form must reproduce the above
		copyright notice, this list of conditions and the following
		disclaimer in the documentation and/or other materials
		provided with the distribution.

	*	Neither the name of the author nor the names of its
		contributors may be used to endorse or promote products
		derived from this software without specific prior written
		permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
	POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <uncertain.h>

#include "matrix-multiplication.h"

typedef enum
{
	kBenchmarkModesCLIArguments = 1 << 0,
} BenchmarkModes;

typedef enum
{
	kBenchmarkDefaultMatrixRows = 8,
	kBenchmarkDefaultMatrixCols = 8,
} BenchmarkDefaults;

double loadDoubleDistFromPath(double *var, const char *filename);
void printMatrix(double m[kBenchmarkDefaultMatrixRows][kBenchmarkDefaultMatrixCols], size_t rows, size_t cols); //FIXME: patch to make mats static
void loadDoubleDistFromPathTemplate(double *var, const char *arg0, size_t arg1, size_t arg2);
static void usage(void);

int main(int argc, char *argv[])
{
	BenchmarkModes	mode = 0;
	uint64_t	rowsA;
	uint64_t	colsA;
	uint64_t	rowsB;
	uint64_t	colsB;

	char	*arrayAIdentifier;
	char	*arrayBIdentifier;

	double	matrixA[kBenchmarkDefaultMatrixRows][kBenchmarkDefaultMatrixCols];
	double	matrixB[kBenchmarkDefaultMatrixRows][kBenchmarkDefaultMatrixCols];
	double	result[kBenchmarkDefaultMatrixRows][kBenchmarkDefaultMatrixCols];

	if (argc == 1 + 6)
	{
		mode |= kBenchmarkModesCLIArguments;
	}
	else if (argc > 1)
	{
		usage();
	}

	if (mode && kBenchmarkModesCLIArguments)
	{
		arrayAIdentifier = argv[1];
		rowsA = (uint64_t)strtoull(argv[2], NULL, 10);
		colsA = (uint64_t)strtoull(argv[3], NULL, 10);
		arrayBIdentifier = argv[4];
		rowsB = (uint64_t)strtoull(argv[5], NULL, 10);
		colsB = (uint64_t)strtoull(argv[6], NULL, 10);
	}
	else
	{
		rowsA = (uint64_t)kBenchmarkDefaultMatrixRows;
		colsA = (uint64_t)kBenchmarkDefaultMatrixCols;
		rowsB = (uint64_t)kBenchmarkDefaultMatrixRows;
		colsB = (uint64_t)kBenchmarkDefaultMatrixCols;
		arrayAIdentifier = "A";
		arrayBIdentifier = "B";
	}

	/*
	 *	Load matrix A
	 */
	for (size_t i = 0; i < rowsA; i++)
	{
		for (size_t j = 0; j < colsA; j++)
		{
			loadDoubleDistFromPathTemplate(&matrixA[i][j], arrayAIdentifier, i, j);
		}
	}

	/*
	 *	Load matrix B
	 */
	for (size_t i = 0; i < rowsB; i++)
	{
		for (size_t j = 0; j < colsB; j++)
		{
			loadDoubleDistFromPathTemplate(&matrixB[i][j], arrayBIdentifier, i, j);
		}
	}

	printf("A = \n");
	printMatrix(matrixA, rowsA, colsA);
	printf("B = \n");
	printMatrix(matrixB, rowsA, colsB);

	matrixMultiplication(result, matrixA, matrixB, rowsA, colsA, colsB);

	printf("result =\n");
	printMatrix(result, rowsA, colsB);

	return 0;
}

void
printMatrix(double m[kBenchmarkDefaultMatrixRows][kBenchmarkDefaultMatrixCols], size_t rows, size_t cols)
{
	for (size_t i = 0; i < rows; i++)
	{
		for (size_t j = 0; j < cols; j++)
		{
			printf("%lf", m[i][j]);
			if (j != cols - 1)
			{
				printf("\t");
			}
		}
		printf("\n");
	}
}

void
loadDoubleDistFromPathTemplate(double *var, const char *arg0, size_t arg1, size_t arg2)
{
	const char *	pathTemplate = "input-%s-%d-%d.txt";
	char *	pathFilled;
	size_t 	needed = 0;

	needed = snprintf(NULL, 0, pathTemplate, arg0, arg1, arg2) + 1;
	pathFilled = malloc(needed);
	if (pathFilled == NULL)
	{
		fprintf(stderr, "Error: could not allocate memory.\n");
		exit(EXIT_FAILURE);
	}
	snprintf(pathFilled, needed, pathTemplate, arg0, arg1, arg2);

	loadDoubleDistFromPath(var, pathFilled);

	free(pathFilled);
	return;
}

double
loadDoubleDistFromPath(double *var, const char *filename)
{
	FILE *inp;
	int sampleCount;
	double *samples;
	double returnValue;

	inp = fopen(filename, "r");
	if (inp == NULL)
	{
		/*
		 *	`fprintf(stderr,...)` may suffer from emulator issue #92.
		 */
		fprintf(stderr, "Error: could not open input file: %s\n", filename);
		exit(EXIT_FAILURE);
	}
	fscanf(inp, "%d\n", &sampleCount);

	printf("Number of samples from %s: %d\n", filename, sampleCount);

	samples = malloc(sampleCount * sizeof(double));
	if (samples == NULL)
	{
		fprintf(stderr, "Error: could not allocate memory for samples.\n");
		exit(EXIT_FAILURE);
	}
	for (int i = 0; i < sampleCount; i++)
	{
		fscanf(inp, "%lf\n", &samples[i]);
	}

	if (var == NULL)
	{
		returnValue = libUncertainDoubleDistFromSamples(samples, sampleCount);
	}
	else
	{
		(*var) = libUncertainDoubleDistFromSamples(samples, sampleCount);
	}

	if (fclose(inp) != 0)
	{
		fprintf(stderr, "Warning: could not close input file.\n");
	}
	free(samples);

	return returnValue;
}

void
usage(void)
{
	fprintf(stderr, "Usage: main [<array-identifier> <array-rows> <array-columns> <array-identifier> <array-rows> <array-columns>]\nSee run.m file.\n");
	exit(EXIT_FAILURE);
}
