#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <uncertain.h>




static void		usage(void);
static double	loadDoubleDistFromPath(double *  var, const char *  filename);

int
main(int argc, char *argv[])
{
	/* 
	 *	We carry out three experiments from PaCal paper (2012) by Jaroszewicz and Korzen
	 */
	
    /* 
	 *	1st Experiment: Product and quotient of two independent Cauchy variables, which are 
	 *	computed as quotient of two independent Gaussians (Figure 4.1 of PaCal paper) 
	 */
    printf("Running 1st Experiment: Product and quotient of two Cauchy random vaiables (Figure 4.1 of PaCal paper)\n\n");
    
    double 	cauchy1, cauchy2;
    double 	cauchyProduct, cauchyQuotient;
	double 	gaussian1, gaussian2, gaussian3, gaussian4;
	double 	gaussianProduct1, gaussianProduct2, gaussianProduct3;

    if (argc != 8)
	{
		usage();

		exit(EXIT_FAILURE);
	}

	/*
     *	Default version of first experiment is for 
	 *	mean value equal to 0.0 and variance equal to 1.0.
	 */
	gaussian1 = loadDoubleDistFromPath(NULL, argv[1]);

	gaussian2 = loadDoubleDistFromPath(NULL, argv[1]);

	gaussian3 = loadDoubleDistFromPath(NULL, argv[1]);

	gaussian4 = loadDoubleDistFromPath(NULL, argv[1]);

	cauchy1 = gaussian1/gaussian2;

	cauchy2 = gaussian3/gaussian4;

	printf("cauchy = %f\n", cauchy1);
	libUncertainDoublePrint(cauchy1);

	printf("cauchy = %f\n", cauchy2);
	libUncertainDoublePrint(cauchy2);

    cauchyProduct=cauchy1*cauchy2;
    cauchyQuotient=cauchy1/cauchy2;
    
    printf("Product of two Cauchy random variables (Compare to Figure 4.1):\n");
    printf("cauchyProduct = %f\n\n", cauchyProduct);
    libUncertainDoublePrint(cauchyProduct);
    
    printf("Quotient of two Cauchy random variables (Compare to Figure 4.1):\n");
    printf("cauchyQuotient = %f\n\n", cauchyQuotient);
    libUncertainDoublePrint(cauchyQuotient);
    /* 
	 *	End of 1st Experiment 
	 */
    
	/* 
	 *	2nd Experiment: Product of two independent Gaussians (Figure 4.5 of PaCal paper) 
	 */
	double gaussian11, gaussian12, gaussian21, gaussian22, gaussian31, gaussian32;
	
	printf("Running 2nd Experiment: Product of two Gaussians (Figure 4.5 of PaCal paper)\n\n");
	
	/*
     *	Default version of second experiment is for 
	 *	mean values equal to 1.0, 2.0, and 3.0 and variance equal to 1.0.
	 */
	gaussian11 = loadDoubleDistFromPath(NULL, argv[2]);
	gaussian12 = loadDoubleDistFromPath(NULL, argv[2]);
	gaussian21 = loadDoubleDistFromPath(NULL, argv[3]);
	gaussian22 = loadDoubleDistFromPath(NULL, argv[3]);
	gaussian31 = loadDoubleDistFromPath(NULL, argv[4]);
	gaussian32 = loadDoubleDistFromPath(NULL, argv[4]);
	
	gaussianProduct1 = gaussian11 * gaussian12;
	gaussianProduct2 = gaussian21 * gaussian22;
	gaussianProduct3 = gaussian31 * gaussian32;
	
	printf("\n");

	printf("N(1,1) multiplied with itself (Compare to the 1st plot of Figure 4.5):\n");
	printf("gaussianProduct1 = %f\n", gaussianProduct1);
	libUncertainDoublePrint(gaussianProduct1);
	
	printf("N(2,1) multiplied with itself (Compare to the 2nd plot of Figure 4.5):\n");
	printf("gaussianProduct2 = %f\n", gaussianProduct2);
	libUncertainDoublePrint(gaussianProduct2);
	
	printf("N(3,1) multiplied with itself (Compare to the 3rd plot of Figure 4.5):\n");
	printf("gaussianProduct3 = %f\n", gaussianProduct3);
	libUncertainDoublePrint(gaussianProduct3);
	
	/* 
	 *	End of 2nd Experiment 
	 */
	
	/* 
	 *	3rd Experiment: Coefficient of thermal expansion (Figure 4.6 of PaCal paper) 
	 */
	printf("Running 3rd Experiment: Coefficient of thermal expansion (Figure 4.6 of PaCal paper)\n\n");

	double initialLength, finalLength, temperatureDifference, thermalExpansionCoefficient;
	
	initialLength = loadDoubleDistFromPath(NULL, argv[5]);
	finalLength = loadDoubleDistFromPath(NULL, argv[6]);
	
	/* 
	 *	Temperature difference is Gaussian with mean 2 and standard deviation 1. 
	 */
	temperatureDifference = loadDoubleDistFromPath(NULL, argv[7]);
	
	thermalExpansionCoefficient = (finalLength/initialLength-1) / temperatureDifference;
	
	printf("Distribution of initial length L_a is uniform on [9,10]:\n");
	printf("initialLength = %f\n\n", initialLength);
	libUncertainDoublePrint(initialLength);

	printf("Distribution of final length L_b is uniform on [11,12]:\n");
	printf("finalLength = %f\n\n", finalLength);
	libUncertainDoublePrint(finalLength);

	printf("Distribution of temperature difference deltaT is N(2,1):\n");
	printf("temperatureDifference = %f\n\n", temperatureDifference);
	libUncertainDoublePrint(temperatureDifference);
	
	printf("The thermal expansion coefficient K:\n");
	printf("thermalExpansionCoefficient = %f\n\n", thermalExpansionCoefficient);
	libUncertainDoublePrint(thermalExpansionCoefficient);

	/* 
	 *	End of 3rd Experiment 
	 */

	return 0;
}

double
loadDoubleDistFromPath(double *  var, const char *  filename)
{
	FILE *		inp;
	int			sampleCount;
	double *	samples;
	double		returnValue;

	inp = fopen(filename, "r");
	if (inp == NULL)
	{
		printf("Could not open input file: %s\n", filename);
		exit(EXIT_FAILURE);
	}

	fscanf(inp, "%d\n", &sampleCount);

	printf("Number of samples from %s: %d\n", filename, sampleCount);

	samples = malloc(sampleCount * sizeof(double));
	if (samples == NULL)
	{
		perror("error: could not allocate memory for samples");
		exit(EXIT_FAILURE);
	}

	for (int i = 0; i < sampleCount; i++)
	{
		fscanf(inp, "%lf\n", &samples[i]);
	}

	if (var == NULL )
		returnValue = libUncertainDoubleDistFromSamples(samples, sampleCount);
	else
		(*var) = libUncertainDoubleDistFromSamples(samples, sampleCount);

	if (fclose(inp) != 0)
	{
		perror("warning: could not close input file");
	}

	free(samples);

	return returnValue;
}

void
usage(void)
{
	fprintf(stderr, "Usage: main\nInputs loaded from working directory.\nSee run.m for more information.\n");
}