#ifndef LINUX
#include <uncertain.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <math.h>

#define INITIAL_POS_GPS_LAT 	0.0
#define INITIAL_POS_GPS_LON 	0.0
#define INITIAL_POS_GPS_ALT 	0.0

#define EARTH_RADIUS 			6371000 //metres;
#define PI 						3.1415926

enum {
	kBenchmarkDefaultStrLen 	            	= 128,
    kBenchmarkDefaultGradientDescentIterations  = 1000,
    kBenchmarkSamplesPerDistribution			= 32,
	kBenchmarkSamplesPerFileLine				= 64,
	kBenchmarkDeltaTimeThresholdMs				= 1000,
	kBenchmarkDefaultNumberOfPositions			= 16,
	kBenchmarkPrintPointDistributions			= 0,
	kBenchmarkUseAltitudeInSpeecEstimation		= 0,
	kBenchmarkUpperBoundOfExaminedUserPositions	= -1 /*	-1 stands for unlimited	*/
};


double calcDistanceUsingGPSCoordinatesWithoutAltitude(double waypoint1_Lat, double waypoint1_Lon, double waypoint2_Lat, double waypoint2_Lon)
{
    double distanceDistr;
	
	waypoint1_Lat = (waypoint1_Lat / 180) * PI;
    waypoint2_Lat = (waypoint2_Lat / 180) * PI;
    waypoint1_Lon = (waypoint1_Lon / 180) * PI;
    waypoint2_Lon = (waypoint2_Lon / 180) * PI;

	double deltaLat = waypoint2_Lat - waypoint1_Lat;

	double deltaLon = waypoint2_Lon - waypoint1_Lon;

	double a = (sin(deltaLat/2) * sin(deltaLat/2)) + (sin(deltaLon/2) * sin(deltaLon/2) * cos(waypoint1_Lat) * cos(waypoint2_Lat));

	double c = 2 * atan(sqrt(a) / sqrt(1-a));

	distanceDistr = EARTH_RADIUS * c;
  	
    return distanceDistr;
}

double calcDistanceUsingGPSCoordinates(double waypoint1_Lat, double waypoint1_Lon, double waypoint1_Alt, double waypoint2_Lat, double waypoint2_Lon, double waypoint2_Alt)
{
    double distanceDistr;
	
	waypoint1_Lat = (waypoint1_Lat / 180) * PI;
    waypoint2_Lat = (waypoint2_Lat / 180) * PI;
    waypoint1_Lon = (waypoint1_Lon / 180) * PI;
    waypoint2_Lon = (waypoint2_Lon / 180) * PI;

    distanceDistr = sqrt(pow(EARTH_RADIUS+waypoint1_Alt, 2) + pow(EARTH_RADIUS+waypoint2_Alt, 2) - 2*(EARTH_RADIUS+waypoint1_Alt) * (EARTH_RADIUS+waypoint2_Alt) *
						(sin(waypoint1_Lat) * sin(waypoint2_Lat) + cos(waypoint1_Lat) * cos(waypoint2_Lat) * cos(waypoint1_Lon-waypoint2_Lon)));

    return distanceDistr;
}

int 
main(int argc, char *argv[])
{
	double	gpsLatSamples[kBenchmarkSamplesPerFileLine+1], gpsLonSamples[kBenchmarkSamplesPerFileLine+1];
	double	gpsAltSamples[kBenchmarkSamplesPerFileLine+1], gpsSpeedSamples; /*	gpsSpeedSamples is discarded	*/
	char 	inputBuffer[1280], fileName[256], activityStr[32];
	FILE *	inputUserFile;
	int		performSpeedCalc = 0, id, userName, activityId, positionsCnt = 0;
	double	timeStamp, gpsBearing, gpsAccuracy, gpsLatIncrement, gpsLonIncrement, gpsAltIncrement;
	double	gpsStartLatDistr, gpsStartLonDistr, gpsStartAltDistr;
	double	gpsFinishLatDistr, gpsFinishLonDistr, gpsFinishAltDistr;
	double	gpsPreviousLat, gpsPreviousLon, gpsPreviousAlt;
	double	timeStampInit=-1, deltaTimeStamp;
	double 	gpsCalcSpeedDistr[kBenchmarkDefaultNumberOfPositions], gpsCalcSpeedParticle;
	int		numOfSamplesPerDistribution=kBenchmarkSamplesPerDistribution;
	
	if (argc < 2)
	{
		printf("Please provide input file name.\n");

		exit(-1);
	}	

	strcpy(fileName, argv[1]);

	printf("Input filename: %s\n", fileName);

	inputUserFile = fopen(fileName, "r");

	if(inputUserFile == NULL)
    {
        printf("Error opening inputUserFile\n");
		
        exit(-1);
    }

	numOfSamplesPerDistribution = atoi(argv[2]);
	
	gpsPreviousLat = INITIAL_POS_GPS_LAT;
		
	gpsPreviousLon = INITIAL_POS_GPS_LON;
	
	gpsPreviousAlt = INITIAL_POS_GPS_ALT;

	/*	Ignore first line	*/
	fscanf(inputUserFile, "%s", inputBuffer);

	/*	id,username,timestamp,gps_lat_increment,gps_long_increment,gps_alt_increment,gps_speed,gps_bearing,gps_accuracy,activity_id,activity -> 11	*/
	while (fscanf(inputUserFile, "%s\n", inputBuffer) != EOF)
	{
		if (positionsCnt == kBenchmarkUpperBoundOfExaminedUserPositions)
		{
			break;
		}

		sscanf(inputBuffer, "%d,%d,%lf,%lf,%lf,%lf,%lf,%lf,%lf,%d,%s\n", 
			&id, &userName, &timeStamp, &gpsLatIncrement, &gpsLonIncrement, &gpsAltIncrement,
			&gpsSpeedSamples, &gpsBearing, &gpsAccuracy, &activityId, activityStr);

		/*	Current GPS point -- Bad name	*/
		gpsPreviousLat += gpsLatIncrement;
	
		gpsPreviousLon += gpsLonIncrement;

		gpsPreviousAlt += gpsAltIncrement;

		if (timeStampInit < 0) /*	Get initial point	*/
		{
			timeStampInit = timeStamp;
			
			gpsLatSamples[0] = gpsPreviousLat;
	
			gpsLonSamples[0] = gpsPreviousLon;

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsAltSamples[0] = gpsPreviousAlt;
			}

			/*	Read generated Lat samples	*/
			for (int samplesCnt=1; samplesCnt<kBenchmarkSamplesPerFileLine; samplesCnt++)
			{
				fscanf(inputUserFile, "%lf,", &gpsLatIncrement);

				gpsLatSamples[samplesCnt] = gpsLatSamples[0] + gpsLatIncrement;
			}

			fscanf(inputUserFile, "%lf\n", &gpsLatIncrement);

			gpsLatSamples[kBenchmarkSamplesPerFileLine] = gpsLatSamples[0] + gpsLatIncrement;

			/*	Read generated Lon samples	*/
			for (int samplesCnt=1; samplesCnt<kBenchmarkSamplesPerFileLine; samplesCnt++)
			{
				fscanf(inputUserFile, "%lf,", &gpsLonIncrement);

				gpsLonSamples[samplesCnt] = gpsLonSamples[0] + gpsLonIncrement;
			}

			fscanf(inputUserFile, "%lf\n", &gpsLonIncrement);

			gpsLonSamples[kBenchmarkSamplesPerFileLine] = gpsLonSamples[0] + gpsLonIncrement;

			/*	Read generated Alt samples	*/
			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				for (int samplesCnt=1; samplesCnt<kBenchmarkSamplesPerFileLine; samplesCnt++)
				{
					fscanf(inputUserFile, "%lf,", &gpsAltIncrement);

					gpsAltSamples[samplesCnt] = gpsAltSamples[0] + gpsAltIncrement;
				}

				fscanf(inputUserFile, "%lf\n", &gpsAltIncrement);

				gpsAltSamples[kBenchmarkSamplesPerFileLine] = gpsAltSamples[0] + gpsAltIncrement;
			}

#ifndef LINUX			
			gpsStartLatDistr = libUncertainDoubleDistFromSamples(gpsLatSamples, numOfSamplesPerDistribution);

			gpsStartLonDistr = libUncertainDoubleDistFromSamples(gpsLonSamples, numOfSamplesPerDistribution);
			
			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsStartAltDistr = libUncertainDoubleDistFromSamples(gpsAltSamples, numOfSamplesPerDistribution);
			}
#else
			gpsStartLatDistr = 0.0;

			gpsStartLonDistr = 0.0;

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsStartAltDistr = 0.0;
			}
			
			/*	Calculate mean value of input samples	*/
			for (int i=0; i<numOfSamplesPerDistribution; i++)
			{
				gpsStartLatDistr += gpsLatSamples[i];

				gpsStartLonDistr += gpsLonSamples[i];

				if (kBenchmarkUseAltitudeInSpeecEstimation)
				{
					gpsStartAltDistr += gpsAltSamples[i];
				}
			}

			gpsStartLatDistr /= numOfSamplesPerDistribution;

			gpsStartLonDistr /= numOfSamplesPerDistribution;

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsStartAltDistr /= numOfSamplesPerDistribution;
			}
#endif
		}
		else if (timeStamp - timeStampInit > kBenchmarkDeltaTimeThresholdMs) /*	Get point after threshold ms	*/
		{
			/*	Get initial point	*/
			gpsLatSamples[0] = gpsPreviousLat;
	
			gpsLonSamples[0] = gpsPreviousLon;

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsAltSamples[0] = gpsPreviousAlt;
			}

			/*	Read generated Lat samples	*/
			for (int samplesCnt=1; samplesCnt<kBenchmarkSamplesPerFileLine; samplesCnt++)
			{
				fscanf(inputUserFile, "%lf,", &gpsLatIncrement);

				gpsLatSamples[samplesCnt] = gpsLatSamples[0] + gpsLatIncrement;
			}

			fscanf(inputUserFile, "%lf\n", &gpsLatIncrement);

			gpsLatSamples[kBenchmarkSamplesPerFileLine] = gpsLatSamples[0] + gpsLatIncrement;

			/*	Read generated Lon samples	*/
			for (int samplesCnt=1; samplesCnt<kBenchmarkSamplesPerFileLine; samplesCnt++)
			{
				fscanf(inputUserFile, "%lf,", &gpsLonIncrement);

				gpsLonSamples[samplesCnt] = gpsLonSamples[0] + gpsLonIncrement;
			}

			fscanf(inputUserFile, "%lf\n", &gpsLonIncrement);

			gpsLonSamples[kBenchmarkSamplesPerFileLine] = gpsLonSamples[0] + gpsLonIncrement;

			/*	Read generated Alt samples	*/
			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				for (int samplesCnt=1; samplesCnt<kBenchmarkSamplesPerFileLine; samplesCnt++)
				{
					fscanf(inputUserFile, "%lf,", &gpsAltIncrement);

					gpsAltSamples[samplesCnt] = gpsAltSamples[0] + gpsAltIncrement;
				}

				fscanf(inputUserFile, "%lf\n", &gpsAltIncrement);

				gpsAltSamples[kBenchmarkSamplesPerFileLine] = gpsAltSamples[0] + gpsAltIncrement;
			}

#ifndef LINUX
			gpsFinishLatDistr = libUncertainDoubleDistFromSamples(gpsLatSamples, numOfSamplesPerDistribution);

			gpsFinishLonDistr = libUncertainDoubleDistFromSamples(gpsLonSamples, numOfSamplesPerDistribution);

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsFinishAltDistr = libUncertainDoubleDistFromSamples(gpsAltSamples, numOfSamplesPerDistribution);
			}
#else
			gpsFinishLatDistr = 0.0;

			gpsFinishLonDistr = 0.0;

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsFinishAltDistr = 0.0;
			}
			
			/*	Calculate mean value of input samples	*/
			for (int i=0; i<numOfSamplesPerDistribution; i++)
			{
				gpsFinishLatDistr += gpsLatSamples[i];

				gpsFinishLonDistr += gpsLonSamples[i];

				if (kBenchmarkUseAltitudeInSpeecEstimation)
				{
					gpsFinishAltDistr += gpsAltSamples[i];
				}
			}

			gpsFinishLatDistr /= numOfSamplesPerDistribution;

			gpsFinishLonDistr /= numOfSamplesPerDistribution;

			gpsFinishAltDistr /= numOfSamplesPerDistribution;
#endif
			performSpeedCalc = 1;
		}
		else
		{
			/* Discard the extra particle points	*/
			fscanf(inputUserFile, "%s\n", inputBuffer);

			fscanf(inputUserFile, "%s\n", inputBuffer);
		}
		
		if (performSpeedCalc == 1)
		{
			printf("gpsStartLatDistr = %f\n", gpsStartLatDistr);
#ifndef LINUX
			if (kBenchmarkPrintPointDistributions)
			{
				libUncertainDoublePrint(gpsStartLatDistr);
			}	
#endif
			printf("gpsStartLonDistr = %f\n", gpsStartLonDistr);
#ifndef LINUX
			if (kBenchmarkPrintPointDistributions)
			{
				libUncertainDoublePrint(gpsStartLonDistr);
			}
#endif

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				printf("gpsStartAltDistr = %f\n", gpsStartAltDistr);
#ifndef LINUX
				if (kBenchmarkPrintPointDistributions)
				{
					libUncertainDoublePrint(gpsStartAltDistr);
				}
#endif				
			}

			printf("gpsFinishLatDistr = %f\n", gpsFinishLatDistr);
#ifndef LINUX
			if (kBenchmarkPrintPointDistributions)
			{
				libUncertainDoublePrint(gpsFinishLatDistr);
			}			
#endif	
			printf("gpsFinishLonDistr = %f\n", gpsFinishLonDistr);
#ifndef LINUX
			if (kBenchmarkPrintPointDistributions)
			{
				libUncertainDoublePrint(gpsFinishLonDistr);
			}
#endif

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				printf("gpsFinishAltDistr = %f\n", gpsFinishAltDistr);
#ifndef LINUX
				if (kBenchmarkPrintPointDistributions)
				{
					libUncertainDoublePrint(gpsFinishAltDistr);
				}
#endif				
			}
		
			deltaTimeStamp = timeStamp - timeStampInit;
			
			if (!kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsCalcSpeedDistr[positionsCnt] = calcDistanceUsingGPSCoordinatesWithoutAltitude(
									gpsStartLatDistr, 
									gpsStartLonDistr, 
									gpsFinishLatDistr, 
									gpsFinishLonDistr
								) / deltaTimeStamp; 

				printf("Position %d: Mean value of speed estimation(w/o altitude): %f deltaTimeStamp: %f\n\n", positionsCnt, gpsCalcSpeedDistr[positionsCnt], deltaTimeStamp);
			}
			else
			{
				gpsCalcSpeedDistr[positionsCnt] = calcDistanceUsingGPSCoordinates(
									gpsStartLatDistr, 
									gpsStartLonDistr,
									gpsStartAltDistr,  
									gpsFinishLatDistr, 
									gpsFinishLonDistr,
									gpsFinishAltDistr
								) / deltaTimeStamp;

				printf("Position %d: Mean value of speed estimation(with altitude): %f deltaTimeStamp: %f\n\n", positionsCnt, gpsCalcSpeedDistr[positionsCnt], deltaTimeStamp);
			}
		
#ifndef LINUX			
			libUncertainDoublePrint(gpsCalcSpeedDistr[positionsCnt]);
#endif

			if (!kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsCalcSpeedParticle = calcDistanceUsingGPSCoordinatesWithoutAltitude(
									libUncertainDoubleNthMode(gpsStartLatDistr, 1), 
									libUncertainDoubleNthMode(gpsStartLonDistr, 1), 
									libUncertainDoubleNthMode(gpsFinishLatDistr, 1), 
									libUncertainDoubleNthMode(gpsFinishLonDistr, 1)
								) / deltaTimeStamp;

				printf("Position %d: Speed estimation using mean values (w/o altitude): %f deltaTimeStamp: %f\n\n", positionsCnt, gpsCalcSpeedParticle, deltaTimeStamp);
			}
			else
			{
				gpsCalcSpeedParticle = calcDistanceUsingGPSCoordinates(
									libUncertainDoubleNthMode(gpsStartLatDistr, 1), 
									libUncertainDoubleNthMode(gpsStartLonDistr, 1), 
									libUncertainDoubleNthMode(gpsStartAltDistr, 1), 
									libUncertainDoubleNthMode(gpsFinishLatDistr, 1), 
									libUncertainDoubleNthMode(gpsFinishLonDistr, 1), 
									libUncertainDoubleNthMode(gpsFinishAltDistr, 1)
								) / deltaTimeStamp;
				
				printf("Position %d: Speed estimation using mean values (with altitude): %f deltaTimeStamp: %f\n\n", positionsCnt, gpsCalcSpeedParticle, deltaTimeStamp);
			}

			/*	Remove that to always maintain the initial position as the starting point	*/
			gpsStartLatDistr = gpsFinishLatDistr;

			gpsStartLonDistr = gpsFinishLonDistr;

			if (kBenchmarkUseAltitudeInSpeecEstimation)
			{
				gpsStartAltDistr = gpsFinishAltDistr;
			}

			timeStampInit = timeStamp;

			performSpeedCalc = 0;

			positionsCnt++;
		}
	}
	
	fclose(inputUserFile);

	return 0;
}

