-- ------------------------------------------------------------------
-- Title: Quick Sequential Organ Failure Assessment (qSOFA)
-- This query extracts the quick sequential organ failure assessment.
-- This score was a recent revision of SOFA, aiming to detect patients at risk of sepsis.
-- The score is calculated on the first day of each ICU patients' stay - though needn't be.
-- ------------------------------------------------------------------

-- Reference for qSOFA:
--    Singer M, et al. The Third International Consensus Definitions for Sepsis and Septic Shock (Sepsis-3)
--    Seymour CW, et al. Assessment of Clinical Criteria for Sepsis: For the Third International Consensus Definitions for Sepsis and Septic Shock (Sepsis-3)

-- Variables used in qSOFA:
--  GCS, respiratory rate, systolic blood pressure

-- The following views required to run this query:
--  1) gcsfirstday - generated by gcs-first-day.sql
--  2) vitalsfirstday - generated by vitals-first-day.sql

-- Note:
--  The score is calculated for *all* ICU patients, with the assumption that the user will subselect appropriate ICUSTAY_IDs.
--  For example, the score is calculated for neonates, but it is likely inappropriate to actually use the score values for these patients.

DROP MATERIALIZED VIEW IF EXISTS QSOFA CASCADE;
CREATE MATERIALIZED VIEW QSOFA AS
with scorecomp as
(
select ie.icustay_id
  , v.SysBP_Min
  , v.RespRate_max
  , gcs.MinGCS
from icustays ie
left join vitalsfirstday v
  on ie.icustay_id = v.icustay_id
left join gcsfirstday gcs
  on ie.icustay_id = gcs.icustay_id
)
, scorecalc as
(
  -- Calculate the final score
  -- note that if the underlying data is missing, the component is null
  -- eventually these are treated as 0 (normal), but knowing when data is missing is useful for debugging
  select icustay_id
  , case
      when SysBP_Min is null then null
      when SysBP_Min   <= 100 then 1
      else 0 end
    as SysBP_score
  , case
      when MinGCS is null then null
      when MinGCS   <= 13 then 1
      else 0 end
    as GCS_score
  , case
      when RespRate_max is null then null
      when RespRate_max   >= 22 then 1
      else 0 end
    as RespRate_score
  from scorecomp
)
select ie.subject_id, ie.hadm_id, ie.icustay_id
, coalesce(SysBP_score,0)
 + coalesce(GCS_score,0)
 + coalesce(RespRate_score,0)
 as qSOFA
, SysBP_score
, GCS_score
, RespRate_score
from icustays ie
left join scorecalc s
  on ie.icustay_id = s.icustay_id
order by ie.icustay_id;
