LOG_QUAD_RULES = [(0, 0), (1, 1), (2, 2), (3, 1), (3, 3), (4, 2), (2, 4),
                  (0, 6), (4, 4), (5, 5), (6, 4), (7, 3), (6, 6),
                  (7, 7), (8, 8), (9, 9), (10, 10), (14, 6), (16, 4), (17, 3),
                  (11, 11), (16, 6), (18, 4), (19, 3), (12, 12), (15, 9),
                  (18, 6), (11, 13), (10, 14), (8, 16)]
LOG_LOG_QUAD_RULES = [(0, 0), (1, 2), (3, 1), (3, 2), (1, 3), (1, 4), (3, 3),
                      (5, 2), (7, 1), (11, 0), (9, 1), (7, 2), (5, 3), (3, 4),
                      (1, 5), (-1, 6)]

SQRT_QUAD_RULES = [(0, 0), (1, 1), (2, 2), (3, 3), (4, 4), (5, 5), (6, 6),
                   (7, 7), (8, 8), (9, 9)]
SQRTINV_QUAD_RULES = [(0, 0), (1, 1), (2, 2), (3, 3), (4, 4), (5, 5), (6, 6),
                      (7, 7), (8, 8), (9, 9)]


def log_quadrature_rule(N_poly, N_poly_log):
    """ Returns quadrature rule that is exact on 0^1 for
    p(x) + q(x)log(x) for deg(p) <= N_poly and deg(q) <= N_log.
    """
    lvls = (N_poly, N_poly_log)
    if lvls == (0, 0):
        return ((0.36787944117144232159552377016146086744581, ),
                ((1.00000000000000000000000000000000000000000, )))
    elif lvls == (1, 1):
        return ((
            0.08829686513765301175959513851848532315174,
            0.67518649090988720103627431696220487135249,
        ), (
            0.29849989370552491470847414328891897594758,
            0.70150010629447508529152585671108102405241,
        ))
    elif lvls == (2, 2):
        return ((
            0.02881166253095183117432844630588929438337,
            0.30406372961213765261086235863923723179495,
            0.81166922534407811686370517776050307614502,
        ), (
            0.10333070796492864676925159266381529251176,
            0.45463652597009870884069112142559509297114,
            0.44203276606497264439005728591058961451709,
        ))
    elif lvls == (3, 1):
        return ((
            0.03282362947812551317618097972175411710835,
            0.32677056929847683230968036425559808031592,
            0.82489036017782240029771247322766247606204,
        ), (
            0.11619324581447927325866999692014372918072,
            0.46747300569024018814078114540939614604442,
            0.41633374849528053860054885767046012477485,
        ))
    elif lvls == (3, 3):
        return ((
            0.01180259099784491826491730110952772437920,
            0.14282567997748369513685136917635909949385,
            0.48920152265457447871903130569940406739951,
            0.87867997406918370280768890626512031709831,
        ), (
            0.04339102877841439110189836963212064031500,
            0.24045209765946067597845006156666699704258,
            0.42140345225977593197881502421072436888031,
            0.29475342130234900094083654459048799376209,
        ))
    elif lvls == (4, 2):
        return ((
            0.01284924375750415937383837890289480150238,
            0.15169707873452995498631862341834886019539,
            0.50393294130801526330238410905835290729887,
            0.88419588782938008089584135224413219057247,
        ), (
            0.04700160891435411212393313504874682721943,
            0.25088787645469871671217291271696733534236,
            0.41935751136225065268677142171321693090078,
            0.28275300326869651847712253052106890653741,
        ))
    elif lvls == (2, 4):
        return ((
            0.013904928643030615382885268259473625091453882,
            0.160349490887963483676505367519454214852094296,
            0.517467627411374566430614521495157077743211264,
            0.888946038033987381600655151114803974181931342,
        ), (
            0.050620979401532056471584403233876960506663080,
            0.260657417636668755800682574557614655741771103,
            0.416542391582308648653563555609731595055439754,
            0.272179211379490539074169466598776788696126061,
        ))
    elif lvls == (0, 6):
        return ((
            0.049983035809587703085079094929382151643342,
            0.292850770727967806248112369454770262602022,
            0.641027672768554437281381513267552687401731,
            0.919940902909003456861028136089433754599186,
        ), (
            0.145576094093366312386007427957530846446686,
            0.321891831694662364893522461720874779018749,
            0.343523098402553137031962439837145807678499,
            0.189008975809418185688507670484448566856064,
        ))
    elif lvls == (4, 4):
        return ((
            0.00565222820508009713592725619673322436446,
            0.07343037174265227340615889388832093647469,
            0.28495740446255815371452760192605091016289,
            0.61948226408477838140680894305137332715616,
            0.91575808300469833378460918092797263341106,
        ), (
            0.02104694579185462911900268264212980706242,
            0.13070554074444669759107625499218677731647,
            0.28970230167131415684159035105657171744707,
            0.35022037012039871028554680413529463815675,
            0.20832484167198580616278390717381706001727,
        ))
    elif lvls == (5, 5):
        return ((
            0.00302580213754625870972997037525626840177,
            0.04097825415595061505346596310891297982215,
            0.17086329552687729472514982978613052222720,
            0.41325570884479324766648145516363018010160,
            0.70909514679062854395004591708377772175057,
            0.93823959037716709135502059471596547503225,
        ), (
            0.01135133881727260944049112382835582058174,
            0.07524106995491652291735628910920604024498,
            0.18879004161541635460950794377178886980344,
            0.28582072182722731198668348008454198267329,
            0.28448642789140880004515166984442400213528,
            0.15431039989375840100080949336168328456123,
        ))
    elif lvls == (6, 4):
        return ((
            0.00316017997969991764079805141973502020609,
            0.04252060743228469318303528219024580208609,
            0.17562400291443728015956588553535445941918,
            0.42031291702635120927072502833559353089265,
            0.71447887294581044995603823047159305752689,
            0.93965931927734119476687004099702658458337,
        ), (
            0.01184081311949377672150498543290110457332,
            0.07772229838410788643375856312797245606819,
            0.19222967305912322652587642630208060218322,
            0.28635345560126447125736074811207567327140,
            0.28091173256910775909597050664282713865076,
            0.15094202726690287996552877038214302525309,
        ))
    elif lvls == (7, 3):
        return ((
            0.00361942815689334528529302213024837902846,
            0.04762986103502181020972085354641735906024,
            0.19065563011677167668296059800213739059521,
            0.44136720881132660942758989185187672183547,
            0.72971726859297974074223290413498982770434,
            0.94354055877984985220655900515539549142586,
        ), (
            0.01350366950860238463323740876808054778768,
            0.08573477577701790667589547291393984964577,
            0.20227625337971153972988202182675208173757,
            0.28674132528818051244656319890421752727162,
            0.27009628904086946103141852107239155615509,
            0.14164768700561819548300337651461843740223,
        ))
    elif lvls == (6, 6):
        return ((
            0.00175965211846577428056264284948845223884,
            0.02446965071251336742764533734970449814707,
            0.10674805685878895418025978108310126399844,
            0.27580764129591738307785951205694010320150,
            0.51785514215183371615866896198180745377262,
            0.77181548536238490027464686949438183988203,
            0.95284134058109055899430658850308799116525,
        ), (
            0.00663266631902570511783904989050545154508,
            0.04579970797847533412557673481204533079388,
            0.12384020807131819455048956492193932808793,
            0.21210192602381193010791487545556985671573,
            0.26139064567200772564658060685851239803835,
            0.23163618029090938431881552610404399695298,
            0.11859866564445172613278364195738363786601,
        ))
    elif lvls == (7, 7):
        return ((
            0.00109069394192182289416511189527296165688,
            0.01544065354637409035654803129696733338088,
            0.06943486210070215589606820183704622774352,
            0.18744324425543703147418501397707839129148,
            0.37330442134309300687742241961309464279837,
            0.60049401369939721600757270345009107680283,
            0.81687733973466662645996840427629228518826,
            0.96283975926944796797672939381176215955140,
        ), (
            0.00412430118519834301985673277790633209473,
            0.02927037967468729611607963459474011124474,
            0.08311406745317000356959934495432356781311,
            0.15372167034228774045046056968680847507329,
            0.21349760952226085537723210057428492875948,
            0.23187027244357504563899737162078026441063,
            0.19053623904036773900755846964036097504358,
            0.09386546033845297682021577615079534556040,
        ))
    elif lvls == (8, 8):
        return ((
            0.00071107328870842923712719481144114271305,
            0.01019453302625490675705263439721390757082,
            0.04683386722112451208004788277264385778717,
            0.13036783136513152790927251167462566167484,
            0.27044725718891175341157923962100861098280,
            0.45831945709512795557047884231048355766553,
            0.66544463307035119311410115688403432223432,
            0.85011672984926898621352180781176804438987,
            0.96997704487058066904886010149227245320778,
        ), (
            0.00269489114902097253765081827466997206478,
            0.01949806475263514725048677490738025162488,
            0.05727368794913121851622494268319581220230,
            0.11155101434875819084452900903927327682925,
            0.16717487686324059413316947505369536328513,
            0.20369711869471113272844957892027359963509,
            0.20338245316419985992970422662716057693300,
            0.15865527983010627445081398546773458675504,
            0.07607261324819660960897118902661656067049,
        ))
    elif lvls == (9, 9):
        return ((
            0.00048296171068962949431831492727491752332,
            0.00698862921431576529132141505233085270390,
            0.03261139659467762873636756333311119502903,
            0.09282575738916595754304180899809018017197,
            0.19832725689540379524514488767882390493256,
            0.34888014297935319343087270520799659528213,
            0.53044055578795607735936289304815047537414,
            0.71676464851165508512715245960808578970658,
            0.87523455750623356818999184018286099911840,
            0.97524569868439287031164134845672934509662,
        ), (
            0.00183340007378984497434303924418708875693,
            0.01345312234599178938388397441120775272856,
            0.04049719431695833328218443450460775397747,
            0.08182236965890360616068411173063619462800,
            0.12919234277013753914262284470404141233106,
            0.16954531954725874717618674242665670393394,
            0.18910021653299560922416373215189726387401,
            0.17796575396147055085186835168716864603405,
            0.13372477061546151976426264031473240654640,
            0.06286551017703246003980012882486477718951,
        ))
    elif lvls == (10, 10):
        return ((
            0.00033932415013438556870174182341138987298,
            0.00494463807238220962318895575153983777212,
            0.02334658913510800709233212941284857214133,
            0.06757652642306404189451850394487230404961,
            0.14759528165052643154482340051832030095394,
            0.26692816830234757243752979641799037040739,
            0.41984102340899340310191927381115704558925,
            0.59097736922630627918350635190134959544998,
            0.75770013450251907728523355959708959505447,
            0.89462676131362450842509292035111260653978,
            0.97924361632111143264048128866088076065811,
        ), (
            0.00128973634219495730047286563205175915422,
            0.00956470186440151170867627991951187664310,
            0.02932398914942156022713177893293523196290,
            0.06087210046049704400086667151951852094058,
            0.09980233008453562466013423726106897834836,
            0.13791440941591388725449464512352523240867,
            0.16531158508385042266763372068708621154710,
            0.17310985679474128031739270544466135312103,
            0.15600833092113443234233602973375529608117,
            0.11400004591177338792399573810001614352500,
            0.05280291397153589159686532764586939626783,
        ))
    elif lvls == (14, 6):
        return ((
            0.000434069151702750253735233648457923333747,
            0.006235195491471156099860052468330066345653,
            0.028784560495265538993951938103605913722779,
            0.080950126502783894722811392245053625992811,
            0.171118818644087937876894405673024610859784,
            0.299158272925192431233141686604458166226787,
            0.455483150814167341706803032616111658822876,
            0.622880764646775108446306667062834791576118,
            0.905415590415528441733065733326134184410428,
            0.779999448744388043953819345632717416729186,
            0.981492633898872818015792981470714124874634,
        ), (
            0.001645531589447522207116242316366433421966,
            0.011944590316567324429519419842228285631232,
            0.035404846881700783579721434220110213285694,
            0.070394229245229134901656209688995028216923,
            0.109935442656684909387931647360254315380401,
            0.144512530104402884764944162156243939940152,
            0.165207642103203928819659571799127352909448,
            0.165959326461432766474406687908799632790329,
            0.103196918289543109146180438486454911810647,
            0.144642039204382910882491123126998104867106,
            0.047156903147404725406373063094421781746098,
        ))
    elif lvls == (16, 4):
        return ((
            0.000619233810731850925385769946834165182643,
            0.008645856529897286060130817771305235488439,
            0.038245053178278020142390302826557703417106,
            0.102210633640749533004652716625762438197557,
            0.204969551296373805822749108673955853280491,
            0.341168862983487922396416399486233522347220,
            0.497940260292428357738869251732416190028443,
            0.658105620759537447074572147206005605794457,
            0.916191394409970452576171691249671282999875,
            0.803207537792306896559717694329319093298969,
            0.983688089077539327083998945352699733259879,
        ), (
            0.002335077407358409953962434462212895050488,
            0.016247737057635748200244452878870373093673,
            0.045207569391347264244859377436830390186240,
            0.083477818961859498713221756300991183228892,
            0.121102668680636038797064446850026038614450,
            0.149095757878978155303357335460677948875899,
            0.161533427003921165320371541915928090265526,
            0.155676228047009027372949256539331555535438,
            0.092061580114198812412920951116433472610076,
            0.131646992927974136005974279675472931845528,
            0.041615142529081743675074167363225120693783,
        ))
    elif lvls == (17, 3):
        return ((
            0.010976820033086541052006628523034379901385,
            0.046590647555832600244283996435832197095489,
            0.119051823387609015004770574431251877254848,
            0.000809480223707175594013685722882790325413,
            0.229093739682745265916499745580980463081435,
            0.368493157179753381364426079481702474435568,
            0.523618241490737061759826790432549231270434,
            0.678275588181247896264897233437398569138745,
            0.921976238697456228803741020202406633201808,
            0.815988501866132203898662593382330534719040,
            0.984850691052772494807427511740136473719497,
        ), (
            0.020226296444505504076388277993167295069344,
            0.052997793042732686606331496928490430590690,
            0.091999027939189527720640285085767729969875,
            0.003035457420998474561972521185755207124481,
            0.126650816368952248642211641385693426400128,
            0.149829950189923037011186448287224464473824,
            0.157683968483035018794038199522781097007485,
            0.148851550473572592452528171005632297965788,
            0.085971601579700224542807525173949965926079,
            0.124082455923981073805624794451411961704338,
            0.038671082133409611786270638980126123767963,
        ))
    elif lvls == (11, 11):
        return ((
            0.000245284264977222214486999753643254347020,
            0.003593698020213691584953180824439056033669,
            0.017122925951596143704174347792003156327822,
            0.050205612323188193845736701920531341397219,
            0.111523585557362564421810450989050630107070,
            0.206003002905123975875249593328067481797187,
            0.332462697513606540097367548945215173118647,
            0.482606700965931942548503036927670018299377,
            0.641679470123992858986334235641107731404563,
            0.790709087183355406866399898023802984723841,
            0.909883828765562592119662746024572556141848,
            0.982347937715761951022141891215901227156130,
        ), (
            0.000933199883067142806309743480285368200955,
            0.006977495915143715985530451668181416880378,
            0.021694689021998533977157173992450559844052,
            0.045970694395591124692663622761661633031949,
            0.077527038695831784588421352717640519296356,
            0.111252518183739606826313138071923190483961,
            0.140273106008583333229227269735827131918570,
            0.157517130086829627554151648724681189257382,
            0.157408342223648975971552080600918143223231,
            0.137283832717785858586031372088803223895249,
            0.098196695474189500678766421272805578732159,
            0.044965257393590795103875724884822045235751,
        ))
    elif lvls == (16, 6):
        return ((
            0.000342587650696471235365129809618542079927,
            0.004932398239379700003860261466034555387010,
            0.022863757921931253349960272316808933175326,
            0.064707300968197410656490808290668632931473,
            0.138034671307054801658068310499695324863063,
            0.244359854287579760453337764584005665359473,
            0.378285433843258910332721042267622649722403,
            0.528518638730399957614839088218495991087720,
            0.679957967180239351818123104202605274781557,
            0.816167013567541400566394481934849109130746,
            0.921809305317133346921700666755300937370434,
            0.984791357303470103946238592949875235212733,
        ), (
            0.001299248489092345717922422781493927260485,
            0.009464332246944598397914538701065376723835,
            0.028239597220770429989079968785045747952203,
            0.056762265648750862058523234909068022551981,
            0.090157902095283350658139423270959114050226,
            0.121573800180752897337827920240470890245100,
            0.144358796351019323109570353424528365111204,
            0.153548509351857123775744984073118132807376,
            0.146549572383197013913938701077816177234927,
            0.123276460831165932508102444656855619642528,
            0.085963303416974481571277690176778095922714,
            0.038806211784191640961958317902800530497414,
        ))
    elif lvls == (18, 4):
        return ((
            0.000501366081904866202476756061993638554213,
            0.007012420897452078747000833060504110832138,
            0.031122349216906718950568991424429934512293,
            0.083626615198321350711388828811739960049548,
            0.169071654346530007434771518358586665528910,
            0.284653356272132009400380269675736033422610,
            0.421836563816752301857012352779573640366507,
            0.568530179986245279136199441645196654260369,
            0.711035895278626398196042539944054127175409,
            0.835747855459173091101163302766386275079516,
            0.930646321464987219363849020226386934075164,
            0.986565051148505478438285285833805039681577,
        ), (
            0.001891172421214558286590667016792671305620,
            0.013195286463791950810117981026866357167322,
            0.036920776076468457170243420670004413397637,
            0.068857812339363903633252273763211375452427,
            0.101532058185178880835797850968946498676798,
            0.128171227391491965997529241184475724584137,
            0.144147743991988456892406698177598185136640,
            0.146928227628999689994925905120319070795801,
            0.135790569418769095539166588338744738284469,
            0.111608929933239486069519088942666869510053,
            0.076642803488817771307720734041512475354749,
            0.034313392660675783462729550748861620334342,
        ))
    elif lvls == (19, 3):
        return ((
            0.000662513055012361481062860460457065020886,
            0.008998172273767646320980653752205542472780,
            0.038312315697868660993998091178658194425481,
            0.098418351820075087137234303032269978128429,
            0.190917377339996493595684860084886355438423,
            0.310558285848708083086610380523921876329237,
            0.447881729471481232011631859908162576290008,
            0.591171852773711276882483478445138561196966,
            0.727918261140607819703352975683535968980544,
            0.846081242709667448221538925884957105280255,
            0.935222456075500738822194846914945276849617,
            0.987474269678819848501462527551439618750666,
        ), (
            0.002485009018508776221769554921319588659857,
            0.016600770050172297437462313095281774285050,
            0.043738504664052113951111781934720334077426,
            0.076702029184703742587546296509237521798431,
            0.107373539877506411147780776092042131556042,
            0.130300516045431870113980975338376127478930,
            0.142370040690905031689546769171640509124094,
            0.142104134310967967868260390075668973202177,
            0.129363104955857887001858444759804377151266,
            0.105206987591213317368680410516508066519724,
            0.071750518362022184786904024233230500850841,
            0.032004845248658399825098263352170095296156,
        ))
    elif lvls == (12, 12):
        return ((
            0.000181654204416518112080458509185224719404,
            0.002672848790349611252669102383303945775082,
            0.012826941530277098219578266251314880073384,
            0.037993120751545488265716059328992648219579,
            0.085521720028976179450043814221237069759282,
            0.160606356416897680638663697584169686550148,
            0.264441443622600745325883394446408751906580,
            0.393113430336054641176681234875490715774455,
            0.537498494929546295669541788197461917746471,
            0.684235402702098707236365651289346747665772,
            0.817621486873285493443958673719021355302135,
            0.922088408240674764050737714075712380094731,
            0.984805887437127093363561440057650849842381,
        ), (
            0.000691641401275958947975789199388204444173,
            0.005204839576802010985606912756299889569405,
            0.016362516262130272039569144683437444205505,
            0.035232037087913650469060435939773861504994,
            0.060722504576943526693942168025250930712265,
            0.089664989996729992376015022798095729989720,
            0.117354801468484283410646882123819996818803,
            0.138451946844571611747936922099321668091528,
            0.148058863100483433833154698534859367072349,
            0.142755436151675163928461626605619521888691,
            0.121377592952669847824691380018182981767754,
            0.085378247902486978415728115775765690401463,
            0.038744582677833269327210901440184713533344,
        ))
    elif lvls == (15, 9):
        return ((
            0.000200333136088013515539504906203465358966,
            0.002936291604114984229950809725530317386755,
            0.014003862096056670294700175997879465814219,
            0.041138854335766059056605373140988279688874,
            0.091698005437539125127790574304951002399250,
            0.170343728292354223001054208417141234852018,
            0.277313853634692165143892461039413700793699,
            0.407662548492578495757518984485751492690476,
            0.551592841265789721507753258637267092091033,
            0.695752337884027763270648810727863645302866,
            0.825192267898665645446067016830830753373634,
            0.925612244853673050699402866181388885402676,
            0.985525718608446094036291003954325446044976,
        ), (
            0.000762226319863538473467524974130715200639,
            0.005702900011001280397974440652027489271531,
            0.017761725593070808220232963802690121307974,
            0.037774412498879543814662522477805407191822,
            0.064150745243399620017316861967926940949601,
            0.093193267800595236092592774206379649463532,
            0.119916795138392187454913721637907185458200,
            0.139125417329121299842657562707174676458392,
            0.146476187171839157767506613988456844725807,
            0.139313016053273893442881964890572209384204,
            0.117146751464978051490368249864625529243224,
            0.081747649255433458332589438118354024232682,
            0.036928906120151924652835360711949207112386,
        ))
    elif lvls == (18, 6):
        return ((
            0.000277271187667465203313120832436237292906,
            0.003998561577819322692135625359066200102298,
            0.018589650052302669700065483594955763258651,
            0.052851314828572085803572016874244144257898,
            0.113486956266655214422317146123354070563942,
            0.202730180923838864366194243458237352863299,
            0.317629783592143897152149954193984506973915,
            0.450676003245072966266037041868663588718947,
            0.591142004167921399930713962504733864724774,
            0.726663761183599622467333429225230752597257,
            0.844810416439255408345232670096094566546952,
            0.934519509669943928381490535588046046084936,
            0.987320008689363096252166497900783039830656,
        ), (
            0.001051838564204279660855049876825327440374,
            0.007681469488636963875904959566221885332125,
            0.023029116161782099627055937569121213615638,
            0.046652279966357881128717798939321410109541,
            0.075002148831379585121074018933279379409591,
            0.102980302518081884808251774006661260377371,
            0.125543775046438032384910109591835500311685,
            0.138733995713682869523660106261541911128175,
            0.140110896795215643378143290703763996955969,
            0.128836687147918522650109161435146925969029,
            0.105592473495141027868119183457402729995726,
            0.072397089387361496350670678079698084903101,
            0.032387926883799713622527931579180374451668,
        ))
    elif lvls == (11, 13):
        return ((
            0.000184531894190741422704304195853625226436,
            0.002713585546932306562407975288765308572891,
            0.013010034428054037904657894528659324761180,
            0.038486529454887349004927231616530606958403,
            0.086500267105340853711883038446009914610933,
            0.162166968826660772520246193623321341377008,
            0.266530298901488052456771119896250697637587,
            0.395504524089548964784115830114643589385445,
            0.539843382850620133346361542720085353934923,
            0.686172805881525822403775298257430013030380,
            0.818906838356305025125412198755519058832863,
            0.922690741127097600002618185384318636630700,
            0.984929410443476841824253361951501332831001,
        ), (
            0.000702523056123373612668569652746055113509,
            0.005282055199721900824258522849124345207523,
            0.016581501428491466417883223895010435673370,
            0.035635342575365655189671743865304419568791,
            0.061276294010325601161559821106421005439280,
            0.090249298868263818179845509789113576888568,
            0.117797015201554558867872545526090719352176,
            0.138592731704063328721655097360950350035135,
            0.147823161959138978886735368031559092782814,
            0.142196189474587475464010678977007484694819,
            0.120669665143340496512069541514255126047196,
            0.084760914818085308582609614938252803987770,
            0.038433306560938037579159762494164585209043,
        ))
    elif lvls == (10, 14):
        return ((
            0.000192103103801067933797698661626371625012,
            0.002820510004401877847944164242285481074946,
            0.013488741974073773932945464172717987904293,
            0.039769716928225992556709810470890731551786,
            0.089028348502471550048877216364432166668015,
            0.166167991830835161251148517316854227130300,
            0.271840735768562694348449033675398589813540,
            0.401530431477056886839237219148493582056062,
            0.545702370089204455769896433951977838470171,
            0.690975643882388993538448874954059946327286,
            0.822072111868903229936823742367215987357052,
            0.924166711292056849796876157222926660806216,
            0.985231224507709036835252333507436759861358,
        ), (
            0.000731140452602652638225813772008640758399,
            0.005484391937803255343932151372298891684792,
            0.017151833717459142057450827192038706879180,
            0.036676496102542340934336167745952044435187,
            0.062688830485605730770300325632927321445352,
            0.091714890870443162272428275286669912341589,
            0.118875290222521926806105869520735908900629,
            0.138894826996234234119099969228314940699323,
            0.147185487210891192497793027237095356085498,
            0.140774503365702925707185404849029449243423,
            0.118907758060183997366815680580147919532506,
            0.083242341343730396699251371418217925050406,
            0.037672209234279042787075116164562982943710,
        ))
    elif lvls == (8, 16):
        return ((
            0.000225271929580716244381863589134302011751,
            0.003284691046326309447362656467553342861239,
            0.015536279388579501595161797737003465416853,
            0.045149009156630269667899458119794131114071,
            0.099369817934623893380723295014583467006869,
            0.182087093278690918926404814276571540152630,
            0.292356757447828151751171967304071214343105,
            0.424135510618149996446551611772962364584055,
            0.567081250534302192272377204505238881050184,
            0.708078856891950892633950822789599670670295,
            0.833123105338135208816192622846838601595525,
            0.929246890829122718813422993678369600883620,
            0.986261655679735483955230751145199823518070,
        ), (
            0.000856304175876042223535931627329101675873,
            0.006357190345861744470302420915614409004055,
            0.019555273699236951995118903580155799455870,
            0.040920281825531721973930289946207601809018,
            0.068194526688659145599938087202655231944664,
            0.097086703969880612158437404093386064497878,
            0.122431177309009849806996974921080101222499,
            0.139376309177432366941221925319023456941791,
            0.144297192761491742019607912284557992257180,
            0.135331104357490139631571619855171219484804,
            0.112566928273987592925562710018130352698289,
            0.077958344539767851913798480519953803572394,
            0.035068662875774238339977339716734865435680,
        ))

    else:
        assert (False)


def log_log_quadrature_rule(N_poly, N_log):
    """ Returns quadrature rule that is exact on 0^1 for
    p(x) + q(x)log(x) + k(x)log(1-x) for deg(p) <= N_poly, deg(k) <= deg(q) <= N_log.
    """
    lvls = (N_poly, N_log)
    if lvls == (0, 0):
        return ((
            0.061151359021200679621983711149640788306168520381328374043636166,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.938848640978799320378016288850359211693831479618671625956363833,
        ), (
            0.208571756330286321730564759745458571876900255557976909886006629,
            0.582856487339427356538870480509082856246199488884046180227986741,
            0.208571756330286321730564759745458571876900255557976909886006629,
        ))
    elif lvls == (1, 2):
        return ((
            0.029358667367612155139465127104696648538108166305157159558772598,
            0.282459932137251345036698162259816603914212116759730466812268967,
            0.717540067862748654963301837740183396085787883240269533187731032,
            0.970641332632387844860534872895303351461891833694842840441227401,
        ), (
            0.102928607795495886856025918485997172750949852204720855456640758,
            0.397071392204504113143974081514002827249050147795279144543359241,
            0.397071392204504113143974081514002827249050147795279144543359241,
            0.102928607795495886856025918485997172750949852204720855456640758,
        ))

    elif lvls == (3, 1):
        return ((
            0.027796240603499428576801036014692502663682166488758811344471140,
            0.278451385012661581461577393560679082733795135355447417227892814,
            0.721548614987338418538422606439320917266204864644552582772107185,
            0.972203759396500571423198963985307497336317833511241188655528859,
        ), (
            0.098479015835843615418621605772317648130901425289959144878821505,
            0.401520984164156384581378394227682351869098574710040855121178494,
            0.401520984164156384581378394227682351869098574710040855121178494,
            0.098479015835843615418621605772317648130901425289959144878821505,
        ))
    elif lvls == (3, 2):
        return ((
            0.013701756886165988586680514893451189667768277910074829197282108,
            0.157402034095278881671782964877113246903216701377845415762861418,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.842597965904721118328217035122886753096783298622154584237138581,
            0.986298243113834011413319485106548810332231722089925170802717891,
        ), (
            0.049872115793175488374138193145306814506552218024648932767422993,
            0.254509275676902547584246263140195066310782883303430469677067352,
            0.391237217059843928083231087428996238365329797343841195111019306,
            0.254509275676902547584246263140195066310782883303430469677067352,
            0.049872115793175488374138193145306814506552218024648932767422993,
        ))
    elif lvls == (1, 3):
        return ((
            0.015162779926077379271119200799015052071811019603117045455612856,
            0.163885133545424780576435237703121124434731204198621988423729374,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.836114866454575219423564762296878875565268795801378011576270625,
            0.984837220073922620728880799200984947928188980396882954544387143,
        ), (
            0.054332187754320470491011704280299409205989139976435420756370199,
            0.255703217663275678369865034387716500113158286815287452265557841,
            0.379929189164807702278246522663968181361705146416554253956143919,
            0.255703217663275678369865034387716500113158286815287452265557841,
            0.054332187754320470491011704280299409205989139976435420756370199,
        ))
    elif lvls == (1, 4):
        return ((
            0.008750134824547052948923710927176823212162914472229792666954682,
            0.101029911011116747659326753644386314693745003373878827591366728,
            0.342018327198089677891645464600288110976865658707177130463190376,
            0.657981672801910322108354535399711889023134341292822869536809623,
            0.898970088988883252340673246355613685306254996626121172408633271,
            0.991249865175452947051076289072823176787837085527770207333045317,
        ), (
            0.031759114620320366598446033163859503920042831821724314556326109,
            0.166257128745225856189277492145628090388457868769592407379871852,
            0.301983756634453777212276474690512405691499299408683278063802038,
            0.301983756634453777212276474690512405691499299408683278063802038,
            0.166257128745225856189277492145628090388457868769592407379871852,
            0.031759114620320366598446033163859503920042831821724314556326109,
        ))
    elif lvls == (3, 3):
        return ((
            0.007604050966061114516540113866315348829271927967440058001198276,
            0.093901163911642027527632824880810839281307908293804715499894510,
            0.336339482173969990415235156365949083593719828477061452432870935,
            0.663660517826030009584764843634050916406280171522938547567129064,
            0.906098836088357972472367175119189160718692091706195284500105489,
            0.992395949033938885483459886133684651170728072032559941998801723,
        ), (
            0.028049332488176653335088551281407322124053407311309150321352893,
            0.160896772594535913555911271850846265766953031085416115454578261,
            0.311053894917287433109000176867746412108993561603274734224068844,
            0.311053894917287433109000176867746412108993561603274734224068844,
            0.160896772594535913555911271850846265766953031085416115454578261,
            0.028049332488176653335088551281407322124053407311309150321352893,
        ))
    elif lvls == (5, 2):
        return ((
            0.007924334686765955444446476510244160769580481221013508125269966,
            0.096154525630818659562946549380243433715365903635848048081709244,
            0.338303911289052955506965627625999402776686063781093849377111142,
            0.661696088710947044493034372374000597223313936218906150622888857,
            0.903845474369181340437053450619756566284634096364151951918290755,
            0.992075665313234044555553523489755839230419518778986491874730033,
        ), (
            0.029120542013624490996623918122568733886520272560282084512326927,
            0.162867847639386338048820341494496573653395227995573934974157129,
            0.308011610346989170954555740382934692460084499444143980513515943,
            0.308011610346989170954555740382934692460084499444143980513515943,
            0.162867847639386338048820341494496573653395227995573934974157129,
            0.029120542013624490996623918122568733886520272560282084512326927,
        ))
    elif lvls == (7, 1):
        return ((
            0.009675493107397894656750094525758179045772826021002407138018143,
            0.107524835788475831249492568389632384836438608369769547078201945,
            0.347487441622930598472675858539538220807299946087649800421980781,
            0.652512558377069401527324141460461779192700053912350199578019218,
            0.892475164211524168750507431610367615163561391630230452921798054,
            0.990324506892602105343249905474241820954227173978997592861981856,
        ), (
            0.034868924203187536189696598645390077291658495394174318422462295,
            0.171778027727069251873305890821623004973500884118510919977686254,
            0.293353048069743211936997510532986917734840620487314761599851450,
            0.293353048069743211936997510532986917734840620487314761599851450,
            0.171778027727069251873305890821623004973500884118510919977686254,
            0.034868924203187536189696598645390077291658495394174318422462295,
        ))
    elif lvls == (11, 0):
        return ((
            0.010868728320343345046022152359422064420183606569237051697691195,
            0.096566622935821336374829342492736963845799284838462511720261212,
            0.272297987161904370600396094403470328633275757317630823940377268,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.727702012838095629399603905596529671366724242682369176059622731,
            0.903433377064178663625170657507263036154200715161537488279738787,
            0.989131271679656654953977847640577935579816393430762948302308804,
        ), (
            0.036854619092694344827531837268099034152680060362224619970937720,
            0.135016262207243582914456946817206155652786271456107322913847640,
            0.209731038039358692259663375803563002660176663182516654566166281,
            0.236796161321406759996695680222263615068714009998302805098096715,
            0.209731038039358692259663375803563002660176663182516654566166281,
            0.135016262207243582914456946817206155652786271456107322913847640,
            0.036854619092694344827531837268099034152680060362224619970937720,
        ))
    elif lvls == (9, 1):
        return ((
            0.006609183801671364912711057196672202857395699920257882805701207,
            0.074656280680178530158877992159603831919359887314776860838009661,
            0.250114178484491397671413051554564299082240740285445466252057021,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.749885821515508602328586948445435700917759259714554533747942978,
            0.925343719319821469841122007840396168080640112685223139161990338,
            0.993390816198328635087288942803327797142604300079742117194298792,
        ), (
            0.023880579220911036189386250345638373712646617589146017960350226,
            0.121117657458479709192391857573482114643267184943631699972725077,
            0.223260410019178605689509830759615310103822445138082610041376174,
            0.263482706602861297857424122642528403080527504658279344051097043,
            0.223260410019178605689509830759615310103822445138082610041376174,
            0.121117657458479709192391857573482114643267184943631699972725077,
            0.023880579220911036189386250345638373712646617589146017960350226,
        ))
    elif lvls == (7, 2):
        return ((
            0.005098984660015853912240952018596381307190372233349806431067164,
            0.063462633250946747400944469068077804907838586873306027732891912,
            0.234963246268821890507664278551771323541905704303176436639037064,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.765036753731178109492335721448228676458094295696823563360962935,
            0.936537366749053252599055530931922195092161413126693972267108087,
            0.994901015339984146087759047981403618692809627766650193568932835,
        ), (
            0.018818197488752156435907327740921251008450422825964647954839964,
            0.109818899773814683513563654959888682842843090935611837791802063,
            0.229625970569744759297921563884691042645196936596808693805529423,
            0.283473864335376801505214906828998047007019099283229640895657097,
            0.229625970569744759297921563884691042645196936596808693805529423,
            0.109818899773814683513563654959888682842843090935611837791802063,
            0.018818197488752156435907327740921251008450422825964647954839964,
        ))
    elif lvls == (5, 3):
        return ((
            0.004582303076197091878865104933759099825664520130181921162474819,
            0.059113672627967872657022643310427083501172782526249263928867485,
            0.228180137349392106229518049615940690774372057680968634973509814,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.771819862650607893770481950384059309225627942319031365026490185,
            0.940886327372032127342977356689572916498827217473750736071132514,
            0.995417696923802908121134895066240900174335479869818078837525180,
        ), (
            0.017036822641759681160340769268947567348819298329525692405012089,
            0.104776866228196457708273902985315988647924712869153813695961429,
            0.231688766402711013849497729702764397224898179028097970345563300,
            0.292995089454665694563775196085944093556715619546445047106926362,
            0.231688766402711013849497729702764397224898179028097970345563300,
            0.104776866228196457708273902985315988647924712869153813695961429,
            0.017036822641759681160340769268947567348819298329525692405012089,
        ))
    elif lvls == (3, 4):
        return ((
            0.004650301268862259547506934298610690936891958228837997812348258,
            0.059695973270776936289357184238632958539717550281010647043787497,
            0.229106391463773583595109620564299694925105117844571412616714753,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.770893608536226416404890379435700305074894882155428587383285246,
            0.940304026729223063710642815761367041460282449718989352956212502,
            0.995349698731137740452493065701389309063108041771162002187651741,
        ), (
            0.017272143720561288469090944463409001188911709513593457839606363,
            0.105463845769295135333080472946282850982888083110064291090180438,
            0.231422415243886839608549477570225592282936274445690303792486249,
            0.291683190532513473178558210040165111090527865861303894555453898,
            0.231422415243886839608549477570225592282936274445690303792486249,
            0.105463845769295135333080472946282850982888083110064291090180438,
            0.017272143720561288469090944463409001188911709513593457839606363,
        ))
    elif lvls == (1, 5):
        return ((
            0.005278769481224410563124807650883341316861333570234691702274852,
            0.064341328902512037572766410532428232622594480004477001155680307,
            0.235554853786443021275457964619754807776047519166422535885274613,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.764445146213556978724542035380245192223952480833577464114725386,
            0.935658671097487962427233589467571767377405519995522998844319692,
            0.994721230518775589436875192349116658683138666429765308297725147,
        ), (
            0.019369759427953668352450450357254757506922957762470822508154135,
            0.110214131621570994191786753111147161635319256285132538164507182,
            0.228900849267194222898665423376134799344733305420540995458446746,
            0.283030519366562229114194746310926563026048961063711287737783871,
            0.228900849267194222898665423376134799344733305420540995458446746,
            0.110214131621570994191786753111147161635319256285132538164507182,
            0.019369759427953668352450450357254757506922957762470822508154135,
        ))
    elif lvls == (-1, 6):
        return ((
            0.010677753969509299127709077805651522728837700730172176052374743,
            0.085105788440601675332635047100099044718153887524951636477783931,
            0.256845732665995340696899766351881517082716396015186932196940567,
            0.500000000000000000000000000000000000000000000000000000000000000,
            0.743154267334004659303100233648118482917283603984813067803059432,
            0.914894211559398324667364952899900955281846112475048363522216068,
            0.989322246030490700872290922194348477271162299269827823947625256,
        ), (
            0.033266338381890194158449745078175551956908691303358774279451935,
            0.122067272159615474443527207474165092527612196575300877122454499,
            0.216781224919705788451758194268493112418964294166550174858907915,
            0.256792348512880862027025315917682579369795644869375769468349741,
            0.216781224919705788451758194268493112418964294166550174858907915,
            0.122067272159615474443527207474165092527612196575300877122454499,
            0.033266338381890194158449745078175551956908691303358774279451935,
        ))
    else:
        assert False


def sqrt_quadrature_rule(N_poly, N_poly_sqrt):
    """ Returns quadrature rule that is exact on 0^1 for
    p(x) + q(x)sqrt(x) for deg(p) <= N_poly and deg(q) <= N_poly_sqrt.
    """
    lvls = (N_poly, N_poly_sqrt)
    if lvls == (0, 0):
        return ((0.44444444444444444444444444444444444444, ),
                (1.00000000000000000000000000000000000000, ))
    elif lvls == (1, 1):
        return ((
            0.12606123086601862821632591103529303296,
            0.71393876913398137178367408896470696703,
        ), (
            0.36391723651204566121126199584967270044,
            0.63608276348795433878873800415032729955,
        ))
    elif lvls == (2, 2):
        return ((
            0.04508850417969317363296212165898076482,
            0.34872938419345759429201074127532493213,
            0.83067190754521657901380264726977593569,
        ), (
            0.13965395980290824506776254435815380982,
            0.45848221271917249338784118911264649289,
            0.40186382747791926154439626652919969727,
        ))
    elif lvls == (3, 3):
        return ((
            0.01953281968139194254742626358633432225,
            0.17339692801470563680061319669168082895,
            0.52295602692399076029118349203754173810,
            0.88905249698484993196571531928938138228,
        ), (
            0.06236194190001616434775029419315062126,
            0.25969509521646488165291240577925802922,
            0.40692913602054272158280895187170955021,
            0.27101382686297623241652834815588179929,
        ))
    elif lvls == (4, 4):
        return ((
            0.00970916313338208523477402787479390605,
            0.09274200880402887885216061240974461504,
            0.31587231391646161088563085922903369833,
            0.64318247791077182341748247714786996505,
            0.92196511061552089086615037044599583203,
        ), (
            0.03149582904338455237058463313697596258,
            0.14781774014523334070126643868340715764,
            0.29277397416933961739607573871191777197,
            0.33434927618873913098335124619539763716,
            0.19356318045330335854872194327230147063,
        ))
    elif lvls == (5, 5):
        return ((
            0.00533693493892329590380320659566764974,
            0.05325301043356392178869000900466147413,
            0.19477082834341022743852795834693923717,
            0.43958930092157689053171714715964142322,
            0.72577007234281065252822092445827511423,
            0.94222659858184518932383365384309912515,
        ), (
            0.01747660362721906351203460662479774148,
            0.08791033110101795101635324861086019025,
            0.19732230178131052824116902109671957589,
            0.28158510757639785623815367814185770165,
            0.27108499446303723368138079327610085541,
            0.14462066145101736731090865224966393528,
        ))
    elif lvls == (6, 6):
        return ((
            0.00316547571817082924999048003945700254,
            0.03248670695779345965898158819632354634,
            0.12434419111914291171674106810481172374,
            0.29937709080667702891538954437513979252,
            0.53906458432668453800104277775725971169,
            0.78379317891207303508478954562659818232,
            0.95554654993723597515084277367818781858,
        ), (
            0.01042872440561480817938274860756603627,
            0.05481671344374695088866209671645257295,
            0.13276939293098293996729597752727982681,
            0.21425013139174733998182450599971146454,
            0.25478179459917665383746897154992328269,
            0.22101851638174920484224147273921554935,
            0.11193472684698210230312422685985126735,
        ))
    elif lvls == (7, 7):
        return ((
            0.00199218996482702762204019573254393602,
            0.02084161617235884581067010638066802636,
            0.08226844131096154602254182717434130047,
            0.20685515168009757982722253253575523121,
            0.39446920588505375076723860559828063632,
            0.61731116554960597657988357626554211446,
            0.82569278556024739704026819800539309498,
            0.96475629508792054068999655000297739026,
        ), (
            0.00659038288449759770948463202217814023,
            0.03568580531197241678710559440208448516,
            0.09087863900939779616611084219138819921,
            0.15839919898463832024081637871978445343,
            0.21209471887186002140906219282231643234,
            0.22501159894177473234354419720856224745,
            0.18223804727274725269144492356575952070,
            0.08910160872311186265243123906792652143,
        ))
    elif lvls == (8, 8):
        return ((
            0.00131462899507382996291110845270313437,
            0.01394264523208896290416480532340773023,
            0.05625292212591594920223379846444097949,
            0.14583444643676914585843136139533937646,
            0.28947584931292823938697757432460598539,
            0.47655885020296432173666612267177611714,
            0.67878376425204414027125589364173076102,
            0.85675870432935571633030947675700049708,
            0.97138289825413393257419667337342755175,
        ), (
            0.00436169417154626174509810224003393198,
            0.02412000856957075810345430457380053454,
            0.06360964298210800252338021387173996796,
            0.11680239059033022113146304580328029603,
            0.16871664368984069878850003991717498171,
            0.20061761838673657930830044946517357804,
            0.19706748434469141032034604073854885090,
            0.15214851021861632516015860750792858121,
            0.07255600704655974291929919588231927759,
        ))
    elif lvls == (9, 9):
        return ((
            0.00090174277255559040062605867416579422,
            0.00966072992118867725888968404602886354,
            0.03960938987163694505494557118866803889,
            0.10501199191802636928364712016070713484,
            0.21461097119064956826082680084561075236,
            0.36646091497846359745137604622369268056,
            0.54588502435592935340408860152347834562,
            0.72741887932994511031450552808343034347,
            0.88034670494394913144868983612400312242,
            0.97630680264509329884576076220050970862,
        ), (
            0.00299828120481278805649627200821278483,
            0.01683863956596637483503273103740214403,
            0.04554918290652606517297529885212039260,
            0.08680381281430125184491223517276793341,
            0.13210615112670075833214900500975559272,
            0.16911421938165466486129116433100173671,
            0.18539378735544683168951398385942172453,
            0.17242260057835207285666672142382858353,
            0.12857430901816514574965307725773038337,
            0.06019901604807404660130951104775872423,
        ))

    else:
        assert (False)


def sqrtinv_quadrature_rule(N_poly, N_poly_sqrt):
    """ Returns quadrature rule that is exact on 0^1 for
    p(x) + q(x)1/sqrt(x) for deg(p) <= N_poly and deg(q) <= N_poly_sqrt.
    """
    lvls = (N_poly, N_poly_sqrt)
    if lvls == (0, 0):
        return ((0.25, ), (1., ))
    elif lvls == (1, 1):
        return ((
            0.0446581987385204510787589430823546055095,
            0.6220084679281462155879077235843120611571,
        ), (
            0.2113248654051871177454256097490212721761,
            0.7886751345948128822545743902509787278238,
        ))
    elif lvls == (2, 2):
        return ((
            0.0127016653792583114820734600217600389167,
            0.2500000000000000000000000000000000000000,
            0.7872983346207416885179265399782399610832,
        ), (
            0.0626120363218101730455963666787555771759,
            0.4444444444444444444444444444444444444444,
            0.4929435192337453825099591888767999783796,
        ))
    elif lvls == (3, 3):
        return ((
            0.0048207809894260143104510102394241237311,
            0.1089062557068338513905285800482630658350,
            0.4488872992916901161931943391515077530356,
            0.8659570925834785895343974991322336288268,
        ), (
            0.0241522034128332440720359725464196811082,
            0.2152140822717850020194694080312726812948,
            0.4369310725907611406074666427467279114698,
            0.3237026417246206133010279766755797261271,
        ))
    elif lvls == (0, 6):
        return ((
            0.0115150808760773921808404249155790057849,
            0.1880776026782814208067009643264663043843,
            0.5629806281290880804626829932208512394622,
            0.9037176918122826866982002220265949460613,
        ), (
            0.0549546712954705452039598503677135056942,
            0.3035315518468962788103417138214853268745,
            0.4029287718907212999613269813155084883433,
            0.2385850049669118760243714544952926790878,
        ))
    elif lvls == (-1, 7):
        return ((
            0.0336482680675068532985540048193035156251,
            0.2761843138724644405463052851573544729585,
            0.6346774762346366440178864667605024322530,
            0.9221566084920587288039209099295062458299,
        ), (
            0.1330575402859590547556376150838596515152,
            0.3297260192299490372135270788725851765490,
            0.3543270307231364048765173368779188557652,
            0.1944174731754821131443188160141885761498,
        ))
    elif lvls == (4, 4):
        return ((
            0.0022005553270232058216764041665882388175,
            0.0532526444285810300391307241113886539137,
            0.2500000000000000000000000000000000000000,
            0.5917219545342641210754451448115974588810,
            0.9083804012656871986193032824659812039432,
        ), (
            0.0111142584286220539859749860528626590297,
            0.1104509102493861466004927682745926578987,
            0.2844444444444444444444444444444444444444,
            0.3681777602499803214407987465610455350135,
            0.2258126266275670335282890546670547036134,
        ))
    elif lvls == (5, 5):
        return ((
            0.0011400916279895714252925662242950660621,
            0.0286947699546412286194721707269028350274,
            0.1449251859501533839940429183605870622213,
            0.3835443720333502926245446400412989976399,
            0.6899041564209057422808717657468081820338,
            0.9336096058311415992375941207182896751969,
        ), (
            0.0057848130996318758183086142045647930651,
            0.0611113173361872055053385561515139630791,
            0.1781303461739846294853146600663139778952,
            0.2897835883987064179045556839232370169164,
            0.2996502557119514020644949576862021485824,
            0.1655396792795384692219875279681681004616,
        ))
    elif lvls == (6, 6):
        return ((
            0.0006475011465280875329023496812149224648,
            0.0167015320044136710452174789998645300917,
            0.0882549960354370219273958904622738271861,
            0.2500000000000000000000000000000000000000,
            0.4941001474128341888340023025392352905335,
            0.7582327176038081109090822522806529371658,
            0.9497554134892866120590920337290661848656,
        ), (
            0.0032948801242805316690478002972990784805,
            0.0361475604598452845505114570818537187001,
            0.1134330879287148705817939680105127420805,
            0.2089795918367346938775510204081632653061,
            0.2683969625764040743685758074784623917978,
            0.2435578310294313833509563143419258637868,
            0.1261900860445891616015636323817829398480,
        ))
    elif lvls == (7, 7):
        return ((
            0.0003942238742465663591997931976400662433,
            0.0103361303518457912086946484522103896446,
            0.0562798735099516172277067966947154437186,
            0.1666947457690518108549004300247338885728,
            0.3501293882647016157943765723849178692396,
            0.5818122834262806030454458458839617927606,
            0.8070026077654725308002485849280408264818,
            0.9606840803717827980427606617671130566716,
        ), (
            0.0020098998513176012901859731514339510202,
            0.0226087595459031340530486629936407282737,
            0.0744218181314563843655993312751543624928,
            0.1480775066176912277936658208676328932182,
            0.2146062767606707551714846284095627189758,
            0.2392848277464309029723628707114469507674,
            0.1997722749074713364913073314326001561563,
            0.0992186364390586578623453811585282390951,
        ))
    elif lvls == (8, 8):
        return ((
            0.0002534425870529336255564239715688952929,
            0.0067214494411323074739461290888953350539,
            0.0373704122629985250896994644956543308859,
            0.1141583589453679815108514305530074629091,
            0.2499999999999999999999999999999999999999,
            0.4384117823491769105493894451963440714811,
            0.6507418449635889223984015038371285156716,
            0.8427525567677681017733759171586302115980,
            0.9684136820946790234611326268752417653424,
        ), (
            0.0012938785297983554465682517898820403914,
            0.0148103394363078619395127929354477153741,
            0.0503797700865841687284417170992412179471,
            0.1055337340098043362894070845468565601235,
            0.1651196775006298815822625346434870244394,
            0.2068133430301985037792233220375871054752,
            0.2102309263163512935903011523193916318246,
            0.1658378212585495421189592383074650941401,
            0.0799805098317760565253239063206416102842,
        ))
    elif lvls == (9, 9):
        return ((
            0.0001702173135062931683435511289333354696,
            0.0045519737523278637405119125579802049616,
            0.0256945562245544742357506472751484849546,
            0.0802601948484877821944637566740416998159,
            0.1811037227109888041387953905626295993688,
            0.3299780616926200150236213916923495839863,
            0.5136555889777349729937296998398258620160,
            0.7051041245235788804700780123900220607239,
            0.8696153404413123744726086009814732534892,
            0.9740767458306780132463075632133853888979,
        ), (
            0.0008698434107202897294977572823965540887,
            0.0100832309490842198560250033063175236495,
            0.0351184957693975604616957121809478237278,
            0.0762838816843755547956746188871090537466,
            0.1257641255536424976203062236993478791804,
            0.1697600991611103725535867709519904502405,
            0.1929828376256208002955523026823602991131,
            0.1839678667465844835338392220472153687308,
            0.1393681182014963732897513363513798087530,
            0.0658015008979678478640710526109352387691,
        ))
    elif lvls == (10, 10):
        return ((
            0.0001184978315303126344201247021113431249,
            0.0031887140927853569985595860502674526010,
            0.0182044850239269767635263724675162930707,
            0.0578171332359678588370823203382681548970,
            0.1333918002540150545562886625450369209046,
            0.2500000000000000000000000000000000000000,
            0.4029349562063600268878206479458984455842,
            0.5769132624427796747628079897968777093773,
            0.7483564905979763008569426244986697511203,
            0.8902513138608806560737173553541947192327,
            0.9783471559775873054383581258249687338963,
        ), (
            0.0006059897026026935588487257503473005348,
            0.0070913602237641987885041102406272872703,
            0.0251350199000182039859362730184009619947,
            0.0560718920185618657350700374045898227066,
            0.0959836890921681301790375441027120412006,
            0.1364625433889503153572417641681710945780,
            0.1668208554180785320016513257877971541720,
            0.1771218725734286141834536674385853167251,
            0.1611551910277160474401613684132549296965,
            0.1184890092411404258461901889833128129273,
            0.0550625774135709729239049946922012781937,
        ))
    elif lvls == (11, 11):
        return ((
            0.0000850025523458157338532034582117100584,
            0.0022983751314813111136438883808146692705,
            0.0132361948357330745512105106670466005033,
            0.0425766177135455917356872286229550776118,
            0.0999092534147219623437365412059716848446,
            0.1913041473961156560145183095307850087741,
            0.3165375559075845714869596789946381387575,
            0.4677407524129021560964280778496892461009,
            0.6298945720001630390323896475634893579809,
            0.7831388690300377615881043438798646764882,
            0.9064156315019561677921097544999108618081,
            0.9816456367990650664244022936074925330186,
        ), (
            0.0004349416410824727003077416012434377265,
            0.0051268179891616654214182212508173380456,
            0.0184167976586347361213951852961060978329,
            0.0419217746411992953795374103731940735288,
            0.0738033134092821998185902729084573391470,
            0.1089727560228136264214534669181685427235,
            0.1401742897905891585791089691247826681069,
            0.1596892231290726089422596260164207171123,
            0.1612456520818666263695270454366043029776,
            0.1416615308847114902132573442472529740390,
            0.1018125080061567655388364969431788861689,
            0.0467403947454293544943082198837736225904,
        ))
    else:
        assert (False)


def gauss_sqrtinv_quadrature_rule(N):
    """ Gauss rule of N pts that is exact on 0^1 with weight sqrt(x)^-1 """
    if N == 1:
        return ((0.3333333333333333333333333333333333333333333, ),
                (2.0000000000000000000000000000000000000000000, ))
    elif N == 2:
        return ((
            0.1155871099970479351674458383995416377412887,
            0.7415557471458092076896970187433155051158541,
        ), (
            1.3042903097250922852538721015560011855293026,
            0.6957096902749077147461278984439988144706973,
        ))
    elif N == 3:
        return ((
            0.0569391159670073532371751168037721197227028,
            0.4371978527510939418006878729474220341226577,
            0.8694993949182623413257733738851694825182756,
        ), (
            0.9358278691453820947797406879791019896233112,
            0.7215231460962772151396670276754322233230437,
            0.3426489847583406900805922843454657870536450,
        ))
    elif N == 4:
        return ((
            0.0221635688072176383248335645099583667103941,
            0.1878315676524455103763869130277351236639007,
            0.4615973614962667094116573193303410913570132,
            0.7483346283872804764262410270789069169018070,
            0.9484939262883686128293022286846374487353058,
        ), (
            0.5910484494295057403477859893026766588420934,
            0.5385334386199927101824538431389387057195198,
            0.4381727250319640879910698684563263849175437,
            0.2989026983011611862915526793153946648051132,
            0.1333426886173762751871376197866635857157296,
        ))
    elif N == 5:
        return ((
            0.0156834066074004550029559770508462950633585,
            0.1353000116552482368803292381113218618911971,
            0.3449423794274172615361537523728888711854356,
            0.5927501277315416722786297090938225539833138,
            0.8174280132668749578115072857614510621573700,
            0.9634612787028217643165109941314084861541073,
        ), (
            0.4982940916268055700011248720859024216609218,
            0.4669850730767096175216997978497561125188199,
            0.4063348534461318434981289116195967530130362,
            0.3201566570866924526693050590867181437440234,
            0.2138786519906368619205094363879924484291403,
            0.0943506727730236543892319229700341206340581,
        ))
    elif N == 6:
        return ((
            0.0116758719401466697457303327348505753968074,
            0.1018327040027696222521393024839321232654808,
            0.2654811572689373108541435577454016305052973,
            0.4723715370044845444841089829827442018582275,
            0.6842620156531486132199956656757587009037342,
            0.8619913332033932913005513711712565630592873,
            0.9727557512974903185137011575764265753815355,
        ), (
            0.4305277069263155803917528866325200705499951,
            0.4103969274425912079318481313224361114206781,
            0.3710767949558756274834331802503140724978452,
            0.3144063343163870691392038772476843132113360,
            0.2430371413758063693788296181449532519133386,
            0.1603161743195204196112665541257086191673955,
            0.0702389206635037260636657522763835612394112,
        ))
    elif N == 7:
        return ((
            0.0090273770256471513503257112500901210456601,
            0.0793005598114866532769237445412301454635046,
            0.2097793686155100679293253806734347952971680,
            0.3817710533971155500827427559058291457155431,
            0.5706358201621721774414928584623759219067175,
            0.7493173785474033214084246027997486835958985,
            0.8922219742137978534717933249936908308625073,
            0.9789142101623510960067135568574713238549360,
        ), (
            0.3789012209101369925707934464165662102938179,
            0.3652068300898471777335273359384398787671124,
            0.3383130387900050763786241580607199244232789,
            0.2991919776331534641630034610949570979409821,
            0.2492579425110677441049525643840328402897737,
            0.1903170233649855696198502152044924527105270,
            0.1245070478772957857256876739887553885499730,
            0.0543049188235081897035611449120362070245347,
        ))
    elif N == 8:
        return ((
            0.0071868028362263904124851648338526653746392,
            0.0634466706931120502323152178772448202476140,
            0.1695390189659999149454372426446861525481725,
            0.3133433833212179068011672695931429914269072,
            0.4784309655375749127847379616283021575795006,
            0.6459416610770250646329341345785738955742047,
            0.7967391631975201158231268451879049936244050,
            0.9135994225742659189405471324715498224732049,
            0.9832014832256291539986776026133139297227800,
        ), (
            0.3382847659262871836813129402699732206682116,
            0.3285529674916654459721075529318551808246779,
            0.3093693502525304898508360076727495443864367,
            0.2812858293413013024094626075038944561910048,
            0.2451104134229569203690382536004031104563277,
            0.2018840882125743311256279698496692141256022,
            0.1528514605097781130582593552332730512106358,
            0.0994290977899395929066698924052772833617324,
            0.0432320270529666206266854205329049387753704,
        ))
    elif N == 9:
        return ((
            0.0058563084367956140478585343942154330871501,
            0.0518863939803236903211178728730159228580699,
            0.1396562407429770351070584042479359149678954,
            0.2609850936822263779581172655468410453464028,
            0.4045642847657475115569649763562785181603358,
            0.5570113146004432724102288039915533655838566,
            0.7041172924000903788480461219971880577939760,
            0.8321716520870234720248666946536349326975555,
            0.9292418765798955148935718619425161810778733,
            0.9863044145193489277039643357916924232986789,
        ), (
            0.3055067742614517013961686639101951869838972,
            0.2983459729452074935756574740039388733853598,
            0.2841922186367641026585966501343298660690308,
            0.2633772768983532537969889994963262698322210,
            0.2363890639230368346247547554227645740100824,
            0.2038602396344808700735002709606997523333833,
            0.1665534831534094974495162864440924122003556,
            0.1253440966682181271390130703740832127032021,
            0.0812028596007738826620799045498642197581812,
            0.0352280142783042366237239247037056327242862,
        ))
    elif N == 10:
        return ((
            0.0048635662431629196410193075345674774084251,
            0.0432059569830094156374498468444364310942117,
            0.1169201056091435028728458005454538643309347,
            0.2202949026522509100628383350511495029238416,
            0.3453212438337658748004121634460592577008432,
            0.4823125587285287949681196309683526864795996,
            0.6206553197832813171991139348592761934989261,
            0.7496314197386707643749413570076232076214281,
            0.8592488575036643986850097988284484065556020,
            0.9410173694607198221818157077542589350687433,
            0.9886217227196162330648062101836298512709324,
        ), (
            0.2785037457112639867508204966836199157478404,
            0.2730829966920303427051476624630347931727353,
            0.2623470095741247414659299850606148917514837,
            0.2465047536210248485711219723096289439188899,
            0.2258645921610784367868012148435686382285266,
            0.2008282888857619298641576756610725647016221,
            0.1718832124341354548288873627454057323782888,
            0.1395929368490409761899228378604353147975501,
            0.1045866703053665718806241025464225122243001,
            0.0675498031696283095866044937318258026983386,
            0.0292559905965444013699821960943708903804238,
        ))
    elif N == 11:
        return ((
            0.0041032855232113356015324412716498979853270,
            0.0365264215043977140225543399094007543148270,
            0.0992518900301393904616341252708474496591514,
            0.1881768072585076817211584017554990583732400,
            0.2974845814519667041946925497185763356965944,
            0.4200253816810056370272298881782474093012027,
            0.5477838189598086429056903554854523709038822,
            0.6724032570011448843475648497498180170046296,
            0.7857324865144900191228158418631196776149138,
            0.8803591349359288595590093649361442816067642,
            0.9500957578259142332168296026272575238047060,
            0.9903976028453997914363095158297319045858247,
        ), (
            0.2558763906935043139481123304493907437034224,
            0.2516749126936565922427507650223673774528066,
            0.2433409458556067824089263069525248512140591,
            0.2310113361074512027066889678135671197245406,
            0.2148885402319312695651546848932124455892573,
            0.1952373042082277765397613289284943088558379,
            0.1723803230639065518343704059674853343701611,
            0.1466929628221606114680672305062330362386730,
            0.1185971698308735614927355170002171690824002,
            0.0885548776348396123372054964226764577186256,
            0.0570627772578673263626156319037565728983955,
            0.0246824595999743990936113341400745831518200,
        ))
    elif N == 12:
        ((
            0.0035082039676451715488882516347519421788139,
            0.0312790423380307537400715432077185219486451,
            0.0852668262832194510904011594311453451840534,
            0.1624507173081227701065810901493418595676249,
            0.2585119604912543482843810804350196090547496,
            0.3680755384069753353611667499405917763641926,
            0.4850109290560469747487644984888147997910415,
            0.6027751415261857682113788793684989069566439,
            0.7147788421775322651565132553162504736805091,
            0.8147551098876009860468642661616267856504858,
            0.8971102975594896586731135348170744329428376,
            0.9572380808594426184299014562403898376616097,
            0.9917883297462970358568761955930894345089885,
        ), (
            0.2366428305585245530327421714009373692997805,
            0.2333208869705931640893250150807244272639297,
            0.2267236330926393330988814368851962904919007,
            0.2169436810571531813131588534559951645788764,
            0.2041183221888508464768281405068615846544176,
            0.1884276007118282969273297661346063970983314,
            0.1700917886269704784208955301599633931678406,
            0.1493682995313194917741515922056966772774943,
            0.1265480926591496710789073798140901914444456,
            0.1019516505942956239966398014481467590517586,
            0.0759247665887255279006062824977009893815408,
            0.0488357021852638175792316550395768004809918,
            0.0211027452346860143113023753705039558086914,
        ))
    else:
        assert (False)


def gauss_x_quadrature_rule(N):
    """ Gauss rule of N pts that is exact on 0^1 with weight x """
    if N == 1:
        return ((0.666666666666666666666666666666666666666666666, ),
                (0.500000000000000000000000000000000000000000000, ))
    elif N == 2:
        return ((
            0.355051025721682190180271592529410860803405251,
            0.844948974278317809819728407470589139196594748,
        ), (
            0.181958618256022830605630997924836350223168125,
            0.318041381743977169394369002075163649776831874,
        ))
    elif N == 3:
        return ((
            0.212340538239152943974758110124000376651919941,
            0.590533135559265289135073747931170105948095104,
            0.911412040487296052604453856230543803114270668,
        ), (
            0.069826979901454122533881272179076904912611287,
            0.229241106359586246693920594556323246448747403,
            0.200931913738959630772198133264599848638641308,
        ))
    elif N == 4:
        return ((
            0.139759864343780552152087081124880233261621977,
            0.416409567631083179943302331337082751306567360,
            0.723156986361876172319954002314374858889110830,
            0.942895803885482317806878807445884378764922054,
        ), (
            0.031180970950008082173875147096575310634893758,
            0.129847547608232440826456202889629014613552971,
            0.203464568010271360791404475935854775105519777,
            0.135506913431488116208264174077940899646033492,
        ))
    elif N == 5:
        return ((
            0.098535085798826426123498897887752690380513569,
            0.304535726646363905485385176278832838605998445,
            0.562025189752613855994987479994771147483209725,
            0.801986582126391827464207863204703763961813641,
            0.960190142948531257659193309906666832295737345,
        ), (
            0.015747914521692276185292316568487981291490311,
            0.073908870072616670350633219341703578823544061,
            0.146386987084669808698037869355958885985990993,
            0.167174638094369565491675623097698818583935946,
            0.096781590226651679274360971636150735315038686,
        ))
    elif N == 6:
        return ((
            0.073054328680258885148126034180317988943827580,
            0.230766137969945499083116639884354548162857856,
            0.441328481228449867918606658194478059925631711,
            0.663015309718845700902947027919224294802431249,
            0.851921400331515708150023147504020787769751225,
            0.970683572840215108027949723086835089626269607,
        ), (
            0.008738301813609531756017303312398870744647057,
            0.043955165550508975508176624305430095127996335,
            0.098661150890655264120584510548359787948966543,
            0.140792553788198928119076839070928850827294952,
            0.135542497231518616840690396638050427707687230,
            0.072310330725508683655454326124831967643407880,
        ))
    elif N == 7:
        return ((
            0.056262560536922146465652191032311175779765514,
            0.180240691736892364987579942809181784542060620,
            0.352624717113169637373907770171241202808021883,
            0.547153626330555383001448557652348854640385927,
            0.734210177215410531523210608306610002563003118,
            0.885320946839095768090359762932485372922270175,
            0.977520613561287501891174500429154940077826092,
        ), (
            0.005214362202807404089691374303783018136426012,
            0.027408356721873475444331048358226286477086481,
            0.066384696465491469983647988763639913408018382,
            0.107125065695873669990912252999855732271542257,
            0.127390897299588326918734485774961641349506401,
            0.110509258190874602421120736369607774678343292,
            0.055967363423491051151562113429925633679077172,
        ))
    elif N == 8:
        return ((
            0.044633955289969850733121021858307761523577133,
            0.144366257042145571485218520228214969713521839,
            0.286824757144430518948686239749092658553116209,
            0.454813315196573350967727770046786944867019943,
            0.628067835416727697569146039517370755331899936,
            0.785691520604369241642458732418329677325570282,
            0.908676392100206043996258541925458646892959524,
            0.982220084852636548186794898962320938733511601,
        ), (
            0.003295191442248798854742316011089070116458878,
            0.017842902655986208393552797201042242584902711,
            0.045439319504698898083055421095694099606401172,
            0.079199599492319160120408189359892226719494007,
            0.106047359435930010704531096411158216173665091,
            0.112505799470887366171772098604281123729265659,
            0.091119023636373626345722461782879760351915841,
            0.044550804361555931326215619533963260717896637,
        ))
    elif N == 9:
        return ((
            0.036257812883209460941164300768081452075871122,
            0.118078978789998700192285111994735223692879414,
            0.237176984814960385317306692853274467076488817,
            0.381882765304705975360770248396495519719332595,
            0.538029598918989065116856891319435691892766932,
            0.690332420072362182940379532770515951183610915,
            0.823883343837004718136824253927431413588613952,
            0.925612610290803955364081814044001414919023086,
            0.985587590351123451367173259189186760588255268,
        ), (
            0.002180847085773130872549051120016965994673329,
            0.012060004284785379051727152286900267274304678,
            0.031804821491054001261690106935869983982543434,
            0.058401195295165110565731522901640148015179655,
            0.084358321844920349394250019958587490855231358,
            0.100308809193368289654150224732586789020286162,
            0.098533742172345705160173020369274425451178032,
            0.076074255109308162580079303753964290606605301,
            0.036278003523279871459649597941159638799998047,
        ))
    elif N == 10:
        return ((
            0.030029032161486497043064357634407940694654843,
            0.098289012209853229651201021590232018531882323,
            0.199021078963101154862053698382757475326997737,
            0.324055538323334892642849491065238528496004723,
            0.463261234284339367126904822288110977218801768,
            0.605360153114213157038047894922392497785060846,
            0.738840323991543759733948341945154276801364283,
            0.852888550356929759572400564420180497866228638,
            0.938267928122851874477370632805749858547127919,
            0.988082386567584403090254413041014023969972152,
        ), (
            0.001499140602406394028248136004106392416542194,
            0.008419319782983187417516365518701072015764350,
            0.022774591453263032586487649426060196300449466,
            0.043401906407150625922456117586383966709234207,
            0.066053075563350379166074502504877796364153435,
            0.084557109690827332430645582165500868356189574,
            0.092696893677723415844756991929710862268821806,
            0.086211300289176036428333360711914291766310487,
            0.064287154509082572874826538628865191685129349,
            0.030099508024037023300654755523879362117405129,
        ))
    elif N == 11:
        return ((
            0.025273620397520349753331186461629101915463182,
            0.083041613447405146706865372981972582247994890,
            0.169175100377181425969433456094343296056010695,
            0.277796715109032074436678692195389092862763554,
            0.401502720232860816772279286326956409725081247,
            0.531862386910415957916889619242246406714728232,
            0.659991842085334811766394766102982472268670660,
            0.777159392956162144492168546542637235588274749,
            0.875380774855556926264700412736090770752555869,
            0.947964548872819447416457304227035677552814074,
            0.989981719538319594156975270132195215185208059,
        ), (
            0.001063667293244541407533880961225323985669718,
            0.006043920960479775792732962176576883994369868,
            0.016663623451680692415747124573825603864024502,
            0.032641546713833346017953865389764253893956367,
            0.051591360672297580193086641744607792826060599,
            0.069531875158183049799050889633934099430924473,
            0.081879102988063470389586851943461204964999025,
            0.084659422884017621219239388551910431893547803,
            0.075620048057186998321377054841660461552338966,
            0.054938091132871498258746681665650567928069690,
            0.025367340688141426184944658517383375666038984,
        ))
    else:
        assert (False)


def gauss_log_quadrature_rule(N):
    """ Gauss rule of N pts that is exact on 0^1 with weight log(x). """
    if N == 0:
        return ((0.25, ), (-1, ))
    elif N == 1:
        return ((
            0.11200880616697618295720548894767738145955234705971531137512,
            0.60227690811873810275708022533803690425473336722599897433915,
        ), (
            -0.7185393190303844406655102008909857306799273887061971988212,
            -0.2814606809696155593344897991090142693200726112938028011787,
        ))
    elif N == 2:
        return ((
            0.06389079308732540499611660313628497820824549053476281670086,
            0.36899706371561876554619764585669599415161900192512528470316,
            0.76688030393894145542368265981691083598634879965757712270725,
        ), (
            -0.5134045522323633251293004975665356814264511793307779296494,
            -0.3919800412014875548062871809662384281240086247415834919777,
            -0.0946154065661491200644123214672258904495401959276385783728,
        ))
    elif N == 3:
        return ((
            0.04144848019938322080332131015635901100846145653441767301719,
            0.24527491432060225193967575952330997290477760077514434060034,
            0.55616545356027583718018435437603993902340903924569824176642,
            0.84898239453298517464784918808468292065364696392138377734817,
        ), (
            -0.3834640681451351248500465223430427353478161951448529235048,
            -0.3868753177747626273360082345543494063769672633879468570350,
            -0.1904351269501424153613600145473887189806609377187731526645,
            -0.0392254871299598324525852285552191392945556037484270667955,
        ))
    elif N == 4:
        return ((
            0.02913447215197205330372676211544169239610793130238244668465,
            0.17397721332089762870113971082856517443039224068336551195941,
            0.41170252028490204317493192464565864761615211948171312910085,
            0.67731417458282038070180266799800618028289928763756396739705,
            0.89477136103100828363888620445517246089508556524206477909042,
        ), (
            -0.2978934717828944572722578778879382024163841263092887522186,
            -0.3497762265132241803750718703073098856265391849650381263406,
            -0.2344882900440524188869068579425969864966379272951062288596,
            -0.0989304595166331469761807114403955438620722363956020897754,
            -0.0189115521431957964895826824217593815983665250349648028055,
        ))
    elif N == 5:
        return ((
            0.02163400584411694899569585585374682999178614215949874854694,
            0.12958339115495079613115850500929293027133804972276751947758,
            0.31402044991476550879824818841962747704248460906009992579556,
            0.53865721735180214454894189399258300864954587891207844903103,
            0.75691533737740285216454415613856233302346594376982905363470,
            0.92266885137212023733387323150705590049740577288421576263901,
        ), (
            -0.2387636625785475697222683033295032875495617101538807031819,
            -0.3082865732739467929693831092108497342650481014054125585896,
            -0.2453174265632103859849325401879030907965497691874423407489,
            -0.1420087565664766854213455760300400003905821953728040647318,
            -0.0554546223248862900151353549662062994528518621808570342153,
            -0.0101689586929322758869351162754975875454063616996032985322,
        ))
    elif N == 6:
        return ((
            0.01671935540825851594166736093204211562709208830436300353159,
            0.10018567791567512158688503175678285491190695419643070274907,
            0.24629424620793059904666854723908609172798651967271983166856,
            0.43346349325703310583288248260116183353617194998457670639946,
            0.63235098804776608846180581224488753785128488946055887762554,
            0.81111862674010557652622679678204641498923293428527113585652,
            0.94084816674334772176013411337898914887671058287940583308351,
        ), (
            -0.1961693894252482075254273775846386469078557954159792670738,
            -0.2703026442472729821452717195334951355508511167350128738368,
            -0.2396818730076909483080727852520423052525875236134995556375,
            -0.1657757748104329065608696877359693325728721415123324951136,
            -0.0889432271376579644357238403457822028277777451583164897404,
            -0.0331943043565710670254494111033878307344477324647256052745,
            -0.0059327870151259239991851784446845461536079451001337133230,
        ))
    elif N == 7:
        return ((
            0.01332024416089246501225267252425049701507305734389899718926,
            0.07975042901389493840982772914236310018217059433917385386661,
            0.19787102932618805379447615951575422699054476430887523158154,
            0.35415399435190941967146360353786139771232014216931695110020,
            0.52945857523491727770614969999613786188867911326324804040233,
            0.70181452993909996383715267030958480671056497295687922780820,
            0.84937932044110667604830920230122431585946513682081544424015,
            0.95332645005635978876737967851396008045334975559037044154674,
        ), (
            -0.1644166047280028868314725683258872853823186431767331478434,
            -0.2375256100233060205013485619604628094927810377731820983186,
            -0.2268419844319191263687804029359766042222704040273727925580,
            -0.1757540790060702449880562120059387141010772144963043598244,
            -0.1129240302467590518550004420863005494534828735844119704080,
            -0.0578722107177820723985279672939768585750711478174622876498,
            -0.0209790737421329780434615241149875618061479549594979570516,
            -0.0036864071040276190133523212764696169668507241650353863458,
        ))
    elif N == 15:
        return ((0.38978344871159159240536052676e-2,
                 0.230289456168732398203176309832e-1,
                 0.582803983062404123483532298386e-1,
                 0.108678365091054036487713613047e0,
                 0.172609454909843937760843776226e0,
                 0.247937054470578495147671753052e0,
                 0.332094549129917155984755859335e0,
                 0.422183910581948600115088366787e0,
                 0.515082473381462603476277704044e0,
                 0.607556120447728724086384921723e0,
                 0.696375653228214061156318166601e0,
                 0.778432565873265405203868167738e0,
                 0.850850269715391083233822761293e0,
                 0.911086857222271905418818994079e0,
                 0.957025571703542157591520509344e0,
                 0.987047800247984476758697436483e0),
                (-0.607917100435912328511733870890e-1,
                 -0.102915677517582144387691736211e0,
                 -0.122355662046009193557547513198e0,
                 -0.127569246937015988717042209204e0,
                 -0.123013574600070915423123358514e0,
                 -0.111847244855485722621848957720e0,
                 -0.965963851521243412529681907273e-1,
                 -0.793566643514731387824404347462e-1,
                 -0.618504945819652070951357191096e-1,
                 -0.454352465077266686288169623362e-1,
                 -0.310989747515818064092426891639e-1,
                 -0.194597659273608420781029120208e-1,
                 -0.107762549632055256455506226941e-1,
                 -0.497254289008764171249909926004e-2,
                 -0.167820111005119451503084228480e-2,
                 -0.282353764668436321778087564400e-3))
    elif N == 31:
        return ((0.1075611083792196962969780373553878690976e-2,
                 0.6250810087022611420940359811684815126704e-2,
                 0.1582844084981408575603401372785976836696e-1,
                 0.2976350301565074196920186506588590223925e-1,
                 0.4795207502179678192923031352370847973507e-1,
                 0.7024383508069178239463134943884032441943e-1,
                 0.9644673387230614455071755687709502101873e-1,
                 0.1263301772382888556709278232365368842018e0,
                 0.1596278853770582378128111070103633133645e0,
                 0.1960407756731627626960113483112120389235e0,
                 0.2352399914320458764156562386343717777778e0,
                 0.2768701174819714987658255988228550259588e0,
                 0.3205525883330845015709914502202628604843e0,
                 0.3658892770173910082133641815334431991424e0,
                 0.4124662429659580067469184415009931980453e0,
                 0.4598576113661137314622512303978238733299e0,
                 0.5076295526509711075738730933786303666873e0,
                 0.5553443282868045538962418695015974923909e0,
                 0.6025643674279367465110819474804192581146e0,
                 0.6488563380842230983052106899037648712549e0,
                 0.6937951760695034616434227737078402891711e0,
                 0.7369680350963850585966715689237591954311e0,
                 0.7779781219045459497510938206516371451763e0,
                 0.8164483812191657216014057207144758328438e0,
                 0.8520249965975079109067754071361479688612e0,
                 0.8843806747877038831677326523661267697008e0,
                 0.9132176830007792711779927289133526307307e0,
                 0.9382706102208146884698734198344224679876e0,
                 0.9593088243610789707145969510141673550912e0,
                 0.9761385919089608938952455774936787826763e0,
                 0.9886047831996289258777422250254816769313e0,
                 0.9965916301800251900735592491319261428168e0),
                (-0.2038902353154669360678095145948087844663e-1,
                 -0.3744167643666325870266865609601524037065e-1,
                 -0.4879551120357912426760113662188071256940e-1,
                 -0.5652398407509892722913139058302321848332e-1,
                 -0.6156760171990661038317854151263326068903e-1,
                 -0.6448943356574232921334360558511203208587e-1,
                 -0.6568343028833496966372219195799556302988e-1,
                 -0.6545403708436423656843745959206738557993e-1,
                 -0.6405254161395674276283699710478802598272e-1,
                 -0.6169550988558143005800175958400712459791e-1,
                 -0.5857473633393504772421084464043350574132e-1,
                 -0.5486278410751183662840545889662091191021e-1,
                 -0.5071608278942955352752500229780703275378e-1,
                 -0.4627661690561036129053563601495176467280e-1,
                 -0.4167278621118906992105934402426830566907e-1,
                 -0.3701978322627793155373837697682580785800e-1,
                 -0.3241970322894102028788082667921822295651e-1,
                 -0.2796152596501113562905552282563381102978e-1,
                 -0.2372106188106418474816880475992574515212e-1,
                 -0.1976092596118249418335306238886411545973e-1,
                 -0.1613058237539254893728891101001641037148e-1,
                 -0.1286648927967597408651098857782524749538e-1,
                 -0.9992363018151793691338425248060605016733e-2,
                 -0.7519573343586885468907038123040124561000e-2,
                 -0.5447675287399510074194534043451625066202e-2,
                 -0.3765078475839582375623345112611816893960e-2,
                 -0.2449850691860755055343519843183533769294e-2,
                 -0.1470649124225212622569588080214504896458e-2,
                 -0.7877699015661481354428014814900550126355e-3,
                 -0.3543041047381739788832616667250280035202e-3,
                 -0.1173864734710108664221842255272621582887e-3,
                 -0.1952190916544675885462589822731828488056e-4))
    else:
        assert (False)
