% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/onelump.R
\name{onelump}
\alias{onelump}
\title{One-lump Transient Heat Budget for Constant Environment}
\usage{
onelump(t, Tc_init, Ww_g, geom, Tair, Trad, vel, Qsol, Zen, ...)
}
\arguments{
\item{t}{= seq(1,3600,60), time intervals (s) at which output is required}

\item{Tc_init}{= 5, initial temperature (°C)}

\item{Ww_g}{= 500, animal weight (g)}

\item{geom}{= 2, organism shape, 0-5, Determines whether standard or custom shapes/surface area/volume relationships are used: 0=plate, 1=cyl, 2=ellips, 3=lizard (desert iguana), 4=frog (leopard frog), 5=custom (see parameter 'shape_coefs')}

\item{Tair}{= 30, air temperature (°C)}

\item{Trad}{= 30, radiant temperature (°C), averaging ground and sky}

\item{vel}{= 0.1, wind speed (m/s)}

\item{Qsol}{= 500, solar radiation (W/m2)}

\item{Zen}{= 20, zenith angle of sun (90 is below horizon), degrees}

\item{k_flesh}{= 0.5, thermal conductivity of flesh (W/mK, range: 0.412-2.8)}

\item{q}{= 0, metabolic heat production rate W/m3}

\item{c_body}{= 3073, specific heat of flesh J/(kg-K)}

\item{emis}{= 0.95, emissivity of animal (0-1)}

\item{rho_body}{= 932, animal density (kg/m3)}

\item{alpha}{= 0.85, solar absorptivity, decimal percent}

\item{shape_coefs}{= c(10.4713,.688,0.425,0.85,3.798,.683,0.694,.743), custom shape coefficients. Operates if geom=5, and consists of 4 pairs of values representing the parameters a and b of a relationship AREA=a*Ww_g^b, where AREA is in cm2 and Ww_g is in g. The first pair are a and b for total surface area, then a and b for ventral area, then for silhouette area normal to the sun, then silhouette area perpendicular to the sun}

\item{shape_b}{= 1/5, proportionality factor (-) for going from volume to area, represents ratio of width:height for a plate, length:diameter for cylinder, b axis:a axis for ellipsoid}

\item{shape_c}{= 1/5, proportionality factor (-) for going from volume to area, represents ratio of length:height for a plate, c axis:a axis for ellipsoid}

\item{posture}{= 'n' pointing normal 'n', parallel 'p' to the sun's rays, or 'a' in between?}

\item{orient}{= 1, does the object orient toward the sun? (0,1)}

\item{fatosk}{= 0.4, configuration factor to sky (-) for infrared calculations}

\item{fatosb}{= 0.4, configuration factor to substrate for infrared calculations}

\item{alpha_sub}{= 0.2, substrate solar reflectivity, decimal percent}

\item{pdif}{= 0.1, proportion of solar energy that is diffuse (rather than direct beam)}

\item{press}{= 101325, air pressure (Pa)}
}
\value{
Tc Core temperature (°C)

Tcf Final (steady state) temperature (°C), if conditions remained constant indefinitely

tau Time constant (s)

dTc Rate of change of core temperature (°C/s)
}
\description{
Transient, 'one-lump', heat budget for computing rate of change of temperature
under a constant environment
Michael Kearney, Raymond Huey and Warren Porter developed this R function and example in September 2017.
}
\examples{
library(NicheMapR)

# compare heating rates of a variety of different-sized objects

t=seq(1,3600*2,60) # times (in seconds) to report back values - every minute for 2 hrs
tmins <- t/60

par(mfrow = c(1,2))
Ww_g <- 5 # body weight, g
Tc_init <- 20 # initial body temperature, Â°C
geom <- 2 # shape (2 = ellipsoid)
Tair <- 20 # air temperature, Â°C
Trad <- Tair # radiant temperature, Â°C
vel <- 1 # wind speed, m/s
Qsol <- 500 # horizontal plane solar radiation, W/m2
Zen <- 20 # zenith angle of sun, degrees
alpha <- 0.85 # solar absorptivity, -

Tbs<-onelump(t=t, alpha = alpha, Tc_init = Tc_init, Ww_g = Ww_g,
  geom = geom, Tair = Tair, Trad = Trad, vel = vel, Qsol = Qsol, Zen = Zen)
plot(Tbs$Tc ~ tmins, type= 'l' ,col = 1, ylim = c(20, 30), ylab = 'Temperature, Â°C',xlab='time, min', las = 1)
text(80, 27, "    500 g")
text(80, 24.5, "5 g")
text(90, 20.5, "Tair for both sizes", col = "blue")
text(90, 26, "   vel = 1.0 m/s")
text(90, 23.5, "     vel = 1.0 m/s")

Ww_g <- 500 # body weight, g
Tbs<-onelump(t=t, alpha = alpha, Tc_init = Tc_init, Ww_g = Ww_g,
  geom = geom, Tair = Tair, Trad = Trad, vel = vel, Qsol = Qsol, Zen = Zen)
points(Tbs$Tc~tmins,type='l',lty = 2, col=1)
abline(Tair,0, lty = 1, col = 'blue')
abline(h = Tair + .1, lty = 2, col = 'blue')

Ww_g <- 5 # body weight, g
Tair <- 25 # air temperature, Â°C
vel <-0.5 # wind speed, m/s

Tbs<-onelump(t=t, alpha = alpha, Tc_init = Tc_init, Ww_g = Ww_g,
  geom = geom, Tair = Tair, Trad = Trad, vel = vel, Qsol = Qsol, Zen = Zen)
plot(Tbs$Tc~tmins,type='l',col=1,ylim=c(20,30),ylab='Temperature, Â°C',xlab='time, min', las = 1)
abline(h = Tair, lty = 1, col = 'blue')

Ww_g <- 500 # body weight, g
Tair <- 20 # air temperature, Â°C
vel <-1 # wind speed, m/s

Tbs<-onelump(t=t, alpha = alpha, Tc_init = Tc_init, Ww_g = Ww_g,
  geom = geom, Tair = Tair, Trad = Trad, vel = vel, Qsol = Qsol, Zen = Zen)
points(Tbs$Tc~tmins,type='l',lty = 2, col=1)
abline(h = Tair, lty = 2, col = 'blue')

text(65, 29.65, "5 g")
text(85, 27, "500 g")
text(90, 20.5, "Tair for large animal", col = "blue")
text(90, 25.4, "Tair for small animal", col = "blue")
text(80, 28.65, "vel = 0.5 m/s")
text(93, 26, "vel = 1.0 m/s")
}
