% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ellipsoid_endo.R
\name{ellipsoid}
\alias{ellipsoid}
\title{Ellipsoid endotherm model}
\usage{
ellipsoid(
  posture = 4.5,
  mass = 0.5,
  coreT = 37,
  furdepth = 5,
  furcond = 0.04,
  O2eff = 0.2,
  stress = 0.6,
  airT = 20,
  windspd = 1,
  rh = 50,
  Q10 = 3,
  basal = NA,
  basmult = 1
)
}
\arguments{
\item{posture}{= 4.5, Shape, ratio of long to short axis of a prolate ellipsoid}

\item{mass}{= 0.5, Body Mass (kg)}

\item{coreT}{= 37, Core temperature (deg C)}

\item{furdepth}{= 5, Fur depth (mm)}

\item{furcond}{= 0.04, Conductivity of fur (W/Cm)}

\item{O2eff}{= 0.2, Oxygen extraction efficiency (decimal \%)}

\item{stress}{= 0.6, Fraction of basal metabolic rate at which evaporative water loss is required (decimal \%)}

\item{airT}{= 20, Air temperature (deg C)}

\item{windspd}{= 1, Wind speed (m/s)}

\item{rh}{= 50, Relative humidity (\%)}

\item{basal}{= NA, user specified basal metabolic rate (W)}

\item{basmult}{= 1, multiplier to adjust mouse-elephant predicted basal metabolic rate}
}
\value{
\itemize{
\item 1 AirTemp - Air temperature for calculation (deg C)
\item 2 Windspeed - Wind speed for calculation (m/s)
\item 3 RelHum - Relative humidity for calculation (\%)
\item 4 Tcore - Core temperature for calculation (deg C)
\item 5 UCT - Upper Critical Temperature (deg C)
\item 6 LCT - Lower Critical Temperature (deg C)
\item 7 Qresp_gph - Respiration rate (W)
\item 8 Qresp_W - Respiration rate (W)
\item 9 Qresp_kjph - Respiration rate (kJ/h)
\item 10 Tskin - Skin temperature (deg C)
\item 11 Qgen - required heat generation (W)
\item 12 QgenFinal - Required heat generation capped at basal (W)
\item 13 mlO2ph - Oxygen consumption rate (ml/h)
\item 14 PctBasal - Metabolic rate (\% of basal)
\item 15 status - 1=cold, 2=happy, 3=hot
\item 16 H2Oloss_W - Watts of heat to be dumped
\item 17 H2O_gph - Water loss rate required to dump heat based on latent heat of vaporization
\item 18 massph_percent - Percent of body mass lost as water per hour
\item 19 timetodeath - Time to death (hours) from desiccation (15\% desiccated) if no water to drink
}
}
\description{
An implementation of the model described in
Porter, W. P., and M. Kearney. 2009. Size, shape, and the thermal niche of endotherms.
Proceedings of the National Academy of Sciences 106:19666-19672.
with additional rough calculations for water loss. Note this model is only relevant for
conditions where there is no solar radiation and air/ground/sky temperatures are equal.
Originally coded into R from Excel by John Baumgartner.
Requires the NicheMapR functions WETAIR.rh, DRYAIR and VAPPRS
Modified to allow use with rasters 2 Aug 2018
Modified to allow user-specified basal metabolic rates and Q10 effects 2 Aug 2018
}
\details{
Note that the parameter 'stress' is a fudge factor that accounts for physiological thermoregulatory
adjustments during heat stress (e.g. core temperature increases, changes in flesh conductivity, changes
in posture) that are not captured dynamically in this function (but some of these could be modelled
dynamically on the basis of this function).
}
\examples{

# compute metabolic and water loss rate as a function of air temperature
endo<-ellipsoid(airT=seq(5,45), windspd = 0.1)
endo<-as.data.frame(endo)
plot(endo$AirTemp,endo$QgenFinal,type='l', xlab = "Air Temperature (deg C)", ylab = "Metabolic Rate (W)", main="Metabolic Rate vs. Air Temperaure")
plot(endo$AirTemp,endo$H2O_gph,type='l', xlab = "Air Temperature (deg C)", ylab = "Water Loss Rate (g/h)", main="Water Loss Rate vs. Air Temperaure")

# compute thermoneutral zone as a function of body mass
masses<-10^seq(-3,2,0.5) # log sequence of masses
endo<-ellipsoid(mass = masses)
endo<-as.data.frame(endo)
ylims<-c(min(endo$UCT,endo$LCT),max(endo$UCT,endo$LCT))
plot(masses,endo$UCT, col='red',type='l', ylim=ylims, xlab="body mass (kg)",ylab="temperature (deg C)", main = "Upper and Lower Critical Temperatures vs Mass")
points(masses,endo$LCT, type='l', col='blue')

micro<-micro_global(loc = c(139.5, -25.9)) # run the microclimate model at Birdsville with default settings

metout<-as.data.frame(micro$metout) # above ground microclimatic conditions, min shade
shadmet<-as.data.frame(micro$shadmet) # above ground microclimatic conditions, max shade
soil<-as.data.frame(micro$soil) # soil temperatures, minimum shade
shadsoil<-as.data.frame(micro$shadsoil) # soil temperatures, maximum shade

# append dates
days<-rep(seq(1,12),24)
days<-days[order(days)]
dates<-days+metout$TIME/60/24-1 # dates for hourly output
dates2<-seq(1,12,1) # dates for daily output
metout<-cbind(dates,metout)
soil<-cbind(dates,soil)
shadmet<-cbind(dates,shadmet)
shadsoil<-cbind(dates,shadsoil)

endo<-cbind(metout[,1:3],ellipsoid(airT = shadmet$TALOC, windspd = shadmet$VLOC, rh = shadmet$RHLOC))

with(endo,{plot(H2O_gph ~ dates,xlab = "Date and Time", ylab = "Water Loss Rate (g/h)"
, type = "l",main=paste("Evaporative Water Loss",sep=""))})
with(endo,{plot(QgenFinal ~ dates,xlab = "Date and Time", ylab = "Metabolic Rate (W)"
, type = "l",main=paste("Metabolic Heat Generation",sep=""))})
with(endo,{plot(PctBasal ~ dates,xlab = "Date and Time", ylab = "Metabolic Rate (\% of basal)"
, type = "l",main=paste("Metabolic Heat Generation",sep=""))})
with(endo,{plot(Tskin ~ dates,xlab = "Date and Time", ylab = "Skin and Core Temperature (deg C)"
, type = "l",main=paste("Skin and Core Temperature",sep=""),ylim=c(min(cbind(endo$Tcore,endo$Tskin)),
 max(cbind(endo$Tcore,endo$Tskin))))})
with(endo,{points(Tcore ~ dates,xlab = "Date and Time",lty=2, type = "l")})
with(endo,{plot(timetodeath ~ dates,xlab = "Date and Time", ylab = "Time to Death (h)"
, type = "l",main=paste("Time to Death by Desiccation",sep=""), ylim=c(0,24))})
}
