#!/usr/bin/env Rscript

hlp = " Do significant species have a larger effect on metabolism? "

# Paths
in.data = "./data/data.Robj"
out.dir = file.path("./output/metabolomics/scfas/")
dir.create(out.dir, recursive = TRUE, showWarnings = FALSE)

# Load previous results; immunomodulatory species and metabolomics correlations
in.mbx = "./output/metabolomics/correlations/results.csv"
in.anova.train = "./output/response/trained/results.csv"
in.anova.spec = "./output/response/specific/results.csv"

# Thresholds
padj.msp = .25     # Adjusted p-value for immunomodulatory species
padj.mbx = .05     # Adjusted p-value for metabolomics correlations
min.prev = 0.2     # Minimum prevalence

# Load data object
load(in.data)
obj = .GlobalEnv$obj

# Scfas
SCFAS = c("mb_0002", "mb_0007")

# Colors
TRAINED = "#0dafdb"
SPECIFIC = "#db7b14"
THRESHOLDS = c("white", "#8CC1E2", "#1982C4", "#314A7A")
names(THRESHOLDS) = c("N/A", "P < 0.20", "P < 0.10", "P < 0.05")
IMMUNOMODULATORY = c("white", SPECIFIC, TRAINED)
names(IMMUNOMODULATORY) = c("N/A", "Specific", "Trained")

# Libs
require(ggplot2)
require(ggpubr)
require(pheatmap)
require(reshape2)

# Read trained and specific response significant species
ant = read.csv(in.anova.train, row.names = 1, stringsAsFactors = F)
ans = read.csv(in.anova.spec, row.names = 1, stringsAsFactors = F)
row.names(ant) = ant$Feature
row.names(ans) = ans$Feature
ant = ant[ant$Padj < padj.msp,]
ans = ans[ans$Padj < padj.msp,]
ant$Mode = "Trained"
ans$Mode = "Specific"
ann = rbind(ant, ans)
both = ann$Feature[duplicated(ann$Feature)]
ann[ann$Feature %in% both, "Mode"] = "Both"
ann = ann[!duplicated(ann$Feature),]
row.names(ann) = ann$Feature

# Load metabolites
mbx = read.csv(in.mbx, stringsAsFactors = F)
mbx = mbx[mbx$Column %in% ann$Feature,]
mbx = mbx[mbx$Row %in% SCFAS,]
mbx$padj = p.adjust(mbx$pvalue, method = "fdr")
data = merge(mbx, ann, by.x = "Column", by.y = "Feature")

# Correlation vs. bugs
agg = aggregate(data$spearman, by=list(MSP=data$label.x), max)
row.names(agg) = agg$MSP
data$Rank = agg[data$label.x, "x"]

# Plot correlations
fname = file.path(out.dir, "scfas_msps_correlation.pdf")
p = ggplot(data, aes(x=spearman, y=reorder(label.x, Rank))) + 
  geom_vline(xintercept = 0, linetype="dashed") +
  geom_point(aes(col=Mode), show.legend = F) + 
  facet_grid(rows=vars(Mode), cols=vars(Top.annotation.name), space = "free", scales = "free_y") + 
  xlab("Spearman corr.") + ylab("") + 
  theme(text = element_text(size=7)) + 
  xlim(-.2, .2)
ggsave(fname, width = 5, height = 5, plot = p) 
message(sprintf("Written %s", fname))





