#!/usr/bin/env Rscript

hlp = "Plot relative abundances at multiple levels. 
  Tailor each figure individually."

# Paths
in.data = "./data/data.Robj"
out.dir = file.path("./output/composition/")
dir.create(out.dir, recursive = TRUE, showWarnings = FALSE)

# Libs
require(reshape2)
require(ggplot2)

# Colors
COLORS = c("#52A675", "#1982C4", "#FF595E", "#6A4C93", "#FFCA3A", "#45355A", "#8AC926")
names(COLORS) = c("Firmicutes", "Bacteroidetes", "Actinobacteria", "Protebacteria",
                  "Verrucomicrobia",  "Euryarchaeota", "Lentisphaerae")

# Load data object
load(in.data)
obj = .GlobalEnv$obj
meta = obj$metadata

# Load phylum level
Sx = obj$assays$MSPCore$data
phylum = obj$assays$MSPCore$metadata$phylum

# Average by Phyla
agg = aggregate(t(Sx), by=list(Phylum=phylum), sum)
row.names(agg) = agg$Phylum
Px = t(agg[,2:ncol(agg)])
colnames(Px) = row.names(agg)

# Relative abundance
Px = sweep(Px, 1, rowSums(Px), "/")
Px[is.na(Px)] = 0

# Plot phylum level
tax = "Phylum"
mtp = data.frame(t(Px), check.names = F)
mtp$taxonomy = row.names(mtp)
mlt = melt(mtp, id.vars = "taxonomy")
colnames(mlt) = c("Taxonomy", "Sample", "Abundance")
mlt$Sample = as.character(mlt$Sample)
agg = aggregate(mlt$Abundance, by=list(Taxonomy=mlt$Taxonomy), sum)
row.names(agg) = agg$Taxonomy
agg$fraction =agg$x / sum(agg$x)
mlt[tax] = sprintf("%s (%.2f %%)", 
                    mlt$Taxonomy, 
                    100 * agg[mlt$Taxonomy, "fraction"])

# Compute abundance of top phylum per sample
top = agg$Taxonomy[agg$x == max(agg$x)]
mltop = mlt[mlt$Taxonomy == top,]
row.names(mltop) = mltop$Sample
mlt$Top = mltop[mlt$Sample, "Abundance"]

# Add cluster
mlt$Cluster = meta[mlt$Sample, "Cluster"]
mlt = mlt[!is.na(mlt$Cluster),]
mlt$Cluster = sprintf("C%d", mlt$Cluster)

# Plot relative abundance - sorted by top phylum
fname = file.path(out.dir, sprintf("abundance-%s.pdf", tax))
qplot(data=mlt, x=reorder(Sample, -Top),  
        fill=reorder(Phylum, -Abundance), y=Abundance, geom="col",
      position = "fill") + 
  xlab("Sample") + ylab("Relative abundance") + 
  theme(axis.text.x=element_blank(),
        axis.ticks.x=element_blank()) + 
  theme(panel.grid.major = element_blank(), 
        panel.grid.minor = element_blank(),
        panel.background = element_blank()) + 
  guides(fill=guide_legend(title="Phylum")) + 
  theme(text = element_text(size=7)) + 
  facet_grid(~Cluster, space="free", scales = "free") +
  scale_fill_manual(values = as.vector(COLORS))
ggsave(fname, width = 4.5, height = 3)
message(sprintf("Written %s", fname))

